/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationcontext', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/class',
    'io.ox/office/editframework/utils/operationerror'
], function (Utils, Class, OperationError) {

    'use strict';

    // private static functions ===============================================

    /**
     * Returns whether the passed array contains non-negative integers only.
     *
     * @param {Array<Any>} position
     *  A JSON array extracted from a document operation.
     *
     * @returns {Boolean}
     *  Whether the passed array contains non-negative integers only.
     */
    function isValidPos(position) {
        return (position.length > 0) && position.every(function (element) {
            return (typeof element === 'number') && (element >= 0) && (element === Math.floor(element));
        });
    }

    // class OperationContext =================================================

    /**
     * A small wrapper for a JSON document operation object providing useful
     * helper methods, used as calling context for operation handler callback
     * functions.
     *
     * @constructor
     *
     * @property {EditModel} docModel
     *  The document model targeted by the document operation.
     *
     * @property {Object} operation
     *  The JSON operation object wrapped by this instance.
     *
     * @property {Boolean} external
     *  Whether the operation has been received from the server.
     *
     * @property {Boolean} importing
     *  Whether the operation is part of the initial document import process.
     */
    var OperationContext = Class.extendable(function (docModel, operation, external, importing) {

        this.docModel = docModel;
        this.operation = operation;
        this.external = !!external;
        this.importing = !!importing;

    }); // class OperationContext

    // public methods ---------------------------------------------------------

    /**
     * Throws an OperationError exception. Can be used as convenience shortcut
     * to prevent importing the OperationError code module.
     *
     * @param {String} message
     *  The message text inserted into the OperationError exception. Can be a
     *  format string that will be passed with all following parameters to the
     *  method _.printf().
     *
     * @param {Any} [...]
     *  Additional values to be inserted into the format string passed in the
     *  parameter 'message'.
     *
     * @throws {OperationError}
     *  This method always throws an OperationError exception.
     */
    OperationContext.prototype.error = function () {
        throw new OperationError(_.printf.apply(_, arguments), this.operation);
    };

    /**
     * Throws an OperationError exception, if the passed condition is falsy.
     *
     * @param {Any} condition
     *  If falsy, an OperationError exception will be thrown with the passed
     *  message text.
     *
     * @param {String} message
     *  The message text inserted into the OperationError exception thrown in
     *  case the condition is falsy. Can be a format string that will be passed
     *  with all following parameters to the method _.printf().
     *
     * @param {Any} [...]
     *  Additional values to be inserted into the format string passed in the
     *  parameter 'message'.
     *
     * @throws {OperationError}
     *  If the passed condition is falsy. Will contain the passed message text.
     */
    OperationContext.prototype.ensure = function (condition) {
        if (!condition) { this.error.apply(this, _.toArray(arguments).slice(1)); }
    };

    /**
     * Returns whether the wrapped JSON operation contains a property with the
     * specified name.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Boolean}
     *  Whether the operation contains a property with the specified name.
     */
    OperationContext.prototype.has = function (propName) {
        return propName in this.operation;
    };

    /**
     * Returns the value of a required property of the wrapped JSON operation,
     * regardless of its data type.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Any}
     *  The value of the specified operation property, if existing.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a property with the specified
     *  name.
     */
    OperationContext.prototype.get = function (propName) {
        var value = this.operation[propName];
        this.ensure(!_.isUndefined(value), 'missing property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional property of the wrapped JSON operation,
     * regardless of its data type.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Any} [defValue]
     *  The default value that will be returned if the operation does not
     *  contain the specified property. If omitted, the undefined value will be
     *  returned in this case.
     *
     * @returns {Any}
     *  The value of the specified operation property, if existing.
     */
    OperationContext.prototype.getOpt = function (propName, defValue) {
        var value = this.operation[propName];
        return _.isUndefined(value) ? defValue : value;
    };

    /**
     * Returns the value of a required Boolean property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Boolean}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid Boolean property with
     *  the specified name.
     */
    OperationContext.prototype.getBool = function (propName) {
        var value = this.operation[propName];
        this.ensure(_.isBoolean(value), 'missing boolean property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional Boolean property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Boolean} [defValue=false]
     *  The default value that will be returned if the operation does not
     *  contain the specified property.
     *
     * @returns {Boolean}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not Boolean.
     */
    OperationContext.prototype.getOptBool = function (propName, defValue) {
        return this.has(propName) ? this.getBool(propName) : (_.isBoolean(defValue) && defValue);
    };

    /**
     * Returns the value of a required floating-point number property of the
     * wrapped JSON operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Number}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid floating-point number
     *  property with the specified name.
     */
    OperationContext.prototype.getNum = function (propName) {
        var value = this.operation[propName];
        this.ensure(Utils.isFiniteNumber(value), 'missing number property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional floating-point number property of the
     * wrapped JSON operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Number} [defValue=0]
     *  The default value that will be returned if the operation does not
     *  contain the specified property.
     *
     * @returns {Number}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a number.
     */
    OperationContext.prototype.getOptNum = function (propName, defValue) {
        return this.has(propName) ? this.getNum(propName) : (typeof defValue === 'number') ? defValue : 0;
    };

    /**
     * Returns the value of a required integer property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Number}
     *  The value of the specified operation property. If the property is a
     *  floating-point number, it will be truncated to an integer.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid number property with
     *  the specified name.
     */
    OperationContext.prototype.getInt = function (propName) {
        return Math.floor(this.getNum(propName));
    };

    /**
     * Returns the value of an optional integer property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Number} [defValue=0]
     *  The default value that will be returned if the operation does not
     *  contain the specified property.
     *
     * @returns {Number}
     *  The value of the specified operation property. If the property is a
     *  floating-point number, it will be truncated to an integer.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a number.
     */
    OperationContext.prototype.getOptInt = function (propName, defValue) {
        return Math.floor(this.getOptNum(propName, defValue));
    };

    /**
     * Returns the value of a required string property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Boolean} [allowEmpty=false]
     *  Whether the empty string is allowed. By default, the property must be a
     *  string with at least one character.
     *
     * @returns {String}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid string property with
     *  the specified name.
     */
    OperationContext.prototype.getStr = function (propName, allowEmpty) {
        var value = this.operation[propName];
        this.ensure(_.isString(value), 'missing string property \'%s\'', propName);
        this.ensure(allowEmpty || value, 'empty string property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional string property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {String} [defValue='']
     *  The default value that will be returned if the operation does not
     *  contain the specified property.
     *
     * @param {Boolean} [allowEmpty=false]
     *  Whether the empty string is allowed. By default, the existing property
     *  must be a string with at least one character.
     *
     * @returns {String}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid string.
     */
    OperationContext.prototype.getOptStr = function (propName, defValue, allowEmpty) {
        if (!_.isString(defValue)) { allowEmpty = defValue; defValue = ''; }
        return this.has(propName) ? this.getStr(propName, allowEmpty) : defValue;
    };

    /**
     * Returns the value of a required enumeration property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Function} EnumClass
     *  The expected enumeration class (subclass of Enum).
     *
     * @returns {Enum}
     *  The enumeration value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid string property with
     *  the specified name, or the string value cannot be converted to a value
     *  of the passed enumeration class.
     */
    OperationContext.prototype.getEnum = function (propName, EnumClass) {
        var strValue = this.getStr(propName);
        var enumValue = EnumClass.parse(strValue);
        this.ensure(enumValue, 'invalid enumeration value \'%s\' in property \'%s\'', strValue, propName);
        return enumValue;
    };

    /**
     * Returns the value of an optional enumeration property of the wrapped
     * JSON operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Function} EnumClass
     *  The expected enumeration class (subclass of Enum).
     *
     * @param {Enum|Null} [defValue=null]
     *  The default value that will be returned if the operation does not
     *  contain the specified property.
     *
     * @returns {Enum}
     *  The enumeration value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid string, or the string value
     *  cannot be converted to a value of the passed enumeration class.
     */
    OperationContext.prototype.getOptEnum = function (propName, EnumClass, defValue) {
        return this.has(propName) ? this.getEnum(propName, EnumClass) : (defValue || null);
    };

    /**
     * Returns the value of a required object property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Object}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid object property with
     *  the specified name.
     */
    OperationContext.prototype.getObj = function (propName) {
        var value = this.operation[propName];
        this.ensure(_.isObject(value) && !_.isArray(value), 'missing object property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional object property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Object|Null} [defValue=null]
     *  The default value that will be returned if the operation does not
     *  contain the specified property. If omitted, null will be returned
     *  instead (NOT an empty object!).
     *
     * @returns {Object|Null}
     *  The value of the specified operation property; or null, if the property
     *  is missing, and no default value has been passed.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not an object.
     */
    OperationContext.prototype.getOptObj = function (propName, defValue) {
        return this.has(propName) ? this.getObj(propName) : _.isObject(defValue) ? defValue : null;
    };

    /**
     * Returns the value of a required array property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Array<Any>}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid array property with
     *  the specified name.
     */
    OperationContext.prototype.getArr = function (propName) {
        var value = this.operation[propName];
        this.ensure(_.isArray(value), 'missing array property \'%s\'', propName);
        return value;
    };

    /**
     * Returns the value of an optional array property of the wrapped JSON
     * operation.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @param {Array<Any>|Null} [defValue=null]
     *  The default value that will be returned if the operation does not
     *  contain the specified property. If omitted, null will be returned
     *  instead (NOT an empty array!).
     *
     * @returns {Array<Any>|Null}
     *  The value of the specified operation property; or null, if the property
     *  is missing, and no default value has been passed.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not an array.
     */
    OperationContext.prototype.getOptArr = function (propName, defValue) {
        return this.has(propName) ? this.getArr(propName) : _.isArray(defValue) ? defValue : null;
    };

    /**
     * Returns the value of a required document position property of the
     * wrapped JSON operation. A document position is expected to be an array
     * with non-negative integers.
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Array<Number>}
     *  The value of the specified operation property.
     *
     * @throws {OperationError}
     *  If the wrapped operation does not contain a valid document position
     *  property with the specified name.
     */
    OperationContext.prototype.getPos = function (propName) {
        var position = this.getArr(propName);
        this.ensure(isValidPos(position), 'missing position property \'%s\'', propName);
        return position;
    };

    /**
     * Returns the value of an optional document position property of the
     * wrapped JSON operation. A document position is expected to be an array
     * with non-negative integers
     *
     * @param {String} propName
     *  The name of the operation property.
     *
     * @returns {Array<Number>|Null}
     *  The value of the specified document position property; or null, if the
     *  property is missing.
     *
     * @throws {OperationError}
     *  If the wrapped operation contains a property with the specified name,
     *  but the property value is not a valid document position.
     */
    OperationContext.prototype.getOptPos = function (propName) {
        return this.has(propName) ? this.getPos(propName) : null;
    };

    // exports ================================================================

    return OperationContext;

});
