/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/attributepool', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/triggerobject'
], function (Utils, ValueMap, TriggerObject) {

    'use strict';

    // private global functions ===============================================

    /**
     * Invokes a callback function for each passed attribute family.
     *
     * @param {String|Array<String>|Object} attrFamilies
     *  A single attribute family as string (e.g. 'character'), or an array of
     *  attribute families (e.g.: ['character', 'paragraph']), or a set object
     *  with attribute families as property keys (e.g.: {character:true,
     *  paragraph:true}).
     *
     * @param {Function} callback
     *  The callback function invoked for each attribute family.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    function forEachFamily(attrFamilies, callback, context) {
        if (_.isString(attrFamilies)) {
            callback.call(context, attrFamilies);
        } else if (_.isArray(attrFamilies)) {
            attrFamilies.forEach(callback, context);
        } else if (_.isObject(attrFamilies)) {
            _.forEach(attrFamilies, function (flag, attrFamily) {
                callback.call(context, attrFamily);
            });
        }
    }

    // class Definition =======================================================

    /**
     * The definition properties of a single formatting attribute.
     *
     * @property {String} name
     *  The name of the attribute.
     *
     * @property {Any} def
     *  The default value of the attribute.
     *
     * @property {String} scope
     *  The scope of the attribute. MUST be one of 'element', 'style', or 'any'.
     *
     * @property {Function|Null} merge
     *  A merge callback function used to merge two values of this attribute.
     *  If omitted, the new value will replace the old value completely.
     *
     * @property {Array<String>|Boolean} undo
     *  The names of all attributes that need to be inserted into the undo
     *  operation that will be generated when changing the value of this
     *  attribute. The boolean value true is a simple replacement for all
     *  attributes that have been registered for the attribute family of this
     *  attribute.
     */
    function Definition(name) {

        this.name = name;
        this.def = undefined;
        this.scope = 'any';
        this.merge = null;
        this.undo = null;

    } // class Definition

    // private methods ------------------------------------------------------

    Definition.prototype._extendWith = function (definition) {
        _.extend(this, definition);
        if (definition.undo === '*') {
            this.undo = true;
        } else if (typeof definition.undo === 'string') {
            this.undo = definition.undo.split(/\s+/);
        }
        return this;
    };

    // public methods ---------------------------------------------------------

    Definition.prototype.isStyleScope = function () {
        return (this.scope === 'style') || (this.scope === 'any');
    };

    Definition.prototype.isElementScope = function () {
        return (this.scope === 'element') || (this.scope === 'any');
    };

    // class DefinitionMap ====================================================

    /**
     * Represents the definitions of all formatting attributes of a single
     * attribute family.
     *
     * @constructor
     *
     * @extends ValueMap
     */
    var DefinitionMap = ValueMap.extend(function () {
        ValueMap.call(this);
    });

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the specified attribute is supported by this instance,
     * according to the passed options.
     *
     * @param {String} attrName
     *  The attribute name to be checked.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.special=false]
     *      If set to true, the check will include 'special' attributes too
     *      (attributes marked with the 'special' flag in the attribute
     *      definitions denoting an attribute for internal use only, not
     *      for usage in document operations). Otherwise, special
     *      attributes will not be recognized by this method.
     *
     * @returns {Boolean}
     *  Whether the specified attribute is supported by this instance.
     */
    DefinitionMap.prototype.isSupported = function (attrName, options) {
        var definition = this.get(attrName, null);
        var special = !!options && (options.special === true);
        return !!definition && (special || !definition.special);
    };

    /**
     * Invokes the callback function for all attributes that are supported by
     * this instance, according to the passed options.
     *
     * @param {Object|Null} attrValues
     *  A map of attribute values to be visited. If null or undefined, nothing
     *  will be done.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for all attribute values in
     *  the passed ap that are suported by this instance. Receives the
     *  following parameters:
     *  (1) {Any} attrValue
     *      The value of the current attribute.
     *  (2) {String} attrName
     *      The name of the current attribute.
     *  (3) {Object} definition
     *      The definition descriptor of the current attribute.
     *  Will be called in the context of this instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.special=false]
     *      If set to true, all 'special' attributes will be visited too
     *      (attributes marked with the 'special' flag in the attribute
     *      definitions denoting an attribute for internal use only, not for
     *      usage in document operations). Otherwise, special attributes will
     *      not be visited by this method.
     *
     * @returns {DefinitionMap}
     *  A reference to this instance.
     */
    DefinitionMap.prototype.forEachSupported = function (attrValues, callback, options) {
        _.forEach(attrValues, function (attrValue, attrName) {
            if (this.isSupported(attrName, options)) {
                callback.call(this, attrValue, attrName, this.get(attrName));
            }
        }, this);
        return this;
    };

    // class AttributePool ====================================================

    /**
     * Represents the static definitions of all formatting attributes of
     * various attribute families.
     *
     * Triggers the following events:
     * - 'change:defaults':
     *      After the default values of formatting attributes have been changed
     *      via the method AttributePool.changeDefaultValues(). Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newValueSet
     *          A complete value set with the current default values of all
     *          registered attribute families (map of map of values).
     *      (3) {Object} oldValueSet
     *          A complete value set with all previous default values of all
     *          registered attribute families (map of map of values).
     *      (4) {Object} changedValueSet
     *          An incomplete value set with all default values that have
     *          really changed (map of map of values).
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {EditModel} docModel
     *  The document model containing this attribute pool.
     */
    var AttributePool = TriggerObject.extend({ constructor: function (docModel) {

        // a map of maps with all registered attributes, mapped by family, then by name
        var attributeRegistry = new ValueMap();

        // the default values of all registered attributes, mapped by family
        var defaultValueMap = new ValueMap();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Registers new formatting attributes for a specific attribute family,
         * or updates the settings of formatting attributes that have been
         * registered already.
         *
         * @param {String} attrFamily
         *  The name of the attribute family the new formatting attributes are
         *  associated with. MUST NOT be the string 'styleId' which is reserved
         *  to be used as style sheet identifier in attribute sets.
         *
         * @param {Object} definitions
         *  The definitions of all formatting attributes to be registered. MUST
         *  be an object with the names of the formatting attribute to be added
         *  or updated as object keys, and the settings for the attributes as
         *  values. Each value MUST be an object with the following properties:
         *  - {Any} [definition.def]
         *      The default value of the attribute which will be used if
         *      neither a style sheet nor an explicit value exists for the
         *      attribute. MUST be specified when defining a new attribute, may
         *      be omitted when updating the definition of an attribute already
         *      registered. MUST be a JSON value (including arrays and objects
         *      with JSON values) except the special value null (!) that can be
         *      added to document operations.
         *  - {String} [definition.scope='any']
         *      Specifies where the formatting attribute can be used. If set to
         *      'element', the attribute can be set explicitly only (not in
         *      style sheets). If set to 'style', the formatting  attribute can
         *      be contained in style sheets only. By default, the formatting
         *      attribute can be set explicitly, as well as in style sheets.
         *  - {Function} [definition.merge]
         *      A callback function that will be called while merging attribute
         *      sets, for example while collecting attributes from style sheets
         *      and explicit attribute sets. Will be called to merge two
         *      existing values of this formatting attribute, where the second
         *      value has to overwrite the first value in some way. Will be
         *      called in the context of this instance. The function receives
         *      the 'old' attribute value in the first parameter, and the 'new'
         *      attribute value in the second parameter. By default, the new
         *      value wins and the old value will be overwritten completely.
         *  - {String} [definition.undo]
         *      The names of all attributes of the specified attribute family
         *      that need to be inserted into the undo operation that will be
         *      generated when changing the value of this attribute. Can be set
         *      to a single asterisk which will cause to restore all explicit
         *      attributes of the attribute family.
         *  Additionally, the attribute definition may contain any other
         *  user-defined properties which will be stored internally.
         *
         * @returns {AttributePool}
         *  A reference to this instance.
         */
        this.registerAttributes = function (attrFamily, definitions) {

            // existing or new map for the specified attribute family
            var definitionMap = attributeRegistry.getOrConstruct(attrFamily, DefinitionMap);
            // get or create the submap for default attribute values
            var defaultValues = defaultValueMap.getOrConstruct(attrFamily, Object);

            // process all passed definitions
            _.forEach(definitions, function (definition, attrName) {
                // create or update the definition of the attribute
                definition = definitionMap.getOrConstruct(attrName, Definition, attrName)._extendWith(definition);
                // store the default attribute value in the attribute map
                defaultValues[attrName] = definition.def;
            });

            return this;
        };

        /**
         * Returns the definitions for the specified attribute family.
         *
         * @param {String} attrFamily
         *  The name of the attribute family.
         *
         * @returns {DefinitionMap|Null}
         *  The definitions for the specified attribute family; or null, if the
         *  passed attribute family has not been registered.
         */
        this.getRegisteredAttributes = function (attrFamily) {
            return attributeRegistry.get(attrFamily, null);
        };

        /**
         * Returns whether the specified attribute is supported by this pool,
         * according to the passed options
         *
         * @param {String} attrFamily
         *  The family of the attribute to be checked.
         *
         * @param {String} attrName
         *  The attribute name to be checked.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, the check will include 'special' attributes too
         *      (attributes marked with the 'special' flag in the attribute
         *      definitions denoting an attribute for internal use only, not
         *      for usage in document operations). Otherwise, special
         *      attributes will not be recognized by this method.
         *
         * @returns {Boolean}
         *  Whether the specified attribute is supported by this pool.
         */
        this.isSupportedAttribute = function (attrFamily, attrName, options) {
            var definitionMap = attributeRegistry.get(attrFamily, null);
            return !!definitionMap && definitionMap.isSupported(attrName, options);
        };

        /**
         * Returns the default value of a single attribute.
         *
         * @param {String} attrFamily
         *  The family of the attribute whose default value will be returned.
         *
         * @param {String} attrName
         *  The name of the attribute whose default value will be returned.
         *
         * @returns {Any|Null}
         *  A deep copy of the default value of the specified attribute; or
         *  null, if the attribute does not exist.
         */
        this.getDefaultValue = function (attrFamily, attrName) {
            var defaultValues = defaultValueMap.get(attrFamily, null);
            return (defaultValues && (attrName in defaultValues)) ? _.copy(defaultValues[attrName], true) : null;
        };

        /**
         * Returns the default attribute values of a single attribute family.
         *
         * @param {String} attrFamily
         *  The attribute family whose default attributes will be returned.
         *
         * @returns {Object|Null}
         *  A simple map with the default attribute values of the specified
         *  attribute family; or null, if the passed attribute family has not
         *  been registered.
         */
        this.getDefaultValues = function (attrFamily) {
            var defaultValues = defaultValueMap.get(attrFamily, null);
            return defaultValues ? _.copy(defaultValues, true) : null;
        };

        /**
         * Returns an attribute set with the default values of all specified
         * attribute families.
         *
         * @param {String|Array<String>|Object} attrFamilies
         *  A single attribute family as string (e.g. 'character'), or an array
         *  of attribute families (e.g.: ['character', 'paragraph']), or a set
         *  object with attribute families as property keys (e.g.:
         *  {character:true, paragraph:true}).
         *
         * @returns {Object}
         *  The complete attribute set containing all default values of the
         *  specified attribute families.
         */
        this.getDefaultValueSet = function (attrFamilies) {

            // the resulting attribute set
            var attributeSet = {};

            // process the input parameter according to its type
            forEachFamily(attrFamilies, function (attrFamily) {
                var attrValues = this.getDefaultValues(attrFamily);
                if (attrValues) { attributeSet[attrFamily] = attrValues; }
            }, this);

            return attributeSet;
        };

        /**
         * Changes the default values for specific registered formatting
         * attributes. These values override the defaults of the attribute
         * definitions passed in the method AttributePool.registerAttributes(),
         * and will be used before the values of any style sheet attributes and
         * explicit element attributes will be resolved.
         *
         * @param {Object} attributeSet
         *  An incomplete set of attribute values to be changed.
         *
         * @returns {AttributePool}
         *  A reference to this instance.
         */
        this.changeDefaultValues = function (valueSet) {

            // copy of the old attribute defaults for change listeners
            var oldValueMap = defaultValueMap.clone(function (attrValues) { return _.clone(attrValues); });
            // all changed attributes
            var changedValueMap = new ValueMap();

            // update defaults of all attribute families
            _.forEach(valueSet, function (attrValues, attrFamily) {

                // only process the attribute families that have been registered
                var definitionMap = attributeRegistry.get(attrFamily, null);
                var defaultValues = defaultValueMap.get(attrFamily, null);
                if (!definitionMap || !defaultValues || !_.isObject(attrValues)) { return; }

                // add all changed default values to the maps
                var changedValues = changedValueMap.insert(attrFamily, {});
                definitionMap.forEachSupported(attrValues, function (attrValue, attrName) {
                    if ((attrValue !== null) && !_.isEqual(attrValue, defaultValues[attrName])) {
                        defaultValues[attrName] = _.copy(attrValue, true);
                        changedValues[attrName] = _.copy(attrValue, true);
                    }
                });

                // remove empty entry from the map of changed attributes (nothing changed)
                if (_.isEmpty(changedValues)) {
                    changedValueMap.remove(attrFamily);
                }
            }, this);

            // notify all change listeners for document attributes
            if (!changedValueMap.empty()) {
                this.trigger('change:defaults', _.copy(defaultValueMap.toObject(), true), oldValueMap.toObject(), changedValueMap.toObject());
            }

            return this;
        };

        /**
         * Builds a map with all supported formatting attributes of the
         * specified attribute family, set to the null value.
         *
         * @param {String} attrFamily
         *  The attribute family whose null attributes will be returned.
         *
         * @returns {Object}
         *  A simple map with the attributes of the specified attribute family,
         *  set to the null value; or null, if the passed attribute family has
         *  not been registered. Attributes that have been registered with the
         *  'special' flag will NOT be included into the result.
         */
        this.buildNullValues = function (attrFamily) {

            // the attribute definitions of the current family
            var definitionMap = attributeRegistry.get(attrFamily, null);
            if (!definitionMap) { return null; }

            // create the attribute map filled with null values
            var nullValues = {};
            definitionMap.forEach(function (definition, attrName) {
                if (!definition.special) { nullValues[attrName] = null; }
            });
            return nullValues;
        };

        /**
         * Builds an attribute set containing all formatting attributes of the
         * specified attribute families, set to the null value.
         *
         * @param {String|Array<String>|Object} attrFamilies
         *  A single attribute family as string (e.g. 'character'), or an array
         *  of attribute families (e.g.: ['character', 'paragraph']), or a set
         *  object with attribute families as property keys (e.g.:
         *  {character:true, paragraph:true}).
         *
         * @returns {Object}
         *  The complete attribute set with null values for all attributes of
         *  the specified attribute families. Attributes that have been
         *  registered with the 'special' flag will NOT be included into the
         *  result.
         */
        this.buildNullValueSet = function (attrFamilies) {

            // the resulting attribute set
            var attributeSet = {};

            // process the input parameter according to its type
            forEachFamily(attrFamilies, function (attrFamily) {
                var attrValues = this.buildNullValues(attrFamily);
                if (attrValues) { attributeSet[attrFamily] = attrValues; }
            }, this);

            return attributeSet;
        };

        /**
         * Extends the passed attribute value map with another value map. If
         * the definition of a specific attribute contains a merger callback
         * function, and both value maps contain a value, uses that merger
         * function to merge the values from both value maps, otherwise the
         * value of the second map will be copied to the first map.
         *
         * @param {String} attrFamily
         *  The attribute family the passed value maps are related to.
         *
         * @param {Object} attrValues1
         *  (in/out) The first value map that will be extended in-place.
         *
         * @param {Object|Null} attrValues2
         *  The second value map that will be inserted into the first map. If
         *  set to null, the first value map will bot be changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clone=false]
         *      If set to true, the first value map will be cloned deeply,
         *      instead of being extended in-place. This method will return the
         *      new cloned value map, the value map passed in the parameter
         *      'attrValues1' will not be modified.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result.
         *  @param {Boolean} [options.autoClear=false]
         *      If set to true, attributes with the value null in the second
         *      value map will be removed from the first value map. By default,
         *      attributes with the value null will be inserted into the first
         *      value map.
         *
         * @returns {Object}
         *  A reference to the first passed and extended value map.
         */
        this.extendAttributes = function (attrFamily, attrValues1, attrValues2, options) {

            // whether to delete null attributes from first attribute set
            var autoClear = Utils.getBooleanOption(options, 'autoClear', false);

            // clone the first attribute set if specified
            if (Utils.getBooleanOption(options, 'clone', false)) {
                attrValues1 = _.copy(attrValues1, true);
            }

            // the attribute definitions of the current family; restrict to valid attribute families
            var definitionMap = attributeRegistry.get(attrFamily, null);
            if (!definitionMap || !_.isObject(attrValues2)) { return attrValues1; }

            // copy attribute values of second attribute set
            definitionMap.forEachSupported(attrValues2, function (attrValue2, attrName, definition) {

                // try to find merger function from attribute definition
                var mergeFunc = definition.merge;

                // either set return value from merger, or copy the attribute directly
                if (autoClear && (attrValue2 === null)) {
                    delete attrValues1[attrName];
                } else if (mergeFunc && (attrName in attrValues1) && (attrValue2 !== null) && (attrValues1[attrName] !== null)) {
                    attrValues1[attrName] = mergeFunc(attrValues1[attrName], attrValue2);
                } else {
                    attrValues1[attrName] = attrValue2;
                }
            }, options);

            return attrValues1;
        };

        /**
         * Extends the passed attribute set with the existing values of the
         * second attribute set. If the definition of a specific attribute
         * contains a merger callback function, and both attribute sets contain
         * an attribute value, uses that merger function to merge the values
         * from both attribute sets, otherwise the value of the second
         * attribute set will be copied to the first attribute set.
         *
         * @param {Object} attributeSet1
         *  (in/out) The first attribute set that will be extended in-place.
         *
         * @param {Object|Null} attributeSet2
         *  The second attribute set, whose attribute values will be inserted
         *  into the first attribute set. If set to null, the first attribute
         *  set will bot be changed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clone=false]
         *      If set to true, the first attribute set will be cloned deeply,
         *      instead of being extended in-place. This method still returns
         *      the new attribute set, but the attribute set passed in the
         *      parameter 'attributeSet1' will not be modified.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *  @param {Boolean} [options.autoClear=false]
         *      If set to true, formatting attributes with the value null in
         *      the second attribute set will be removed from the first
         *      attribute set. By default, attributes with the value null will
         *      be inserted into the first attribute set.
         *
         * @returns {Object}
         *  A reference to the first passed and extended attribute set.
         */
        this.extendAttributeSet = function (attributeSet1, attributeSet2, options) {

            // clone the first attribute set deeply if specified, prevent further cloning
            if (Utils.getBooleanOption(options, 'clone', false)) {
                attributeSet1 = _.copy(attributeSet1, true);
                options = _.extend({}, options, { clone: false });
            }

            // add attributes existing in attributeSet2 to attributeSet1
            _.forEach(attributeSet2, function (attrValues2, attrFamily) {

                // copy style identifier directly (may be string or null)
                if (attrFamily === 'styleId') {
                    attributeSet1.styleId = attrValues2;
                    return;
                }

                // copy attribute values of second attribute set
                var attrValues1 = attributeSet1[attrFamily] || (attributeSet1[attrFamily] = {});
                this.extendAttributes(attrFamily, attrValues1, attrValues2, options);

                // delete empty value map from the attribute set
                if (_.isEmpty(attrValues1)) {
                    delete attributeSet1[attrFamily];
                }
            }, this);

            return attributeSet1;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = attributeRegistry = defaultValueMap = null;
        });

    } }); // class AttributePool

    // exports ================================================================

    return AttributePool;

});
