/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author York Richter <york.richtert@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/drawinglabels', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/container/valuemap',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, IO, ValueMap, gt) {

    'use strict';

    // maps drawing object types as defined/used in operations to GUI names and icons
    var DRAWING_TYPE_INFOS = {
        chart:     { icon: 'fa-bar-chart-o', label: /*#. bar charts, line charts, pie charts, etc. */ gt.pgettext('drawing', 'Chart') },
        image:     { icon: 'fa-picture-o',   label: /*#. bitmaps, vector graphics, etc. */ gt.pgettext('drawing', 'Image') },
        shape:     { icon: 'fa-picture-o',   label: /*#. rectangles, circles, stars, etc. */ gt.pgettext('drawing', 'Shape') },
        connector: { icon: 'fa-picture-o',   label: /*#. lines, arrows, straight/bent/curved connectors, etc. */ gt.pgettext('drawing', 'Line') },
        textframe: { icon: 'fa-list-alt',    label: /*#. rectangles, circles, etc. with arbitrary text contents */ gt.pgettext('drawing', 'Text frame') },
        diagram:   { icon: 'fa-sitemap',     label: /*#. complex diagrams, organigrams, etc. */ gt.pgettext('drawing', 'Diagram') }
    };

    // fall-back settings for unknown/unsupported drawing objects
    var DRAWING_TYPE_DEFAULT_INFO = { icon: 'fa-picture-o', label: gt.pgettext('drawing', 'Drawing') };

    // chart type descriptors for all supported chart type
    var CHART_TYPE_STYLES = {
        'column standard':         { cat: 1, series: { type: 'column',  cjs: 'column',            markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: vertical bars (clustered) */                                                                     gt.pgettext('chart-type', 'Column') },
        'column stacked':          { cat: 1, series: { type: 'column',  cjs: 'stackedColumn',     markersOnly: false }, stacking: 'stacked',                    title: /*#. Chart type: vertical bars (stacked from bottom to top) */                                                    gt.pgettext('chart-type', 'Column (stacked)') },
        'column percentStacked':   { cat: 1, series: { type: 'column',  cjs: 'stackedColumn100',  markersOnly: false }, stacking: 'percentStacked',             title: /*#. Chart type: vertical bars (stacked from bottom to top with percent scaling) */                               gt.pgettext('chart-type', 'Column (percent)') },
        'bar standard':            { cat: 1, series: { type: 'bar',     cjs: 'bar',               markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: horizontal bars (clustered) */                                                                   gt.pgettext('chart-type', 'Bar') },
        'bar stacked':             { cat: 1, series: { type: 'bar',     cjs: 'stackedBar',        markersOnly: false }, stacking: 'stacked',                    title: /*#. Chart type: horizontal bars (stacked from left to right) */                                                  gt.pgettext('chart-type', 'Bar (stacked)') },
        'bar percentStacked':      { cat: 1, series: { type: 'bar',     cjs: 'stackedBar100',     markersOnly: false }, stacking: 'percentStacked',             title: /*#. Chart type: horizontal bars (stacked from left to right with percent scaling) */                             gt.pgettext('chart-type', 'Bar (percent)') },
        'line standard':           { cat: 2, series: { type: 'line',    cjs: 'line',              markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: data points connected with lines */                                                              gt.pgettext('chart-type', 'Line') },
        'line standard curved':    { cat: 2, series: { type: 'line',    cjs: 'spline',            markersOnly: false }, stacking: 'standard', curved: true,     title: /*#. Chart type: data points connected with curved lines */                                                       gt.pgettext('chart-type', 'Line (curved)') },
        'line standard marker':    { cat: 2, series: { type: 'line',    cjs: 'spline',            markersOnly: true  }, stacking: 'standard', curved: false,    title: /*#. Chart type: data point markers without connected lines */                                                    gt.pgettext('chart-type', 'Line (marker)') },
        'scatter standard':        { cat: 2, series: { type: 'scatter', cjs: 'line',              markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: data points with free X/Y coordinates connected with lines */                                    gt.pgettext('chart-type', 'Scatter') },
        'scatter standard curved': { cat: 2, series: { type: 'scatter', cjs: 'spline',            markersOnly: false }, stacking: 'standard', curved: true,     title: /*#. Chart type: data points with free X/Y coordinates connected with curved lines */                             gt.pgettext('chart-type', 'Scatter (curved)') },
        'scatter standard marker': { cat: 2, series: { type: 'scatter', cjs: 'spline',            markersOnly: true  }, stacking: 'standard', curved: false,    title: /*#. Chart type: data points with free X/Y markers without connected lines */                                     gt.pgettext('chart-type', 'Scatter (marker)') },
        'bubble standard':         { cat: 3, series: { type: 'bubble',  cjs: 'bubble',            markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: data points with free X/Y coordinates drawn as circles/bubbles */                                gt.pgettext('chart-type', 'Bubble') },
        'pie standard':            { cat: 3, series: { type: 'pie',     cjs: 'pie',               markersOnly: false }, stacking: 'standard', varyColors: true, title: /*#. Chart type: single pie */                                                                                    gt.pgettext('chart-type', 'Pie') },
        'donut standard':          { cat: 3, series: { type: 'donut',   cjs: 'doughnut',          markersOnly: false }, stacking: 'standard', varyColors: true, title: /*#. Chart type: one or multiple concentric circles */                                                            gt.pgettext('chart-type', 'Donut') },
        'area standard':           { cat: 3, series: { type: 'area',    cjs: 'area',              markersOnly: false }, stacking: 'standard',                   title: /*#. Chart type: filled areas between X axis and data points (areas overlay each other) */                        gt.pgettext('chart-type', 'Area') },
        'area stacked':            { cat: 3, series: { type: 'area',    cjs: 'stackedArea',       markersOnly: false }, stacking: 'stacked',                    title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top) */                      gt.pgettext('chart-type', 'Area (stacked)') },
        'area percentStacked':     { cat: 3, series: { type: 'area',    cjs: 'stackedArea100',    markersOnly: false }, stacking: 'percentStacked',             title: /*#. Chart type: filled areas between X axis and data points (stacked from bottom to top with percent scaling) */ gt.pgettext('chart-type', 'Area (percent)') }
    };

    // set missing Boolean properties explicitly to false
    _.each(CHART_TYPE_STYLES, function (CHART_TYPE) {
        CHART_TYPE.curved = CHART_TYPE.curved || false;
        CHART_TYPE.varyColors = CHART_TYPE.varyColors || false;
    });

    // geometries of predefined drawing shapes
    var PRESET_GEOMETRIES = null;

    // description labels for all custom shape types, grouped by shape categories
    var SHAPE_LABELS_MAP = {

        line: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Lines'),
            shapes: {
                line: {
                    type: 'connector',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Line')
                },
                straightConnector1: {
                    type: 'connector',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Line arrow'),
                    lineAttrs: { tailEndType: 'arrow' }
                },
                straightConnector1_2: {
                    type: 'connector',
                    linkedId: 'straightConnector1',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Line arrow: double'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' }
                },

                bentConnector3: {
                    type: 'connector',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: elbow')
                },
                bentConnector3_1: {
                    type: 'connector',
                    linkedId: 'bentConnector3',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: elbow arrow'),
                    lineAttrs: { tailEndType: 'arrow' }
                },
                bentConnector3_2: {
                    type: 'connector',
                    linkedId: 'bentConnector3',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: elbow double-arrow'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' }
                },

                curvedConnector3: {
                    type: 'connector',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: curved')
                },
                curvedConnector3_1: {
                    type: 'connector',
                    linkedId: 'curvedConnector3',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: curved arrow'),
                    lineAttrs: { tailEndType: 'arrow' }
                },
                curvedConnector3_2: {
                    type: 'connector',
                    linkedId: 'curvedConnector3',
                    tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Connector: curved double-arrow'),
                    lineAttrs: { headEndType: 'arrow', tailEndType: 'arrow' }
                }
            }
        },

        shape: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Basic shapes'),
            shapes: {
                ellipse: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ellipse'),
                triangle: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Isosceles triangle'),
                rtTriangle: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Right triangle'),
                parallelogram: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Parallelogram'),
                trapezoid: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Trapezoid'),
                diamond: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Diamond'),
                pentagon: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Pentagon'),
                hexagon: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Hexagon'),
                heptagon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Heptagon'), iconLabel: '7' },
                octagon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Octagon'), iconLabel: '8' },
                decagon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Decagon'), iconLabel: '10' },
                dodecagon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Dodecagon'), iconLabel: '12' },
                pie: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Circle pie'),
                chord: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Circle segment'),
                teardrop: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Teardrop'),
                frame: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Frame'), outerText: true },
                halfFrame: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Half frame'), outerText: true },
                corner: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Corner'), outerText: true },
                diagStripe: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Diagonal stripe'), outerText: true },
                plus: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Plus'),
                plaque: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Plaque'),
                can: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Cylinder'),
                cube: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Cube'),
                funnel: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Funnel'), // no excel
                bevel: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle bevel'),
                donut: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ring'), outerText: true },
                noSmoking: /*#. predefined drawing shape type */ gt.pgettext('shape', 'No symbol'),
                blockArc: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Block arc'),
                foldedCorner: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Folded corner'),
                smileyFace: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Smiley face'),
                heart: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Heart'),
                lightningBolt: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Lightning bolt'),
                sun: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Sun'),
                moon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Moon'), aspectRatio: 0.5 },
                cloud: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Cloud'),
                arc: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arc'), noFill: true },
                bracketPair: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Double bracket'), noFill: true },
                bracePair: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Double brace'), noFill: true },
                leftBracket: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Left bracket'), aspectRatio: 0.1, noFill: true },
                rightBracket: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Right bracket'), aspectRatio: 0.1, noFill: true },
                leftBrace: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Left brace'), aspectRatio: 0.2, noFill: true },
                rightBrace: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Right brace'), aspectRatio: 0.2, noFill: true },
                // no excel
                squareTabs: { tooltip: /*#. predefined drawing shape type (4 small squares, one per corner, like photo corners) */ gt.pgettext('shape', 'Square tabs'), noFill: true },
                cornerTabs: { tooltip: /*#. predefined drawing shape type (4 triangles, one per corner, like photo corners with straight borders inside) */ gt.pgettext('shape', 'Corner tabs'), noFill: true },
                plaqueTabs: { tooltip: /*#. predefined drawing shape type (4 quarters of a circle, one per corner, like photo corners with rounded borders inside) */ gt.pgettext('shape', 'Plaque tabs'), noFill: true },
                pieWedge: /*#. predefined drawing shape type (one quarter of a circle) */ gt.pgettext('shape', 'Quarter pie'),
                chartPlus: { tooltip: /*#. predefined drawing shape type ("plus" sign) */ gt.pgettext('shape', 'Plus sign'), noFill: true },
                chartX: { tooltip: /*#. predefined drawing shape type (X letter, "times" sign) */ gt.pgettext('shape', 'Times sign'), noFill: true }
            }
        },

        rect: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Rectangles'),
            shapes: {
                rect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle'),
                roundRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: rounded corners'),
                snip1Rect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: single corner snipped'),
                snip2SameRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners snipped'),
                snip2DiagRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: diagonal corners snipped'),
                snipRoundRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners one rounded one snipped'),
                round1Rect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: single corner rounded'),
                round2SameRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: top corners rounded'),
                round2DiagRect: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Rectangle: diagonal corners rounded')
            }
        },

        arrow: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Block arrows'),
            shapes: {
                rightArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: right'),
                leftArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: left'),
                upArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: up'),
                downArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: down'),
                leftRightArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: left-right'), aspectRatio: 2 },
                upDownArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: up-down'), aspectRatio: 0.5 },
                quadArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: quad'),
                leftRightUpArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: left-right-up'), aspectRatio: 1.4 },
                bentArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: right-bent'), outerText: true },
                uturnArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: u-turn'), outerText: true },
                leftUpArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: left-up'),
                bentUpArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: up-bent'),
                curvedRightArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: curved right'), outerText: true },
                curvedLeftArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: curved left'), outerText: true },
                curvedUpArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: curved up'), outerText: true },
                curvedDownArrow: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: curved down'), outerText: true },
                leftCircularArrow:  { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: circular right'), outerText: true },
                circularArrow:  { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: circular left'), outerText: true },
                stripedRightArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: striped right'),
                notchedRightArrow: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: notched right'),
                homePlate: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: pentagon'),
                chevron: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: chevron'), outerText: true },
                swooshArrow:  { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Arrow: swoosh'), outerText: true }
            }
        },

        flowchart: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Flow charts'),
            shapes: {
                flowChartProcess: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: process'),
                flowChartAlternateProcess: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: alternate process'),
                flowChartDecision: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: decision'),
                flowChartInputOutput: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: data'),
                flowChartPredefinedProcess: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: predefined process'),
                flowChartInternalStorage: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: internal storage'),
                flowChartDocument: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: document'),
                flowChartMultidocument: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: multidocument'),
                flowChartTerminator: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: terminator'),
                flowChartPreparation: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: preparation'),
                flowChartManualInput: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: manual input'),
                flowChartManualOperation: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: manual operation'),
                flowChartConnector: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: connector'),
                flowChartOffpageConnector: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: off-page connector'),
                flowChartPunchedCard: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: Card'),
                flowChartPunchedTape: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: punched tape'),
                flowChartSummingJunction: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: summing junction'),
                flowChartOr: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: or'),
                flowChartCollate: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: collate'),
                flowChartSort: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: sort'),
                flowChartExtract: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: extract'),
                flowChartMerge: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: merge'),
                flowChartOnlineStorage: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: stored data'),
                flowChartDelay: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: delay'),
                flowChartMagneticTape: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic tape'),
                flowChartMagneticDisk: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic disk'),
                flowChartMagneticDrum: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: magnetic drum'),
                flowChartDisplay: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: display'),
                flowChartOfflineStorage: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Flowchart: offline storage')
            }
        },

        starbanner: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Stars and banners'),
            shapes: {
                irregularSeal1: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Explosion: 12 points'),
                irregularSeal2: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Explosion: 14 points'),
                star4: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 4 points'),
                star5: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 5 points'),
                star6: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 6 points'),
                star7: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 7 points'), iconLabel: '7' },
                star8: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 8 points'), iconLabel: '8' },
                star10: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 10 points'), iconLabel: '10' },
                star12: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 12 points'), iconLabel: '12' },
                star16: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 16 points'), iconLabel: '16' },
                star24: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 24 points'), iconLabel: '24' },
                star32: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Star: 32 points'), iconLabel: '32' },
                ribbon2: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ribbon: tilted up'),
                ribbon: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ribbon: tilted down'),
                ellipseRibbon2: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ribbon: curved and tilted up'),
                ellipseRibbon: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ribbon: curved and tilted down'),
                leftRightRibbon: { tooltip: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Ribbon: left-right arrow'), aspectRatio: 2 },
                verticalScroll: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Scroll: vertical'),
                horizontalScroll: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Scroll: horizontal'),
                wave: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Wave'),
                doubleWave: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Double wave')
            }
        },

        math: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Equation'),
            shapes: {
                mathPlus: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Plus sign'),
                mathMinus: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Minus sign'),
                mathMultiply: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Multiplication sign'),
                mathDivide: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Division sign'),
                mathEqual: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Equal'),
                mathNotEqual: /*#. predefined drawing shape type */ gt.pgettext('shape', 'Not equal')
            }
        },

        callouts: {
            title: /*#. category name for shape objects */ gt.pgettext('shape', 'Callouts'),
            shapes: {}
        }
    };

    // settings for predefined shape types, mapped by shape identifier
    var PRESET_SHAPE_MAP = new ValueMap();

    // the identifiers of preset two-point shapes (to be used if drawing type is 'shape' instead of 'connector')
    var TWO_POINT_SHAPES = Utils.makeSet([
        'line',
        'straightConnector1',
        'bentConnector2',
        'bentConnector3',
        'bentConnector4',
        'bentConnector5',
        'curvedConnector2',
        'curvedConnector3',
        'curvedConnector4',
        'curvedConnector5'
    ]);

    // the identifiers of preset two-point shapes (to be used if drawing type is 'shape' instead of 'connector')
    var NO_TEXT_SHAPES = Utils.makeSet([
        'chartPlus',
        'chartX'
    ]);

    // static class DrawingLabels =============================================

    /**
     * Provides generic caption texts for control groups, menu headers, and
     * other GUI elements in a single map for convenience.
     */
    var DrawingLabels = {};

    // drawing types ----------------------------------------------------------

    /**
     * Returns an appropriate CSS icon class for the passed drawing type.
     *
     * @param {String|Null} [type]
     *  The type identifier of a drawing object. If omitted or set to null, the
     *  default icon for a generic drawing object will be returned.
     *
     * @returns {String}
     *  A CSS icon class for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeIcon = function (type) {
        return ((type && DRAWING_TYPE_INFOS[type]) || DRAWING_TYPE_DEFAULT_INFO).icon;
    };

    /**
     * Returns an appropriate text label for the passed drawing type.
     *
     * @param {String|Null} [type]
     *  The type identifier of a drawing object. If omitted or set to null, the
     *  default label for a generic drawing object will be returned.
     *
     * @returns {String}
     *  A text label for the passed drawing type.
     */
    DrawingLabels.getDrawingTypeLabel = function (type) {
        return ((type && DRAWING_TYPE_INFOS[type]) || DRAWING_TYPE_DEFAULT_INFO).label;
    };

    // constant labels and tooltips -------------------------------------------

    /**
     * Standard options for the 'Delete drawing' label.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_LABEL = /*#. delete a drawing object from the document */ gt.pgettext('drawing', 'Delete');

    /**
     * Standard options for the 'Delete drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Delete the drawing object');

    /**
     * Standard options for the 'Insert drawing' label.
     *
     * @constant
     */
    DrawingLabels.INSERT_DRAWING_LABEL = DrawingLabels.getDrawingTypeLabel('image');

    /**
     * Standard options for the 'Insert drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.INSERT_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Insert an image');

    /**
     * Standard options for the 'Group drawing' label.
     *
     * @constant
     */
    DrawingLabels.GROUP_DRAWING_LABEL = /*#. group the selected drawings */ gt.pgettext('drawing', 'Group');

    /**
     * Standard options for the 'Group drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.GROUP_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Group the selected drawings');

    /**
     * Standard options for the 'Ungroup drawing' label.
     *
     * @constant
     */
    DrawingLabels.UNGROUP_DRAWING_LABEL = /*#. ungroup the selected drawing(s) */ gt.pgettext('drawing', 'Ungroup');

    /**
     * Standard options for the 'Ungroup drawing' tooltip.
     *
     * @constant
     */
    DrawingLabels.UNGROUP_DRAWING_TOOLTIP = gt.pgettext('drawing', 'Ungroup the selected drawings');

    /**
     * Standard options for the 'Crop' label.
     *
     * @constant
     */
    DrawingLabels.CROP_IMAGE_LABEL = /*#. Crop image to remove unwanted areas */ gt.pgettext('drawing', 'Crop');

    /**
     * Standard options for the 'Crop image' tooltip.
     *
     * @constant
     */
    DrawingLabels.CROP_IMAGE_TOOLTIP = /*#. Crop image to remove unwanted areas */ gt.pgettext('drawing', 'Crop image');

// constants for controls -------------------------------------------------

    /**
     * Standard options for the 'Delete drawing' button.
     *
     * @constant
     */
    DrawingLabels.DELETE_DRAWING_BUTTON_OPTIONS = {
        icon: 'fa-trash-o',
        tooltip: DrawingLabels.DELETE_DRAWING_TOOLTIP
    };

    /**
     * Standard options for insert image.
     *
     * @constant
     */
    DrawingLabels.INSERT_IMAGE_OPTIONS = {
        label: DrawingLabels.INSERT_DRAWING_LABEL,
        tooltip: DrawingLabels.INSERT_DRAWING_TOOLTIP
    };

    /**
     * Standard options for insert comment.
     *
     * @constant
     */
    DrawingLabels.INSERT_COMMENT_OPTIONS = {
        label: gt('Comment'),
        tooltip: /*#. insert a comment into the text */ gt('Insert a comment')
    };

    /**
     * Standard options for a 'Insert chart' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_CHART_OPTIONS = {
        label: DrawingLabels.getDrawingTypeLabel('chart'),
        tooltip: gt.pgettext('drawing', 'Insert a chart')
    };

    /**
     * Standard options for the 'Insert chart' button.
     *
     * @constant
     */
    DrawingLabels.INSERT_CHART_BUTTON_OPTIONS = {
        width: null,
        icon: DrawingLabels.getDrawingTypeIcon('chart'),
        label: DrawingLabels.INSERT_CHART_OPTIONS.label,
        tooltip: DrawingLabels.INSERT_CHART_OPTIONS.tooltip,
        updateCaptionMode: 'none',
        value: 'column standard'
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_DATA_POINTS_BUTTON_OPTIONS = {
        //#. menu title: options to format data points (bars, lines, pie segments) in chart objects
        label: gt.pgettext('chart-format', 'Data points')
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_SHOW_POINT_LABELS_BUTTON_OPTIONS = {
        //#. check box label: show/hide small text labels next to the single data points in charts objects
        label: gt.pgettext('chart-format', 'Show data point labels'),
        attributes: { class: 'highlight' }
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_VARY_POINT_COLORS_BUTTON_OPTIONS = {
        //#. check box label: if active, a bar/pie chart will have different colors for each data point (default is the same color for all points)
        label: gt.pgettext('chart-format', 'Vary colors by point'),
        attributes: { class: 'highlight' }
    };

    /**
     * @constant
     */
    DrawingLabels.CHART_LABELS_BUTTON_OPTIONS = {
        //#. button label to open a menu to modify chart objects
        label: gt.pgettext('chart-format', 'Labels'),
        //#. tooltip: menu to modify chart objects
        tooltip: gt.pgettext('chart-format', 'Settings for labels and axes'),
        toggle: true,
        haspopup: true
    };

    /**
     * A map with chart type descriptors for all supported chart type. Each
     * map element contains the following properties:
     * - {Number} cat
     *      The type category, used for grouping in the type picker control.
     * - {String} title
     *      The localized name of the chart type.
     * - {String} type
     *      The base chart type, as used in document operations.
     * - {String} stacking
     *      The stacking mode: 'standard', 'stacked', or 'percentStacked'.
     * - {String} cjs
     *      The chart type identifier for the CanvasJS library.
     * - {Boolean} curved
     *      Whether the lines in the chart area are smoothed.
     * - {Boolean} varyColors
     *      Whether the data points of a single series show different colors.
     *
     * @constant
     */
    DrawingLabels.CHART_TYPE_STYLES = CHART_TYPE_STYLES;

    /**
     * A collection of the names of those preset shapes, that can be larger than
     * the drawing itself. In this case it is necessary to recalculate the size of
     * the canvas inside the drawing.
     *
     * @constant
     */
    DrawingLabels.CANVAS_EXPANSION_PRESET_SHAPES = {
        accentBorderCallout1: 1,
        accentBorderCallout2: 1,
        accentBorderCallout3: 1,
        accentCallout1: 1,
        accentCallout2: 1,
        accentCallout3: 1,
        bentConnector3: 1,
        bentConnector4: 1,
        bentConnector5: 1,
        borderCallout1: 1,
        borderCallout2: 1,
        borderCallout3: 1,
        callout1: 1,
        callout2: 1,
        callout3: 1,
        curvedConnector3: 1,
        curvedConnector4: 1,
        curvedConnector5: 1,
        cloudCallout: 1,
        teardrop: 1,
        wedgeEllipseCallout: 1,
        wedgeRectCallout: 1,
        wedgeRoundRectCallout: 1
    };

    // custom shapes ----------------------------------------------------------

    /**
     * A map with tooltips and other settings for all custom shapes, grouped by
     * category names.
     *
     * @constant
     */
    DrawingLabels.SHAPE_LABELS_MAP = SHAPE_LABELS_MAP;

    /**
     * Return detailed information for a predefined shape type.
     *
     * @param {String} presetShapeId
     *  The identifier of a predefined shape type.
     *
     * @returns {Object|Null}
     *  A descriptor object for the specified predefined shape type; or null,
     *  if the passed identifier is not supported. The descriptor will contain
     *  the following properties:
     *  - {String} id
     *      The root identifier of the shape geometry. If the identifier passed
     *      to this method is a sub type (for example a line connector that
     *      adds arrow styles to the line ends), this identifier specifies the
     *      base geometry of the shape.
     *  - {String} type
     *      The drawing type identifier for the shape. Will be either 'shape'
     *      for regular shapes, or 'connector' for connector line shapes.
     *  - {Object} shape
     *      The JSON geometry data needed to render the shape.
     *  - {String} tooltip
     *      The tooltip string to be shown in the GUI for the shape.
     *  - {Number} aspectRatio
     *      The default aspect ratio (quotient of width and height) according
     *      to the default format defined in the property 'format'.
     *  - {Boolean} noFill
     *      Whether the area of the shape will NOT be filled by default.
     *  - {Boolean} outerText
     *      Whether the text frame will be located outside the shape geometry
     *      (especially used for filigree shapes, e.g. narrow bent arrows).
     *  - {Object} [lineAttrs]
     *      Additional line attributes to be added to the shape (usually, this
     *      property contains arrow attributes for connector sub types).
     *  - {String} [iconLabel]
     *      A label to be rendered into the shape center when generating an
     *      icon for the shape. Can be used for example to specify the number
     *      of corners for polygons, or the number of tips for star shapes.
     */
    DrawingLabels.getPresetShape = function (presetShapeId) {
        return PRESET_SHAPE_MAP.get(presetShapeId, null);
    };

    /**
     * Returns whether the specified predefined shape type is a connector line.
     *
     * @param {String} presetShapeId
     *  The identifier of a predefined shape type.
     *
     * @returns {Boolean}
     *  Whether the specified predefined shape type is a connector line.
     */
    DrawingLabels.isPresetConnector = function (presetShapeId) {
        var presetData = DrawingLabels.getPresetShape(presetShapeId);
        return (!!presetData && (presetData.type === 'connector')) || (presetShapeId in TWO_POINT_SHAPES);
    };

    /**
     * Returns whether the specified predefined shape type has a two
     * point selection.
     *
     * @param {String} presetShapeId
     *  The identifier of a predefined shape type.
     *
     * @returns {Boolean}
     *  Whether the specified predefined shape type has a two
     *  point selection.
     */
    DrawingLabels.isPresetShapeIdOfTwoPointShape = function (presetShapeId) {
        return presetShapeId in TWO_POINT_SHAPES;
    };

    /**
     * A list of shapes that do not allow text input.
     *
     * @param {String} presetShapeId
     *  The identifier of a predefined shape type.
     *
     * @returns {Boolean}
     *  Whether the specified predefined shape type can contain a paragraph.
     */
    DrawingLabels.isNoTextShape = function (presetShapeId) {
        return DrawingLabels.isPresetConnector(presetShapeId) || (presetShapeId in NO_TEXT_SHAPES);
    };

    /**
     * Returns the default aspect ratio for the specified predefined shape
     * type.
     *
     * @param {String} presetShapeId
     *  The identifier of a predefined shape type.
     *
     * @returns {Number}
     *  The default aspect ratio for the specified predefined shape type.
     */
    DrawingLabels.getPresetAspectRatio = function (presetShapeId) {
        var presetData = DrawingLabels.getPresetShape(presetShapeId);
        return presetData ? presetData.aspectRatio : 1;
    };

    // exports ================================================================

    return IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries').then(function (geometries) {

        PRESET_GEOMETRIES = geometries;

        _.each(SHAPE_LABELS_MAP, function (categoryDef) {
            _.each(categoryDef.shapes, function (shapeDef, presetShapeId, shapeMap) {

                // resolve base shape identifier linked to the current shape
                var shapeId = Utils.getStringOption(shapeDef, 'linkedId', presetShapeId);

                // get shape geometry definition
                var shapeGeometry = PRESET_GEOMETRIES[shapeId];
                if (!shapeGeometry) {
                    Utils.error('DrawingLabels: missing geometry for preset shape "' + shapeId + '"');
                    delete shapeMap[shapeId]; // remove shapes without geometry definition from the UI
                    return;
                }

                // create the map entry with default settings
                var presetData = PRESET_SHAPE_MAP.insert(presetShapeId, {
                    id: shapeId,
                    type: 'shape',
                    shape: shapeGeometry,
                    aspectRatio: 1,
                    noFill: false,
                    outerText: false
                });

                // add the tooltip (as string), or the extended shape settings (as object)
                if (_.isObject(shapeDef)) {
                    _.extend(presetData, shapeDef);
                    presetData.noFill = presetData.noFill || (presetData.type === 'connector');
                    presetData.outerText = presetData.noFill || presetData.outerText;
                    delete presetData.linkedId;
                } else {
                    presetData.tooltip = shapeDef;
                }
            });
        });

        return DrawingLabels;
    });

});
