/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/chartcolorsetpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Forms, RadioList, AppObjectMixin, ChartStyleUtil, gt) {

    'use strict';

    // number of color boxes per list entry in the ColorSetPicker control
    var COLOR_BOX_COUNT = 3;

    // all predefined color sets
    var CHART_COLOR_SETS = ChartStyleUtil.getColorSet();

    // class ChartColorSetPicker ==============================================

    /**
     * A drop-down menu for picking different predefined chart color sets.
     *
     * @constructor
     *
     * @extends RadioList
     */
    function ChartColorSetPicker(docView, initOptions) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // base constructors --------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            label: /*#. Menu title: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Color set'),
            tooltip: /*#. Menu tooltip: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Predefined color set'),
            smallerVersion: {
                label: gt.pgettext('chart-format', 'Color')
            }
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Inserts the special mark-up for a color set into the specified list
         * entry of the drop-down menu.
         */
        function createItemHandler(event, buttonNode, value, options) {

            // the color set data this list item is based on
            var colorSet = options.colorSet;
            // the content node for the color boxes
            var markup = null;

            // create all color boxes
            markup = '<div>';
            _.times(COLOR_BOX_COUNT, function (index) {

                // extract the base color from the color set
                var color = _.copy(colorSet.colors[index % colorSet.colors.length], true);

                if (colorSet.type === 'single') {
                    if (index === 0) {
                        color.transformations = [{ type: 'shade', value: 80000 }];
                    } else if (index === 2) {
                        color.transformations = [{ type: 'tint', value: 80000 }];
                    }
                }

                markup += '<span class="color-box" style="background-color:' + docModel.getCssColor(color, 'fill') + ';"></span>';
            });
            markup += '</div>';

            // insert the color boxes into the caption of the button node
            Forms.getCaptionNode(buttonNode).prepend(markup);
        }

        /**
         * Fills the drop-down menu with all available color sets.
         */
        var initializeColorTable = this.createDebouncedMethod('ChartColorSetPicker.initializeColorTable', null, function () {
            self.clearMenu();
            _.each(CHART_COLOR_SETS, function (colorSet, index) {
                self.createOptionButton('cs' + index, { tooltip: colorSet.name, colorSet: colorSet });
            });
        }, { delay: 500 });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('color-set-picker');
        this.getMenu().on('create:item', createItemHandler);

        // insert color buttons in the drop-down menu after import and changed themes
        this.waitForImportSuccess(function () {
            initializeColorTable();
            this.listenTo(docModel.getThemeCollection(), 'triggered', initializeColorTable);
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = initOptions = docModel = null;
        });

    } // class ChartColorSetPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ChartColorSetPicker });

});
