/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/baseobject', function () {

    'use strict';

    // class BaseObject =======================================================

    /**
     * An abstract base class for all classes that want to register destructor
     * code that will be executed when the public method BaseObject.destoy()
     * has been invoked.
     *
     * @constructor
     */
    function BaseObject() {

        var // all event source objects and listeners
            listeners = null,

            // all destructor callbacks
            destructors = null;

        // methods ------------------------------------------------------------

        /**
         * Registers an event listener at the specified event source object. On
         * destruction of this instance, all event listeners registered with
         * this method will be removed from the source object.
         *
         * @param {Object} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener (for example, jQuery objects,
         *  or any object extended with the core Events mix-in class).
         *
         * @param {String} type
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names.
         *
         * @param {Function} listener
         *  The event listener that will be invoked for the events triggered by
         *  the event source object.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.listenTo = function (source, type, listener) {
            (listeners || (listeners = [])).push({ source: source, type: type, listener: listener });
            source.on(type, listener);
            return this;
        };

        /**
         * Registers a destructor callback function that will be invoked when
         * the method BaseObject.destroy() has been called.
         *
         * @param {Function} destructor
         *  A destructor callback function. The BaseObject.destroy() method
         *  will invoke all registered destructor callbacks in reverse order of
         *  registration. The callback will be invoked in the context of this
         *  instance.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.registerDestructor = function (destructor) {
            (destructors || (destructors = [])).unshift(destructor);
            return this;
        };

        /**
         * Destroys this object. Invokes all destructor callback functions that
         * have been registered for this instance in reverse order. Afterwards,
         * all public properties and methods of this instance will be deleted,
         * and a single property 'destroyed' will be inserted and set to the
         * value true.
         */
        this.destroy = function () {

            // unregister all event listeners
            _(listeners).each(function (data) {
                // source object may have been destroyed already
                if (data.source.off) { data.source.off(data.type, data.listener); }
            });
            listeners = null;

            // invoke all destructor callbacks
            _(destructors).each(function (destructor) { destructor.call(this); }, this);
            destructors = null;

            // delete all public members, to detect any misuse after destruction
            _(this).each(function (member, name) { delete this[name]; }, this);
            this.destroyed = true;
        };

    } // class BaseObject

    // exports ================================================================

    return _.makeExtendable(BaseObject);

});
