/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/combofield',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/tk/popup/listmenu'
    ], function (Utils, KeyCodes, TextField, MenuMixin, ListMenu) {

    'use strict';

    // class ComboField =======================================================

    /**
     * Creates a text field control with attached drop-down list showing
     * predefined values for the text field.
     *
     * @constructor
     *
     * @extends TextField
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the control. Supports all
     *  options of the TextField base class, and the ListMenu class used for
     *  the drop-down menu element. Additionally, the following options are
     *  supported:
     *  @param {Boolean} [initOptions.typeAhead]
     *      If set to true, the label of the first list item that starts with
     *      the text currently edited will be inserted into the text field.
     *      The remaining text appended to the current text will be selected.
     *  @param {Function} [initOptions.equality=_.isEqual]
     *      A comparison function that returns whether an arbitrary value
     *      should be considered being equal to the value of a list item in the
     *      drop-down menu. If omitted, uses _.isEqual() which compares arrays
     *      and objects deeply.
     */
    function ComboField(initOptions) {

        var // self reference
            self = this,

            // search the list items and insert label into text field while editing
            typeAhead = Utils.getBooleanOption(initOptions, 'typeAhead', false),

            // comparator for list item values
            equality = Utils.getFunctionOption(initOptions, 'equality'),

            // the drop-down button that will be added to the group
            menuButton = Utils.createButton();

        // base constructors --------------------------------------------------

        TextField.call(this, initOptions);
        MenuMixin.call(this, new ListMenu(this.getNode(), Utils.extendOptions(initOptions, { itemDesign: 'list' })), { button: menuButton });

        // private methods ----------------------------------------------------

        /**
         * Scrolls the drop-down menu to make the specified list item visible.
         */
        function scrollToListItem(buttonNode) {
            if ((buttonNode.length > 0) && self.isMenuVisible()) {
                self.getMenu().scrollToChildNode(buttonNode);
            }
        }

        /**
         * Handles 'menu:beforeopen' events.
         */
        function menuBeforeOpenHandler() {
            self.getMenu().getSectionNodes().css({ minWidth: self.getNode().outerWidth() });
        }

        /**
         * Handles 'menu:open' events and moves the focus to the text field.
         */
        function menuOpenHandler() {
            self.getTextFieldNode().focus();
            scrollToListItem(Utils.getSelectedButtons(self.getMenu().getItemNodes()));
        }

        /**
         * Update handler that activates a list item if it matches the passed
         * value.
         */
        function itemUpdateHandler(value) {

            var // activate a button representing a list item
                buttonNode = Utils.selectOptionButton(self.getMenu().getItemNodes(), value, equality);

            // scroll to make the element visible
            scrollToListItem(buttonNode);
        }

        /**
         * Handles keyboard events in the text field. Moves the active list
         * entry according to cursor keys.
         */
        function textFieldKeyHandler(event) {

            var // distinguish between event types (ignore keypress events)
                keydown = event.type === 'keydown';

            function moveListItem(delta) {

                var // all list items (button elements)
                    buttonNodes = self.getMenu().getItemNodes(),
                    // index of the active list item
                    index = buttonNodes.index(Utils.getSelectedButtons(buttonNodes));

                // show the menu
                self.showMenu();
                // calculate new index, if old index is valid
                if (index >= 0) {
                    index += delta;
                }
                index = Utils.minMax(index, 0, buttonNodes.length - 1);
                // call the update handler to update the text field and list selection
                self.setValue(Utils.getControlValue(buttonNodes.eq(index)));
                // select entire text field
                self.getTextFieldNode().select();
            }

            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                if (keydown) { moveListItem(-1); }
                return false;
            case KeyCodes.DOWN_ARROW:
                if (keydown) { moveListItem(1); }
                return false;
            case KeyCodes.PAGE_UP:
                if (keydown) { moveListItem(-ListMenu.PAGE_SIZE); }
                return false;
            case KeyCodes.PAGE_DOWN:
                if (keydown) { moveListItem(ListMenu.PAGE_SIZE); }
                return false;
            case KeyCodes.ESCAPE:
                if (keydown && self.isMenuVisible()) {
                    self.hideMenu();
                    // Bug 28215: IE needs explicit selection again, otherwise text cursor is hidden
                    Utils.setTextFieldSelection(self.getTextFieldNode(), self.getTextFieldNode().val().length);
                    // let the Group base class not trigger the 'group:cancel' event
                    event.preventDefault();
                }
                break;
            case KeyCodes.SPACE:
                // Bug 28208: SPACE key with open drop-down menu and selected list item: trigger change
                var buttonNode = Utils.getSelectedButtons(self.getMenu().getItemNodes());
                if (keydown && self.isMenuVisible() && (buttonNode.length > 0)) {
                    self.hideMenu();
                    self.triggerChange(buttonNode, { preserveFocus: true });
                    // do not insert the space character into the text field
                    return false;
                }
                break;
            }
        }

        /**
         * Handler that will be called after the text field has been validated
         * while editing. Will try to insert auto-completion text according to
         * existing entries in the drop-down list.
         */
        function textFieldValidationHandler(event, oldFieldState) {

            var // the text field element
                textField = self.getTextFieldNode(),
                // current text of the text field
                fieldText = textField.val(),
                // current selection of the text field
                selection = Utils.getTextFieldSelection(textField),
                // the list item button containing the text of the text field
                buttonNode = $(),
                // the button value
                buttonValue = null,
                // the textual representation of the button value
                buttonValueText = null;

            // find the first button whose text representation starts with the entered text
            buttonNode = self.getMenu().getItemNodes().filter(function () {
                var buttonValueText = self.valueToText(Utils.getControlValue($(this)));
                return _.isString(buttonValueText) && (buttonValueText.length >= fieldText.length) &&
                    (buttonValueText.substr(0, fieldText.length).toLowerCase() === fieldText.toLowerCase());
            }).first();

            // get value and text representation from the button
            if (buttonNode.length > 0) {
                buttonValue = Utils.getControlValue(buttonNode);
                buttonValueText = self.valueToText(buttonValue);
            }

            // try to add the remaining text of an existing list item, but only
            // if the text field does not contain a selection, and something
            // has been appended to the old text
            if (typeAhead && _.isString(buttonValueText) && (buttonValueText.length > 0) &&
                    (selection.start === fieldText.length) && (oldFieldState.start < selection.start) &&
                    (oldFieldState.value.substr(0, oldFieldState.start) === fieldText.substr(0, oldFieldState.start))) {
                textField.val(buttonValueText);
                Utils.setTextFieldSelection(textField, fieldText.length, buttonValueText.length);
                fieldText = buttonValueText;
            }

            // select entry in drop-down list, if value (not text representation) is equal
            itemUpdateHandler(self.getFieldValue());
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new string entry to the drop-down list.
         *
         * @param value
         *  The value to be shown in the drop-down list. Will be converted to
         *  a string using the current validator of the text field.
         *
         * @param {Object} [options]
         *  Additional options for the list entry. Supports all options that
         *  are supported by the method ListMenu.createItemNode(), except the
         *  option 'value' which will be set to the 'value' parameter passed to
         *  this method. If the option 'label' is not specified, the label of
         *  the list entry will be set to the string value provided by the
         *  current validator of the text field.
         *
         * @returns {ComboField}
         *  A reference to this instance.
         */
        this.createListEntry = function (value, options) {
            options = Utils.extendOptions({ label: this.valueToText(value) }, options);
            this.getMenu().createItemNode('values', Utils.extendOptions(options, { value: value }));
            // the inserted list item may match the value in the text field
            itemUpdateHandler(this.getFieldValue());
            return this;
        };

        // initialization -----------------------------------------------------

        // add the drop-down button to the group (do not make it focusable)
        this.addChildNodes(menuButton);

        // add special marker class used to adjust formatting
        this.getNode().addClass('combo-field');

        // highlight list entry, handle click events in the list
        this.registerUpdateHandler(itemUpdateHandler);
        this.registerChangeHandler('click', { source: this.getMenuNode(), selector: Utils.BUTTON_SELECTOR });

        // register event handlers
        this.on({
            'menu:beforeopen': menuBeforeOpenHandler,
            'menu:open': menuOpenHandler,
            'textfield:validated': textFieldValidationHandler
        });
        this.getTextFieldNode()
            .on('keydown keypress keyup', textFieldKeyHandler);

        // keep focus in text field when clicking in the drop-down menu
        this.getMenuNode().on('focusin', function () {
            self.getTextFieldNode().focus();
        });

    } // class ComboField

    // exports ================================================================

    // derive this class from class TextField
    return TextField.extend({ constructor: ComboField });

});
