/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/cellcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/cellmodel'
    ], function (Utils, TriggerObject, Border, SheetUtils, CellModel) {

    'use strict';

    // class CellCollection ===================================================

    /**
     * Collects layout information about a range of visible cells.
     *
     * Triggers the following events:
     * - 'change:boundrange'
     *      After the associated column or row header pane has changed its
     *      visible interval or has performed an operation that changes the
     *      layout or visibility of the columns or rows in the current
     *      interval. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} boundRange
     *          The logical address of the cell range currently covered by this
     *          cell collection.
     *      (3) {Object} [operation]
     *          A description of the operation performed by the header pane.
     *          See the description of the 'change'interval' event triggered by
     *          the HeaderPane class for details.
     * - 'clear:boundrange'
     *      After the associated column or row header pane has been hidden, the
     *      bounding range of this cell collection has been cleared, and all
     *      cell entries have been removed.
     * - 'change:cells'
     *      After the contents or formatting of some cells in this collection
     *      have been changed. Receives an array of logical cell range
     *      addresses of all changed cells as first parameter.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this cell collection.
     */
    function CellCollection(app) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            model = null,
            view = null,

            // the collections of the active sheet
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,

            // all cell entries mapped by column/row index
            map = null,

            // the current bounding cell range and intervals
            boundRange = null,
            colInterval = null,
            rowInterval = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private class EntryModel -------------------------------------------

        var EntryModel = CellModel.extend({ constructor: function (data) {

            _(this).extend(CellCollection.extractCellData(data));

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            // do not trigger any events (collection entries are potential mass objects)
            CellModel.call(this, app, Utils.getObjectOption(data, 'attrs'), { silent: true });

        }}); // class EntryModel

        // methods (prototype, collection entries are mass objects) - - - - - -

        /**
         * Returns a descriptor object for the cell entry, suitable to be
         * passed to iterator functions.
         *
         * @returns {Object}
         *  A descriptor object for this collection entry, with the properties
         *  'address', 'display', 'result', 'formula', 'format', 'attributes',
         *  and 'explicit'.
         */
        EntryModel.prototype.getData = function (address) {
            return {
                address: address,
                display: this.display,
                result: this.result,
                formula: this.formula,
                format: this.format,
                attributes: this.getMergedAttributes(),
                explicit: this.getExplicitAttributes()
            };
        };

        var // default model for entries missing in the collection
            defaultEntry = new EntryModel();

        // private methods ----------------------------------------------------

        /**
         * Removes all cell entries from this collection.
         */
        function clearMap() {
            _(map).invoke('destroy');
            map = {};
        }

        /**
         * Tries to detect the type of the passed cell text with a few simple
         * tests, and returns a partial cell entry object containing the typed
         * value and formula properties.
         *
         * @param {Any} [value]
         *  If specified, the new value for the cell entry.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the passed cell value must be a string and will
         *      be parsed to determine the data type.
         *
         * @returns {Object|Null}
         *  A cell entry object containing the properties 'display' with the
         *  passed cell text, 'result' with the converted value (null, number,
         *  boolean, error code, or string), and 'formula' if the passed value
         *  may be a formula. If no value was passed, returns null.
         */
        function parseCellValue(value, parse) {

            var // the decimal separator character
                DEC = app.getDecimalSeparator(),
                // the Boolean literal FALSE in formulas
                FALSE_LITERAL = app.getBooleanLiteral(false),
                // the Boolean literal TRUE in formulas
                TRUE_LITERAL = app.getBooleanLiteral(true),
                // the resulting data object
                cellData = null;

            // parse value if passed
            if (_.isString(value)) {
                cellData = { display: value, result: null, formula: null };
                if (value.length === 0) {
                    cellData.result = null;
                } else if (/^=./.test(value)) {
                    cellData.display = '\u2026'; // result of the formula unknown
                    cellData.result = 0; // result of the formula unknown
                    cellData.formula = value;
                } else if (parse && /^[\-+]?[0-9][^\n]*$/.test(value)) {
                    // TODO: remove current group separator
                    cellData.result = parseFloat(value.replace(DEC, '.'));
                } else if (parse && (value.toUpperCase() === TRUE_LITERAL)) {
                    cellData.display = TRUE_LITERAL;
                    cellData.result = true;
                } else if (parse && (value.toUpperCase() === FALSE_LITERAL)) {
                    cellData.display = FALSE_LITERAL;
                    cellData.result = false;
                } else {
                    // error codes, simple strings, unrecognized formatted values (e.g. dates)
                    cellData.result = value;
                }
            } else if (_.isNull(value)) {
                cellData = { display: '', result: null, formula: null };
            } else if (_.isNumber(value)) {
                cellData = { display: String(value).replace(/\./, DEC), result: value, formula: null };
            } else if (_.isBoolean(value)) {
                cellData = { display: value ? TRUE_LITERAL : FALSE_LITERAL, result: value, formula: null };
            }

            return cellData;
        }

        /**
         * Returns the cell entry with the specified cell address from the
         * collection.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {EntryModel|Null}
         *  The cell entry; or null, if the entry does not exist.
         */
        function getCellEntry(address) {
            var cellEntry = map[SheetUtils.getCellKey(address)];
            return cellEntry ? cellEntry : null;
        }

        /**
         * Inserts a new entry into this collection. An existing entry will be
         * deleted.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {EntryModel} cellEntry
         *  The new cell entry to be inserted into this collection.
         */
        function putCellEntry(address, cellEntry) {
            var key = SheetUtils.getCellKey(address);
            if (key in map) { map[key].destroy(); }
            map[key] = cellEntry;
        }

        /**
         * Removes an entry from this collection and destroys it.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         */
        function deleteCellEntry(address) {
            var key = SheetUtils.getCellKey(address);
            if (key in map) { map[key].destroy(); delete map[key]; }
        }

        /**
         * Moves an entry in this collection to a new position. An existing
         * entry at the target position will be deleted.
         *
         * @param {Number[]} fromAddress
         *  The logical address of the cell entry to be moved.
         *
         * @param {Number[]} toAddress
         *  The logical address of the target position.
         */
        function moveCellEntry(fromAddress, toAddress) {
            var fromKey = SheetUtils.getCellKey(fromAddress),
                toKey = SheetUtils.getCellKey(toAddress);
            if (fromKey !== toKey) {
                if (toKey in map) { map[toKey].destroy(); delete map[toKey]; }
                if (fromKey in map) { map[toKey] = map[fromKey]; delete map[fromKey]; }
            }
        }

        /**
         * Creates a new entry for this collection. An existing entry will be
         * deleted.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [data]
         *  A data object from the server response of a view update request. If
         *  missing, creates a default entry representing an empty unformatted
         *  cell. May contain the properties 'display', 'result', 'formula',
         *  'format', and 'attrs'.
         *
         * @returns {EntryModel}
         *  A new collection entry. Always contains the properties 'display',
         *  'result', 'formula', and 'format'.
         */
        function createCellEntry(address, data) {
            var cellEntry = new EntryModel(data);
            putCellEntry(address, cellEntry);
            return cellEntry;
        }

        /**
         * Returns the cell entry with the specified cell address from the
         * collection for read/write access. If the cell entry is missing, it
         * will be created, and will be initialized with the default formatting
         * of the column or row.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {EntryModel}
         *  The collection entry corresponding to the passed cell address.
         */
        function getOrCreateCellEntry(address) {

            var // the existing cell entry
                cellEntry = getCellEntry(address),
                // the row descriptor with default formatting for empty cells
                rowEntry = null,
                // the data used to initialize a new cell entry
                data = null;

            // no entry found: create a new entry with either default row or column formatting
            if (!cellEntry) {
                rowEntry = rowCollection.getEntry(address[1]);
                if (rowEntry.attributes.row.customFormat) {
                    data = { attrs: rowEntry.explicit };
                } else {
                    data = { attrs: colCollection.getEntry(address[0]).explicit };
                }
                cellEntry = createCellEntry(address, data);
            }

            return cellEntry;
        }

        /**
         * Deletes all cell entries that exist in the passed range.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.outside=false]
         *      If set to true, the method will delete all cell entries located
         *      outside the specified range, and will keep the range
         *      unmodified.
         *
         * @returns {Array}
         *  An array with the the logical addresses of all existing cells that
         *  have been deleted.
         */
        function deleteEntriesInRange(range, options) {

            var // whether to remove cells outside the passed range
                outside = Utils.getBooleanOption(options, 'outside', false),
                // the merged ranges covering the passed range
                mergedRanges = mergeCollection.getMergedRanges(range),
                // addresses of all deleted cells
                deletedCells = [];

            // add passed range to the array of merged ranges (decision whether
            // to actually delete a collection entry is made against that array)
            mergedRanges.push(range);

            // process all cell entries, to catch entries moved into hidden columns/rows
            _(map).each(function (cellEntry, key) {

                var // the logical address of the cell entry
                    address = SheetUtils.parseCellKey(key);

                // delete the cell if is inside the range or inside a merged range
                // covering the range (or, keep all these entries in outside mode)
                if (outside !== SheetUtils.rangesContainCell(mergedRanges, address)) {
                    cellEntry.destroy();
                    delete map[key];
                    deletedCells.push(address);
                }
            });

            return deletedCells;
        }

        /**
         * Moves all cell entries that exist in the passed range, by the
         * specified amount of columns and rows.
         *
         * @returns {Array}
         *  An array with the the logical addresses of the old and new
         *  positions of all existing cells that have been moved.
         */
        function moveEntriesInRange(range, moveCols, moveRows) {

            var // a temporary map used to collect moved entries (prevents overwriting existing entries)
                tempMap = {},
                // the bounding range containing all moved cells
                movedCells = [];

            // move all cell entries into the temporary map
            _(map).each(function (cellEntry, key) {

                var // the logical address of the cell entry
                    address = SheetUtils.parseCellKey(key);

                if (SheetUtils.rangeContainsCell(range, address)) {

                    // store the old cell address
                    movedCells.push(address);

                    // move the cell to the temporary map (or delete it, if it
                    // will be move outside the bounding range of the collection)
                    address[0] += moveCols;
                    address[1] += moveRows;
                    if (SheetUtils.rangeContainsCell(boundRange, address)) {
                        tempMap[SheetUtils.getCellKey(address)] = cellEntry;
                        movedCells.push(address);
                    } else {
                        cellEntry.destroy();
                    }
                    delete map[key];
                }
            });

            // insert all cell entries back into the map
            _(tempMap).each(function (cellEntry, key) {
                map[key] = cellEntry;
            });

            return movedCells;
        }

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Object} range
         *  The logical address of the cell range to be inserted.
         *
         * @param {String} direction
         *  Either 'columns' to move the existing cells horizontally (to the
         *  right or left according to the sheet orientation), or 'rows' to
         *  move the existing cells down.
         *
         * @returns {Array}
         *  An array with the logical cell addresses of all changed cells (the
         *  addresses of inserted cells with formatting copied from the
         *  preceding column/row, and the old and new addresses of existing
         *  moved cells).
         */
        function insertCells(range, columns) {

            var // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                lastIndex = columns ? model.getMaxCol() : model.getMaxRow(),

                // number of columns the entries will be moved
                moveCols = columns ? SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true),

                // the intersection of available range and inserted range
                intersectRange = SheetUtils.getIntersectionRange(range, boundRange),
                // the column/row collection and interval in the specified move direction
                collection1 = null, interval1 = null,
                // the column/row collection in the opposite direction
                collection2 = null, interval2 = null,

                // the names of the border attributes in move direction
                innerBorderName1 = columns ? 'borderLeft' : 'borderTop',
                innerBorderName2 = columns ? 'borderRight' : 'borderBottom',
                // the names of the border attributes in the opposite direction
                outerBorderName1 = columns ? 'borderTop' : 'borderLeft',
                outerBorderName2 = columns ? 'borderBottom' : 'borderRight',

                // the addresses of all changed cells
                changedCells = [];

            // move the existing cell entries
            moveRange.end[addrIndex] = lastIndex - (columns ? moveCols : moveRows);
            if (moveRange.start[addrIndex] <= moveRange.end[addrIndex]) {
                changedCells = changedCells.concat(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            // insert new cell entries according to preceding cells (but not if
            // cells are inserted at the leading border of the current bounding range)
            if (intersectRange && (intersectRange.start[addrIndex] > boundRange.start[addrIndex])) {

                // get the column/row collections and intervals
                collection1 = columns ? colCollection : rowCollection;
                interval1 = (columns ? SheetUtils.getColInterval : SheetUtils.getRowInterval)(intersectRange);
                collection2 = columns ? rowCollection : colCollection;
                interval2 = (columns ? SheetUtils.getRowInterval : SheetUtils.getColInterval)(intersectRange);

                // process all entries in the opposite direction (e.g., create
                // new cells column-by-column, when moving in row direction)
                collection2.iterateVisibleEntries(interval2, function (oppositeEntry) {

                    var // the explicit attributes to be set at the new cells
                        newAttributes = null,
                        // the cell entry before the inserted range
                        prevCellEntry = null,
                        // the explicit cell attributes of the preceding cell entry
                        prevCellAttrs = null,
                        // the cell entry after the inserted range
                        nextCellEntry = null,
                        // the explicit cell attributes of the following cell entry
                        nextCellAttrs = null,
                        // a helper cell address to obtain cell entries from the collection
                        tempAddress = [0, 0];

                    // returns whether the passed cell attribute map contains two equal border attributes
                    function hasEqualBorders(cellAttributes, borderName1, borderName2) {
                        var border1 = cellAttributes[borderName1], border2 = cellAttributes[borderName2];
                        return _.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2);
                    }

                    // returns the border attribute, if it is equal in both passed cell attribute maps
                    function getEqualBorder(cellAttributes1, borderName1, cellAttributes2, borderName2) {
                        var border1 = cellAttributes1[borderName1], border2 = cellAttributes2[borderName2];
                        return (_.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2)) ? border1 : null;
                    }

                    // initialize the address index in the opposite direction
                    tempAddress[1 - addrIndex] = oppositeEntry.index;

                    // get preceding cell entry (do nothing, if no preceding cell is present)
                    tempAddress[addrIndex] = range.start[addrIndex] - 1;
                    prevCellEntry = getCellEntry(tempAddress);
                    if (!prevCellEntry) { return; }
                    prevCellAttrs = prevCellEntry.getExplicitAttributes().cell;

                    // get following cell entry
                    tempAddress[addrIndex] = range.end[addrIndex] + 1;
                    nextCellEntry = getCellEntry(tempAddress);
                    nextCellAttrs = nextCellEntry ? nextCellEntry.getExplicitAttributes().cell : null;

                    // copy explicit attributes of the preceding cell, used for the new cells
                    newAttributes = prevCellEntry.getExplicitAttributes();

                    // prepare explicit border attributes for the new cells
                    if (_.isObject(prevCellAttrs)) {

                        // remove old borders
                        delete newAttributes.cell[innerBorderName1];
                        delete newAttributes.cell[innerBorderName2];
                        delete newAttributes.cell[outerBorderName1];
                        delete newAttributes.cell[outerBorderName2];

                        // copy borders, that are equal in preceding and following cell entries
                        if (_.isObject(nextCellAttrs)) {

                            var border = null;

                            // insert columns: clone top border if equal in preceding and following column
                            // insert rows: clone left border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName1, nextCellAttrs, outerBorderName1))) {
                                newAttributes.cell[outerBorderName1] = _.copy(border, true);
                            }

                            // insert columns: clone bottom border if equal in preceding and following column
                            // insert rows: clone right border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName2, nextCellAttrs, outerBorderName2))) {
                                newAttributes.cell[outerBorderName2] = _.copy(border, true);
                            }

                            // insert columns: clone left/right border if equal in preceding and following column
                            // insert rows: clone top/bottom border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, innerBorderName2, nextCellAttrs, innerBorderName1))) {
                                // ... but only if *both* borders in either preceding or following column/row are equal
                                if (hasEqualBorders(prevCellAttrs, innerBorderName1, innerBorderName2) || hasEqualBorders(nextCellAttrs, innerBorderName1, innerBorderName2)) {
                                    newAttributes.cell[innerBorderName1] = _.copy(border, true);
                                    newAttributes.cell[innerBorderName2] = _.copy(border, true);
                                }
                            }
                        }

                        // remove empty cell attribute map
                        if (_.isEmpty(newAttributes.cell)) {
                            delete newAttributes.cell;
                        }
                    }

                    // no cell entries, if no explicit attributes left
                    if (_.isEmpty(newAttributes)) { return; }

                    // create equal cell entries inside the interval
                    collection1.iterateVisibleEntries(interval1, function (moveEntry) {
                        tempAddress[addrIndex] = moveEntry.index;
                        changedCells.push(_.clone(tempAddress));
                        createCellEntry(tempAddress, { attrs: newAttributes });
                    }, { hidden: true });

                }, { hidden: true });
            }

            return changedCells;
        }

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Object} range
         *  The logical address of the cell range to be deleted.
         *
         * @param {String} direction
         *  Either 'columns' to move the remaining cells horizontally (to the
         *  left or right according to the sheet orientation), or 'rows' to
         *  move the remaining cells up.
         *
         * @returns {Array}
         *  An array with the logical cell addresses of all changed cells (the
         *  addresses of the deleted cells, and the old and new addresses of
         *  existing moved cells).
         */
        function deleteCells(range, columns) {

            var // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                lastIndex = columns ? model.getMaxCol() : model.getMaxRow(),
                // number of columns the entries will be moved
                moveCols = columns ? -SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : -SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true),
                // the addresses of all changed cells
                changedCells = deleteEntriesInRange(range);

            // move the following cell entries
            moveRange.start[addrIndex] = range.end[addrIndex] + 1;
            moveRange.end[addrIndex] = lastIndex;
            if (moveRange.start[addrIndex] <= moveRange.end[addrIndex]) {
                changedCells = changedCells.concat(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            return changedCells;
        }

        /**
         * Updates the cell entries after changing the formatting attributes of
         * entire columns or rows.
         */
        function changeCellAttributesInInterval(interval, columns, attributes, options) {

            var // the column/row range covering the passed interval
                range = columns ? model.makeColRange(interval) : model.makeRowRange(interval),
                // all merged ranges covering the range partly or completely
                mergedRanges = mergeCollection.getMergedRanges(range),
                // the column interval inside the passed range
                rangeColInterval = SheetUtils.getIntersectionInterval(colInterval, SheetUtils.getColInterval(range)),
                // the row interval inside the passed range
                rangeRowInterval = SheetUtils.getIntersectionInterval(rowInterval, SheetUtils.getRowInterval(range)),
                // special treatment for border attributes applied to entire ranges
                rangeBorders = Utils.getBooleanOption(options, 'rangeBorders', false),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false),
                // additional options for border attributes
                attributeOptions = { visibleBorders: visibleBorders };

            function updateOuterCells(cellAttributes, borderName, index) {
                if (borderName in cellAttributes) {
                    var range = columns ? SheetUtils.makeRangeFromIntervals(interval, index) : SheetUtils.makeRangeFromIntervals(index, interval),
                        attributes = { cell: Utils.makeSimpleObject(borderName, cellAttributes[borderName]) },
                        attributeOptions = { visibleBorders: visibleBorders };
                    self.iterateCellsInRanges(range, function (cellData) {
                        self.updateCellEntry(cellData.address, undefined, attributes, attributeOptions);
                    });
                }
            }

            // do nothing if the interval is not visible
            if (!rangeColInterval || !rangeRowInterval) { return; }

            // update all existing cell entries
            self.iterateCellsInRanges(range, function (cellData) {

                // do not update any merged ranges here
                if (_(mergedRanges).any(function (mergedRange) {
                    return SheetUtils.rangeContainsCell(mergedRange, cellData.address);
                })) { return; }

                // add options for inner/outer border treatment, update the cell entry
                attributeOptions.innerLeft = rangeBorders && (cellData.address[0] > range.start[0]);
                attributeOptions.innerRight = rangeBorders && (cellData.address[0] < range.end[0]);
                attributeOptions.innerTop = rangeBorders && (cellData.address[1] > range.start[1]);
                attributeOptions.innerBottom = rangeBorders && (cellData.address[1] < range.end[1]);
                self.updateCellEntry(cellData.address, undefined, attributes, attributeOptions);

            }, { hidden: true, existing: true });

            // update reference cell of all merged ranges located completely inside the passed range
            _(mergedRanges).each(function (mergedRange) {

                // do not handle merged ranges not contained completely
                if (!SheetUtils.rangeContainsRange(range, mergedRange)) { return; }

                // add options for inner/outer border treatment, update the cell entry
                attributeOptions.innerLeft = rangeBorders && (mergedRange.start[0] > range.start[0]);
                attributeOptions.innerRight = rangeBorders && (mergedRange.end[0] < range.end[0]);
                attributeOptions.innerTop = rangeBorders && (mergedRange.start[1] > range.start[1]);
                attributeOptions.innerBottom = rangeBorders && (mergedRange.end[1] < range.end[1]);
                self.updateCellEntry(mergedRange.start, undefined, attributes, attributeOptions);
            });

            // column formatting: create cell entries for all rows with explicit default formats
            if (columns) {
                rowCollection.iterateVisibleEntries(rangeRowInterval, function (rowEntry) {

                    // skip rows without explicit formatting
                    if (!rowEntry.attributes.row.customFormat) { return; }

                    // add options for inner/outer border treatment
                    attributeOptions.innerTop = rangeBorders && (rowEntry.index > range.start[1]);
                    attributeOptions.innerBottom = rangeBorders && (rowEntry.index < range.end[1]);

                    // update cells in all columns of the row
                    colCollection.iterateVisibleEntries(rangeColInterval, function (colEntry) {

                        var // the cell address
                            address = [colEntry.index, rowEntry.index];

                        // do not modify existing cell entries (already updated above)
                        if (getCellEntry(address)) { return; }

                        // add options for inner/outer border treatment, update the cell entry
                        attributeOptions.innerLeft = rangeBorders && (colEntry.index > range.start[0]);
                        attributeOptions.innerRight = rangeBorders && (colEntry.index < range.end[0]);
                        self.updateCellEntry(address, undefined, attributes, attributeOptions);

                    }, { hidden: true });
                }, { hidden: true });
            }

            // add explicit borders at sheet ranges in 'rangeBorders' mode
            if (rangeBorders && _.isObject(attributes.cell)) {
                updateOuterCells(attributes.cell, columns ? 'borderTop' : 'borderLeft', 0);
                updateOuterCells(attributes.cell, columns ? 'borderBottom' : 'borderRight', columns ? model.getMaxRow() : model.getMaxCol());
            }
        }

        /**
         * Joins the passed cell addresses to an array of range addresses, and
         * triggers a 'change:cells' event, if at least one cell address has
         * been passed to this method.
         *
         * @param {Array} cells
         *  The logical addresses of all changed cells.
         */
        function triggerChangeCellsEvent(cells) {

            var // join cell addresses to range addresses
                ranges = SheetUtils.joinCellsToRanges(cells);

            if (ranges.length > 0) {
                self.trigger('change:cells', ranges);
            }
        }

        /**
         * Triggers a 'change:boundrange' event, passing the current bounding
         * range of this collection; or a 'clear:boundrange' event, if the
         * bounding range is currently not available. Multiple changes of the
         * bounding range will be debounced and triggered only once (e.g.,
         * after changing column and row interval at the same time while
         * scrolling or zooming), unless an operation has been passed which
         * will be notified immediately.
         *
         * @param {Object} [operation]
         *  A sheet operation that has caused the changed bounding range. If
         *  specified, must be an object as specified in the description of the
         *  constructor of this class. Will be ignored, if the bounding range
         *  is not available.
         */
        var triggerBoundRangeEvent = (function () {

            var // current timer for deferred event
                timer = null,
                // the cached 'refresh' operation for debounced event
                cachedOperation = null,
                // all cached cell ranges which need to be updated from server
                cachedRanges = null;

            // triggers a change/clear event, and requests server update for changed cells
            function triggerAndRequestUpdate(operation) {
                if (boundRange) {
                    self.trigger('change:boundrange', _.copy(boundRange, true), operation);
                    if (_.isArray(cachedRanges) && (cachedRanges.length > 0)) {
                        view.requestUpdate({ cells: cachedRanges });
                    }
                } else {
                    self.trigger('clear:boundrange');
                }
            }

            // the actual 'triggerBoundRangeEvent()' method
            function triggerBoundRangeEvent(operation, requestRanges) {

                var // whether a 'refresh' operation has been passed
                    refreshAll = _.isObject(operation) && (operation.name === 'refresh'),
                    // whether an operation with interval has been passed
                    hasOperation = !refreshAll && _.isObject(operation);

                // cache the current 'refresh' operation for debounced event
                cachedOperation = refreshAll ? operation : (!boundRange || hasOperation) ? null : cachedOperation;

                // cache the new request ranges (forget old ranges of outdated bounding range)
                cachedRanges = requestRanges;

                // bound range not initialized, or operation passed: cancel pending timer
                if (timer && (!boundRange || hasOperation)) {
                    timer.abort();
                    timer = null;
                }

                // bound range not initialized: trigger a 'clear:boundrange' event
                if (!boundRange) {
                    triggerAndRequestUpdate();
                    return;
                }

                // operation with interval passed: trigger immediately
                if (hasOperation) {
                    triggerAndRequestUpdate(operation);
                    return;
                }

                // do nothing, if a timer is already running
                if (timer) { return; }

                // set up a timer for triggering only one event for multiple
                // changes of the bounding range in the current script (e.g.,
                // after changing column and row interval at the same time)
                timer = app.executeDelayed(function () {
                    timer = null;
                    triggerAndRequestUpdate(cachedOperation);
                });
            }

            return triggerBoundRangeEvent;
        }());

        /**
         * Adds the reference cells of all merged ranges to this collection
         * that are located in the specified cell ranges.
         */
        function updateReferenceCells(ranges) {

            var // the merged ranges starting in the passed ranges
                mergedRanges = mergeCollection.getMergedRanges(ranges, { reference: true });

            // create missing collection entries for the reference cells
            _.chain(mergedRanges).pluck('start').each(getOrCreateCellEntry);
        }

        /**
         * Updates the intervals of the cells contained in this cell
         * collection. Deletes all cell entries not contained in the collection
         * anymore.
         *
         * @param {Boolean} columns
         *  True, if the passed interval is a column interval; or false, if the
         *  passed interval is a row interval.
         *
         * @param {Object} [newInterval]
         *  The new column/row interval, with the zero-based index properties
         *  'first' and 'last'. If omitted, the column/row interval will be
         *  cleared, and all collection entries will be removed.
         *
         * @param {Object} [operation]
         *  The sheet operation that caused the changed interval.
         */
        function updateInterval(columns, newInterval, operation) {

            var // the original interval
                oldInterval = columns ? colInterval : rowInterval,
                // the cell ranges that need to be refreshed from server
                requestRanges = [],
                // whether a 'refresh' operation has been passed
                refreshAll = _.isObject(operation) && (operation.name === 'refresh'),
                // whether an operation with interval has been passed
                hasOperation = !refreshAll && _.isObject(operation);

            // makes a cell range from the passed interval
            function makeRange(interval) {
                return columns ?
                    SheetUtils.makeRangeFromIntervals(interval, rowInterval) :
                    SheetUtils.makeRangeFromIntervals(colInterval, interval);
            }

            // store the passed interval
            newInterval = _.clone(newInterval);
            if (columns) { colInterval = newInterval; } else { rowInterval = newInterval; }

            // clear bounding range and cell map, if column or row interval are missing
            if (!colInterval || !rowInterval) {
                if (boundRange) {
                    boundRange = null;
                    clearMap();
                    triggerBoundRangeEvent();
                }
                return;
            }

            // update the bounding range
            boundRange = SheetUtils.makeRangeFromIntervals(colInterval, rowInterval);

            // insert, delete, and move cells according to the passed operation
            if (hasOperation && (operation.first <= newInterval.last)) {
                switch (operation.name) {
                case 'insert':
                    insertCells(makeRange(operation), columns);
                    break;
                case 'delete':
                    deleteCells(makeRange(operation), columns);
                    break;
                case 'change':
                    if (_.isObject(operation.attrs) && (('cell' in operation.attrs) || ('character' in operation.attrs) || ('styleId' in operation.attrs))) {
                        changeCellAttributesInInterval(operation, columns, operation.attrs, operation);
                    }
                    break;
                }
            }

            // remove all cell entries outside the bounding range
            deleteEntriesInRange(boundRange, { outside: true });

            // add new cells now covered by the bounding range to the requested ranges
            if (oldInterval) {
                if (newInterval.first < oldInterval.first) {
                    requestRanges.push(makeRange({ first: newInterval.first, last: Math.min(newInterval.last, oldInterval.first - 1) }));
                }
                if (oldInterval.last < newInterval.last) {
                    requestRanges.push(makeRange({ first: Math.max(oldInterval.last + 1, newInterval.first), last: newInterval.last }));
                }
            } else {
                requestRanges.push(boundRange);
            }

            // add reference cells of merged ranges starting in the new ranges
            updateReferenceCells(requestRanges);

            // add current direction to the operation
            if (hasOperation) {
                operation = _.clone(operation);
                operation.direction = columns ? 'columns' : 'rows';
            }

            // notify listeners, if the bounding range has changed, or if an operation has been passed
            if (_.isObject(operation) || !oldInterval || !_.isEqual(oldInterval, newInterval)) {
                triggerBoundRangeEvent(operation, requestRanges);
            }
        }

        /**
         * Initializes this collection after the active sheet has been changed.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {

            var // the model of the active sheet
                sheetModel = activeSheetModel;

            // get collections from the current active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();

            // clear the map and set bounding range to default values
            boundRange = colInterval = rowInterval = null;
            clearMap();
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * inserted into the sheet.
         */
        function insertMergedHandler(event, ranges) {

            var // the parts of the passed ranges covered by the collection
                changedRanges = [];

            // move value and formatting of first non-empty cell in each range to reference cell
            SheetUtils.iterateIntersectionRanges(ranges, boundRange, function (intersectionRange, originalRange) {

                var // the first entry will be moved to the reference cell
                    found = false;

                self.iterateCellsInRanges(intersectionRange, function (cellData) {
                    if (!found && !CellCollection.isBlank(cellData)) {
                        // move first non-empty cell entry to the reference position
                        found = true;
                        moveCellEntry(cellData.address, originalRange.start);
                    } else if (!_.isEqual(cellData.address, originalRange.start)) {
                        // remove all other cell entries (but the first) from the collection
                        deleteCellEntry(cellData.address);
                    }
                }, { hidden: true, existing: true });

                changedRanges.push(intersectionRange);
            });

            // notify listeners
            self.trigger('change:cells', changedRanges);
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * deleted from the sheet.
         */
        function deleteMergedHandler(event, ranges, options) {

            var // the parts of the passed ranges covered by the collection
                changedRanges = [];

            // do nothing for merged ranges that have been deleted implicitly
            // while deleting columns or rows (will be handled
            if (Utils.getBooleanOption(options, 'implicit', false)) { return; }

            // copy formatting of reference cell to all other cells formerly hidden by the merged range
            SheetUtils.iterateIntersectionRanges(ranges, boundRange, function (intersectionRange, originalRange) {

                var cellEntry = getCellEntry(originalRange.start),
                    attributes = cellEntry ? cellEntry.getExplicitAttributes() : null;

                if (!_.isEmpty(attributes)) {
                    self.iterateCellsInRanges(intersectionRange, function (cellData) {
                        self.updateCellEntry(cellData.address, undefined, attributes);
                    }, { hidden: true });
                }

                changedRanges.push(intersectionRange);
            });

            // notify listeners
            self.trigger('change:cells', changedRanges);
        }

        /**
         * Invokes the passed cell iterator with the correct collection entry.
         */
        function invokeIterator(colEntry, rowEntry, originalRange, iterator, context, existing) {

            var // the cell address
                address = [colEntry.index, rowEntry.index],
                // the cell entry instance
                cellEntry = getCellEntry(address),
                // the descriptor object to be passed to the iterator
                cellData = null;

            // skip missing cells if specified
            if (existing && !cellEntry) { return; }

            // build the cell descriptor
            if (cellEntry) {
                cellData = cellEntry.getData(address);
            } else {
                cellData = defaultEntry.getData(address);
                // if the row entry contains attributes, use them (ignore column default attributes)
                cellData.attributes = rowEntry.attributes.row.customFormat ? rowEntry.attributes : colEntry.attributes;
                cellData.explicit = rowEntry.attributes.row.customFormat ? rowEntry.explicit : colEntry.explicit;
            }

            // extend with column and row layout information
            cellData.left = colEntry.offset;
            cellData.width = colEntry.size;
            cellData.top = colEntry.offset;
            cellData.height = colEntry.size;

            // invoke the iterator function
            return iterator.call(context, cellData, originalRange);
        }

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this cell collection is valid.
         */
        function registerEventHandler(source, type, handler) {
            source.on(type, function () {
                if (_.isObject(boundRange)) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Initialization of class members.
         */
        function initHandler() {

            // the spreadsheet model and view
            model = app.getModel();
            view = app.getView();

            // initialize class members on change of active sheet
            view.on('change:activesheet', changeActiveSheetHandler);

            // update cell entries after merging or unmerging ranges
            registerEventHandler(view, 'insert:merged', insertMergedHandler);
            registerEventHandler(view, 'delete:merged', deleteMergedHandler);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the logical address of the cell range covered by this cell
         * collection. The range address will be build by the columns and rows
         * available in the associated column and row collections.
         *
         * @returns {Object}
         *  The range address covered by the column and row collection.
         */
        this.getRange = function () {
            return _.copy(boundRange, true);
        };

        /**
         * Returns whether this cell collection covers the passed cell range
         * completely.
         *
         * @param {Object} range
         *  The logical cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether this collection covers the passed cell range completely.
         */
        this.containsRange = function (range) {
            return SheetUtils.rangeContainsRange(boundRange, range);
        };

        /**
         * Returns the merged default attribute set used for undefined cells.
         *
         * @returns {Object}
         *  The merged default attribute set for undefined cells, containing
         *  the default column and row attributes.
         */
        this.getDefaultAttributes = function () {
            return defaultEntry.getMergedAttributes();
        };

        /**
         * Returns a descriptor object for the cell with the specified address,
         * if it exists in this collection (cell not empty or formatted).
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.always=false]
         *      If set to true, will always return a cell descriptor object for
         *      every address passed to this method, also for cells not
         *      existing in the collection and for cells outside the bounding
         *      range of this collection. Building a default cell descriptor is
         *      expensive, as it accesses the column and row collection to
         *      resolve the default formatting attributes used by the cell.
         *
         * @returns {Object|Null}
         *  The cell descriptor object, containing the properties 'address',
         *  'display', 'result', 'formula', 'format', and 'attributes'; or
         *  null, if the cell does not exist in this collection.
         */
        this.getCellEntry = function (address, options) {

            var // existing collection entry
                cellEntry = getCellEntry(address),
                // the cell descriptor object
                cellData = cellEntry ? cellEntry.getData(address) : null,
                // descriptor from row or column collection
                colRowEntry = null;

            // return descriptor of existing collection entries
            if (cellData) { return cellData; }

            // build descriptor for empty cells, if specified
            if (Utils.getBooleanOption(options, 'always', false)) {
                // get cell descriptor from the default cell entry
                cellData = defaultEntry.getData(address);
                // if the row entry contains attributes, use them (ignore column default attributes)
                colRowEntry = rowCollection.getEntry(address[1]);
                if (!colRowEntry.attributes.row.customFormat) {
                    colRowEntry = colCollection.getEntry(address[0]);
                }
                cellData.attributes = colRowEntry.attributes;
                cellData.explicit = colRowEntry.explicit;
            }

            return cellData;
        };

        /**
         * Invokes the passed iterator function for all visible cells in this
         * collection for read-only access.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells. Receives a cell
         *  descriptor object with the following parameters:
         *  - {Number[]} cellData.address
         *      The logical address of the cell.
         *  - {String} cellData.display
         *      The formatted display string of the cell.
         *  - {Any} cellData.result
         *      The typed result value of the cell, or the formula result.
         *  - {String|Null} cellData.formula
         *      The formula definition, if the cell contains a formula.
         *  - {Object} cellData.attributes
         *      The resulting merged attribute set for the cell. If the cell is
         *      undefined, this object will be set to the appropriate default
         *      cell formatting attributes of the row or column.
         *  - {Object} cellData.explicit
         *      The explicit attributes of the cell. If the cell is undefined,
         *      this object will be set to the appropriate default cell
         *      formatting attributes of the row or column.
         *  - {Number} cellData.left
         *      The horizontal position of the cell in the sheet, in pixels.
         *  - {Number} cellData.width
         *      The width of the cell (ignoring merged ranges), in pixels.
         *  - {Number} cellData.top
         *      The vertical position of the cell in the sheet, in pixels.
         *  - {Number} cellData.height
         *      The height of the cell (ignoring merged ranges), in pixels.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the cells will be visited in reversed order.
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the cells in all hidden columns/rows that
         *      directly precede a visible column/row will be visited too.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCells = function (iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit hidden entries
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false);

            // invoke iterator for all cells in this collection
            return rowCollection.iterateVisibleEntries(rowInterval, function (rowEntry) {
                return colCollection.iterateVisibleEntries(colInterval, function (colEntry) {
                    return invokeIterator(colEntry, rowEntry, undefined, iterator, context, existing);
                }, { reverse: reverse, hidden: hidden });
            }, { reverse: reverse, hidden: hidden });
        };

        /**
         * Invokes the passed iterator function for specific visible cells in
         * one column or row, and moving into the specified direction.
         *
         * @param {Number[]} address
         *  The logical address of the cell that will be visited first (the
         *  option 'options.skipStartCell' can be used if iteration shall start
         *  with the nearest visible neighbor of the cell, and not with the
         *  cell itself).
         *
         * @param {String} direction
         *  How to move to the next cells while iterating. Supported values are
         *  'up', 'down', 'left', or 'right'.
         *
         * @param {Function} iterator
         *  The iterator function called for all visited cells. Receives a cell
         *  descriptor object as first parameter (see method
         *  CellCollection.iterateCells() for details). If the iterator returns
         *  the Utils.BREAK object, the iteration process will be stopped
         *  immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the cells in all hidden columns/rows that
         *      directly precede a visible column/row will be visited too.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection.
         *  @param {Boolean} [options.skipStartCell=false]
         *      If set to true, iteration will start at the nearest visible
         *      neighbor of the cell specified in the 'address' parameter
         *      instead of that cell.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateNextCells = function (address, direction, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to visit hidden entries
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // number of cells to skip before iteration starts
                skipCount = Utils.getBooleanOption(options, 'skipStartCell', false) ? 1 : 0,
                // the fixed collection entry
                fixedEntry = null,
                // the column/row interval to be visited
                interval = null;

            function getFixedEntry(collection, interval, index) {
                return SheetUtils.intervalContainsIndex(interval, index) ? collection.getVisibleEntry(index) : null;
            }

            switch (direction) {
            case 'up':
                if ((fixedEntry = getFixedEntry(colCollection, colInterval, address[0]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(rowInterval, { first: 0, last: address[1] - skipCount }))) {
                        return rowCollection.iterateVisibleEntries(interval, function (rowEntry) {
                            return invokeIterator(fixedEntry, rowEntry, undefined, iterator, context, existing);
                        }, { reverse: true, hidden: hidden });
                    }
                }
                break;

            case 'down':
                if ((fixedEntry = getFixedEntry(colCollection, colInterval, address[0]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(rowInterval, { first: address[1] + skipCount, last: model.getMaxRow() }))) {
                        return rowCollection.iterateVisibleEntries(interval, function (rowEntry) {
                            return invokeIterator(fixedEntry, rowEntry, undefined, iterator, context, existing);
                        }, { hidden: hidden });
                    }
                }
                break;

            case 'left':
                if ((fixedEntry = getFixedEntry(rowCollection, rowInterval, address[1]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(colInterval, { first: 0, last: address[0] - skipCount }))) {
                        return colCollection.iterateVisibleEntries(interval, function (colEntry) {
                            return invokeIterator(colEntry, fixedEntry, undefined, iterator, context, existing);
                        }, { reverse: true, hidden: hidden });
                    }
                }
                break;

            case 'right':
                if ((fixedEntry = getFixedEntry(rowCollection, rowInterval, address[1]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(colInterval, { first: address[0] + skipCount, last: model.getMaxCol() }))) {
                        return colCollection.iterateVisibleEntries(interval, function (colEntry) {
                            return invokeIterator(colEntry, fixedEntry, undefined, iterator, context, existing);
                        }, { hidden: hidden });
                    }
                }
                break;

            default:
                Utils.error('CellCollection.iterateNextCells(): invalid direction "' + direction + '"');
            }
        };

        /**
         * Invokes the passed iterator function for all visible cells in the
         * passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times (see method
         *  'CellCollection.iterateCellsInRows()' for an alternative). Receives
         *  the following parameters:
         *  (1) {Object} cellData
         *      The cell descriptor object (see CellCollection.iterateCells()
         *      method for details).
         *  (2) {Object} originalRange
         *      The logical address of the range containing the current cell
         *      (one of the ranges contained in the 'ranges' parameter passed
         *      to this method).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cells in each range will be
         *      visited in reversed order.
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, the cells in all hidden columns/rows that
         *      directly precede a visible column/row will be visited too.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRanges = function (ranges, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit hidden entries
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // whether only over outer cells shall be iterated
                outerCellsOnly = Utils.getBooleanOption(options, 'outerCellsOnly', false);

            // iterate the intersection of the passed ranges and the range available in this collection
            return SheetUtils.iterateIntersectionRanges(ranges, boundRange, function (intersectRange, originalRange) {
                var colInt = SheetUtils.getColInterval(intersectRange),
                    rowInt = SheetUtils.getRowInterval(intersectRange);
                return rowCollection.iterateVisibleEntries(rowInt, function (rowEntry) {
                    return colCollection.iterateVisibleEntries(colInt, function (colEntry) {
                        if (!outerCellsOnly || SheetUtils.rangeContainsOuterCell(originalRange, [colEntry.index, rowEntry.index])) {
                            return invokeIterator(colEntry, rowEntry, originalRange, iterator, context, existing);
                        }
                    }, { reverse: reverse, hidden: hidden });
                }, { reverse: reverse, hidden: hidden });
            }, { reverse: reverse });
        };

        /**
         * Invokes the passed iterator function exactly once for each visible
         * cell covered by the passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called exactly once for each visible cell in
         *  the passed ranges (even if the cell is covered by multiple ranges).
         *  The cells will be visited row by row from top to bottom, and inside
         *  the rows with ascending column index. Example: If the entire
         *  columns B and D are passed to this method, the iterator function
         *  will be called for cells B1, D1, B2, D2, and so on. Receives the
         *  following parameters:
         *  (1) {Object} cellData
         *      The cell descriptor object (see CellCollection.iterateCells()
         *      method for details).
         *  (2) {Object} rowBandRange
         *      The logical address of a range containing the current cell.
         *      These ranges are created with the algorithm that divides and
         *      merges the original ranges in order to create a list of
         *      distinct ranges. The range address has been reduced to visible
         *      columns and rows, but may contain hidden columns or rows
         *      inside.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the rows will be visited in reversed order from
         *      the bottom-most row to the top-most row, and the cells in each
         *      row will be visited from right-most column to left-most column.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRows = function (ranges, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // ordered row bands with columns intervals
                rowBands = SheetUtils.getUnifiedRowBands(SheetUtils.getIntersectionRanges(ranges, boundRange));

            // reduce bands to visible rows, remove row bands without a corresponding row entry completely
            Utils.iterateArray(rowBands, function (rowBand, index) {
                var visibleInterval = rowCollection.getVisibleInterval(rowBand);
                if (visibleInterval) { _(rowBand).extend(visibleInterval); } else { rowBands.splice(index, 1); }
            }, { reverse: true });

            // call the iterator function for all resulting ranges in the row bands
            return Utils.iterateArray(rowBands, function (rowBand) {

                var // reduce row band to visible rows
                    rowInterval = rowCollection.getVisibleInterval(rowBand);

                // early exit if no visible row present in the row band
                if (!rowInterval) { return; }

                // reduce to visible columns, remove column intervals without corresponding entries
                Utils.iterateArray(rowBand.intervals, function (colInterval, index) {
                    colInterval = colCollection.getVisibleInterval(colInterval);
                    if (colInterval) { rowBand.intervals[index] = colInterval; } else { rowBand.intervals.splice(index, 1); }
                }, { reverse: true });

                // early exit if no visible column present in the intervals
                if (rowBand.intervals.length === 0) { return; }

                // call iterator function for all cells in the row band
                return rowCollection.iterateVisibleEntries(rowInterval, function (rowEntry) {
                    // iterate row by row through all column intervals
                    return Utils.iterateArray(rowBand.intervals, function (colInterval) {
                        var rowBandRange = SheetUtils.makeRangeFromIntervals(colInterval, rowInterval);
                        return colCollection.iterateVisibleEntries(colInterval, function (colEntry) {
                            return invokeIterator(colEntry, rowEntry, rowBandRange, iterator, context, existing);
                        }, { reverse: reverse });
                    }, { reverse: reverse });
                }, { reverse: reverse });
            }, { reverse: reverse });
        };

        /**
         * Checks the presence of locked cell(s) in given ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @returns {boolean}
         */
        this.hasProtectedCellsInRanges = function (ranges) {
            if (!ranges) return false;
            var found = false;
            self.iterateCellsInRanges(ranges, function (cellData) {
                if (!cellData.attributes.cell.unlocked) {
                    found = true;
                    return Utils.BREAK;
                }
            });
            return found;
        };

        /**
         * Updates this cell collection according to the new column interval
         * displayed in the column header pane.
         *
         * @param {Object} interval
         *  The new column interval shown in the column header pane.
         *
         * @param {Object} [operation]
         *  A sheet operation that has caused the changed column interval. If
         *  specified, must be an object with the following properties:
         *  @param {String} operation.name
         *      The name of the operation. Must be one of 'insert' (new columns
         *      inserted into the sheet), 'delete' (columns deleted from the
         *      sheet), or 'change' (column formatting changed, including
         *      column width and hidden state).
         *  @param {Number} operation.first
         *      The zero-based index of the first column affected by the
         *      operation.
         *  @param {Number} operation.last
         *      The zero-based index of the last column affected by the
         *      operation.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setColInterval = function (interval, operation) {
            updateInterval(true, interval, operation);
            return this;
        };

        /**
         * Updates this cell collection according to the new row interval
         * displayed in the row header pane.
         *
         * @param {Object} interval
         *  The new row interval shown in the row header pane.
         *
         * @param {Object} [operation]
         *  A sheet operation that has caused the changed row interval. If
         *  specified, must be an object with the following properties:
         *  @param {String} operation.name
         *      The name of the operation. Must be one of 'insert' (new rows
         *      inserted into the sheet), 'delete' (rows deleted from the
         *      sheet), or 'change' (row formatting changed, including row
         *      height and hidden state).
         *  @param {Number} operation.first
         *      The zero-based index of the first row affected by the
         *      operation.
         *  @param {Number} operation.last
         *      The zero-based index of the last row affected by the operation.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setRowInterval = function (interval, operation) {
            updateInterval(false, interval, operation);
            return this;
        };

        /**
         * Invalidates the column interval of this cell collection, and removes
         * all cell collection entries.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.clearColInterval = function () {
            updateInterval(true);
            return this;
        };

        /**
         * Invalidates the row interval of this cell collection, and removes
         * all cell collection entries.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.clearRowInterval = function () {
            updateInterval(false);
            return this;
        };

        /**
         * Imports the passed cell data received from a 'docs:update' event
         * notification.
         *
         * @param {Array} ranges
         *  The cell range addresses containing the passed cell data.
         *
         * @param {Array} cellData
         *  An array with cell layout descriptor objects, as received from the
         *  'docs:update' event notification.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.importCellEntries = function (ranges, importData) {

            var // the deleted and inserted cells in all passed ranges
                changedCells = _.chain(ranges).map(deleteEntriesInRange).flatten(true).value(),
                // the reference cells of merged ranges covering the passed ranges, as sorted array
                mergeRefCells = _.chain(mergeCollection.getMergedRanges(ranges)).pluck('start').map(SheetUtils.getCellName).value();

            // returns whether the passed cell address is contained in the passed ranges
            function isCoveredCell(address) {
                return SheetUtils.rangeContainsCell(boundRange, address);
            }

            // returns whether the passed cell address is contained in the 'mergeRefCells' array
            function isMergeRefCell(address) {
                return _(mergeRefCells).indexOf(SheetUtils.getCellName(address), true) >= 0;
            }

            // sort merge reference cells by cell name
            mergeRefCells.sort();

            // map all existing cells by their position, update bounding range
            _(importData).each(function (cellData) {
                if (_.isObject(cellData) && model.isValidAddress(cellData.pos) && (isCoveredCell(cellData.pos) || isMergeRefCell(cellData.pos))) {
                    createCellEntry(cellData.pos, cellData);
                    changedCells.push(cellData.pos);
                }
            });

            // create missing entries for reference cells of empty merged ranges
            updateReferenceCells(ranges);

            // notify all change listeners
            triggerChangeCellsEvent(changedCells);
            return this;
        };

        /**
         * Handler for 'setCellContents' operations generated and applied
         * locally. Changes the contents and formatting attributes of the cells
         * in this collection, and triggers a 'change:cells' event for all
         * changed cells.
         *
         * @param {Number[]} start
         *  The logical address of the first cell to be changed.
         *
         * @param {Object} operation
         *  The complete 'setCellContents' operations object.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setCellContents = function (start, operation) {

            var // whether to parse the cell contents
                parse = _.isString(operation.parse),
                // the addresses of all changed cells
                changedCells = [],
                // the row interval to be modified
                rowInterval = { first: start[1], last: start[1] + operation.contents.length - 1 };

            // restrict to bounding range
            rowInterval = SheetUtils.getIntersectionInterval(rowInterval, SheetUtils.getRowInterval(boundRange));
            if (!rowInterval || (start[0] > boundRange.end[0])) { return this; }

            // process all visible rows, and hidden rows preceding a visible row
            rowCollection.iterateVisibleEntries(rowInterval, function (rowEntry) {

                var // the contents of the current row
                    rowContents = operation.contents[rowEntry.index - start[1]],
                    // the column interval to be modified
                    colInterval = { first: start[0], last: start[0] + rowContents.length - 1 };

                // restrict to bounding range
                colInterval = SheetUtils.getIntersectionInterval(colInterval, SheetUtils.getColInterval(boundRange));
                if (!colInterval) { return; }

                // process all visible columns, and hidden columns preceding a visible column
                colCollection.iterateVisibleEntries(colInterval, function (colEntry) {

                    var // the address of the current cell
                        address = [colEntry.index, rowEntry.index],
                        // the new contents of the current cell
                        cellContents = rowContents[colEntry.index - start[0]];

                    // update the affected cell entry with the parsed value
                    self.updateCellEntry(address, parseCellValue(cellContents.value, parse), cellContents.attrs);
                    changedCells.push(address);

                }, { hidden: true });
            }, { hidden: true });

            // notify all change listeners
            triggerChangeCellsEvent(changedCells);
            return this;
        };

        /**
         * Handler for 'fillCellRange' operations generated and applied
         * locally. Fills all cells in the specified cell range with the same
         * value and formatting attributes.
         *
         * @param {Object} range
         *  The logical address of a single cell range.
         *
         * @param {Object} operation
         *  The complete 'fillCellRange' operations object.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fillCellRange = function (range, operation) {

            var // the parsed value for all cell in all ranges
                parsedValue = parseCellValue(operation.value, _.isString(operation.parse)),
                // the new attributes passed in the operation
                attributes = operation.attrs,
                // the intersection of the passed range with the bounding range
                intersectRange = SheetUtils.getIntersectionRange(range, boundRange),
                // the merged ranges covering the intersection range
                mergedRanges = null,
                // special treatment for border attributes applied to entire ranges
                rangeBorders = Utils.getBooleanOption(operation, 'rangeBorders', false),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(operation, 'visibleBorders', false);

            // do nothing, if nothing changed in the bounding range
            if (!intersectRange) { return this; }

            // update all affected cell entries
            if (_.isObject(parsedValue) || _.isObject(attributes)) {
                mergedRanges = mergeCollection.getMergedRanges(intersectRange);

                this.iterateCellsInRanges(intersectRange, function (cellData) {

                    var // merged range containing the cell
                        mergedRange = _(mergedRanges).find(function (mergedRange) {
                            return SheetUtils.rangeContainsCell(mergedRange, cellData.address);
                        }),
                        // additional options for border attributes
                        attributeOptions = { visibleBorders: visibleBorders };

                    // do not update cells hidden by merged ranges
                    if (mergedRange && !_.isEqual(mergedRange.start, cellData.address)) { return; }

                    // add options for inner/outer border treatment, update the cell entry
                    mergedRange = mergedRange || { start: cellData.address, end: cellData.address };
                    attributeOptions.innerLeft = rangeBorders && (mergedRange.start[0] > range.start[0]);
                    attributeOptions.innerRight = rangeBorders && (mergedRange.end[0] < range.end[0]);
                    attributeOptions.innerTop = rangeBorders && (mergedRange.start[1] > range.start[1]);
                    attributeOptions.innerBottom = rangeBorders && (mergedRange.end[1] < range.end[1]);
                    self.updateCellEntry(cellData.address, parsedValue, attributes, attributeOptions);

                }, { hidden: true });
            }

            // notify all change listeners
            this.trigger('change:cells', intersectRange);
            return this;
        };

        /**
         * Handler for 'clearCellRange' operations generated and applied
         * locally. Clears value and formatting of all cells in the specified
         * cell range.
         *
         * @param {Object} range
         *  The logical address of a single cell range.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.clearCellRange = function (range) {

            var // all existing entries that have been deleted
                changedCells = deleteEntriesInRange(range);

            // notify all change listeners
            triggerChangeCellsEvent(changedCells);
            return this;
        };

        /**
         * Handler for 'autoFill' operations generated and applied locally.
         *
         * @param {Object} range
         *  The logical address of a single cell range.
         *
         * @param {Object} operation
         *  The complete 'autoFill' operations object.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.autoFill = function (range, operation) {

            var autofillRange = {},
                replaceString = '...',
                replaceResult = null,
                localCellEntry = null,
                attrs = null,
                isSingleCell = (!range.end) || (_.isEqual(range.start, range.end));

            if (isSingleCell) {
                localCellEntry = getCellEntry(range.start);
                if (! localCellEntry) { return this; }
                if (_.isString(localCellEntry.result) && localCellEntry.result[0] !== '#') {
                    replaceString = localCellEntry.result;
                    attrs = localCellEntry.getExplicitAttributes();
                } else if (_.isNumber(localCellEntry.result)) {
                    replaceResult = localCellEntry.result;
                    replaceString = String(replaceResult);
                    attrs = localCellEntry.getExplicitAttributes();
                }
            }

            // Recalculate auto fill range
            autofillRange.start = _.clone(range.start);
            autofillRange.end = _.clone(operation.target);
            SheetUtils.adjustRange(autofillRange);

            // modifying the number order
            if (replaceResult && !_.isEqual(range.start, autofillRange.start)) {
                replaceResult = replaceResult - SheetUtils.getCellCount(autofillRange);
                replaceString = String(replaceResult);
            }

            // iterating over all cells in auto fill range
            self.iterateCellsInRanges(autofillRange, function (cellData) {
                var newCellData = null;
                if (! SheetUtils.rangeContainsCell(range, cellData.address)) {
                    newCellData = { display: replaceString };
                    if (_.isNumber(replaceResult)) {
                        replaceResult++;
                        newCellData.result = replaceResult;
                        newCellData.display = String(replaceResult);
                    }
                    self.updateCellEntry(cellData.address, newCellData, attrs);
                }
            });

            // notify all change listeners
            this.trigger('change:cells', autofillRange);
            return this;
        };

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Object} range
         *  The logical address of the cell range to be inserted.
         *
         * @param {String} direction
         *  Either 'columns' to move the existing cells horizontally (to the
         *  right or left according to the sheet orientation), or 'rows' to
         *  move the existing cells down.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.insertCells = function (range, direction) {

            var // the addresses of all changed (inserted and moved) cells
                changedCells = insertCells(range, direction === 'columns');

            // notify all change listeners
            triggerChangeCellsEvent(changedCells);
            return this;
        };

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Object} range
         *  The logical address of the cell range to be deleted.
         *
         * @param {String} direction
         *  Either 'columns' to move the remaining cells horizontally (to the
         *  left or right according to the sheet orientation), or 'rows' to
         *  move the remaining cells up.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.deleteCells = function (range, direction) {

            var // the addresses of all changed (deleted and moved) cells
                changedCells = deleteCells(range, direction === 'columns');

            // notify all change listeners
            triggerChangeCellsEvent(changedCells);
            return this;
        };

        /**
         * Updates the value and/or formatting attributes of a cell entry.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [parsedValue]
         *  The parsed value for the cell entry, in the attributes 'display',
         *  'result', 'formula', and 'format'.
         *
         * @param {Object} [attributes]
         *  If specified, additional explicit attributes for the cell.
         *
         * @param {Object} [options]
         *  Additional options for changing formatting attributes. Will be
         *  passed to the method 'CellModel.setCellAttributes()'.
         */
        this.updateCellEntry = function (address, parsedValue, attributes, options) {

            var // the cell entry to be updated
                cellEntry = null;

            // update the cell value
            if (_.isObject(parsedValue)) {
                // do not create new entry, if cell will be cleared
                cellEntry = _.isNull(parsedValue.result) ? getCellEntry(address) : getOrCreateCellEntry(address);
                if (cellEntry) {
                    _(cellEntry).extend(parsedValue);
                }
            }

            // update formatting attributes
            if (_.isObject(attributes)) {
                cellEntry = cellEntry || getOrCreateCellEntry(address);
                cellEntry.setCellAttributes(attributes, options);
            }
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', initHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(map).invoke('destroy');
            defaultEntry.destroy();
            model = view = map = defaultEntry = colCollection = rowCollection = mergeCollection = null;
        });

    } // class CellCollection

    // static constants and methods -------------------------------------------

    /**
     * Standard cell data object for empty cells.
     */
    CellCollection.DEFAULT_CELL_DATA = {
        display: '',
        result: null,
        formula: null,
        format: { cat: 'standard', places: 0, group: false, red: false }
    };

    /**
     * Extracts the cell information from the passed data object.
     *
     * @param {Object} [data]
     *  A cell data object, as received for instance from a server response.
     *
     * @returns {Object}
     *  A descriptor object with the properties 'display', 'result', 'formula',
     *  and 'format'.
     */
    CellCollection.extractCellData = function (data) {
        return {
            display: Utils.getStringOption(data, 'display', ''),
            result: Utils.getOption(data, 'result', null),
            formula: Utils.getStringOption(data, 'formula', null),
            format: _({}).extend(CellCollection.DEFAULT_CELL_DATA.format, Utils.getObjectOption(data, 'format'))
        };
    };

    /**
     * Returns whether the cell with the passed descriptor does not contain any
     * value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  true.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor is a blank cell.
     */
    CellCollection.isBlank = function (cellData) {
        return !cellData || _.isNull(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a numeric cell,
     * including formula cells resulting in a number, including positive and
     * negative infinity.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a numeric result.
     */
    CellCollection.isNumber = function (cellData) {
        return !!cellData && _.isNumber(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a Boolean cell,
     * including formula cells resulting in a Boolean value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a Boolean result.
     */
    CellCollection.isBoolean = function (cellData) {
        return !!cellData && _.isBoolean(cellData.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is an error code
     * cell, including formula cells resulting in an error code.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has an error code as
     *  result.
     */
    CellCollection.isError = function (cellData) {
        return !!cellData && _.isString(cellData.result) && (cellData.result[0] === '#');
    };

    /**
     * Returns whether the cell with the passed descriptor is a text cell,
     * including formula cells resulting in a text value.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a text result.
     */
    CellCollection.isText = function (cellData) {
        return !!cellData && _.isString(cellData.result) && (cellData.result[0] !== '#');
    };

    /**
     * Returns whether the passed cell descriptor will wrap the text contents
     * of a cell.
     */
    function hasWrappingAttributes(cellData) {
        return cellData.attributes.cell.wrapText || (cellData.attributes.cell.alignHor === 'justify');
    }

    /**
     * Returns whether the cell with the passed descriptor wraps its text
     * contents at the left and right cell borders. This happens for all cells
     * with text result value, which either have the cell attribute 'wrapText'
     * set to true, or contain a justifying horizontal alignment.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell wraps its text contents automatically.
     */
    CellCollection.isWrappedText = function (cellData) {
        return CellCollection.isText(cellData) && hasWrappingAttributes(cellData);
    };

    /**
     * Returns whether the cell with the passed descriptor overflows its text
     * contents over the left and right cell borders. This happens for all
     * cells with text result value, which do not have the cell attribute
     * 'wrapText' set to true, and do not contain a justifying horizontal
     * alignment.
     *
     * @param {Object} [cellData]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell contains text that can overflow the cell borders.
     */
    CellCollection.isOverflowText = function (cellData) {
        return CellCollection.isText(cellData) && !hasWrappingAttributes(cellData);
    };

    /**
     * Returns the effective CSS text alignment of the passed cell data,
     * according to the cell result value and the alignment formatting
     * attributes.
     *
     * @param {Object} cellData
     *  The cell descriptor object, as received from the cell collection.
     *
     * @param {Boolean} [editMode=false]
     *  If set to true, formula cells will be left-aligned (used for example in
     *  in-place cell edit mode).
     *
     * @returns {String}
     *  The effective horizontal CSS text alignment ('left', 'center', 'right',
     *  or 'justify').
     */
    CellCollection.getCssTextAlignment = function (cellData, editMode) {

        switch (cellData.attributes.cell.alignHor) {
        case 'left':
        case 'center':
        case 'right':
        case 'justify':
            return cellData.attributes.cell.alignHor;
        case 'auto':
            // formulas in cell edit mode always left aligned, regardless of result type
            if (editMode && _.isString(cellData.formula)) { return 'left'; }
            // errors (centered) or strings (left aligned)
            if (_.isString(cellData.result)) { return (cellData.result[0] === '#' ? 'center' : 'left'); }
            // numbers (right aligned)
            if (_.isNumber(cellData.result)) { return 'right'; }
            // booleans (centered)
            if (_.isBoolean(cellData.result)) { return 'center'; }
            // empty cells (left aligned)
            return 'left';
        }

        Utils.warn('CellCollection.getTextAlignment(): unknown alignment attribute value "' + cellData.attributes.cell.alignHor + '"');
        return 'left';
    };

    /**
     * Returns the CSS text decoration value according to the formatting
     * attributes in the passed cell descriptor.
     *
     * @param {Object} cellData
     *  The cell descriptor object, as received from the cell collection.
     *
     * @returns {String}
     *  The effective CSS text decoration (underline and strike-out mode).
     */
    CellCollection.getCssTextDecoration = function (cellData) {
        var textDecoration = 'none';
        if (cellData.attributes.character.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
        if (cellData.attributes.character.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }
        return textDecoration;
    };

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CellCollection });

});
