/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/paneutils', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // configuration for pane sides (header panes)
        PANE_SIDE_DATA = {
            left:   { nextSide: 'right',  panePos1: 'topLeft',    panePos2: 'bottomLeft',  columns: true,  anchorName: 'anchorLeft'   },
            right:  { nextSide: 'left',   panePos1: 'topRight',   panePos2: 'bottomRight', columns: true,  anchorName: 'anchorRight'  },
            top:    { nextSide: 'bottom', panePos1: 'topLeft',    panePos2: 'topRight',    columns: false, anchorName: 'anchorTop'    },
            bottom: { nextSide: 'top',    panePos1: 'bottomLeft', panePos2: 'bottomRight', columns: false, anchorName: 'anchorBottom' }
        },

        // configuration for pane positions (grid panes)
        PANE_POS_DATA = {
            topLeft:     { colSide: 'left',  rowSide: 'top',    nextColPos: 'topRight',    nextRowPos: 'bottomLeft'  },
            topRight:    { colSide: 'right', rowSide: 'top',    nextColPos: 'topLeft',     nextRowPos: 'bottomRight' },
            bottomLeft:  { colSide: 'left',  rowSide: 'bottom', nextColPos: 'bottomRight', nextRowPos: 'topLeft'     },
            bottomRight: { colSide: 'right', rowSide: 'bottom', nextColPos: 'bottomLeft',  nextRowPos: 'topRight'    }
        };

    // static class PaneUtils =================================================

    var PaneUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed pane side represents a column header pane.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The position 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    PaneUtils.isColumnSide = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].columns;
    };

    /**
     * Returns the name of the scroll anchor view property for the passed pane
     * side.
     *
     * @param {String} paneSide
     *  The pane side identifier (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The name of the scroll anchor sheet view property.
     */
    PaneUtils.getScrollAnchorName = function (paneSide) {
        return PANE_SIDE_DATA[paneSide].anchorName;
    };

    /**
     * Returns the column pane side identifier for the passed grid pane
     * position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    PaneUtils.getColPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].colSide;
    };

    /**
     * Returns the row pane side identifier for the passed grid pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane side 'top' for the pane positions 'topLeft' and 'topRight', or
     *  'bottom' for the pane positions 'bottomLeft' and 'bottomRight'.
     */
    PaneUtils.getRowPaneSide = function (panePos) {
        return PANE_POS_DATA[panePos].rowSide;
    };

    /**
     * Returns the pane position of the horizontal neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the horizontal neighbor (for example, returns
     *  'bottomLeft' for the pane position 'bottomRight').
     */
    PaneUtils.getNextColPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextColPos;
    };

    /**
     * Returns the pane position of the vertical neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The pane position (one of 'topLeft', 'topRight', 'bottomLeft', or
     *  'bottomRight').
     *
     * @returns {String}
     *  The pane position of the vertical neighbor (for example, returns
     *  'topRight' for the pane position 'bottomRight').
     */
    PaneUtils.getNextRowPanePos = function (panePos) {
        return PANE_POS_DATA[panePos].nextRowPos;
    };

    /**
     * Returns the pane position identifier according to the passed pane side
     * identifiers.
     *
     * @param {String} colPaneSide
     *  The column pane side ('left' or 'right').
     *
     * @param rowPaneSide
     *  The row pane side ('top' or 'bottom').
     *
     * @returns {String}
     *  The pane position for the passed pane side identifiers.
     */
    PaneUtils.getPanePos = function (colPaneSide, rowPaneSide) {
        return rowPaneSide + Utils.capitalize(colPaneSide);
    };

    /**
     * Returns the pane position nearest to the specified source pane position,
     * but matching the specified target pane side.
     *
     * @param {String} panePos
     *  The source pane position (one of 'topLeft', 'topRight', 'bottomLeft',
     *  or 'bottomRight').
     *
     * @param {String} paneSide
     *  The target pane side (one of 'left', 'right', 'top', or 'bottom').
     *
     * @returns {String}
     *  The grid pane position of the nearest neighbor, matching the specified
     *  pane side (for example, returns 'topRight' for the source pane position
     *  'bottomRight' and target pane side 'top').
     */
    PaneUtils.getNextPanePos = function (panePos, paneSide) {
        var panePosData = PANE_POS_DATA[panePos],
            paneSideData = PANE_SIDE_DATA[paneSide];
        return ((paneSide === panePosData.colSide) || (paneSide === panePosData.rowSide)) ? panePos :
            (paneSideData.columns ? panePosData.nextColPos : panePosData.nextRowPos);
    };

    // CSS helper functions ---------------------------------------------------

    /**
     * Returns the CSS position and size attributes for the passed rectangle in
     * pixels, as HTML mark-up value for the 'style' element attribute.
     *
     * @param {Object} rectangle
     *  The rectangle to be converted to CSS position and size attributes, in
     *  the properties 'left', 'top', 'width', and 'height'.
     *
     * @returns {String}
     *  The value for the 'style' element attribute in HTML mark-up text.
     */
    PaneUtils.getRectangleStyleMarkup = function (rectangle) {
        return 'left:' + rectangle.left + 'px;top:' + rectangle.top + 'px;width:' + rectangle.width + 'px;height:' + rectangle.height + 'px;';
    };

    // tracking events --------------------------------------------------------

    /**
     * Returns the GUI selection mode according to the modifier keys in the
     * passed GUI event.
     *
     * @param {jQuery.Event} event
     *  The event object received by any jQuery event handlers (keyboard,
     *  mouse, tracking).
     *
     * @returns {String}
     *  The selection mode according to the keyboard modifier keys:
     *  - 'select': Standard selection without modifier keys.
     *  - 'append': Append new range to current selection (CTRL/META key).
     *  - 'extend': Extend current active range (SHIFT key).
     */
    PaneUtils.getSelectionMode = function (event) {
        if (!event.shiftKey && (event.ctrlKey || event.metaKey)) { return 'append'; }
        if (event.shiftKey && !event.ctrlKey && !event.metaKey) { return 'extend'; }
        return 'select';
    };

    // exports ================================================================

    return PaneUtils;

});
