/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/numberformatter', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class NumberFormatter ==================================================

    function NumberFormatter(app, fontCollection) {

        var // the exponent character
            EXP = 'E';

        // private methods ----------------------------------------------------

        /**
         * Returns the decimal string representation of the passed value.
         *
         * @param {Number} value
         *  The number whose decimal string representation will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed number
         *  will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} value
         *      The resulting rounded value, as number.
         *  - {String} text
         *      The complete decimal string representation of the number.
         *  - {String} abs
         *      The decimal string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         */
        function formatAsDecimal(value, digits) {

            var // the absolute value, rounded to the specified number of digits after decimal separator
                round = Utils.round(Math.abs(value), Math.pow(10, -digits)),
                // the integral part of the absolute value
                int = Math.floor(round),
                // the fractional part of the absolute value
                frac = Math.round((round - int) * Math.pow(10, digits)),
                // the text representation of the integral part
                intText = String(int),
                // the text representation of the significant fractional part (without leading zeros)
                fracText = String(frac),
                // the zeros between decimal separator and significant fractional part
                fillText = null,
                // the result returned by this method
                result = { abs: intText, digits: (int === 0) ? 0 : intText.length };

            // add the rounded value to the result
            result.value = (value < 0) ? -round : round;

            // no fraction available: do not add decimal separator
            if (frac > 0) {

                // leading zeros for the fractional part
                fillText = Utils.repeatString('0', digits - fracText.length);
                // remove trailing zeros
                fracText = fracText.replace(/0+$/, '');
                // concatenate integer and fractional part, remove trailing zeros
                result.abs += app.getDecimalSeparator() + fillText + fracText;
                // update number of significant digits
                if (int > 0) { result.digits += fillText.length; }
                result.digits += fracText.length;
            }

            // add final string representation (with sign)
            result.text = (value < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Returns the scientific string representation of the passed value.
         *
         * @param {Number} mant
         *  The mantissa of the number whose scientific string representation
         *  will be returned. Must be in the half-open intervals +-[1, 10).
         *
         * @param {Number} exp
         *  The exponent of the number whose scientific string representation
         *  will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed
         *  mantissa will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} value
         *      The resulting rounded value, as number.
         *  - {String} text
         *      The complete scientific string representation of the number.
         *  - {String} abs
         *      The scientific string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      mantissa.
         */
        function formatAsScientific(mant, exp, digits) {

            var // get the formatted mantissa
                result = formatAsDecimal(mant, digits);

            // if mantissa has been rounded up to 10, increase the exponent
            if (Math.abs(result.value) === 10) {
                result.value /= 10;
                result.abs = '1';
                exp += 1;
            }

            // update the rounded value in the result
            result.value *= Math.pow(10, exp);

            // add the exponent (at least two digits, but may be three digits)
            result.abs += EXP + ((exp < 0) ? '-' : '+');
            exp = Math.abs(exp);
            if (exp < 10) { result.abs += '0'; }
            result.abs += String(exp);

            // add final string representation (with sign)
            result.text = (mant < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Formats a number with increasing number of digits following the
         * decimal separator, as long as the resulting number fits according to
         * the provided result validator.
         *
         * @param {Number} minDigits
         *  The minimum number of digits after the decimal separator.
         *
         * @param {Number} maxDigits
         *  The maximum number of digits after the decimal separator.
         *
         * @param {Function} formatCallback
         *  The callback function that receives the current number of digits
         *  following the decimal separator, and must return a result object
         *  containing the properties 'value', 'text', 'abs', and 'digits'. See
         *  methods formatAsDecimal() and formatAsScientific() above for a
         *  detailed description about this result object.
         *
         * @param {Function} validateCallback
         *  The callback function that checks the current result object it
         *  receives. Must return a Boolean value specifying whether the result
         *  object is valid.
         *
         * @returns {Object}
         *  The last matching result object returned by the callback function.
         */
        function findLongestFormat(minDigits, maxDigits, formatCallback, validateCallback) {

            var currDigits = minDigits,
                currResult = null,
                lastResult = null;

            do {
                currResult = formatCallback(currDigits);
                if (!validateCallback(currResult)) { break; }
                lastResult = currResult;
                currDigits += 1;
            } while (currDigits <= maxDigits);

            return lastResult;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the display string of the passed number formatted with the
         * 'Standard' number format, fitting into the specified text width.
         *
         * @param {Number} value
         *  The number whose string representation in 'Standard' number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the text width.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Number} [maxWidth]
         *  The maximum width of the resulting string representation, in
         *  pixels. If omitted, the number will be formatted without a pixel
         *  width restriction (but still with the built-in restriction to a
         *  specific maximum number of text characters).
         *
         * @returns {Object|Null}
         *  A result object containing the formatted number and other data, in
         *  the following properties:
         *  - {String} text
         *      The resulting display text of the number, formatted with the
         *      'Standard' number format.
         *  - {Number} width
         *      The width of the display text, in pixels.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         *  If the passed maximum pixel width is too small to take the passed
         *  number in either decimal or scientific notation, the null value
         *  will be returned instead.
         */
        this.formatAsStandard = function (value, maxLength, attributes, maxWidth) {

            // Bug 31125: This method uses a local cache containing the widths of all
            // characters that will appear in a number formatted with the 'Standard'
            // number format, and will add up these widths to calculate the total
            // width of a formatted number. This works twice as fast as simply using
            // the method FontCollection.getTextWidth() that inserts the text into a
            // DOM element and reads its resulting width. A very little drawback is
            // that the result might not be correct to the last pixel, as it misses
            // advanced font features such as kerning, but this should not be a
            // problem for numbers as hardly any font uses kerning to render digits.

            var // exponent (will be -INF for value zero)
                exp = Math.floor(Math.log(Math.abs(value)) / Math.LN10),
                // mantissa in the half-open intervals +-[1, 10) (will be NaN for zero)
                mant = value / Math.pow(10, exp),
                // whether to restrict the result to a pixel width
                useMaxWidth = _.isNumber(maxWidth),
                // cached widths of all characters that can appear in the display string
                charWidths = getCharacterWidths(),
                // the maximum pixel width of a digit
                digitWidth = useMaxWidth ? Math.max(1, fontCollection.getDigitWidth(attributes)) : null,
                // the pixel width of the decimal separator character
                decWidth = useMaxWidth ? charWidths[app.getDecimalSeparator()] : null,
                // the pixel width of the exponent with two digits
                exp2Width = useMaxWidth ? (charWidths[EXP] + charWidths['+'] + 2 * digitWidth) : null,
                // available, minimum, and maximum number of digits for the passed width
                availDigits = 0, minDigits = 0, maxDigits = 0,
                // the formatter callback function (decimal or scientific)
                formatterFunc = null,
                // the validator callback function (depending on pixel mode)
                validateFunc = null,
                // conversion result in decimal notation
                decResult = null,
                // conversion result in scientific notation
                scientResult = null,
                // the result to be used (either decimal or scientific)
                finalResult = null;

            // returns an object with the widths of all characters that can appear in a result
            function getCharacterWidths() {
                return fontCollection.getCustomFontMetrics(attributes, function () {
                    return _('0123456789+-' + EXP + app.getDecimalSeparator()).reduce(function (memo, char) {
                        memo[char] = fontCollection.getTextWidth(char, attributes);
                        return memo;
                    }, {});
                }, 'standard.widths');
            }

            // returns the width of the passed formatted number
            function getTextWidth(text) {
                var textWidth = _(text).reduce(function (memo, char) { return memo + charWidths[char]; }, 0);
                if (_.isNumber(textWidth)) { return textWidth; }
                Utils.warn('NumberFormatter.getTextWidth(): unexpected character in formatted number: ' + text);
                return fontCollection.getTextWidth(text, attributes);
            }

            // validates the passed result by character count only
            function validateLength(result) {
                return result.abs.length <= maxLength;
            }

            // validates the passed result by character count and pixel width
            function validateLengthAndWidth(result) {
                return (result.abs.length <= maxLength) && (getTextWidth(result.text) <= maxWidth);
            }

            // set the the validator callback function, depending on pixel mode
            validateFunc = useMaxWidth ? validateLengthAndWidth : validateLength;

            // find the best decimal representation
            if (exp < 2 - maxLength) {
                // start with zero representation for numbers equal to or very close to zero
                decResult = { value: value, abs: '0', text: (value < 0) ? '-0' : '0', digits: 0 };
                if (!validateFunc(decResult)) { decResult = null; }
            } else if (exp <= maxLength - 1) {
                // try to find the best explicit representation in a specific interval of exponents
                availDigits = useMaxWidth ? Math.floor((maxWidth - decWidth) / digitWidth) : maxLength;
                minDigits = Math.max(0, -(exp + 2));
                maxDigits = Math.min(16, maxLength - Math.max(2, exp), availDigits - Math.max(0, exp) - 1);
                formatterFunc = _.bind(formatAsDecimal, null, value);
                decResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // find the best scientific representation (exclude zero)
            if (_.isFinite(exp) && ((exp <= -4) || (exp >= 4))) {
                availDigits = maxLength - 5;
                if (useMaxWidth) { availDigits = Math.min(availDigits, Math.floor((maxWidth - decWidth - exp2Width) / digitWidth)); }
                minDigits = Math.max(0, availDigits - 2);
                maxDigits = Math.max(0, availDigits - 1);
                formatterFunc = _.bind(formatAsScientific, null, mant, exp);
                scientResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // prefer decimal notation, if it shows at least as much significant digits as the scientific notation
            finalResult = (decResult && (!scientResult || (decResult.digits >= scientResult.digits))) ? decResult : scientResult;
            return finalResult ? { text: finalResult.text, width: getTextWidth(finalResult.text), digits: finalResult.digits } : null;
        };

    } // class NumberFormatter

    // exports ================================================================

    return NumberFormatter;

});
