/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/mergecollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, ModelObject, SheetUtils) {

    'use strict';

    // class MergeCollection ==================================================

    /**
     * Collects information about all merged cell ranges of a single sheet in a
     * spreadsheet document.
     *
     * Triggers the following events:
     * - 'insert:merged'
     *      After new merged ranges have been inserted into the collection.
     *      Event handlers receive an array of range addresses.
     * - 'delete:merged'
     *      After merged ranges have been deleted from the collection. Event
     *      handlers receive an array of range addresses.
     * - 'change:merged'
     *      After merged ranges in this collection have been modified (moved or
     *      resized). Event handlers receive an array of range addresses.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function MergeCollection(app, sheetModel) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = app.getModel(),

            // array of merged cells, described as range addresses
            // - contains ranges as objects with 'start' and 'end'
            mergedRanges = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * triggers the specified event, if the passed array of range addresses
         * is not empty.
         */
        function triggerEvent(type, ranges) {
            if (ranges.length > 0) {
                self.trigger(type, ranges);
            }
        }

        /**
         * Calculating new merged cell ranges after inserting columns or rows.
         */
        function insertInterval(interval, columns) {

            var // all modified merged ranges
                changedRanges = [],
                // the length of the interval
                size = SheetUtils.getIntervalSize(interval),
                // the index of the cell position to be evaluated
                index = columns ? 0 : 1;

            // iterating over all merged ranges
            _(mergedRanges).each(function (range) {

                if (interval.first > range.end[index]) {
                    // do nothing, new columns/row behind/below merged cell
                } else if (interval.first <= range.start[index]) {
                    // case: shifting cell completely to the right/bottom
                    changedRanges.push(_.copy(range, true)); // old position
                    range.start[index] += size;
                    range.end[index] += size;
                    changedRanges.push(_.copy(range, true));  // new position

                } else {
                    // case: expanding cell
                    range.end[index] += size;
                    changedRanges.push(_.copy(range, true));  // new position (includes old position)
                }

            });

            // notify listeners (with option 'implicit' due do implicit update while inserting columns/rows)
            triggerEvent('change:merged', changedRanges, { implicit: true });
        }

        /**
         * Calculating new merged cell ranges after deleting columns or rows.
         */
        function deleteInterval(interval, columns) {

            var // all modified merged ranges
                changedRanges = [],
                // all deleted merged ranges
                deletedRanges = [],
                // the length of the interval
                intervalSize = SheetUtils.getIntervalSize(interval),
                // the intersection of the merged cell range and the specified interval
                intersectionInterval = null,
                // function to return number of columns/rows of a range
                getCount1 = columns ? SheetUtils.getColCount : SheetUtils.getRowCount,
                // function to return number of columns/rows in opposite direction
                getCount2 = columns ? SheetUtils.getRowCount : SheetUtils.getColCount,
                // the index of the cell position to be evaluated
                addrIndex = columns ? 0 : 1;

            // iterating over all merged ranges
            Utils.iterateArray(mergedRanges, function (range, index) {

                var size = intervalSize;  // refreshing for every range

                if (interval.first > range.end[addrIndex]) {
                    // do nothing, removed columns/rows behind/below merged cell
                } else if (interval.last < range.start[addrIndex]) {
                    // case: shifting cell completely to the top/left
                    changedRanges.push(_.copy(range, true)); // old position
                    range.start[addrIndex] -= size;
                    range.end[addrIndex] -= size;
                    changedRanges.push(_.copy(range, true));  // new position
                } else {
                    // case: shrinking cell or remove merged cell from list
                    // -> calculating the number of rows/columns that will be removed from merged cell
                    intersectionInterval = SheetUtils.getIntersectionInterval({ first: range.start[addrIndex], last: range.end[addrIndex] }, interval);
                    if (intersectionInterval) {
                        size = SheetUtils.getIntervalSize(intersectionInterval);

                        // checking remaining merged cell -> completely removed or simple 1x1 cell remaining
                        if ((getCount1(range) === size) || ((getCount1(range) === size + 1) && (getCount2(range) === 1))) {
                            // this is no longer a merged cell -> removing from list of merged cells
                            deletedRanges.push(_.copy(range, true));
                            mergedRanges.splice(index, 1);
                        } else {
                            changedRanges.push(_.copy(range, true));  // old range position (includes not always new position)
                            // decreasing size of merged cell range
                            range.end[addrIndex] -= size;

                            // there might be columns/rows left/top of the merged cell, that will be removed, too
                            // -> shifting remaining merged cell completely to left/top
                            if (range.start[addrIndex] > interval.first) {
                                size = range.start[addrIndex] - interval.first;
                                range.start[addrIndex] -= size;
                                range.end[addrIndex] -= size;
                            }

                            changedRanges.push(_.copy(range, true));  // new range cell
                        }
                    }
                }
            }, { reverse: true });

            // notify listeners (with option 'implicit' due do implicit update while inserting columns/rows)
            triggerEvent('delete:merged', deletedRanges, { implicit: true });
            triggerEvent('change:merged', changedRanges, { implicit: true });
        }

        // methods ------------------------------------------------------------

        /**
         * Creates and returns a clone of this collection instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model that will contain the new cloned collection.
         *
         * @returns {MergeCollection}
         *  A complete clone of this collection, associated to the specified
         *  sheet model.
         */
        this.clone = function (sheetModel) {
            return new MergeCollection(app, sheetModel, mergedRanges);
        };

        /**
         * Returns, whether the cell at the specified logical address is the upper
         * left corner of a merged cell range.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified logical address is the upper
         * left corner of a merged cell range.
         */
        this.isReferenceCell = function (address) {
            return this.getMergedRange(address, 'reference') !== null;
        };

        /**
         * Returns, whether the cell at the specified logical address is a
         * hidden cell inside a merged cell range.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified logical address is a
         *  hidden cell inside a merged cell range.
         */
        this.isHiddenCell = function (address) {
            return this.getMergedRange(address, 'hidden') !== null;
        };

        /**
         * Returns, whether the cell at the specified logical address is part
         * of a merged cell. It can be the reference cell or a hidden cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell is part of a merged cell range.
         */
        this.isMergedCell = function (address) {
            return this.getMergedRange(address, 'all') !== null;
        };

        /**
         * Returns, whether a specific range is in the list of the
         * merged cells 'mergedRanges'. For the comparison it is
         * necessary, that start and end position of the ranges are
         * identical.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether the specified range can be found in the list
         *  of merged cell ranges.
         */
        this.isMergedRange = function (range) {
            return _(mergedRanges).any(function (mergedRange) {
                return _.isEqual(mergedRange, range);
            });
        };

        /**
         * Returns the logical range address of a merged cell range for a
         * specified cell address. If the cell address is not included into
         * a merged cell range, 'null' is returned.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @param {String} [type='all']
         *  The address type of the specified address. It can be an arbitrary
         *  cell inside a merged cell range (type === 'all'), or it can be
         *  restricted to the upper left reference cell (type === 'reference')
         *  or to a hidden cell (type === 'hidden'). A hidden cell is any cell
         *  inside the merged cell range, that is not the upper left reference
         *  cell. If this address type is not specified, or if it is not one of
         *  the three types 'all', 'reference' or 'hidden', it defaults to
         *  'all'.
         *
         * @returns {Object|Null}
         *  The logical range address of a merged range that covers the cell
         *  address passed to this method; or null, if no merged range exists
         *  at the address.
         */
        this.getMergedRange = function (address, type) {

            var // the searched merged range corresponding to the address
                mergedRange = null;

            switch (type) {
            case 'reference':
                mergedRange = _(mergedRanges).find(function (range) {
                    return _.isEqual(range.start, address);
                });
                break;
            case 'hidden':
                mergedRange = _(mergedRanges).find(function (range) {
                    return SheetUtils.rangeContainsCell(range, address) && (! _.isEqual(range.start, address));
                });
                break;
            default:
                mergedRange = _(mergedRanges).find(function (range) {
                    return SheetUtils.rangeContainsCell(range, address);
                });
            }

            return mergedRange ? _.copy(mergedRange, true) : null;
        };

        /**
         * Returns the logical range addresses of all merged cell ranges that
         * overlap with one of the passed ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array with cell
         *  range addresses.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.reference=false]
         *      If set to true, returns merged ranges whose reference cell is
         *      contained in the passed ranges. Otherwise, returns all merged
         *      ranges that overlap the passed ranges with any cell.
         *
         * @returns {Array}
         *  The logical range addresses of all merged ranges that overlap with
         *  one of the passed ranges.
         */
        this.getMergedRanges = function (ranges, options) {

            var // whether to return merged ranges by reference cell
                reference = Utils.getBooleanOption(options, 'reference', false),
                // the merged ranges not matching any passed range yet
                pendingRanges = _.clone(mergedRanges),
                // the resulting merged ranges
                resultRanges = [];

            // process all passed ranges
            _.chain(ranges).getArray().any(function (range) {

                // iterate backwards through pendingRanges to be able to remove array elements
                Utils.iterateArray(pendingRanges, function (mergedRange, index) {

                    // move overlapping merged range from pendingRanges to resultRanges
                    if (SheetUtils.rangesOverlap(range, mergedRange)) {
                        if (!reference || SheetUtils.rangeContainsCell(range, mergedRange.start)) {
                            resultRanges.push(mergedRange);
                        }
                        pendingRanges.splice(index, 1);
                    }
                }, { reverse: true });

                // early exit of the _.any() loop, if all merged ranges will be returned
                return pendingRanges.length === 0;
            });

            return resultRanges;
        };

        /**
         * Returns the column intervals of all merged ranges in a single row.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Array}
         *  A sorted array of column intervals of all merged ranges covering
         *  the specified row. The column intervals of several adjacent merged
         *  ranges will be joined into a single column interval.
         */
        this.getMergedColIntervals = function (row) {
            return SheetUtils.getColIntervals(this.getMergedRanges(model.makeRowRange(row)));
        };

        /**
         * Returns the row intervals of all merged ranges in a single column.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @returns {Array}
         *  A sorted array of row intervals of all merged ranges covering the
         *  specified column. The row intervals of several adjacent merged
         *  ranges will be joined into a single row interval.
         */
        this.getMergedRowIntervals = function (col) {
            return SheetUtils.getRowIntervals(this.getMergedRanges(model.makeColRange(col)));
        };

        /**
         * Returns whether the passed cell range contains or overlaps with at
         * least one merged cell range.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the passed cell range contains or overlaps with at least
         *  one merged cell range.
         */
        this.rangeOverlapsMergedRange = function (range) {
            return _(mergedRanges).any(function (mergedRange) {
                return SheetUtils.rangesOverlap(range, mergedRange);
            });
        };

        /**
         * Returns whether any of the passed cell ranges contains or overlaps
         * with at least one merged cell range.
         *
         * @param {Array} ranges
         *  Array of logical range addresses.
         *
         * @returns {Boolean}
         *  Whether any of the passed cell ranges contains or overlaps with at
         *  least one merged cell range.
         */
        this.rangesOverlapAnyMergedRange = function (ranges) {
            return _(ranges).any(_.bind(this.rangeOverlapsMergedRange, this));
        };

        /**
         * Expands the passed range address, so that it will include all merged
         * ranges it is currently covering partly.
         *
         * @param {Object} range
         *  The logical address of a cell range that will be expanded to the
         *  merged ranges contained in this collection.
         *
         * @returns {Object}
         *  The logical address of the expanded range.
         */
        this.expandRangeToMergedRanges = function (range) {

            var // flat copy of the merged ranges, to prevent processing them multiple times
                pendingRanges = _.clone(mergedRanges),
                // the length of the pending ranges array
                pendingLength = pendingRanges.length;

            // loop iterator function must be defined outside the do-while loop
            function expandToMergedRange(mergedRange, index) {
                // process all merged ranges that overlap the current range
                if (SheetUtils.rangesOverlap(range, mergedRange)) {
                    range = SheetUtils.getBoundingRange(range, mergedRange);
                    pendingRanges.splice(index, 1);
                }
            }

            // loop as long as the range will still be expanded with other merged ranges
            if (pendingRanges.length > 0) {
                do {
                    // store the current length of the pending ranges array (checking length afterwards)
                    pendingLength = pendingRanges.length;
                    // iterate backwards to be able to remove an element from the array
                    Utils.iterateArray(pendingRanges, expandToMergedRange, { reverse: true });
                } while (pendingRanges.length < pendingLength);
            }

            // return the fully expanded range
            return range;
        };

        /**
         * Shrinks the passed range address, so that it will not include any
         * merged ranges it is currently covering partly.
         *
         * @param {Object} range
         *  The logical address of a cell range that will be shrunk until it
         *  does not partly contain any merged ranges.
         *
         * @param {String} direction
         *  The direction in which the passed range will be shrunk. Supported
         *  values are 'columns' to shrink the range at its left and right
         *  border, or 'rows' to shrink the range at its top and bottom border.
         *
         * @returns {Object|Null}
         *  The logical address of the shrunk range; or null, if no valid range
         *  was left after shrinking (the range has been shrunk to zero
         *  columns/rows).
         */
        this.shrinkRangeFromMergedRanges = function (range, direction) {

            var // flat copy of the merged ranges, to prevent processing them multiple times
                pendingRanges = _.clone(mergedRanges),
                // the length of the pending ranges array
                pendingLength = pendingRanges.length,
                // address array index for passed direction
                addressIndex = (direction === 'columns') ? 0 : 1;

            // loop iterator function must be defined outside the do-while loop
            function shrinkFromMergedRange(mergedRange, index) {

                // do not process merged ranges that are fully contained in the current range
                if (SheetUtils.rangeContainsRange(range, mergedRange)) { return; }
                // remove the merged range from the pending array
                pendingRanges.splice(index, 1);
                // nothing to do, if merged range is completely outside
                if (!SheetUtils.rangesOverlap(range, mergedRange)) { return; }

                // shrink leading border of the range
                if ((mergedRange.start[addressIndex] < range.start[addressIndex]) && (range.start[addressIndex] <= mergedRange.end[addressIndex])) {
                    range.start[addressIndex] = mergedRange.end[addressIndex] + 1;
                }
                // shrink trailing border of the range
                if ((mergedRange.start[addressIndex] <= range.end[addressIndex]) && (range.end[addressIndex] < mergedRange.end[addressIndex])) {
                    range.end[addressIndex] = mergedRange.start[addressIndex] - 1;
                }

                // check if the range has been shrunk to null size
                if (range.start[addressIndex] > range.end[addressIndex]) {
                    range = null;
                    return Utils.BREAK;
                }
            }

            // loop as long as the range will still be shrunk to exclude other merged ranges
            if (pendingRanges.length > 0) {
                do {
                    // store the current length of the pending ranges array (checking length afterwards)
                    pendingLength = pendingRanges.length;
                    // iterate backwards to be able to remove an element from the array
                    Utils.iterateArray(pendingRanges, shrinkFromMergedRange, { reverse: true });
                } while (range && (pendingRanges.length < pendingLength));
            }

            // return the shrunk range
            return range;
        };

        /**
         * Merges the specified cell range, or removes any merged ranges from
         * that range.
         *
         * @param {Object} range
         *  The logical address of the range to be merged or unmerged.
         *
         * @param {String} type
         *  The merge type. Must be one of: 'merge', 'horizontal', 'vertical',
         *  or 'unmerge'.
         *
         * @returns {MergeCollection}
         *  A reference to this instance.
         */
        this.mergeRange = function (range, type) {

            var // all existing merged ranges deleted from the collection
                deletedRanges = [],
                // all merged ranges added to the collection
                insertedRanges = [];

            // inserts the passed range into the collection, and into the 'insertedRanges' array
            function insertMergedRange(range) {

                var // the array index of the passed range in the 'deletedRanges' array
                    deletedIndex = Utils.findFirstIndex(deletedRanges, _.bind(_.isEqual, _, range));

                // if the range has been deleted, remove it from the array, otherwise
                // insert it into the 'insertedRanges' array (this prevents to notify
                // ranges that have been unmerged *and* merged)
                if (deletedIndex >= 0) {
                    deletedRanges.splice(deletedIndex, 1);
                } else {
                    insertedRanges.push(range);
                }

                // push a copy of the range into the collection
                mergedRanges.push(_.copy(range, true));
            }

            // always remove all merged ranges that overlap with the passed range,
            // regardless of passed type (needed to merge over merged ranges)
            mergedRanges = _(mergedRanges).reject(function (mergedRange) {
                var overlaps = SheetUtils.rangesOverlap(range, mergedRange);
                if (overlaps) { deletedRanges.push(mergedRange); }
                return overlaps;
            });

            // handle the different merge types
            switch (type) {
            case 'merge':
                if (SheetUtils.getCellCount(range) > 1) {
                    insertMergedRange(range);
                }
                break;
            case 'horizontal':
                if (SheetUtils.getColCount(range) > 1) {
                    Utils.iterateRange(range.start[1], range.end[1] + 1, function (row) {
                        insertMergedRange({ start: [range.start[0], row], end: [range.end[0], row] });
                    });
                }
                break;
            case 'vertical':
                if (SheetUtils.getRowCount(range) > 1) {
                    Utils.iterateRange(range.start[0], range.end[0] + 1, function (col) {
                        insertMergedRange({ start: [col, range.start[1]], end: [col, range.end[1]] });
                    });
                }
                break;
            case 'unmerge':
                // all merged ranges have been removed already, see above
                break;
            default:
                Utils.warn('MergeCollection.mergeRange(): unknown merge type "' + type + '"');
            }

            // trigger change events with array of changed merged ranges
            triggerEvent('delete:merged', deletedRanges);
            triggerEvent('insert:merged', insertedRanges);

            return this;
        };

        // initialization -----------------------------------------------------

        // recalculate merged cells after inserting or deleting columns or rows
        sheetModel.getColCollection().on({
            'insert:entries': function (event, interval) { insertInterval(interval, true); },
            'delete:entries': function (event, interval) { deleteInterval(interval, true); }
        });
        sheetModel.getRowCollection().on({
            'insert:entries': function (event, interval) { insertInterval(interval, false); },
            'delete:entries': function (event, interval) { deleteInterval(interval, false); }
        });

        // clone merged ranges passed as hidden argument to the c'tor (used by the clone() method)
        if (_.isArray(arguments[MergeCollection.length])) {
            mergedRanges = _.copy(arguments[MergeCollection.length], true);
        }

    } // class MergeCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: MergeCollection });

});
