/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/insertvalidation',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject'
    ], function (Utils, ModelObject) {

    'use strict';

    // class InsertValidation ==================================================

    /**
     * Modifies data validation settings for a range of cells in a specific sheet.
     * Data validation can be used to restrict the values allowed to be entered in the cells,
     * and to allow to show tool tips and messages to the user.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     */
    function InsertValidation(app) {

        //global variables
        var restrictedCellList = [],            //array of restricted cells
            valuesRestrictedCell = [];          //array of valid values for a restricted cell

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /** - no private methods defined here - */

        // methods ------------------------------------------------------------

        /**
         * Checks whether active cell is restricted.
         *
         * @param {Number[]} activeCellAddress
         *  The logical address of the active cell.
         *
         * @returns {Boolean}
         *  Returns "true" if active cell is a restricted cell.
         */
        this.isRestrictedCell = function (activeCellAddress) {

            for (var i = 0; i < restrictedCellList.length; i++) {

                if ((_.isEqual(restrictedCellList[i].range, activeCellAddress)) && (restrictedCellList[i].type === 'source')) {
                    return true; //active is restricted cell & type equals "source"
                }
            }
            return false;
        };

        /**
         * Stores collection of a restricted cell.
         *
         * @param {Object} range
         *  The logical address of the restricted range.
         *
         * @param {String} value1
         *  The formula expression that evaluates to a value or cell range address,
         *  depending on the validation type and comparison operator,
         *  without the leading equality sign.
         *
         * @param {String} type
         * The type of the validation to be performed for the cells.
         */
        this.storeInsertValidationCollection = function (range, value1, type) {

            //fill array of objects with for every restricted cell
            restrictedCellList.push({'range': range.start, 'value1': value1, 'type': type});
        };

        /**
         * Gets valid data collection for a restricted cell.
         *
         * @param {Number[]} activeCellAddress
         *  The logical address of the active cell.
         *
         * @returns  {Array} valuesRestrictedCell
         * An array of objects containing values for a restricted cell.
         */
        this.getValuesForMenu = function (restrictedCellAddress, activeSheet) {

            var def = $.Deferred(),//ajax fix
                getValueFromKey;

            //get the right value1 for restricted cell, and pass it to ajax call
            for (var i = 0; i < restrictedCellList.length; i++) {
                if (_.isEqual(restrictedCellList[i].range, restrictedCellAddress)) {
                    getValueFromKey = [restrictedCellList[i].value1]; //format is ["value1"]
                    break;
                }
            }

            //start-of ajax call
            app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'updateview',
                        requestdata: JSON.stringify({
                            sheet: activeSheet,
                            areas: getValueFromKey
                        })
                    }
                })
                .done(function (response) {

                    valuesRestrictedCell = []; //clear array before new input for a new restricted cell
                    for (var i = 0; i < response.areas[0].length; i++) {

                        valuesRestrictedCell.push({itemValue: response.areas[0][i].result});
                    }

                    //ajax fix
                    def.resolve(valuesRestrictedCell);
                })
                .fail(function () {
                    Utils.error('SpreadshetView.initLocaleData(): could not download locale data for "' + Utils.LOCALE + '"');
                    def.reject('Error ajax call!');
                });
                //end-of ajax call

            //ajax fix
            def.always(function (result) {
                Utils.log('result: ' + JSON.stringify(result));
            });
            return def.promise();
        };

        // initialization -----------------------------------------------------

        /** - no initialization defined here - */

    } // End-of class InsertValidation

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: InsertValidation });

});
