/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mixedborder',
     'io.ox/office/editframework/app/editcontroller',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/controller/drawingcontroller',
     'io.ox/office/spreadsheet/view/hyperlink',
     'io.ox/office/spreadsheet/utils/unitcontroller'
    ], function (Utils, Border, MixedBorder, EditController, SheetUtils, DrawingController, Hyperlink, UnitController) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function SpreadsheetController(app) {

        var // self reference
            self = this,

            // the spreadsheet model
            model = null,

            // the spreadsheet view
            view = null,

            // all the little controller items
            items = {

                'document/search/search': {
                    // enabled in read-only mode
                    set: function (searchQuery) { view.search(searchQuery); },
                    focus: 'never'
                },

                'document/search/nextResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            view.searchIterate('next');
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }],
                    focus: 'never' // do not grab focus after setter
                },

                'document/search/previousResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            view.searchIterate('prev');
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }],
                    focus: 'never'
                },

                'document/search/replaceAll': {
                    parent: 'document/editable',
                    set: function () { view.searchReplace({all: true}); },
                    focus: 'never' // do not grab focus after setter
                },

                'document/search/replaceSelected': {
                    parent: 'document/editable',
                    set: function () { view.searchReplace(); },
                    focus: 'never'
                },

                'document/search/close': {
                    // enabled in read-only mode
                    set: function () { view.searchClear({clearInputs: true}); }
                },

                // view -------------------------------------------------------

                'view/sheet/active': {
                    get: function () { return view.getActiveSheet({ visible: true }); },
                    set: function (index) { view.setActiveSheet(index, { visible: true }); }
                },

                'view/sheet/prev': {
                    enable: function () { return view.getActiveSheet({ visible: true }) > 0; },
                    set: function () { view.activatePreviousSheet(); },
                    shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
                },

                'view/sheet/next': {
                    enable: function () { return view.getActiveSheet({ visible: true }) + 1 < view.getVisibleSheetCount(); },
                    set: function () { view.activateNextSheet(); },
                    shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
                },

                'view/editable/cell': {
                    parent: 'document/editable',
                    enable: function () {
                        return !view.hasDrawingSelection() &&
                            !(view.getActiveSheetModel().isProtected() && view.isSelectionLocked());
                    }
                },

                'view/activecell/attributes': {
                    parent: 'view/editable/cell',
                    get: function () { return view.getCellContents().attributes; }
                },

                'view/subtotals/type': {
                    get: function () { return app.getUserSettingsValue('subtotalType', 'sum'); },
                    set: function (subtotalType) { app.setUserSettingsValue('subtotalType', subtotalType); }
                },

                'view/split/dynamic': {
                    get: function () { return view.hasDynamicSplit(); },
                    set: function (state) { view.setDynamicSplit(state); }
                },

                'view/split/frozen': {
                    get: function () { return view.hasFrozenSplit(); },
                    set: function (value) {
                        if (_.isBoolean(value)) {
                            view.setFrozenSplit(value);
                        } else if (value === 'toggle') {
                            view.setFrozenSplit(!this.getValue());
                        } else if (_.isObject(value)) {
                            view.setFixedFrozenSplit(value.cols, value.rows);
                        }
                    }
                },

                'view/zoom/dec': {
                    enable: function () { return view.getZoom() > SheetUtils.MIN_ZOOM; },
                    set: function () { view.decZoom(); }
                },

                'view/zoom/inc': {
                    enable: function () { return view.getZoom() < SheetUtils.MAX_ZOOM; },
                    set: function () { view.incZoom(); }
                },

                // sheet operations -------------------------------------------

                'sheet/unlocked': {
                    parent: 'document/editable',
                    enable: function () { return !view.getActiveSheetModel().isProtected(); }
                },

                'sheet/attributes': {
                    parent: 'document/editable',
                    get: function () { return view.getSheetAttributes().sheet; }
                },

                'sheet/visible': {
                    parent: 'sheet/attributes',
                    enable: function () { return view.getVisibleSheetCount() > 1; },
                    get: function (attributes) { return attributes.visible; },
                    set: function (visible) { view.setSheetAttributes({ sheet: { visible: visible } }); }
                },

                'sheet/showall': {
                    parent: 'document/editable',
                    enable: function () { return view.getHiddenSheetCount() > 0; },
                    set: function () { view.showAllSheets(); }
                },

                'sheet/insert': {
                    parent: 'document/editable',
                    set: function () { view.insertSheet(); }
                },

                'sheet/delete': {
                    parent: 'sheet/unlocked',
                    enable: function () { return view.getVisibleSheetCount() > 1; },
                    set: function () { view.deleteSheet(); }
                },

                'sheet/name': {
                    parent: 'sheet/unlocked',
                    get: function () { return view.getSheetName(); },
                    set: function (sheetName) { view.setSheetName(sheetName); }
                },

                'sheet/rename/dialog': {
                    parent: 'sheet/unlocked',
                    set: function () { return view.showRenameSheetDialog(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'sheet/copy/dialog': {
                    parent: 'document/editable',
                    set: function () { return view.showCopySheetDialog(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                // column operations ------------------------------------------

                'column/insert': {
                    parent: 'sheet/unlocked',
                    enable: function () { return view.columnsInsertable(); },
                    set: function () { view.insertColumns(); }
                },

                'column/delete': {
                    parent: 'sheet/unlocked',
                    enable: function () { return view.columnsDeleteable(); },
                    set: function () { view.deleteColumns(); }
                },

                'column/attributes': {
                    parent: 'sheet/unlocked',
                    get: function () { return view.getColumnAttributes(); }
                },

                'column/visible': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'column/width': {
                    parent: 'column/attributes',
                    get: function (attributes) { return (attributes.visible) ? attributes.width : 0; },
                    set: function (value) { view.setColumnWidth(value); }
                },

                // row operations ---------------------------------------------

                'row/insert': {
                    parent: 'sheet/unlocked',
                    enable: function () { return view.rowsInsertable(); },
                    set: function () { view.insertRows(); }
                },

                'row/delete': {
                    parent: 'sheet/unlocked',
                    enable: function () { return view.rowsDeleteable(); },
                    set: function () { view.deleteRows(); }
                },

                'row/attributes': {
                    parent: 'sheet/unlocked',
                    get: function () { return view.getRowAttributes(); }
                },

                'row/visible': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'row/height': {
                    parent: 'row/attributes',
                    get: function (attributes) { return (attributes.visible) ? attributes.height : 0; },
                    set: function (value) { view.setRowHeight(value); }
                },

                // cell operations --------------------------------------------

                'math/operators': {
                    set: function () { view.insertAutoSum(); }
                },

                'character/hyperlink/dialog': {
                    parent: 'character/url',
                    set: function () { return Hyperlink.editHyperlink(app); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'character/hyperlink/remove': {
                    parent: 'character/url',
                    set: function () { Hyperlink.removeHyperlink(app); }
                },

                // cell attributes --------------------------------------------

                'cell/stylesheet': {
                    parent: 'view/activecell/attributes',
                    get: function (attributes) { return attributes.styleId; },
                    set: function (styleId) { view.fillCellRanges(undefined, { styleId: styleId }, { clear: true }); }
                },

                'cell/attributes': {
                    parent: 'view/activecell/attributes',
                    get: function (attributes) { return attributes.cell; }
                },

                'cell/fillcolor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { view.setCellAttribute('fillColor', color); }
                },

                'cell/linebreak': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.wrapText; },
                    set: function (wrap) { view.setCellAttribute('wrapText', wrap); }
                },

                'cell/alignhor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignHor; },
                    set: function (alignment) { view.setCellAttribute('alignHor', alignment); }
                },

                'cell/alignvert': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignVert; },
                    set: function (alignment) { view.setCellAttribute('alignVert', alignment); }
                },

                'cell/resetAttributes': {
                    parent: 'cell/attributes',
                    set: function () { view.clearAttributes(); }
                },

                // merging cells
                'cell/merge': {
                    parent: 'view/editable/cell',
                    enable: function () { return view.hasAnyRangeSelected(); },
                    get: function () { return view.hasMergedRangeSelected(); },
                    set: function (type) { view.mergeRanges(type); }
                },

                // cell border attributes -------------------------------------

                // border mode, as boolean map with position keys (left, top, ...)
                'cell/border/mode': {
                    parent: 'view/editable/cell',
                    get: function () { return view.getBorderMode(); },
                    set: function (borderMode) { view.setBorderMode(borderMode); }
                },

                // all mixed border attributes of the selection
                'cell/border/attributes': {
                    parent: 'view/editable/cell',
                    get: function () { return view.getBorderAttributes(); }
                },

                // a single mixed border for all borders in the selection
                'cell/border/value': {
                    parent: 'cell/border/attributes',
                    enable: function (borderAttributes) { return _(borderAttributes).any(MixedBorder.isVisibleBorder); },
                    get: function (borderAttributes) { return MixedBorder.mixBorders(_(borderAttributes).values()); }
                },

                'cell/border/style': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.style; },
                    set: function (value) { view.changeVisibleBorders({ style: value }); }
                },

                // width of the border lines, in points (rounded to 1/2 points)
                'cell/border/width': {
                    parent: 'cell/border/value',
                    get: function (border) { return _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                    set: function (value) { view.changeVisibleBorders({ width: Utils.convertLengthToHmm(value, 'pt') }); }
                },

                'cell/border/color': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.color; },
                    set: function (value) { view.changeVisibleBorders({ color: value }); }
                },

                // cell number format -----------------------------------------

                'cell/numberformat/category': {
                    parent: 'view/editable/cell',
                    get: function () {
                        var category = view.getNumberFormatCategory();
                        view.refreshFormatCodeList(category);
                        return category;
                    },
                    set: function (category) {
                        view.setNumberFormatCategory(category);
                        view.refreshFormatCodeList(category);
                    }
                },

                'cell/numberformat/formatcodeselector': {
                    parent: 'cell/numberformat/category',
                    enable: function (category) { return !_(['standard', 'custom']).contains(category); },
                    get: function () { return view.getNumberFormatCode(); },
                    set: function (formatCode) { view.setCellAttribute('numberFormat', { code: formatCode }); }
                },

                // character attributes ---------------------------------------

                'character/attributes': {
                    parent: 'view/activecell/attributes',
                    get: function (attributes) { return attributes.character; }
                },

                'character/bold': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { view.setCharacterAttribute('bold', state); },
                    shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/italic': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { view.setCharacterAttribute('italic', state); },
                    shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/underline': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { view.setCharacterAttribute('underline', state); },
                    shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/strike': {
                    parent: 'character/attributes',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                    set: function (state) { view.setCharacterAttribute('strike', state ? 'single' : 'none'); }
                },

                'character/color': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { view.setCharacterAttribute('color', color); }
                },

                'character/fontname': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { view.setCharacterAttribute('fontName', fontName); }
                },

                'character/fontsize': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { view.setCharacterAttribute('fontSize', fontSize); }
                },

                'character/url': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.url; },
                    set: function (url) { view.setCharacterAttribute('url', url); }
                },

                // drawing ---------------------------------------------------
                // drawing comes now from the DrawingController

                // format painter ---------------------------------------------

                'format/painter': {
                    get: function () { return view.applyPainterState(); },
                    set: function (state) { view.applyPainter(state); }
                },

                // debug ------------------------------------------------------

                'debug/view/update': {
                    parent: 'debug/enabled',
                    set: function (type) { view.debugRequestUpdate(type); }
                },

                'debug/insert/drawing': {
                    parent: 'debug/enabled',
                    enable: function () { return model.getEditMode(); },
                    set: function (type) { view.debugInsertDrawing(type); }
                }
            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app);
        DrawingController.call(this, app);
        UnitController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.on('docs:init', function () {

            // model and view are not available at construction time
            model = app.getModel();
            view = app.getView();

            // update GUI after changed selection, and after receiving view updates
            view.on('change:sheetviewattributes change:layoutdata celledit:enter celledit:change celledit:leave', function () { self.update(); });
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
