/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingmodelfactory', function () {

    'use strict';

    // class DrawingModelFactory ==============================================

    function DrawingModelFactory(app) {

        var // maps drawing types to constructor functions
            modelConstructors = {};

        // methods ------------------------------------------------------------

        this.clone = function () {
            var factory = new DrawingModelFactory(app);
            _(modelConstructors).each(function (constructor, type) {
                factory.registerModelType(type, constructor);
            });
            return factory;
        };

        /**
         * Registers the constructor function of a drawing model class for a
         * specific drawing type.
         *
         * @param {String} type
         *  The type of a drawing model. Must be one of the types supported by
         *  the document operations that insert new drawing objects into the
         *  document.
         *
         * @param {Function} constructor
         *  The constructor function of the drawing model class. Must be the
         *  class DrawingModel or any of its sub classes. Must take the
         *  parameter 'app' (application instance), and the optional parameter
         *  'attributes' (initial formatting attributes of the drawing object).
         *  An old registered constructor for the type will be replaced.
         *
         * @returns {DrawingModelFactory}
         *  A reference to this instance.
         */
        this.registerModelType = function (type, constructor) {
            modelConstructors[type] = constructor;
            return this;
        };

        /**
         * Unregisters a constructor function of a drawing model class for a
         * specific drawing type.
         *
         * @param {String} type
         *  The type of a drawing model. Must be one of the types supported by
         *  the document operations that insert new drawing objects into the
         *  document.
         *
         * @returns {DrawingModelFactory}
         *  A reference to this instance.
         */
        this.unregisterModelType = function (type) {
            delete modelConstructors[type];
            return this;
        };

        /**
         * Creates a new drawing model instance for the specified type.
         *
         * @param {String} type
         *  The type of the new drawing model. Must be one of the types
         *  supported by the document operations that insert new drawing
         *  objects into the document. If no constructor has been registered
         *  for the type, this method will return null.
         *
         * @param {Object} [attributes]
         *  Attribute set with initial formatting attributes for the new
         *  drawing object.
         *
         * @returns {DrawingModel|Null}
         *  The new drawing model instance created with the constructor
         *  function that has been registered for the specified drawing type;
         *  or null, if the drawing type is not supported.
         */
        this.createModel = function (type, attributes) {
            var model = (type in modelConstructors) ? new modelConstructors[type](app, attributes) : null;
            if (model) { model.getType = function () { return type; }; }
            return model;
        };

    } // class DrawingModelFactory

    // exports ================================================================

    return DrawingModelFactory;

});
