/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/basecontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/object/triggerobject'
    ], function (Utils, KeyCodes, TriggerObject) {

    'use strict';

    // class BaseController ===================================================

    /**
     * A controller contains a collection of items, consisting of unique key
     * and value, and providing arbitrary getter and setter methods for their
     * values.
     *
     * Triggers The following events:
     * - 'change:items'
     *      After the current state (value and/or enabled state) of at least
     *      one controller item has been changed after the controller has
     *      updated itself or the method BaseController.update() has been
     *      invoked. Event listeners receive data about the changed items in a
     *      map with item identifiers as keys, and objects containing the
     *      optional properties 'value' and 'enable' as values.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {BaseApplication} app
     *  The application that has created this controller instance.
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the behavior of this controller. The
     *  following options are supported:
     *  @param {Number} [initOptions.updateDelay=50]
     *      The delay for the debounced Controller.update() method, in
     *      milliseconds. Must not be less than 50.
     */
    function BaseController(app, initOptions) {

        var // self reference
            self = this,

            // all the little controller items
            items = {

                'app/quit': {
                    // quit in a timeout (otherwise this controller becomes invalid while still running)
                    set: function () { _.defer(function () { app.quit(); }); }
                },

                'document/download': {
                    // enabled in read-only mode
                    set: function (format, options) { return app.download(format, options); }
                },

                'document/print': {
                    // enabled in read-only mode
                    set: function () { return app.print(); },
                    shortcut: { keyCode: 'P', ctrlOrMeta: true }
                },

                'document/sendmail': {
                    // enabled in read-only mode
                    set: function () { return app.sendMail(); }
                },

                // toggle the search pane
                'view/searchpane': {
                    // enabled in read-only mode
                    get: function () {
                        return app.getWindow().search.active;
                    },
                    set: function (state) {
                        var win = app.getWindow();
                        state = _.isBoolean(state) ? state : !this.getValue();
                        if (state) {
                            if (win.search.active) {
                                win.nodes.searchField.focus();
                            } else {
                                win.search.open();
                            }
                        } else {
                            win.search.close();
                            app.getView().grabFocus();
                        }
                    },
                    // shortcuts always enable the search pane (no toggling)
                    shortcut: [{ keyCode: 'F', ctrlOrMeta: true, value: true }, { keyCode: 'F3', value: true }],
                    focus: 'never' // do not grab focus after setter
                }
            },

            // the delay time for the debounced BaseController.update() method
            // Bug 30479: must NOT be 0, to be able give some space for deferred
            // focus handling in form controls, for example in text fields which
            // want to commit their value before the controller will reset it.
            updateDelay = Utils.getIntegerOption(initOptions, 'updateDelay', 50, 50),

            // cached item values during a complex update
            resultCache = {},

            // number of item setters currently running (recursion counter)
            runningSetters = 0,

            // shortcut definitions for 'keydown', mapped by key code (for performance)
            keyShortcuts = {},

            // shortcut definitions for 'keypress', mapped by char code (for performance)
            charShortcuts = {};

        // class Item ---------------------------------------------------------

        function Item(key, definition) {

            var // self reference
                item = this,
                // global enabled state of the item
                enabled = true,
                // parent item whose value/state is needed to resolve the own value/state
                parentKey = Utils.getStringOption(definition, 'parent'),
                // handler for enabled state (default to constant true if missing)
                enableHandler = Utils.getFunctionOption(definition, 'enable', true),
                // handler for value getter (default to identity to forward parent value)
                getHandler = Utils.getFunctionOption(definition, 'get', _.identity),
                // handler for value setter
                setHandler = Utils.getFunctionOption(definition, 'set'),
                // whether to block the GUI while setter is running
                busy = Utils.getBooleanOption(definition, 'busy', false),
                // behavior for returning browser focus to application
                focus = Utils.getStringOption(definition, 'focus', 'direct'),
                // additional user data
                userData = Utils.getOption(definition, 'userData');

            // private methods  - - - - - - - - - - - - - - - - - - - - - - - -

            function getParentItem() {
                var parentItem = _.isString(parentKey) ? items[parentKey] : null;
                if (!parentItem && _.isString(parentKey)) {
                    Utils.warn('BaseController.Item.getParentItem(): item "' + key + '" refers to unknown parent "' + parentKey + '"');
                }
                return parentItem;
            }

            /**
             * Returns whether the parent item of this item is enabled.
             */
            function isParentEnabled() {
                var parentItem = getParentItem();
                return !parentItem || parentItem.isEnabled();
            }

            /**
             * Returns the parent value of this item.
             */
            function getParentValue() {
                var parentItem = getParentItem();
                return parentItem ? parentItem.getValue() : undefined;
            }

            function getAndCacheResult(type, handler, parentValue) {

                var // get or create a result object in the cache
                    result = resultCache[key] || (resultCache[key] = {});

                // if the required value does not exist yet, resolve it via the passed handler
                if (!(type in result)) {
                    result[type] = _.isFunction(handler) ? handler.call(item, parentValue) : handler;
                }
                return result[type];
            }

            // methods  - - - - - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns whether this item is effectively enabled, by looking at
             * the own state, and by asking the enable handler of the item.
             */
            this.isEnabled = function () {
                return getAndCacheResult('enable', (enabled && isParentEnabled()) ? enableHandler : false, getParentValue());
            };

            /**
             * Returns the current value of this item.
             */
            this.getValue = function () {
                return getAndCacheResult('value', getHandler, getParentValue());
            };

            /**
             * Executes the setter function of this item (passing in the new
             * value), and moves the browser focus back to the application
             * pane.
             *
             * @param {Any} value
             *  The new value of the item.
             *
             * @param {Object} [options]
             *  A map with options controlling the behavior of this method. All
             *  options will be passed to the item setter function. The
             *  following special options are supported:
             *  @param {Boolean} [options.preserveFocus=false]
             *      If set to true, the browser focus will not be modified,
             *      regardless of the focus mode configured for this item.
             *  @param {HTMLElement} [options.source]
             *      An optional DOM source element that has caused the change.
             *      This may influence the way how to modify the browser focus.
             *
             * @returns {jQuery.Promise}
             *  The Promise of a Deferred object that will be resolved or
             *  rejected according to the result of the item setter function.
             *  If the item is disabled, a rejected Promise will be returned.
             *  If the item setter returns a Promise, this Promise will be
             *  returned. Otherwise, the return value of the item setter
             *  function will be wrapped and returned in a resolved Promise.
             */
            this.execute = function (value, options) {

                var // the result Deferred object
                    def = null;

                // do nothing if item is disabled
                if (this.isEnabled()) {

                    // execute the set handler
                    runningSetters += 1;
                    if (_.isFunction(setHandler)) {
                        def = setHandler.call(this, value, options);
                    }

                    // convert result of the setter to a Deferred object
                    def = $.when(def);
                    def.always(function () {
                        runningSetters -= 1;
                    });

                    // disable this item and show busy screen if setter is still running
                    if (def.state() === 'pending') {
                        if (busy) {
                            app.getView().enterBusy({ delay: 500 });
                        } else {
                            enabled = false;
                            self.update();
                        }
                    }
                } else {
                    // item is disabled
                    def = $.Deferred().reject();
                }

                // focus back to application
                switch (focus) {
                case 'direct':
                    grabApplicationFocus(options);
                    break;
                case 'wait':
                    def.always(function () {
                        // execute in a timeout, needed for dialogs which are closed *after* resolve/reject
                        app.executeDelayed(function () { grabApplicationFocus(options); });
                    });
                    break;
                case 'never':
                    break;
                default:
                    Utils.warn('BaseController.Item.execute(): unknown focus mode "' + focus + '" for item "' + key + '"');
                }

                // post processing after the setter is finished
                if (def.state() === 'pending') {
                    def.always(function () {
                        if (busy) {
                            app.getView().leaveBusy();
                        } else {
                            enabled = true;
                        }
                        self.update();
                    });
                } else {
                    self.update();
                }

                return def.promise();
            };

            /**
             * Returns the unique key of this item.
             *
             * @returns {String}
             *  The unique key this item has been registered with.
             */
            this.getKey = function () {
                return key;
            };

            /**
             * Returns the user data that has been registered with the
             * definition of this item.
             *
             * @returns {Any}
             *  The registered user data if existing, otherwise undefined.
             */
            this.getUserData = function () {
                return userData;
            };

        } // class Item

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Clears all cached item results.
         */
        function clearResultCache() {
            resultCache = {};
        }

        /**
         * Moves the browser focus back to the application pane.
         */
        function grabApplicationFocus(options) {

            var preserveFocus = Utils.getBooleanOption(options, 'preserveFocus', false),
                source = Utils.getOption(options, 'source');

            // Bug 28214: Focus back to application if source GUI element is hidden now.
            // If the source GUI element is hidden now, IE has already changed the
            // 'document.activeElement' property, so it cannot be used to detect visibility.
            // Therefore, all group instances now insert the source DOM node into the value
            // options when triggering the 'group:change' event.
            if (!preserveFocus || (source && !$(source).is(Utils.REALLY_VISIBLE_SELECTOR))) {
                app.getView().grabFocus();
            }
        }

        /**
         * Handles 'keydown' and 'keypress' events and calls the setter of this
         * item, if it contains a matching keyboard shortcut definition.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object. If a matching shortcut definition has been
         *  found, propagation of the event will be stopped, and the browser
         *  default action will be suppressed.
         */
        function keyHandler(event) {

            var // whether to stop propagation and prevent the default action
                stopPropagation = false;

            // executes the item setter defined in the passed shortcut
            function executeItemForShortcut(shortcut) {

                var // the definition data of the shortcut
                    definition = shortcut.definition,
                    // the current value to be passed to the item setter
                    value = null;

                // check whether to actually process the key event according to the event target node
                if (_.isString(definition.selector) && ($(event.target).closest(definition.selector, app.getWindow().nodes.outer[0]).length === 0)) {
                    return;
                }

                // bug 30096: do not allow to execute items while a blocking operation is still running
                if (runningSetters === 0) {
                    // call value resolver function, or take constant value
                    value = _.isFunction(definition.value) ? definition.value(self.getItemValue(shortcut.key)) : definition.value;
                    self.executeItem(shortcut.key, value);
                }

                // stop event propagation unless specified otherwise
                if (!Utils.getBooleanOption(definition, 'propagate', false)) {
                    stopPropagation = true;
                }
            }

            switch (event.type) {
            case 'keydown':
                // process all shortcut definitions for the key code in the passed event
                if (event.keyCode in keyShortcuts) {
                    _(keyShortcuts[event.keyCode]).each(function (shortcut) {
                        // check if the additional modifier keys match the shortcut definition
                        if (KeyCodes.matchModifierKeys(event, shortcut.definition)) {
                            executeItemForShortcut(shortcut);
                        }
                    });
                }
                break;
            case 'keypress':
                // process all shortcut definitions for the char code in the passed
                // event, but ignore 'keypress' events in text fields
                if ((event.charCode in charShortcuts) && !$(event.target).is('input,textarea')) {
                    _(charShortcuts[event.charCode]).each(executeItemForShortcut);
                }
                break;
            }

            return stopPropagation ? false : undefined;
        }

        /**
         * Initializes the key handlers for all registered keyboard shortcuts.
         */
        function initializeKeyHandler() {

            var // the root node of the application window
                windowNode = app.getWindow().nodes.outer;

            // Bug 27528: IE does not allow to prevent default actions of some
            // special key events (especially CTRL+P which always opens the Print
            // dialog, regardless whether the keydown event has been canceled in
            // the bubbling phase). The 'official' hack is to bind a keydown event
            // handler with IE's ancient 'attachEvent()' method, and to suppress
            // the default action by changing the key code in the event object.
            // (add smiley-with-big-eyes here). Note that the 'addEventListener()'
            // method does NOT work for this hack.
            // Bug 27528: IE 11 does not support the attachEvent() method anymore
            // (hack for bug 27528 will fail).
            if (_.browser.IE && _.isFunction(windowNode[0].attachEvent)) {
                windowNode[0].attachEvent('onkeydown', function (event) {
                    var result = keyHandler(_(event).extend({ target: event.srcElement }));
                    // modify the key code to prevent the browser default action
                    if (result === false) { event.keyCode = 0; }
                    return result;
                });
            } else {
                windowNode.on('keydown', keyHandler);
            }

            // register 'keypress' event listener for shortcuts
            windowNode.on('keypress', keyHandler);

            // prevent that Ctrl+A shortcut selects the entire HTML document
            windowNode.on('keydown', function (event) {
                if (!$(event.target).is('input,textarea') && KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    return false;
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Adds the definition for a new item to this controller.
         *
         * @param {String} key
         *  The key of the new item.
         *
         * @param {Object} definition
         *  A map with callback functions defining the behavior of the item.
         *  All callback functions will be executed in the context of the Item
         *  class instance. The following attributes are supported:
         *  @param {String} [definition.parent]
         *      The key of an item that will be used to calculate intermediate
         *      results for the getter function and enabler function (see
         *      below). The key feature of parent items is that if a controller
         *      enables or updates multiple items at once, the getter or
         *      enabler of the same parent item registered at multiple items
         *      will be executed exactly once before the first item getter or
         *      enabler is called, and its result will be cached and passed to
         *      all item getters or enablers that are using this parent item.
         *  @param {Function} [definition.enable]
         *      Predicate function returning true if the item is enabled, and
         *      false otherwise. If a parent item has been specified (see
         *      above) and it returned false, the item is disabled already, and
         *      this function will not be called anymore. Defaults to a
         *      function that always returns true.
         *  @param {Function} [definition.get]
         *      Getter function returning the current value of the item. Can be
         *      omitted for one-way action items (actions without a return
         *      value). If a parent item has been specified (see above), the
         *      cached return value of its getter will be passed to this
         *      getter. May return null to indicate an ambiguous state. May
         *      return undefined to indicate that calculating the value is not
         *      applicable, not possible, not implemented, etc. In the case of
         *      an undefined return value, the current state of the controls in
         *      the view components will not be changed. Defaults to a function
         *      that returns undefined; or, if a parent item has been
         *      registered, that returns its cached value directly.
         *  @param {Function} [definition.set]
         *      Setter function changing the value of an item to the first
         *      parameter of the setter. Can be omitted for read-only items.
         *      Defaults to an empty function. May return a Deferred object or
         *      a Promise to indicate that the setter executes asynchronous
         *      code. This has effect on the properties 'definition.busy' and
         *      'definition.focus', see below.
         *  @param {Object|Array} [definition.shortcut]
         *      One or multiple keyboard shortcut definitions. If the window
         *      root node of the application receives a 'keydown' or 'keypress'
         *      event that matches a shortcut definition, the setter function
         *      of this item will be executed. Can be as single shortcut
         *      definition, or an array of shortcut definitions. Each
         *      definition object supports the following attributes:
         *      - {Number|String} [shortcut.charCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keypress' events, and the value represents the
         *          numeric code point of a Unicode character, or the Unicode
         *          character as a string.
         *      - {Number|String} [shortcut.keyCode]
         *          If specified, the shortcut definition will be matched
         *          against 'keydown' events, and the value represents the
         *          numeric key code, or the upper-case name of a key code, as
         *          defined in the static class KeyCodes. Be careful with digit
         *          keys, for example, the number 9 matches the TAB key
         *          (KeyCodes.TAB), but the string '9' matches the digit '9'
         *          key (KeyCodes['9'] with the key code 57).
         *      - {Boolean|Null} [shortcut.shift=false]
         *          If set to true, the SHIFT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the SHIFT key must not be pressed. If set to null, the
         *          current state of the SHIFT key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.alt=false]
         *          If set to true, the ALT key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the ALT key must not be pressed. If set to null, the
         *          current state of the ALT key will be ignored. Has no effect
         *          when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.ctrl=false]
         *          If set to true, the CTRL key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the CTRL key must not be pressed. If set to null, the
         *          current state of the CTRL key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean|Null} [shortcut.meta=false]
         *          If set to true, the META key must be pressed when the
         *          'keydown' events is received. If set to false (or omitted),
         *          the META key must not be pressed. If set to null, the
         *          current state of the META key will be ignored. Has no
         *          effect when evaluating 'keypress' events.
         *      - {Boolean} [shortcut.altOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the ALT key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.alt' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Boolean} [shortcut.ctrlOrMeta=false]
         *          Convenience option that if set to true, matches if either
         *          the CTRL key, or the META key are pressed. Has the same
         *          effect as defining two separate shortcuts, one with the
         *          'shortcut.ctrl' option set to true, and one with the
         *          'shortcut.meta' option set to true, while keeping the other
         *          option false. Must not be used in a shortcut definition
         *          where these options are set explicitly.
         *      - {Any|Function} [shortcut.value]
         *          The value that will be passed to the setter function of
         *          this item. If multiple shortcuts are defined for an item,
         *          each shortcut definition may define its own value. If this
         *          option contains a function, it will receive the current
         *          value of the controller item as first parameter, and its
         *          return value will be passed to the setter function.
         *      - {Boolean} [shortcut.propagate=false]
         *          If set to true, the event will propagate up to the DOM root
         *          element, and the browser will execute its default action.
         *          If omitted or set to false, the event will be cancelled
         *          immediately after calling the setter function.
         *      - {String} [shortcut.selector]
         *          A CSS selector used to restrict the keyboard shortcut to
         *          specific DOM nodes in the application DOM. The target node
         *          of the keyboard event has to match the selector, or has to
         *          be a descendant node of a DOM node matching the selector.
         *          Otherwise, the event will be ignored completely, and the
         *          browser will execute its default action.
         *  @param {Boolean} [definition.busy=false]
         *      If set to true, the GUI of the application will be blocked and
         *      a busy indicator will be shown while the setter of this item
         *      is running (indicated by a pending Deferred object returned by
         *      the setter). Does not have any effect for synchronous setters.
         *  @param {String} [definition.focus='direct']
         *      Determines how to return the browser focus to the application
         *      pane after executing the setter function of this item. The
         *      following values are supported:
         *      - 'direct': (default) The focus will return directly after the
         *          setter function has been executed, regardless of the return
         *          value of the setter.
         *      - 'never': The focus will not be returned to the application
         *          pane. The setter function is responsible for application
         *          focus handling.
         *      - 'wait': The controller will wait until the Deferred object
         *          returned by the setter function gets resolved or rejected,
         *          and then sets the browser focus to the application pane.
         *  @param {Any} [definition.userData]
         *      Additional user data that will be provided by the method
         *      'Item.getUserData()'. Can be used in all item getter and setter
         *      methods.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinition = function (key, definition) {
            if (_.isString(key) && key && _.isObject(definition)) {
                items[key] = new Item(key, definition);
                if (_.isObject(definition.shortcut)) {
                    _.chain(definition.shortcut).getArray().each(function (shortcut) {
                        var keyCode = Utils.getOption(shortcut, 'keyCode'),
                            charCode = Utils.getOption(shortcut, 'charCode');
                        if (_.isString(keyCode)) {
                            keyCode = KeyCodes[keyCode] || 0;
                        }
                        if (_.isNumber(keyCode) && (keyCode > 0)) {
                            (keyShortcuts[keyCode] || (keyShortcuts[keyCode] = [])).push({ key: key, definition: shortcut });
                        }
                        if (_.isString(charCode)) {
                            charCode = charCode.charCodeAt(0);
                        }
                        if (_.isNumber(charCode) && (charCode > 0)) {
                            (charShortcuts[charCode] || (charShortcuts[charCode] = [])).push({ key: key, definition: shortcut });
                        }
                    });
                }
            }
            return this;
        };

        /**
         * Adds definitions for multiple items to this controller.
         *
         * @param {Object} definitions
         *  A map of key/definition pairs for all new items. Each item will be
         *  defined by calling the method BaseController.registerDefinition().
         *  See this method for more details.
         *
         * @returns {BaseController}
         *  A reference to this controller instance.
         */
        this.registerDefinitions = function (definitions) {
            _(definitions).each(function (definition, key) {
                this.registerDefinition(key, definition);
            }, this);
            return this;
        };

        /**
         * Refreshes the states and values of all registered items, and
         * triggers a 'change:items' event.
         *
         * @returns {BaseController}
         *  A reference to this controller.
         */
        this.update = app.createDebouncedMethod(function () { return self; }, function () {

            // collect states of all matching items in resultCache
            clearResultCache();
            _(items).each(function (item) {
                item.isEnabled();
                item.getValue();
            });

            // notify all listeners
            self.trigger('change:items', resultCache);

        }, { delay: updateDelay, maxDelay: updateDelay * 5 });

        /**
         * Returns the current enabled state of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Boolean}
         *  Whether the specified item exists and is enabled.
         */
        this.isItemEnabled = function (key) {
            // do not clear the cache if currently running in a set handler
            if (runningSetters === 0) { clearResultCache(); }
            return (key in items) && items[key].isEnabled();
        };

        /**
         * Returns the current value of the specified item.
         *
         * @param {String} key
         *  The key of the item.
         *
         * @returns {Any}
         *  The current value of the item, or undefined, if the item does not
         *  exist.
         */
        this.getItemValue = function (key) {
            // do not clear the cache if currently running in a set handler
            if (runningSetters === 0) { clearResultCache(); }
            return (key in items) ? items[key].getValue() : undefined;
        };

        /**
         * Triggers a controller item manually. Executes the setter function of
         * the item associated to the specified key, passing in the new value.
         *
         * @param {String} key
         *  The key of the item to be changed.
         *
         * @param {Any} value
         *  The new value of the item.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. All
         *  options will be passed to the item setter function. The following
         *  special options are supported:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the browser focus will not be modified after
         *      executing the item, regardless of the focus mode configured for
         *      that item.
         *  @param {HTMLElement} [options.source]
         *      An optional DOM source element that has caused the change. This
         *      may influence the way how to modify the browser focus.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  according to the result of the item setter function. If the item is
         *  disabled or does not exist, a rejected Promise will be returned. If
         *  the item setter returns a Promise, this Promise will be returned.
         *  Otherwise, the return value of the item setter function will be
         *  wrapped and returned in a resolved Promise.
         */
        this.executeItem = function (key, value, options) {

            // execute setter of existing item
            if (key in items) {
                // do not clear cache if called recursively from another setter
                if (runningSetters === 0) { clearResultCache(); }
                return items[key].execute(value, options);
            }

            // item does not exist: return focus to application
            grabApplicationFocus(options);
            return $.Deferred().reject();
        };

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // update once after import (successful and failed)
        app.on('docs:import:after', function () {
            initializeKeyHandler();
            self.update();
        });

    } // class BaseController

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BaseController });

});
