/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/listmenu',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/basemenu'
    ], function (Utils, KeyCodes, BaseMenu) {

    'use strict';

    var // the CSS class for menu section nodes
        SECTION_CLASS = 'section-container';

    // class ListMenu =========================================================

    /**
     * Wraps a pop-up menu element, shown on top of the application window, and
     * relative to an arbitrary DOM node. An item list contains a set of button
     * control elements (the list items) that can be stacked vertically, or can
     * be arranged in a tabular layout. The list items can be highlighted. The
     * items will be grouped into vertically stacked section nodes.
     *
     * Instances of this class trigger the following events (additionally to
     * the events triggered by the base class BaseMenu):
     * - 'create:section'
     *      After a new section node has been created with the method
     *      ListMenu.createSectionNode(). Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} sectionNode
     *          The root node of the new section, as jQuery object.
     *      (3) {String} sectionId
     *          The section identifier.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createSectionNode().
     * - 'create:item'
     *      After a button node for a new list item has been created with the
     *      method ListMenu.createItemNode(). Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} buttonNode
     *          The button element representing the new list item, as jQuery
     *          object.
     *      (3) {String} sectionId
     *          The identifier of the menu section containing the new button.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createItemNode().
     *
     * @constructor
     *
     * @extends BaseMenu
     *
     * @param {HTMLElement|jQuery} anchorNode
     *  The DOM node this item list menu is attached to.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the item list menu
     *  element. Supports all options also supported by the base class
     *  BaseMenu. Additionally, the following options are supported:
     *  @param {String} [initOptions.itemDesign='list']
     *      The design mode of the list item elements:
     *      - 'list': All item elements will be stacked vertically, and will
     *          occupy the complete width of the pop-up menu. Highlighted item
     *          elements will be drawn with a changed background color.
     *      - 'grid': All item elements will be arranged in a tabular layout
     *          per menu section. Highlighted items will be drawn with a thick
     *          border, leaving the background untouched.
     *  @param {Boolean|Function} [initOptions.sortItems=false]
     *      If set to true, the item elements will be sorted inside their menu
     *      section by their label texts, ignoring case; items without text
     *      label will be inserted in no special order. If set to a function,
     *      it must return an ordinal value for each item element. The item
     *      elements will be sorted according to these values, either by number
     *      or lexicographically by strings. The function receives the button
     *      element representing the drop-down menu item as jQuery object, and
     *      will be called in the context of this item list instance. If
     *      omitted or set to false, new item elements will be appended to the
     *      section node.
     *  @param {Number} [options.gridColumns=10]
     *      The default number of columns in each menu section in 'grid' design
     *      mode (see option 'itemDesign' above). This value can be overridden
     *      for single menu sections by passing this option to the method
     *      ListMenu.createSectionNode().
     */
    function ListMenu(anchorNode, initOptions) {

        var // self reference
            self = this,

            // the design mode
            design = Utils.getStringOption(initOptions, 'itemDesign', 'list'),

            // functor used to sort the list items
            sortFunctor = Utils.getFunctionOption(initOptions, 'sortItems'),

            // default number of columns in each section node ('grid' design mode)
            defColumns = Utils.getIntegerOption(initOptions, 'gridColumns', 10, 1);

        // base constructor ---------------------------------------------------

        BaseMenu.call(this, anchorNode, Utils.extendOptions({ preferFocusFilter: Utils.SELECTED_SELECTOR }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus to the button element with the specified
         * distance to the button currently focused.
         */
        function moveFocus(event, diff) {

            var // all button nodes in this list
                buttonNodes = self.getItemNodes(),
                // index of the focused button
                index = buttonNodes.index(event.target);

            if (index >= 0) {
                index += (diff % buttonNodes.length) + buttonNodes.length;
                buttonNodes.eq(index % buttonNodes.length).focus();
            }
        }

        /**
         * Handles 'keydown' events if this item list is in list layout.
         */
        function listKeyDownHandler(event) {

            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                moveFocus(event, -1);
                return false;
            case KeyCodes.DOWN_ARROW:
                moveFocus(event, 1);
                return false;
            case KeyCodes.PAGE_UP:
                moveFocus(event, -ListMenu.PAGE_SIZE);
                return false;
            case KeyCodes.PAGE_DOWN:
                moveFocus(event, ListMenu.PAGE_SIZE);
                return false;
            }
        }

        /**
         * Handles 'keydown' events if this item list is in grid layout.
         */
        function gridKeyDownHandler(event) {

            // moves the focus into the preceding or next table row
            function moveFocusByRows(diff) {

                var // the table cell node containing the focused button
                    cellNode = $(Utils.findFarthest(self.getNode(), event.target, 'td')),
                    // the table row node containing the focused button
                    rowNode = cellNode.parent(),
                    // all button elements in the focused row
                    buttonNodes = rowNode.children().children(),
                    // all row nodes in all tables
                    rowNodes = self.getSectionNodes().find('>table>tbody>tr'),
                    // the column index of the cell in the row
                    colIndex = cellNode.index(),
                    // the row index of the cell in the table
                    rowIndex = rowNodes.index(rowNode),
                    // the column index of the cell in the table
                    colRatio = (colIndex >= 0) ? (colIndex / buttonNodes.length) : 0;

                // move focus to the specified row, try to keep column position constant inside row
                if ((rowNodes.length > 1) && (rowIndex >= 0)) {
                    rowIndex += (diff % rowNodes.length) + rowNodes.length;
                    rowNode = rowNodes.eq(rowIndex % rowNodes.length);
                    buttonNodes = rowNode.children().children();
                    colIndex = Math.min(Math.floor(colRatio * rowNode.children().length), buttonNodes.length - 1);
                    buttonNodes.eq(colIndex).focus();
                }
            }

            switch (event.keyCode) {
            case KeyCodes.LEFT_ARROW:
                moveFocus(event, -1);
                return false;
            case KeyCodes.RIGHT_ARROW:
                moveFocus(event, 1);
                return false;
            case KeyCodes.UP_ARROW:
                moveFocusByRows(-1);
                return false;
            case KeyCodes.DOWN_ARROW:
                moveFocusByRows(1);
                return false;
            }
        }

        /**
         * Handles 'keydown' events in the root node of the pop-up menu.
         */
        function keyDownHandler(event) {

            // generic key shortcuts
            switch (event.keyCode) {
            case KeyCodes.HOME:
                self.getItemNodes().first().focus();
                return false;
            case KeyCodes.END:
                self.getItemNodes().last().focus();
                return false;
            }

            // special shortcuts dependent on design mode
            switch (design) {
            case 'list':
                return listKeyDownHandler(event);
            case 'grid':
                return gridKeyDownHandler(event);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the DOM nodes of all menu sections.
         *
         * @returns {jQuery}
         *  The DOM nodes representing all menu sections, as jQuery collection.
         */
        this.getSectionNodes = function () {
            return this.getNode().find('.' + SECTION_CLASS);
        };

        /**
         * Returns the DOM node of the specified menu section.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @returns {jQuery}
         *  The DOM node representing the specified menu section. If the menu
         *  section does not exist, returns an empty jQuery collection.
         */
        this.getSectionNode = function (sectionId) {
            return this.getSectionNodes().filter('[data-section="' + sectionId  + '"]');
        };

        /**
         * Adds a new section to this pop-up menu, if it does not exist yet.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @param {Object} [options]
         *  A map of options to control the appearance of the section. The
         *  following options are supported:
         *  @param {String} [options.label]
         *      If specified, a heading label will be created for the section.
         *  @param {String} [options.classes]
         *      Additional CSS classes that will be added to the section root
         *      node.
         *  @param {Number} [options.gridColumns]
         *      The number of columns in the menu section, if this menu is in
         *      'grid' design mode (see option 'itemDesign' passed to the
         *      constructor). If omitted, the default value for this option as
         *      specified in the constructor options will be used.
         *
         * @returns {jQuery}
         *  The root node of the menu section, as jQuery object.
         */
        this.createSectionNode = function (sectionId, options) {

            var // try to find an existing section node
                sectionNode = this.getSectionNode(sectionId),
                // the header label for the section
                label = null;

            if (sectionNode.length === 0) {

                // create the section root node
                sectionNode = Utils.createContainerNode(SECTION_CLASS + ' group')
                    .attr('data-section', sectionId)
                    .addClass(Utils.getStringOption(options, 'classes', ''))
                    .data('grid-columns', Utils.getIntegerOption(options, 'gridColumns', defColumns, 1));

                // set the label text as attribute (added as :before pseudo element by CSS)
                label = Utils.getStringOption(options, 'label', '');
                if (label.length > 0) {
                    sectionNode.attr('data-section-label', label);
                }

                // insert the new section into the drop-down menu, notify listeners
                this.appendContentNodes(sectionNode);
                this.trigger('create:section', sectionNode, sectionId, options);
            }
            return sectionNode;
        };

        /**
         * Returns all button elements representing the menu items of this
         * list.
         *
         * @param {String} [sectionId]
         *  If specified, the result will contain the item elements of that
         *  menu section only.
         *
         * @returns {jQuery}
         *  All button elements representing the menu items in this item list,
         *  or the buttons of a single menu section, as jQuery collection.
         */
        this.getItemNodes = function (sectionId) {
            var node = _.isString(sectionId) ? this.getSectionNode(sectionId) : this.getNode();
            return node.find(Utils.BUTTON_SELECTOR);
        };

        /**
         * Adds a new item element to this list. If the items will be sorted
         * (see the sort options passed to the constructor), the item element
         * will be inserted according to these settings.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section the new list item will be
         *  inserted into. If a section with this identifier does not exist
         *  yet, it will be created without a header label.
         *
         * @param {Object} [options]
         *  A map of options to control the properties of the button DOM node
         *  representing the new list item. See method Utils.createButton() for
         *  details.
         *
         * @returns {jQuery}
         *  The button element representing the new list item, as jQuery
         *  object.
         */
        this.createItemNode = function (sectionId, options) {

            var // the root node of the menu section
                sectionNode = this.createSectionNode(sectionId),
                // all existing item buttons in the current section
                buttonNodes = sectionNode.find(Utils.BUTTON_SELECTOR),
                // create the button element representing the item
                buttonNode = Utils.createButton(options).addClass(Utils.FOCUSABLE_CLASS),
                // insertion index for sorted items
                sortIndex = -1;

            // inserts the button node in a list
            function insertListItemNode() {
                if ((0 <= sortIndex) && (sortIndex < buttonNodes.length)) {
                    buttonNode.insertBefore(buttonNodes[sortIndex]);
                } else {
                    sectionNode.append(buttonNode);
                }
            }

            // inserts the button node in a table
            function insertGridItemNode() {

                var // the number of columns in the table
                    columns = sectionNode.data('grid-columns'),
                    // the table element containing the grid items
                    tableNode = null,
                    // the last table row
                    rowNode = null;

                // create a new table element for the button if required
                tableNode = sectionNode.find('>table');
                if (tableNode.length === 0) {
                    tableNode = $('<table>').attr('role', 'grid').appendTo(sectionNode);
                }

                // create a new table row element for the button if required
                rowNode = tableNode.find('>tbody>tr').last();
                if ((rowNode.length === 0) || (rowNode.children().last().children().length > 0)) {
                    // IE9 does not allow to write to the 'innerHTML' property for <TR> nodes :-O
                    rowNode = $('<tr role="row">' + Utils.repeatString('<td role="gridcell"></td>', columns) + '</tr>').appendTo(tableNode);
                }

                // insert the new button into the array, and reinsert all buttons into the table
                buttonNodes = buttonNodes.get();
                buttonNodes.splice(sortIndex, 0, buttonNode);
                tableNode.find('>tbody>tr>td').each(function (index) {
                    $(this).append(buttonNodes[index]);
                });
            }

            // find insertion index for sorted items
            if (_.isFunction(sortFunctor)) {
                sortIndex = _.chain(buttonNodes.get())
                    // convert array of button elements to strings returned by sort functor
                    .map(function (node) { return sortFunctor.call(self, $(node)); })
                    // calculate the insertion index of the new list item
                    .sortedIndex(sortFunctor.call(this, buttonNode))
                    // exit the call chain, returns result of sortedIndex()
                    .value();
            } else {
                // else: append to existing items
                sortIndex = buttonNodes.length;
            }

            // insert the new button according to the design mode
            switch (design) {
            case 'list':
                insertListItemNode();
                break;
            case 'grid':
                insertGridItemNode();
            }

            // notify listeners
            this.trigger('create:item', buttonNode, sectionId, options);

            return buttonNode;
        };

        // initialization -----------------------------------------------------

        // add CSS marker for the item design
        this.getNode().addClass('list-menu').attr('data-design', design);

        // register event handlers
        this.getNode().on('keydown', keyDownHandler);

        // default sort functor: sort by button label text, case insensitive
        if (Utils.getBooleanOption(initOptions, 'sortItems', false)) {
            sortFunctor = function (buttonNode) {
                var label = Utils.getControlLabel(buttonNode);
                return _.isString(label) ? label.toLowerCase() : '';
            };
        }

        // special handling for tables in 'grid' design
        if (design === 'grid') {
            this.on({
                'popup:beforeshow popup:beforelayout': function () { self.getSectionNodes().find('>table').css('width', 'auto'); },
                'popup:show popup:layout': function () { self.getSectionNodes().find('>table').css('width', ''); }
            });
        }

    } // class ListMenu

    // constants --------------------------------------------------------------

    /**
     * Number of list items that will be skipped when using the PAGE_UP or
     * PAGE_DOWN key.
     */
    ListMenu.PAGE_SIZE = 5;

    // exports ================================================================

    // derive this class from class BaseMenu
    return BaseMenu.extend({ constructor: ListMenu });

});
