/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/basepopup',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject'
    ], function (Utils, TriggerObject) {

    'use strict';

    var // the order in which available space is checked on each border of the anchor node
        PREFERRED_SIDES = {
            left:   ['left',   'right',  'top',    'bottom'],
            right:  ['right',  'left',   'bottom', 'top'   ],
            top:    ['top',    'bottom', 'left',   'right' ],
            bottom: ['bottom', 'top',    'right',  'left'  ]
        };

    // class BasePopup ========================================================

    /**
     * Wrapper class for a DOM node used as pop-up window, shown on top of the
     * application window, and relative to an arbitrary DOM node.
     *
     * Instances of this class trigger the following events:
     * - 'popup:beforeshow': Before the pop-up node will be shown.
     * - 'popup:show': After the pop-up node has been shown.
     * - 'popup:beforehide': Before the pop-up node will be hidden.
     * - 'popup:hide': After the pop-up node has been hidden.
     * - 'popup:beforelayout': If the absolute position of the anchor node or
     *      the browser window size has changed, and the position and size of
     *      the pop-up node needs to be adjusted.
     * - 'popup:layout': If the absolute position of the anchor node or the
     *      browser window size has changed, and the position and size of the
     *      pop-up node has been adjusted (in auto-layout mode).
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {HTMLElement|jQuery} anchorNode
     *  The DOM node this pop-up node is attached to.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the pop-up node. The
     *  following options are supported:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the pop-up root node.
     *  @param {Boolean} [initOptions.autoLayout=true]
     *      If set to true or omitted, the pop-up node will be positioned and
     *      sized automatically next to the anchor node. If the available space
     *      is not sufficient for the contents of the pop-up node, scroll bars
     *      will be displayed. If set to false, the pop-up node will appear
     *      below the anchor node and will be sized according to the contents
     *      even if the node exceeds the browser window.
     *  @param {String} [initOptions.preferSide='bottom']
     *      The preferred position of the pop-up node relative to the anchor
     *      node. Must be one of the value 'left', 'right', 'top', or 'bottom'.
     *      Used in automatic layout mode only (see option 'autoLayout' above).
     *  @param {String} [initOptions.align='leading']
     *      The preferred alignment of the pop-up node relative to the anchor
     *      node. The value 'leading' tries to align the left borders (if shown
     *      above or below) or top border (if shown left or right) of the
     *      pop-up node and the anchor node, the value 'trailing' tries to
     *      align the right/bottom borders, and the value 'center' tries to
     *      place the pop-up node centered to the anchor node. Used in
     *      automatic layout mode only (see option 'autoLayout' above).
     *  @param {Number} [initOptions.windowPadding=6]
     *      The minimum distance between pop-up node and the borders of the
     *      browser window.
     *  @param {Number} [initOptions.anchorPadding=1]
     *      The distance between pop-up node and anchor node.
     */
    function BasePopup(anchorNode, initOptions) {

        var // self reference (the Group instance)
            self = this,

            // the root pop-up DOM node
            rootNode = $('<div>').addClass('io-ox-office-main popup-container'),

            // the content node inside the root node used to calculate the effective node size
            contentNode = $('<div>').addClass('popup-content').appendTo(rootNode),

            // automatic position and size of the pop-up node
            autoLayout = Utils.getBooleanOption(initOptions, 'autoLayout', true),

            // preferred border of the group node for the drop-down menu
            preferSide = Utils.getStringOption(initOptions, 'preferSide', 'bottom'),

            // alignment of the pop-up node relative to the anchor node
            align = Utils.getStringOption(initOptions, 'align', 'leading'),

            // distance between pop-up node and the borders of the browser window
            windowPadding = Utils.getIntegerOption(initOptions, 'windowPadding', 6, 0),

            // distance between pop-up node and anchor node
            anchorPadding = Utils.getIntegerOption(initOptions, 'anchorPadding', 1, 0),

            // an interval timer to refresh the the pop-up node while it is visible
            refreshTimer = null,

            // the DOM element that was last focused
            lastActiveElement = null,

            // the width and height of the virtual keyboard on touch devices
            keyboardWidth = 0,
            keyboardHeight = 0,

            // last position and size of the anchor node
            lastAnchorPosition = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the position and size of the visible area in the browser
         * window, relative to the entire document page.
         *
         * @param {Number} [padding=0]
         *  Additional padding size that will be removed from the resulting
         *  visible area.
         */
        function getVisibleWindowArea(padding) {

            var // the visible area of the browser window
                visibleArea = { left: window.pageXOffset, top: window.pageYOffset };

            padding = _.isNumber(padding) ? Math.max(padding, 0) : 0;
            visibleArea.left += padding;
            visibleArea.top += padding;

            // The properties window.inner(Width|Height) for their own are not
            // reliable when virtual keyboard on touch devices is visible. This
            // 'hack' works for applications with fixed page size.
            visibleArea.width = Math.max(Math.min(window.innerWidth, document.body.clientWidth - keyboardWidth) - 2 * padding, 0);
            visibleArea.height = Math.max(Math.min(window.innerHeight, document.body.clientHeight - keyboardHeight) - 2 * padding, 0);

            // add right/bottom distance to the result
            visibleArea.right = document.body.clientWidth - visibleArea.width - visibleArea.left;
            visibleArea.bottom = document.body.clientHeight - visibleArea.height - visibleArea.top;

            return visibleArea;
        }

        /**
         * Returns the position of the anchor node, relative to the visible
         * area of the browser window.
         */
        function getAnchorPositionInWindow() {

            var // the node position, relative to the document page
                nodePosition = Utils.getNodePositionInPage(anchorNode),
                // the visible area of the browser window
                visibleArea = getVisibleWindowArea();

            // adjust to visible area of the browser window
            _(['left', 'top', 'right', 'bottom']).each(function (border) {
                nodePosition[border] -= visibleArea[border];
            });

            return nodePosition;
        }

        /**
         * Initializes the size and position of the pop-up node.
         */
        function refreshNodePosition() {

            var // position and size of the anchor node in the page
                anchorPosition = Utils.getNodePositionInPage(anchorNode),
                // the sizes available at every side of the anchor node
                availableSizes = self.getAvailableSizes(),
                // size of the pop-up node contents
                contentWidth = 0, contentHeight = 0,
                // resulting width and height available for the menu node
                availableWidth = 0, availableHeight = 0,
                // whether scroll bars will be shown
                scrollHor = false, scrollVer = false,
                // new CSS properties of the pop-up node
                cssProps = {},
                // the preferred side of the anchor node used for the pop-up node
                preferredSide = null, maxRatio = 0;

            // set all sizes to auto to get effective size of the pop-up node contents
            rootNode.css({ left: 0, top: 0, right: '', bottom: '', width: 'auto', height: 'auto' });
            contentNode.css({ minWidth: '', width: 'auto', minHeight: '', height: 'auto' });
            contentWidth = contentNode.outerWidth();
            contentHeight = contentNode.outerHeight();

            // fix size of content node to prevent any reformatting when shrinking the root node
            contentNode.css({ width: contentWidth, height: contentHeight });

            // calculate the ratio of the menu node being visible at every side of the group
            _(availableSizes).each(function (size) {
                size.ratio = Math.min(size.width / contentWidth, 1) * Math.min(size.height / contentHeight, 1);
            });

            // prefer the best side that can take the menu node, in the given order
            _(PREFERRED_SIDES[preferSide]).any(function (side) {
                var ratio = availableSizes[side].ratio;
                if (ratio >= 1) {
                    preferredSide = side;
                    return true;
                } else if (ratio > maxRatio) {
                    preferredSide = side;
                    maxRatio = ratio;
                }
            });

            // extract available width and height
            availableWidth = availableSizes[preferredSide].width;
            availableHeight = availableSizes[preferredSide].height;

            // first part of the position of the pop-up node (keep in visible area of browser window)
            switch (preferredSide) {
            case 'top':
                cssProps.bottom = document.body.clientHeight - anchorPosition.top + anchorPadding;
                break;
            case 'bottom':
                cssProps.top = anchorPosition.top + anchorPosition.height + anchorPadding;
                break;
            case 'left':
                cssProps.right = document.body.clientWidth - anchorPosition.left + anchorPadding;
                break;
            case 'right':
                cssProps.left = anchorPosition.left + anchorPosition.width + anchorPadding;
                break;
            }

            // add scroll bars if available width or height is not sufficient
            scrollHor = contentWidth > availableWidth;
            scrollVer = contentHeight > availableHeight;
            cssProps.width = Math.min(availableWidth, contentWidth + (scrollVer ? Utils.SCROLLBAR_WIDTH : 0));
            cssProps.height = Math.min(availableHeight, contentHeight + (scrollHor ? Utils.SCROLLBAR_HEIGHT : 0));
            cssProps.overflowX = scrollHor ? 'scroll' : 'hidden';
            cssProps.overflowY = scrollVer ? 'scroll' : 'hidden';

            // second part of the position of the pop-up node (keep in visible area of browser window)
            if (Utils.isVerticalPosition(preferredSide)) {
                switch (align) {
                case 'trailing':
                    cssProps.left = anchorPosition.left + anchorPosition.width - cssProps.width;
                    break;
                case 'center':
                    cssProps.left = anchorPosition.left + Math.floor((anchorPosition.width - cssProps.width) / 2);
                    break;
                default:
                    cssProps.left = anchorPosition.left;
                }
            } else {
                switch (align) {
                case 'trailing':
                    cssProps.top = anchorPosition.top + anchorPosition.height - cssProps.height;
                    break;
                case 'center':
                    cssProps.top = anchorPosition.top + Math.floor((anchorPosition.height - cssProps.height) / 2);
                    break;
                default:
                    cssProps.top = anchorPosition.top;
                }
            }

/*
            Utils.info('BasePopup.refreshNodePosition()');
            Utils.log(' anchor: ' + JSON.stringify(anchorPosition).replace(/"/g, '').replace(/,/g, ', '));
            Utils.log(' window: ' + JSON.stringify(getVisibleWindowArea()).replace(/"/g, '').replace(/,/g, ', '));
            Utils.log(' a-in-w: ' + JSON.stringify(getAnchorPositionInWindow()).replace(/"/g, '').replace(/,/g, ', '));
            Utils.log(' avail:  ' + JSON.stringify(availableSizes).replace(/"/g, '').replace(/,/g, ', '));
            Utils.log(' prefer: ' + preferredSide);
*/

            // apply final CSS formatting
            self.setNodePosition(cssProps);
        }

        /**
         * Checks and updates the position of the pop-up node, according to the
         * current position of the anchor node (a scrollable parent node may
         * have been scrolled). Additionally, hides the pop-up node
         * automatically when the anchor node becomes inaccessible (also
         * indirectly, e.g. when hiding any parent nodes).
         */
        function refreshNode() {

            var // whether to update the position of the pop-up node
                refreshPosition = false,
                // the position of the anchor node in the visible area of the window
                anchorPosition = null;

            if (rootNode.is(Utils.REALLY_VISIBLE_SELECTOR)) {

                // on touch devices, try to detect the size of the virtual keyboard
                if (Modernizr.touch && (lastActiveElement !== document.activeElement)) {
                    lastActiveElement = document.activeElement;
                    refreshPosition = true;
                    // if a text field is focused, try to scroll the browser window
                    // right/down to get the size of the virtual keyboard
                    if ($(lastActiveElement).is('input, textarea')) {
                        var pageX = window.pageXOffset, pageY = window.pageYOffset;
                        window.scrollTo(10000, 10000);
                        keyboardWidth = window.pageXOffset;
                        keyboardHeight = window.pageYOffset;
                        window.scrollTo(pageX, pageY);
                    } else {
                        keyboardWidth = keyboardHeight = 0;
                    }
                }

                // check whether the position of the anchor node has changed
                // (for example, by scrolling a parent node of the anchor node)
                if (!refreshPosition) {
                    anchorPosition = getAnchorPositionInWindow();
                    refreshPosition = !_.isEqual(lastAnchorPosition, anchorPosition);
                    lastAnchorPosition = anchorPosition;
                }

                // refresh the position of the pop-up node, notify all listeners
                if (refreshPosition) {
                    self.trigger('popup:beforelayout');
                    if (autoLayout) { refreshNodePosition(); }
                    self.trigger('popup:layout');
                }
            } else {
                // anchor node is not visible anymore: hide the pop-up node
                self.hide();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the available sizes for the pop-up node at every side of the
         * anchor node.
         *
         * @internal
         *  Intended to be used by sub classes of this class for custom
         *  rendering of the pop-up node.
         *
         * @returns {Object}
         *  The sizes (objects with 'width' and 'height' properties) available
         *  for the pop-up node, mapped by the border side names 'left',
         *  'right', 'top', and 'bottom'.
         */
        this.getAvailableSizes = function () {

            var // position and size of the anchor node in the visible area of the browser window
                anchorPosition = getAnchorPositionInWindow(),
                // visible area of the browser window, reduced by border padding
                availableArea = getVisibleWindowArea(windowPadding),

                // vertical space above, below, left of, and right of the anchor node
                totalPadding = windowPadding + anchorPadding,
                availableAbove = Utils.minMax(anchorPosition.top - totalPadding, 0, availableArea.height),
                availableBelow = Utils.minMax(anchorPosition.bottom - totalPadding, 0, availableArea.height),
                availableLeft = Utils.minMax(anchorPosition.left - totalPadding, 0, availableArea.width),
                availableRight = Utils.minMax(anchorPosition.right - totalPadding, 0, availableArea.width);

            return {
                top: { width: availableArea.width, height: availableAbove },
                bottom: { width: availableArea.width, height: availableBelow },
                left: { width: availableLeft, height: availableArea.height },
                right: { width: availableRight, height: availableArea.height }
            };
        };

        /**
         * Sets the position of the pop-up node according to the passed CSS
         * positioning attributes. The resulting position of the pop-up node
         * will be adjusted according to the current visible area of the
         * browser window.
         *
         * @internal
         *  Intended to be used by sub classes of this class for custom
         *  rendering of the pop-up node.
         *
         * @param {Object} cssProps
         *  The CSS positioning properties for the pop-up node, relative to the
         *  document page. Must contain one horizontal offset (either 'left' or
         *  'right'), and one vertical offset (either 'top' or 'bottom').
         *  Optionally, may contain the new size ('width' and/or 'height') of
         *  the pop-up node. May contain any other CSS properties.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setNodePosition = function (cssProps) {

            var // visible area of the browser window, reduced by border padding
                availableArea = getVisibleWindowArea(windowPadding),
                // the width and height of the pop-up node (may be missing in passed properties)
                nodeWidth = _.isNumber(cssProps.width) ? cssProps.width : rootNode.outerWidth(),
                nodeHeight = _.isNumber(cssProps.height) ? cssProps.height : rootNode.outerHeight();

            // restrict all existing positioning attributes to the visible area
            cssProps.left = _.isNumber(cssProps.left) ? Utils.minMax(cssProps.left, availableArea.left, availableArea.left + availableArea.width - nodeWidth) : '';
            cssProps.right = _.isNumber(cssProps.right) ? Utils.minMax(cssProps.right, availableArea.right, availableArea.right + availableArea.width - nodeWidth) : '';
            cssProps.top = _.isNumber(cssProps.top) ? Utils.minMax(cssProps.top, availableArea.top, availableArea.top + availableArea.height - nodeHeight) : '';
            cssProps.bottom = _.isNumber(cssProps.bottom) ? Utils.minMax(cssProps.bottom, availableArea.bottom, availableArea.bottom + availableArea.height - nodeHeight) : '';

            rootNode.css(cssProps);
            return this;
        };

        /**
         * Returns the root DOM node of this pop-up, as jQuery object.
         *
         * @returns {jQuery}
         *  The root DOM node of this pop-up.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns whether the pop-up node is currently visible.
         */
        this.isVisible = function () {
            return rootNode.parent().length > 0;
        };

        /**
         * Shows the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.show = function () {

            var // position and size of the anchor node in the browser window
                anchorPosition = null;

            // do nothing if the pop-up node is already open
            if (this.isVisible()) { return this; }

            // notify all listeners
            this.trigger('popup:beforeshow');

            // initialize DOM
            $('body').append(rootNode);

            // enable window resize handler which recalculates position and size of the pop-up node
            if (autoLayout) {
                $(window).on('resize', refreshNodePosition);
                refreshNodePosition();
            } else {
                anchorPosition = Utils.getNodePositionInPage(anchorNode);
                rootNode.css({
                    top: anchorPosition.top + anchorPosition.height + anchorPadding,
                    left: anchorPosition.left
                });
            }

            // notify all listeners
            this.trigger('popup:show');

            // start a timer that regularly refreshes the position of the pop-up node, and
            // hides the pop-up node automatically when the anchor node becomes inaccessible
            lastActiveElement = null;
            lastAnchorPosition = getAnchorPositionInWindow();
            refreshTimer = window.setInterval(refreshNode, Modernizr.touch ? 200 : 50);

            return this;
        };

        /**
         * Hides the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.hide = function () {

            // do nothing if the pop-up node is not visible
            if (!this.isVisible()) { return this; }

            // stop the automatic refresh timer
            window.clearInterval(refreshTimer);
            $(window).off('resize', refreshNodePosition);

            // notify all listeners
            this.trigger('popup:beforehide');

            // initialize DOM
            rootNode.detach();

            // notify all listeners
            this.trigger('popup:hide');

            return this;
        };

        /**
         * Toggles the visibility of the pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.toggle = function () {
            return this.isVisible() ? this.hide() : this.show();
        };

        /**
         * Returns the content node of this pop-up node, as jQuery object. All
         * contents to be shown in the pop-up have to be inserted into this
         * content node.
         *
         * @returns {jQuery}
         *  The content node of this pop-up.
         */
        this.getContentNode = function () {
            return contentNode;
        };

        /**
         * Removes all contents from this pop-up node.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.clearContents = function () {
            contentNode.empty();
            return this;
        };

        /**
         * Appends the passed DOM elements to the contents of this pop-up node.
         *
         * @param {HTMLElement|jQuery|String} [nodes ...]
         *  The DOM nodes to be inserted into this pop-up node, as plain HTML
         *  element, as jQuery collection, or as HTML mark-up string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.appendContentNodes = function () {
            contentNode.append.apply(contentNode, _.toArray(arguments));
            return this;
        };

        /**
         * Inserts the passed HTML mark-up into the content node of this pop-up
         * (all existing contents will be removed).
         *
         * @param {String} markup
         *  The HTML mark-up to be inserted into this pop-up node, as string.
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.setContentMarkup = function (markup) {
            contentNode[0].innerHTML = markup;
            return this;
        };

        /**
         * Scrolls the passed child node into the visible area of this pop-up
         * node.
         *
         * @param {HTMLElement|jQuery} childNode
         *  The DOM element that will be made visible by scrolling this pop-up
         *  node. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} [options]
         *  A map of options to control the scroll action. Supports all options
         *  that are supported by the method Utils.scrollToChildNode().
         *
         * @returns {BasePopup}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (childNode, options) {
            Utils.scrollToChildNode(rootNode, childNode, options);
            return this;
        };

        // initialization -----------------------------------------------------

        // additional CSS classes passed to constructor
        rootNode.addClass(Utils.getStringOption(initOptions, 'classes', ''));

        // disable context menu
        rootNode.on('contextmenu', function (event) {
            if (!$(event.target).is('input,textarea')) { return false; }
        });

        // disable dragging of controls (otherwise, it is possible to drag buttons and other controls around)
        rootNode.on('contextmenu dragstart', Utils.BUTTON_SELECTOR + ',input,textarea,label', false);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            this.hide();
            rootNode = contentNode = null;
        });

    } // class BasePopup

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: BasePopup });

});
