/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/radiolist',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/tk/popup/listmenu'
    ], function (Utils, Group, MenuMixin, ListMenu) {

    'use strict';

    // class RadioList ========================================================

    /**
     * Creates a drop-down list control used to hold a set of radio buttons.
     *
     * @constructor
     *
     * @extends Group
     * @extends ListMixin
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the drop-down button and
     *  menu. Supports all options of the Group base class, of the MenuMixin
     *  mix-in class, the ListMenu class used for the drop-down menu, and all
     *  generic formatting options for buttons (see method Utils.createButton()
     *  for details). Additionally, the following options are supported:
     *  @param {Boolean|Function} [initOptions.highlight=false]
     *      If set to true, the drop-down button will be highlighted, if an
     *      existing list item in the drop-down menu is active. The drop-down
     *      button will not be highlighted, if the value of this group is
     *      undetermined (value null), or if there is no option button present
     *      with the current value of this group. If set to false, the
     *      drop-down button will never be highlighted, even if a list item in
     *      the drop-down menu is active. If set to a function, it will be
     *      called every time after a list item will be activated. The
     *      drop-down button will be highlighted if this handler function
     *      returns true. Receives the value of the selected/activated list
     *      item as first parameter (also if this value does not correspond to
     *      any existing list item). Will be called in the context of this
     *      radio group instance.
     *  @param {Any} [initOptions.toggleValue]
     *      If set to a value different to null, the option button that is
     *      currently active can be clicked to be switched off. In that case,
     *      this radio group will activate the button associated to the value
     *      specified in this option, and the action handler will return this
     *      value instead of the value of the button that has been switched
     *      off.
     *  @param {Any} [initOptions.splitValue]
     *      If set to a value different to null, a separate option button will
     *      be inserted next to the drop-down button, representing the value of
     *      this option.
     *  @param {String} [initOptions.updateCaptionMode='all']
     *      Specifies how to update the caption of the drop-down button when a
     *      list item in the drop-down menu has been activated. Supports the
     *      keywords 'all' and 'none', or a space separated list containing the
     *      string tokens 'icon', 'label', and 'labelCss'.
     *      - 'icon': copies the icon of the list item to the drop-down button.
     *      - 'label': copies the label of the list item to the drop-down
     *          button.
     *      - 'labelCss': copies the label CSS formatting of the list item to
     *          the label of the drop-down button.
     *      The keyword 'all' is equivalent to 'icon label labelCss'. The
     *      keyword 'none' is equivalent to the empty string.
     *  @param {Function} [initOptions.equality=_.isEqual]
     *      A comparison function that returns whether an arbitrary value
     *      should be considered being equal to the value of a list item in the
     *      drop-down menu. If omitted, uses _.isEqual() which compares arrays
     *      and objects deeply.
     */
    function RadioList(initOptions) {

        var // self reference
            self = this,

            // whether to highlight the drop-down menu button
            highlight = Utils.getBooleanOption(initOptions, 'highlight', false),

            // custom predicate callback for drop-down menu button highlighting
            highlightHandler = Utils.getFunctionOption(initOptions, 'highlight'),

            // fall-back value for toggle click
            toggleValue = Utils.getOption(initOptions, 'toggleValue'),

            // value of the split button
            splitValue = Utils.getOption(initOptions, 'splitValue'),

            // which parts of a list item caption will be copied to the menu button
            updateCaptionMode = Utils.getStringOption(initOptions, 'updateCaptionMode', 'all'),

            // comparator for list item values
            equality = Utils.getFunctionOption(initOptions, 'equality'),

            // separate split button in split mode
            splitButton = (!_.isUndefined(splitValue) && !_.isNull(splitValue)) ? Utils.createButton(Utils.extendOptions(initOptions, { value: splitValue })) : $(),

            // the drop-down button
            menuButton = Utils.createButton((splitButton.length === 0) ? initOptions : null);

        // base constructors --------------------------------------------------

        Group.call(this, Utils.extendOptions(initOptions, { value: splitValue, highlight: null }));
        MenuMixin.call(this, new ListMenu(this.getNode(), initOptions), Utils.extendOptions(initOptions, { button: menuButton }));

        // private methods ----------------------------------------------------

        /**
         * Scrolls the drop-down menu to make the specified list item visible.
         */
        function scrollToListItem(buttonNode) {
            if ((buttonNode.length > 0) && self.isMenuVisible()) {
                self.getMenu().scrollToChildNode(buttonNode);
            }
        }

        /**
         * Handles 'menu:beforeopen' events.
         */
        function menuBeforeOpenHandler() {
            self.getMenu().getSectionNodes().css({ minWidth: self.getNode().outerWidth() });
        }

        /**
         * Handles 'menu:open' events.
         */
        function menuOpenHandler() {
            scrollToListItem(Utils.getSelectedButtons(self.getMenu().getItemNodes()));
        }

        /**
         * Activates an option button in this radio group.
         *
         * @param {Any} value
         *  The value associated to the button to be activated. If set to null,
         *  does not activate any button (ambiguous state).
         */
        function itemUpdateHandler(value) {

            var // the target caption button
                buttonNode = self.getCaptionButton(),
                // all list item buttons
                buttonNodes = self.getMenu().getItemNodes(),
                // activate an option button
                selectedButtons = Utils.selectOptionButton(buttonNodes, value, equality),
                // the options used to create the list item button
                buttonOptions = selectedButtons.data('options') || {},
                // the options used to set the caption of the drop-down menu button
                captionOptions = _.clone(self.getOptions()),
                // whether the drop-down button will be highlighted
                isHighlighted = false;

            // update visibility of the list items
            buttonNodes.each(function () {
                var options = $(this).data('options'),
                    visible = Utils.getFunctionOption(options, 'visible');
                visible = _.isFunction(visible) ? visible.call(self, value) : Utils.getBooleanOption(options, 'visible', true);
                $(this).css('display', visible ? '' : 'none');
            });

            // highlight the drop-down button (call custom handler, if available)
            if (!_.isUndefined(value) && !_.isNull(value)) {
                if (_.isFunction(highlightHandler)) {
                    isHighlighted = highlightHandler.call(self, value);
                } else {
                    isHighlighted = highlight && (selectedButtons.length > 0);
                }
            } else if (_.isNull(value)) {
                isHighlighted = null;
            }
            Utils.selectButtons(self.getMenuButton().add(buttonNode), isHighlighted);

            // update the caption of the drop-down menu button
            if (updateCaptionMode.length > 0) {
                _(updateCaptionMode).each(function (name) {
                    if (name in buttonOptions) {
                        captionOptions[name] = buttonOptions[name];
                    }
                });
                Utils.setControlCaption(buttonNode, captionOptions);
            }

            // update the data-value attribute of the group from all selected items
            self.getNode().attr('data-value', selectedButtons.map(function () {
                return $(this).attr('data-value');
            }).get().join(','));
        }

        /**
         * Returns the value of the clicked option button, taking the option
         * 'toggleValue' passed to the constructor into account.
         */
        function itemClickHandler(buttonNode) {
            var toggleClick = Utils.isButtonSelected(buttonNode) && !_.isNull(toggleValue) && !_.isUndefined(toggleValue);
            return toggleClick ? toggleValue : Utils.getControlValue(buttonNode);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the button element that contains the caption.
         *
         * @returns {jQuery}
         *  The caption button. If this radio group contains a split button
         *  (see the option 'splitValue' passed to the constructor), the split
         *  button will be returned, otherwise the drop-down button.
         */
        this.getCaptionButton = function () {
            return (splitButton.length > 0) ? splitButton : menuButton;
        };

        /**
         * Adds a new section to the drop-down menu, if it does not exist yet.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @param {Object} [options]
         *  A map of options to control the appearance of the section. Supports
         *  all options that are supported by the method
         *  ListMenu.createSectionNode().
         *
         * @returns {RadioList}
         *  A reference to this instance.
         */
        this.createMenuSection = function (sectionId, options) {
            this.getMenu().createSectionNode(sectionId, options);
            return this;
        };

        /**
         * Adds a new option button to this radio list.
         *
         * @param {String} sectionId
         *  The identifier of the menu section where the new option button will
         *  be inserted.
         *
         * @param {Any} value
         *  The unique value associated to the option button. MUST NOT be null
         *  or undefined.
         *
         * @param {Object} [options]
         *  A map of options to control the properties of the new option
         *  button. Supports all options supported by the method
         *  ListMenu.createItemNode(), except the option 'value' which will be
         *  set to the 'value' parameter passed to this method.
         *
         * @returns {RadioList}
         *  A reference to this instance.
         */
        this.createOptionButton = function (sectionId, value, options) {
            options = Utils.extendOptions(options, { value: value });
            this.getMenu().createItemNode(sectionId, options).data('options', options);
            return this;
        };

        // initialization -----------------------------------------------------

        // add the split button to the group
        if (splitButton.length > 0) {
            this.addFocusableControl(splitButton);
            this.registerChangeHandler('click', { source: splitButton });
            // convert ENTER and SPACE keys to click events
            Utils.setButtonKeyHandler(splitButton);
        }

        // add the drop-down button to the group
        this.addFocusableControl(menuButton);

        // initialize caption update mode
        switch (updateCaptionMode) {
        case 'all':
            updateCaptionMode = ['icon', 'label', 'labelCss'];
            break;
        case 'none':
            updateCaptionMode = [];
            break;
        default:
            updateCaptionMode = updateCaptionMode.split(/\s+/);
        }

        // convert ENTER, SPACE, an TAB keys in drop-down menu to click events
        Utils.setButtonKeyHandler(this.getMenuNode(), { tab: true });

        // highlight list entries, handle click events in the list
        this.registerUpdateHandler(itemUpdateHandler);
        this.registerChangeHandler('click', { source: this.getMenuNode(), selector: Utils.BUTTON_SELECTOR, valueResolver: itemClickHandler });

        // register event handlers
        this.on({
            'menu:beforeopen': menuBeforeOpenHandler,
            'menu:open': menuOpenHandler
        });

        // refresh list item highlighting after inserting new items
        this.getMenu().on('create:item', function () { self.refresh(); });

    } // class RadioList

    // exports ================================================================

    // derive this class from class Group
    return Group.extend({ constructor: RadioList });

});
