/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/editframework/model/format/lineheight',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/text/app/config',
     'io.ox/office/text/view/controls',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox',
     'gettext!io.ox/office/text',
     'less!io.ox/office/text/view/style.less'
    ], function (Utils, KeyCodes, LineHeight, EditView, TextConfig, Controls, BaseControls, Pane, ToolBox, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        TextField = Controls.TextField,
        RadioGroup = Controls.RadioGroup,
        RadioList = Controls.RadioList,

        // tool pane floating over the bottom of the application pane
        bottomOverlayPane = null,

        // the tool box containing page and zoom control groups
        bottomToolBox = null,

        // predefined zoom factors
        ZOOM_FACTORS = [35, 50, 75, 100, 150, 200, 300, 400, 600, 800],

        // log focus/scroll events in debug mode
        LOG_FOCUS_EVENTS = false;

    if (LOG_FOCUS_EVENTS) {
        $(document).on('focusin focusout', function (event) {
            Utils.info(event.type + ' ' + event.target.nodeName + '.' + (event.target.className || '').replace(/ /g, '.'));
        });
    }

    // class TextView =========================================================

    /**
     * @constructor
     *
     * @extends EditView
     */
    function TextView(app) {

        var // self reference
            self = this,

            // the document model
            model = null,

            // the scrollable document content node
            contentRootNode = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // debug operations for replay
            replayOperations = null,

            // edit field for the debug html code for the page
            pageHtmlCode = null,

            // the status label for the page number and zoom factor
            statusLabel = new BaseControls.StatusLabel(app, { type: 'info', fadeOut: true }),

            // current zoom type (percentage or keyword)
            zoomType = 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            deferredInitHandler: deferredInitHandler,
            grabFocusHandler: grabFocusHandler,
            contentScrollable: true,
            contentMargin: 30,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler() {
            model.getNode().focus();
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection() {

            var // the boundaries of the text cursor
                boundRect = model.getSelection().getFocusPositionBoundRect(),
                boundRectHelper = _.clone(boundRect); //copy is needed for modifying properties of obj according to zoom

            //for those browsers using css zoom instead of scale, recalculate cursor coordinates according to zoom factor
            if (boundRectHelper && !(_.browser.Firefox || _.browser.Opera || _.browser.IE)) {
                boundRectHelper.bottom = (boundRect.bottom) * (zoomFactor / 100);
                boundRectHelper.height = boundRect.height * (zoomFactor / 100);
                boundRectHelper.top = (boundRect.top) * (zoomFactor / 100);
                boundRectHelper.left = (boundRect.left) * (zoomFactor / 100);
                boundRectHelper.right = (boundRect.right) * (zoomFactor / 100);
            }

            // make cursor position visible in application pane
            if (boundRectHelper) {
                self.scrollToPageRectangle(boundRectHelper);
            }
        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            model.getSelection().registerInsertText(options);
        }

        /**
         * Configure the global 'Search and Replace' tool bar.
         */
        function configureSearchAndReplace() {

            var win = app.getWindow(),
                controller = app.getController(),
                scrollTimer = null,
                lastKeyTab = false,
                // the ox search panel
                searchDiv = win.nodes.search,
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                searchOffset,
                // additional controls for search & replace
                nextButton = $('<button>')
                    .addClass('btn margin-right')
                    .append(Utils.createIcon('icon-chevron-right'))
                    .attr('tabindex', 0),
                previousButton = $('<button>')
                    .addClass('btn')
                    .append(Utils.createIcon('icon-chevron-left'))
                    .css({'border-radius': '4px 0 0 4px'})
                    .attr('tabindex', 0),
                replaceInput = $('<input>', { type: 'text', name: 'replace' })
                    .css({'border-radius': '14px 14px 14px 14px'})
                    .attr({placeholder: gt('Replace with ...'), tabindex: 0 })
                    .addClass('search-query margin-right input-large'),
                replaceButton = $('<button>')
                    .addClass('btn')
                    .css({'border-radius': '4px 0 0 4px'})
                    .attr('tabindex', 0)
                    .text(gt('Replace')),
                replaceAllButton = $('<button>')
                    .addClass('btn')
                    .attr('tabindex', 0)
                    .text(/*#. in search&replace: replace all occurrences */ gt('All'));

            // Handles the 'search' event from the search panel.
            function searchHandler(event, searchQuery) {
                controller.executeItem('document/search/highlight', searchQuery);
            }

            // Handles the 'search:close' event from the search panel.
            function searchCloseHandler() {
                // since we add the replace input ourself, we need to clear it on close
                searchDiv.css('z-index', '1');
                win.nodes.search.find('input[name="replace"]').val('');
                controller.executeItem('document/search/close');
                if (scrollTimer) { scrollTimer.abort(); }
            }

            // Handles the 'search:open' event from the search panel.
            function searchOpenHandler() {
                searchDiv.css('z-index', '100');

                app.executeDelayed(function () {
                    // execute delayed, otherwise the offset is wrong
                    searchOffset = win.nodes.search.offset();
                    scrollTimer = app.repeatDelayed(scrollHandler, { delay: 1000, repeatDelay: 500 });
                });
            }

            // Handles the 'orientationchange' event of mobile devices
            function orientationChangeHandler(event) {
                // use event.orientation and media queries for the orientation detection.
                // 'window.orientation' depends on where the device defines it's top
                // and therefore says nothing reliable about the orientation.
                if (event && event.orientation === 'landscape') {
                    setLargeSearchPane();
                } else if (event && event.orientation === 'portrait') {
                    setSmallSearchPane();
                } else if (Modernizr.mq('(orientation: landscape)')) {
                    setLargeSearchPane();
                } else if (Modernizr.mq('(orientation: portrait)')) {
                    setSmallSearchPane();
                }
            }

            // Handles the 'scroll' event from the window.
            function scrollHandler() {

                var offset = _.clone(searchOffset);
                // if the window has been scrolled move by scrollY
                if (window.scrollY > searchOffset.top) {
                    offset.top = window.scrollY;
                }
                // set offset for the side pane
                if (!self.isSidePaneVisible()) {
                    offset.left = 0;
                }
                searchDiv.offset(offset);
                return win.search.active;
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    lastKeyTab = true;
                    event.preventDefault();

                    items = $(this).find('[tabindex]');
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;

                } else if (event.keyCode === KeyCodes.ESCAPE) {
                    win.search.close();
                }

                lastKeyTab = false;
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                if (lastKeyTab) {
                    event.stopImmediatePropagation();
                    // if we stop the event propagation we need to update the query here
                    win.search.query = win.search.getQuery();
                    return false;
                }
            }

            function setLargeSearchPane() {
                searchForm.find('input').addClass('input-large').removeClass('input-medium');
            }

            function setSmallSearchPane() {
                searchForm.find('input').addClass('input-medium').removeClass('input-large');
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            searchForm.addClass('f6-target');
            searchInput.removeClass('input-xlarge').addClass('input-large');
            searchInput.parent().css('margin-bottom', 0);

            // set the tool tips
            Utils.setControlTooltip(searchInput, gt('Find text'), 'bottom');
            Utils.setControlTooltip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Utils.setControlTooltip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Utils.setControlTooltip(previousButton, gt('Select previous search result'), 'bottom');
            Utils.setControlTooltip(nextButton, gt('Select next search result'), 'bottom');
            Utils.setControlTooltip(replaceInput, gt('Replacement text'), 'bottom');
            Utils.setControlTooltip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Utils.setControlTooltip(replaceAllButton, gt('Replace all search results'), 'bottom');
            Utils.setControlTooltip(searchForm.find('a.close'), gt('Close search'), 'bottom');

            // add additional controls
            searchForm.find('div.input-append').append(
                $('<div>').addClass('btn-group').css('vertical-align', 'top').append(previousButton, nextButton),
                $('<label>').append(replaceInput),
                $('<div>').addClass('btn-group').css('vertical-align', 'top').append(replaceButton, replaceAllButton)
            );

            // event handling
            previousButton.on('click', function () { controller.executeItem('document/search/previousResult'); });
            nextButton.on('click', function () { controller.executeItem('document/search/nextResult'); });
            replaceButton.on('click', function () { controller.executeItem('document/search/replaceSelected'); });
            replaceAllButton.on('click', function () { controller.executeItem('document/search/replaceAll'); });
            searchForm.on('keydown', keyHandler);
            replaceInput.on('change', function () { win.search.replacement = Utils.cleanString(this.value || ''); });
            searchInput.on('keydown', function (event) { if (KeyCodes.matchKeyCode(event, 'ENTER')) { event.preventDefault(); } });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler);

            win.on('search', searchHandler).on('search:close', searchCloseHandler);

            app.getModel().on('change:editmode', editmodeChangeHandler);

            // when scrolling on mobile devices we need to
            // place the search & replace into the view port
            if (Modernizr.touch) {
                win.on('search:open', searchOpenHandler);
                $(window).on('orientationchange', orientationChangeHandler);
                orientationChangeHandler();
            }
        }

        /**
         * Shows the current zoom factor in the status label for a short time
         *
         */
        function updateZoomStatus() {
            _.defer(function () {
                statusLabel.setValue(
                    //#. %1$d is the current zoom factor, in percent
                    //#, c-format
                    gt('Zoom: %1$d%', _.noI18n(Math.round(zoomFactor)))
                );
            });
        }

        /**
         * Creates necessary margin values when using transform scale css style for Firefox, IE and Opera
         *
         */
        function recalculateDocumentMargin() {
            var
                marginAddition = 0,
                //cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = $(self.getAppPaneNode().find('.page.user-select-text')),
                initialWidth = cachedDocumentNode.width() + parseInt(cachedDocumentNode.css('padding-left'), 10) + parseInt(cachedDocumentNode.css('padding-right'), 10),
                initialHeight = cachedDocumentNode.height() + parseInt(cachedDocumentNode.css('padding-top'), 10) + parseInt(cachedDocumentNode.css('padding-bottom'), 10);

            if (zoomFactor < 99) {
                marginAddition = 6;
            }
            cachedDocumentNode.css({
                'margin':  marginAddition + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px ' + (initialHeight * (zoomFactor / 100 - 1) + marginAddition) + 'px ' + ((initialWidth * (zoomFactor / 100 - 1)) / 2 + marginAddition) + 'px'
            });
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // the root node of the entire application pane
                appPaneNode = self.getAppPaneNode(),
                // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1,
                // deferred and debounced scroll handler, registering insertText operations immediately
                scrollToSelectionDebounced = app.createDebouncedMethod(registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500 });

            // collect operations
            function collectOperations(event, operations, external) {
                if (model.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) {
                    return;
                } else {
                    delayedRecordingActive = true;
                    app.executeDelayed(function () {
                        var // full JSON string
                            fullJSONString = replayOperations.getFieldValue(),
                            // json part string to be appended
                            jsonPartString = '',
                            // current part to be processed
                            operations = pendingOperations.splice(0, 20),
                            // first entry
                            firstEntry = (fullJSONString.length === 0);

                        _(operations).each(function (entry) {

                            // skipping operations, that were 'optimized' in 'sendActions' in the
                            // optimizeOperationHandler. Otherwise 'merged operations' are still
                            // followed by the operations, that will not be sent to
                            // the server, because they were removed by the 'optimizeOperationHandler'.
                            var skipOperation = false,
                                opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                            // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                            // only happen for internal operations (task 29601)
                            if (! entry.external) {
                                if (_.isNumber(opl) && (opl > 1)) {
                                    oplCounter = opl;
                                } else if (oplCounter > 1) {
                                    oplCounter--;
                                    skipOperation = true;
                                }
                            }

                            if (! skipOperation) {
                                if (!firstEntry) {
                                    jsonPartString += ',';
                                }
                                jsonPartString += JSON.stringify(entry.operation);
                                firstEntry = false;
                            }
                        });

                        if (fullJSONString.length >= 2) {
                            if (fullJSONString.charAt(0) === '[') {
                                fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                            }
                        }
                        fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                        replayOperations.setValue(fullJSONString);

                        delayedRecordingActive = false;
                        if (pendingOperations.length > 0) {
                            recordOperations();
                        }
                    }, {delay: 50});
                }
            }

            // initialize other instance fields
            model = app.getModel();
            contentRootNode = self.getContentRootNode();

            // configure the OX search bar for OX Text search&replace
            configureSearchAndReplace();

            // insert the editor root node into the application pane
            self.insertContentNode(model.getNode());

            // make the application pane focusable for global navigation with F6 key
            appPaneNode.addClass('f6-target');

            // handle scroll position when hiding/showing the entire application
            app.getWindow().on({
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    _.defer(function () { appPaneNode.css('opacity', 1); });
                }
            });

            // handle selection change events
            model.on('selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor
                scrollToSelectionDebounced(options);
            });

            // log current selection in debug mode
            if (TextConfig.isDebug()) {

                // create the output nodes in the debug pane
                self.addDebugInfoHeader('selection', { tooltip: 'Current selection' })
                    .addDebugInfoNode('type', { tooltip: 'Type of the selection' })
                    .addDebugInfoNode('start', { tooltip: 'Start position' })
                    .addDebugInfoNode('end', { tooltip: 'End position' })
                    .addDebugInfoNode('dir', { tooltip: 'Direction' });

                // log all selection events
                model.on('selection', function (event, selection) {
                    self.setDebugInfoText('selection', 'type', selection.getSelectionType())
                        .setDebugInfoText('selection', 'start', selection.getStartPosition().join(', '))
                        .setDebugInfoText('selection', 'end', selection.getEndPosition().join(', '))
                        .setDebugInfoText('selection', 'dir', selection.isTextCursor() ? 'cursor' : selection.isBackwards() ? 'backwards' : 'forwards');
                });

                // operations to replay
                replayOperations = new TextField({ tooltip: _.noI18n('Paste operations here'), select: true, fullWidth: false });

                // page html code
                pageHtmlCode = new TextField({ tooltip: _.noI18n('Page html code'), select: true, fullWidth: false });

                // record operations after importing the document
                app.on('docs:import:after', function () {
                    // connect the 'operations:after' event to the collectOperations function
                    app.getModel().on('operations:after', collectOperations);
                });
            }
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            var // ODF does not support inner borders at paragraphs -> special option only for this case
                displayInnerBorders = !app.isODF(),
                // the table tool box container
                tableToolBox = null,
                // the insert tool box container
                insertToolBox = null,
                // optional table size settings
                tableSizeOptions = { maxCols: TextConfig.getMaxTableColumns(), maxRows: TextConfig.getMaxTableRows(), maxCells: TextConfig.getMaxTableCells() },
                // whether it shall be possible to insert tables (Fix for 28945)
                hideInsertTable = (_.isNumber(tableSizeOptions.maxCols) && tableSizeOptions.maxCols === 0) ||
                                  (_.isNumber(tableSizeOptions.maxRows) && tableSizeOptions.maxRows === 0) ||
                                  (_.isNumber(tableSizeOptions.maxCells) && tableSizeOptions.maxCells === 0);

            self.createToolBox('font', { label: gt('Font'), visible: 'document/editable/text' })
                .addGroup('character/fontname', new Controls.FontFamilyPicker(app, { width: 117 }))
                .addRightTab()
                .addGroup('character/fontsize', new Controls.FontHeightPicker({ width: 47 }))
                .newLine()
                .addGroup('character/bold',      new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic',    new Button(Controls.ITALIC_OPTIONS))
                .addGroup('character/underline', new Button(Controls.UNDERLINE_OPTIONS))
                .addGroup('character/strike',    new Button(Controls.STRIKEOUT_OPTIONS))
                .addGap(11)
                .addGroup('character/vertalign', new RadioGroup({ toggleValue: 'baseline' })
                    .createOptionButton('sub',   { icon: 'docs-font-subscript',   tooltip: gt('Subscript') })
                    .createOptionButton('super', { icon: 'docs-font-superscript', tooltip: gt('Superscript') })
                )
                .newLine()
                .addGroup('character/color',     new Controls.ColorPicker(app, 'text', { icon: 'docs-font-color',      tooltip: gt('Text color') }))
                .addGap()
                .addGroup('character/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-font-fill-color', tooltip: /*#. fill color behind single characters */ gt('Text highlight color') }))
                .addRightTab()
                .addGroup('character/resetAttributes', new Button(Controls.CLEAR_FORMAT_OPTIONS));

            self.createToolBox('paragraph', { label: gt('Paragraph'), visible: 'document/editable/text' })
                .addGroup('paragraph/stylesheet', new Controls.ParagraphStylePicker(app, { fullWidth: true }))
                .newLine()
                .addGroup('paragraph/alignment', new RadioList({ icon: 'docs-para-align-left', tooltip: gt('Paragraph alignment'), updateCaptionMode: 'icon' })
                    .createOptionButton('', 'left',    { icon: 'docs-para-align-left',    label: /*#. alignment of text in paragraphs or cells */ gt('Left') })
                    .createOptionButton('', 'center',  { icon: 'docs-para-align-center',  label: /*#. alignment of text in paragraphs or cells */ gt('Center') })
                    .createOptionButton('', 'right',   { icon: 'docs-para-align-right',   label: /*#. alignment of text in paragraphs or cells */ gt('Right') })
                    .createOptionButton('', 'justify', { icon: 'docs-para-align-justify', label: /*#. alignment of text in paragraphs or cells */ gt('Justify') })
                )
                .addGap()
                .addGroup('paragraph/lineheight', new RadioList({ icon: 'docs-para-line-spacing-100', tooltip: gt('Line spacing'), updateCaptionMode: 'icon' })
                    .createOptionButton('', LineHeight.SINGLE,   { icon: 'docs-para-line-spacing-100', label: /*#. text line spacing in paragraphs */ gt('100%') })
                    .createOptionButton('', LineHeight._115,     { icon: 'docs-para-line-spacing-115', label: /*#. text line spacing in paragraphs */ gt('115%') })
                    .createOptionButton('', LineHeight.ONE_HALF, { icon: 'docs-para-line-spacing-150', label: /*#. text line spacing in paragraphs */ gt('150%') })
                    .createOptionButton('', LineHeight.DOUBLE,   { icon: 'docs-para-line-spacing-200', label: /*#. text line spacing in paragraphs */ gt('200%') })
                )
                .addGap()
                .addGroup('paragraph/borders', new Controls.BorderPicker({ tooltip: gt('Paragraph borders'), showInsideHor: displayInnerBorders }))
                .addGap()
                .addGroup('paragraph/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-para-fill-color', tooltip: gt('Paragraph fill color') }))
                .newLine()
                .addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(app))
                .addGap()
                .addGroup('paragraph/list/numbered', new Controls.NumberedListStylePicker(app))
                .addRightTab()
                .addGroup('paragraph/list/decindent', new Button({ icon: 'docs-list-dec-level', tooltip: /*#. indentation of lists (one list level up) */ gt('Demote one level') }))
                .addGroup('paragraph/list/incindent', new Button({ icon: 'docs-list-inc-level', tooltip: /*#. indentation of lists (one list level down) */ gt('Promote one level') }));

            tableToolBox = self.createToolBox('table', { label: gt('Table'), visible: 'document/editable/table' });

            if (!app.isODF()) {
                tableToolBox
                    .addGroup('table/stylesheet', new Controls.TableStylePicker(app))
                    .addRightTab();
            }

            tableToolBox
                .addGroup('table/insert/row',    new Button({ icon: 'docs-table-insert-row',    tooltip: gt('Insert row') }))
                .addGroup('table/insert/column', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('table/delete/row',    new Button({ icon: 'docs-table-delete-row',    tooltip: gt('Delete selected rows') }))
                .addGroup('table/delete/column', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }));


            if (!app.isODF()) {
                tableToolBox
                    .newLine()
                    .addGroup('table/cellborder', new Controls.BorderPicker({ tooltip: gt('Cell borders'), showInsideHor: true, showInsideVert: true }))
                    .addGap()
                    .addGroup('table/borderwidth', new Controls.TableBorderWidthPicker({ tooltip: gt('Cell border width') }))
                    .addGap()
                    .addGroup('table/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }));
            }

            self.createToolBox('drawing', { label: /*#. drawing objects: images, diagrams, ... */ gt('Drawing'), visible: 'document/editable/drawing' })
                .addGroup('drawing/position', new RadioList({ icon: 'docs-drawing-inline', tooltip: /*#. alignment and text floating of drawing objects in text */ gt('Drawing position'), updateCaptionMode: 'icon' })
                    .createMenuSection('inline')
                    .createOptionButton('inline', 'inline',       { icon: 'docs-drawing-inline',       label: gt('Inline with text') })
                    .createMenuSection('left')
                    .createOptionButton('left', 'left:none',      { icon: 'docs-drawing-left-none',    label: gt('Left aligned, no text wrapping') })
                    .createOptionButton('left', 'left:right',     { icon: 'docs-drawing-left-right',   label: gt('Left aligned, text wraps at right side') })
                    .createMenuSection('right')
                    .createOptionButton('right', 'right:none',    { icon: 'docs-drawing-right-none',   label: gt('Right aligned, no text wrapping') })
                    .createOptionButton('right', 'right:left',    { icon: 'docs-drawing-right-left',   label: gt('Right aligned, text wraps at left side') })
                    .createMenuSection('center')
                    .createOptionButton('center', 'center:none',  { icon: 'docs-drawing-center-none',  label: gt('Centered, no text wrapping') })
                    .createOptionButton('center', 'center:left',  { icon: 'docs-drawing-center-left',  label: gt('Centered, text wraps at left side') })
                    .createOptionButton('center', 'center:right', { icon: 'docs-drawing-center-right', label: gt('Centered, text wraps at right side') })
                )
                .addRightTab()
                .addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_OPTIONS));

            insertToolBox = self.createToolBox('insert', { label: gt('Insert'), visible: 'document/editable/text' });
            if (! hideInsertTable) {
                insertToolBox
                    .addGroup('table/insert', new Controls.TableSizePicker(app, tableSizeOptions))
                    .addGap();
            }
            insertToolBox
                .addGroup('image/insert/dialog',        new Button({ icon: 'docs-image-insert',     tooltip: gt('Insert image') }))
                .addGap()
                .addGroup('character/hyperlink/dialog', new Button({ icon: 'docs-hyperlink',        tooltip: gt('Insert/Edit hyperlink') }))
                .addRightTab()
                .addGroup('character/insertTab',        new Button({ icon: 'docs-insert-tab',       tooltip: gt('Insert tab') }))
                .addGroup('character/insertLineBreak',  new Button({ icon: 'docs-insert-linebreak', tooltip: gt('Insert line break') }));

            self.createToolBox('spelling', { label: gt('Spelling'), visible: 'document/spelling/enabled' })
                .addGroup('character/language', new Controls.LanguagePicker({ width: 178 }))
                .addGap()
                .addGroup('document/onlinespelling', new Button({ icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true }));

            self.getOverlayToolBox()
                .addGroup('character/bold',   new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic', new Button(Controls.ITALIC_OPTIONS));

            if (TextConfig.isDebug()) {
                self.createDebugToolBox()
                    .addGap()
                    .addGroup('document/cut',   new Button({ icon: 'icon-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                    .addGroup('document/copy',  new Button({ icon: 'icon-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                    .addGroup('document/paste', new Button({ icon: 'icon-paste', tooltip: _.noI18n('Paste from clipboard') }))
                    .newLine()
                    .addGroup('format/spellchecking', new Button({ label: _.noI18n('ab'), tooltip: gt('Check spelling of selected text') }))
                    .addGap()
                    .addGroup('debug/accessrights', new Controls.AccessRightsPicker(app))
                    .newLine()
                    .addGroup('debug/recordoperations', new Button({ icon: 'icon-play-circle', css: {color: 'red'}, toggle: true, tooltip: _.noI18n('Record operations')}))
                    .addGroup('debug/replayoperations', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Replay operations')}))
                    .addGroup('debug/operations', replayOperations)
                    .newLine()
                    .addGroup('debug/getpagehtmlcode', new Button({ icon: 'docs-undo', tooltip: _.noI18n('Get page html code')}))
                    .addGroup('debug/setpagehtmlcode', new Button({ icon: 'docs-redo', tooltip: _.noI18n('Set page html code')}))
                    .addGroup('debug/pagehtmlcode', pageHtmlCode);
            }

            self.createToolBox('zoom', { fixed: 'bottom' }).addRightTab()
            .addGroup('zoom/dec',  new Button(BaseControls.ZOOMOUT_OPTIONS))
            .addGroup('zoom/type', new Controls.ZoomTypePicker({ preferSide: 'top', align: 'center' }))
            .addGroup('zoom/inc',  new Button(BaseControls.ZOOMIN_OPTIONS));

            // create the bottom overlay pane
            self.addPane(bottomOverlayPane = new Pane(app, 'overlaybottom', { position: 'bottom', classes: 'inline right', overlay: true, transparent: true, hoverEffect: true })
                .addViewComponent(bottomToolBox = new ToolBox(app, 'overlaypages'))
            );

            bottomToolBox
                .addGroup('zoom/dec',  new Button(BaseControls.ZOOMOUT_OPTIONS))
                .addGroup('zoom/type', new Controls.ZoomTypePicker())
                .addGroup('zoom/inc',  new Button(BaseControls.ZOOMIN_OPTIONS));

            // create the status overlay pane
            self.addPane(new Pane(app, 'statuslabel', { position: 'bottom', classes: 'inline right', overlay: true, transparent: true })
                .addViewComponent(new ToolBox(app, 'statuslabel', { landmark: false }).addPrivateGroup(statusLabel))
            );

            if (self.isSidePaneVisible()) {
                bottomOverlayPane.hide();
            }
            self.on('sidepane:visible', function (event, visible) { bottomOverlayPane.toggle(!visible); });
            self.on('refresh:layout', function () {
                if (zoomType === 'width') {
                    _.defer(function () { self.setZoomType('width'); });
                }
            });

            if (_.browser.Firefox || _.browser.IE || _.browser.Opera) {
                app.getModel().on('operations:after', recalculateDocumentMargin);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node) {
            Utils.scrollToChildNode(contentRootNode, node, { padding: 15 });
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            Utils.scrollToPageRectangle(contentRootNode, pageRect, { padding: 15 });
            return this;
        };

        /**
         * Provides the debug operations as string for the debugging
         * feature replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field or null
         *  if there is no debug toolbox.
         */
        this.getDebugOperationsContent = function () {
            if (replayOperations) {
                return replayOperations.getFieldValue();
            }
            return null;
        };

        /**
         * Receiving the html code of the edit field to insert it
         * into the document page.
         *
         * @returns {String}
         *  The content of the debug html code field or null
         *  if there is no debug toolbox or no content in it.
         */
        this.getPageHtmlCode = function () {
            var value = null;
            if (pageHtmlCode) {
                value = pageHtmlCode.getFieldValue();
                if (value === '') { value = null; }
            }
            return value;
        };

        /**
         * Setting html code into the debug edit field.
         *
         * @param {String} htmlCode
         *  The html code as string that will be displayed in
         *  the debug edit field.
         */
        this.setPageHtmlCode = function (htmlCode) {
            if (pageHtmlCode) {
                pageHtmlCode.setFieldValue(htmlCode);
            }
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            var
                cachedDocumentParentNode = this.getAppPaneNode(),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text')),
                documentWidth = cachedDocumentNode.width() + parseInt(cachedDocumentNode.css('padding-left'), 10) + parseInt(cachedDocumentNode.css('padding-right'), 10),
                parentWidth = cachedDocumentParentNode.width(),
                outerDocumentMargin = parseInt(cachedDocumentParentNode.find('.app-content').css('margin-left'), 10) + parseInt(cachedDocumentParentNode.find('.app-content').css('margin-right'), 10);

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'width') {
                this.increaseZoomLevel(((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMargin))) * 100); //20 is width of scrollbar
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                //marginAddition = 0,
                cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text'));

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (_.browser.Firefox || _.browser.Opera || _.browser.IE) {
                Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform-origin', 'top');

                recalculateDocumentMargin();
                cachedAppContentRoot.css({
                    'overflow': zoomInsertion > 99 ? '' : 'hidden'
                });
            } else {
                cachedDocumentNode.css({
                    'zoom': zoomInsertion / 100
                });
            }
            updateZoomStatus();
            zoomType = zoomFactor;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                //marginAddition = 0,
                cachedAppContentRoot = $(this.getAppPaneNode().find('.app-content')),
                cachedDocumentNode = $(this.getAppPaneNode().find('.page.user-select-text'));

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (_.browser.Firefox || _.browser.Opera || _.browser.IE) {
                Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(cachedDocumentNode, 'transform-origin', 'top');

                recalculateDocumentMargin();
                cachedAppContentRoot.css({
                    'overflow': zoomInsertion > 99 ? '' : 'hidden'
                });
            } else {
                cachedDocumentNode.css({
                    'zoom': zoomInsertion / 100
                });
            }
            updateZoomStatus();
            zoomType = zoomFactor;
            return this;
        };

    } // class TextView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: TextView });

});
