/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/text/editor',
     'io.ox/office/text/view/view',
     'io.ox/office/text/app/controller'
    ], function (Utils, EditApplication, TextModel, TextView, TextController) {

    'use strict';

    // class TextApplication ==================================================

    /**
     * The OX Text application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} [appOptions]
     *  A map of static application options, that have been passed to the
     *  static method BaseApplication.createLauncher().
     *
     * @param {Object} [launchOptions]
     *  A map of options to control the properties of the application. Supports
     *  all options supported by the base class EditApplication.
     */
    var TextApplication = EditApplication.extend({ constructor: function (appOptions, launchOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditApplication.call(this, TextModel, TextView, TextController, appOptions, launchOptions, {
            postProcessHandler: postProcessDocument,
            postProcessHandlerStorage: postProcessHandlerStorage,
            postProcessProgressSize: 0.7,
            optimizeOperationsHandler: optimizeOperations,
            operationFilter: operationFilter,
            sendActionsDelay: 1000,
            useStorage: (_.url.hash('use-storage') === 'true')
        });

        // private methods ----------------------------------------------------

        /**
         * Post-processes the document contents and formatting, after all its
         * import operations have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessDocument() {
            return self.getModel().updateDocumentFormatting();
        }

        /**
         * Load performance: Post-processes the document contents and formatting,
         * after the document was loaded from local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully, or rejected when an
         *  error has occurred.
         */
        function postProcessHandlerStorage() {
            return self.getModel().updateDocumentFormattingStorage();
        }

        /**
         * Optimizing the actions (operations) before sending them to the server.
         *
         * @param {Object[]} actionsBuffer
         *  An array with actions.
         *
         * @returns {Object[]}
         *  The optimized array with actions.
         */
        function optimizeOperations(actionsBuffer) {
            return self.getModel().optimizeActions(actionsBuffer);
        }

        /**
         * Load performance: Only selected operations need to be executed, if document
         * can be loaded from local storage.
         *
         * @param {Object[]} operation
         *  An operation object.
         *
         * @returns {Boolean}
         *  Whether the specified operation needs to be executed, although the document
         *  is loaded from the local storage.
         */
        function operationFilter(operation) {
            return self.getModel().operationFilter(operation);
        }

    }}); // class TextApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch text applications.
     */
    TextApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/text', TextApplication, { icon: 'icon-align-left', search: true });
    };

    // exports ================================================================

    return TextApplication;

});
