/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/selectionmixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils'
    ], function (Utils, KeyCodes, SheetUtils, PaneUtils) {

    'use strict';

    // mix-in class SelectionMixin ============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides methods to work
     * with cell selection.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function SelectionMixin(app, rootNode) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = null,

            // the model instance and collections of the active sheet
            sheetModel = null,
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            drawingCollection = null,

            // callback handlers to override cell selection behavior
            getCellSelectionHandler = null,
            setCellSelectionHandler = null;

        // private methods ----------------------------------------------------

        /**
         * Selects the previous or next drawing frame.
         */
        function selectNextDrawing(backwards) {

            var // last drawing selected in the active sheet
                position = _.last(self.getSelectedDrawings()),
                // number of drawings in the sheet
                count = drawingCollection.getModelCount();

            // TODO: selection in embedded objects
            if ((count > 1) && _.isArray(position) && (position.length === 1) && (0 <= position[0]) && (position[0] < count)) {
                position[0] = (backwards ? (position[0] + count - 1) : (position[0] + 1)) % count;
                self.selectDrawing(position);
                self.scrollToSelectedDrawing(position);
            }
        }

        /**
         * Returns a selection object that is suitable for keyboard navigation.
         * Prefers the registered cell selection handler.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.allowEmpty=false]
         *      If set to true, and the registered cell selection handler
         *      returns a selection without any ranges (property 'ranges' set
         *      to an empty array), this selection object will be returned
         *      unmodified. Otherwise, an empty selection returned by that
         *      handler will be replaced by a selection consisting of the
         *      current active cell.
         */
        function getActiveSelection(options) {

            var // the custom selection provided by the registered cell selection handler
                selection = _.isFunction(getCellSelectionHandler) ? getCellSelectionHandler.call(self) : null,
                // the active cell of the real sheet selection
                activeCell = null;

            if (!_.isObject(selection)) {
                // no custom selection: use real selection of this sheet
                selection = self.getSelection();
            } else if ((selection.ranges.length === 0) && !Utils.getBooleanOption(options, 'allowEmpty', false)) {
                // replace empty custom selection with active cell
                activeCell = self.getActiveCell();
                selection = {
                    ranges: [{ start: _.clone(activeCell), end: _.clone(activeCell) }],
                    activeRange: 0,
                    activeCell: activeCell
                };
            }

            return selection;
        }

        /**
         * Clears the current selection and selects the specified cell. If the
         * cell is hidden, tries to select the nearest visible cell.
         */
        function selectVisibleCell(address) {
            if ((address = self.getVisibleCell(address, 'prevNext'))) {
                self.selectCell(address);
                self.scrollToSelectedCell(address);
            }
        }

        /**
         * Expands the active range of the current selection to the passed cell
         * address.
         */
        function expandToVisibleCell(address, shrinkEnd) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the active range in the selection
                activeRange = selection.ranges[selection.activeRange];

            if ((address = self.getVisibleCell(address, 'prevNext'))) {

                // start position of active range: extend to passed address,
                // if it is located before, but do not shrink the active range
                activeRange.start[0] = Math.min(address[0], activeRange.start[0]);
                activeRange.start[1] = Math.min(address[1], activeRange.start[1]);

                // end position of active range: extend to passed address, if it
                // is located after the range, but shrink to active cell if not
                activeRange.end[0] = Math.max(address[0], shrinkEnd ? selection.activeCell[0] : activeRange.end[0]);
                activeRange.end[1] = Math.max(address[1], shrinkEnd ? selection.activeCell[1] : activeRange.end[1]);

                // commit new selection, adjust scrolling position
                self.changeActiveRange(activeRange);
                self.scrollToSelectedCell(address);
            }
        }

        /**
         * Searches for the next/last filled cell in the current cell
         * collection.
         */
        function findNextUsedCell(address, direction) {

            var // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),

                // the index of the cell address array element in move direction
                addressIndex = columns ? 0 : 1,
                // the cell collection of the active grid pane
                cellCollection = self.getActiveGridPane().getCellCollection(),
                // address of the last used cell
                lastUsedCell = self.getLastUsedCell(),
                // number of visited empty and filled cells
                emptyCount = 0, filledCount = 0;

            address = _.clone(address);

            // jump to end of sheet, if start cell is outside used area (forward mode)
            if (forward && ((address[0] >= lastUsedCell[0]) || (address[1] >= lastUsedCell[1]))) {
                address[addressIndex] = columns ? model.getMaxCol() : model.getMaxRow();
                return address;
            }

            // jump to end of used area, if start cell is outside used area (backward mode)
            if (!forward && (address[addressIndex] > lastUsedCell[addressIndex])) {
                address[addressIndex] = (address[1 - addressIndex] <= lastUsedCell[1 - addressIndex]) ? lastUsedCell[addressIndex] : 0;
                return address;
            }

            // cell outside the cell collection: no iteration possible
            if (!SheetUtils.rangeContainsCell(cellCollection.getRange(), address)) {
                return null;
            }

            // visit all available cells in the cell collection
            cellCollection.iterateNextCells(address, direction, function (cellData) {

                var // whether the current cell is empty
                    isEmpty = _.isNull(cellData.result);

                // update number of visited empty/filled cells
                if (isEmpty) { emptyCount += 1; } else { filledCount += 1; }

                // found an empty cell after at least two filled cells: jump to the last filled cell
                if (isEmpty && (emptyCount === 1) && (filledCount >= 2)) {
                    // return with the old state of the 'address' variable (last filled cell)
                    return Utils.BREAK;
                }

                // otherwise: always remember address of current visited cell
                address = cellData.address;

                // found a filled cell after any number of empty cells: jump to the filled cell
                if (!isEmpty && (emptyCount > 0) && (filledCount > 0)) {
                    return Utils.BREAK;
                }
            });

            // jump to end of sheet, if new cell is outside used area
            if (forward && (address[addressIndex] > lastUsedCell[addressIndex])) {
                address[addressIndex] = columns ? model.getMaxCol() : model.getMaxRow();
            }

            return address;
        }

        /**
         * Clears the current selection and moves the active cell into the
         * specified direction. If the target cell is hidden, moves the cursor
         * to the next available visible cell.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Moves the active cell to the next visible cell.
         *  - 'page':
         *      Moves the active cell by the width/height of the visible area,
         *      and adjusts the scroll position in a way that the new active
         *      cell remains on the same screen position if possible.
         *  - 'used':
         *      Moves the active cell to the next available non-empty cell; or,
         *      if none available, to the first or last cell in the sheet.
         */
        function selectNextCell(direction, mode) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the cell to start movement from
                activeCell = _.isArray(selection.originCell) ? selection.originCell : selection.activeCell,
                // the merged range covering the active cell
                mergedRange = mergeCollection.getMergedRange(activeCell),

                // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),

                // the index of the cell address array element in move direction
                addressIndex = columns ? 0 : 1,
                // the column/row collection in the move direction
                collection = columns ? colCollection : rowCollection,
                // the active (focused) grid pane
                gridPane = self.getActiveGridPane(),
                // the header pane in the move direction
                headerPane = self.getActiveHeaderPane(columns),

                // the new column/row index used to look for a visible column/row
                index = activeCell[addressIndex],
                // position and size of the active cell (for page scroll)
                cellPosition = null, cellOffset = 0, cellSize = 0,
                // position and size of the visible area in this grid pane (for page scroll)
                visiblePosition = null, visibleOffset = 0, visibleSize = 0,

                // the new column/row entry to move to
                entryData = null,
                // the target cell to move to (for 'used' mode)
                targetCell = null;

            // returns the descriptor of a new entry when moving in 'page' mode
            function getPageTargetEntry(offset) {
                return collection.getEntryByOffset(offset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
            }

            switch (mode) {
            case 'cell':
                // start from leading/trailing border of a merged range
                if (mergedRange) {
                    index = (forward ? mergedRange.end : mergedRange.start)[addressIndex];
                }
                index += (forward ? 1 : -1);

                // insert new column/row entry (old selection will collapse to active cell, if no other cell has been found)
                if ((entryData = collection.getVisibleEntry(index, forward ? 'next' : 'prev'))) {
                    activeCell[addressIndex] = entryData.index;
                }

                // set selection to the new active cell
                self.selectCell(activeCell, { storeOrigin: true });
                self.scrollToSelectedCell(activeCell);
                break;

            case 'page':
                // get position and size of the active cell
                cellPosition = mergedRange ? sheetModel.getRangeRectangle(mergedRange) : sheetModel.getCellRectangle(activeCell);
                cellOffset = cellPosition[columns ? 'left' : 'top'];
                cellSize = cellPosition[columns ? 'width' : 'height'];

                // get position and size of the visible area in this grid pane
                visiblePosition = gridPane.getVisibleRectangle();
                visibleOffset = visiblePosition[columns ? 'left' : 'top'];
                visibleSize = visiblePosition[columns ? 'width' : 'height'];

                // if the active cell is located inside the visible area, find the column/row one page away
                if ((visibleOffset <= cellOffset) && (cellOffset + cellSize <= visibleOffset + visibleSize)) {
                    entryData = getPageTargetEntry(cellOffset + cellSize / 2);
                    activeCell[addressIndex] = entryData.index;
                    self.selectCell(activeCell, { storeOrigin: true });
                    headerPane.scrollRelative(entryData.offset - cellOffset);
                }
                // otherwise, scroll one page and select the top visible entry
                else {
                    entryData = getPageTargetEntry(visibleOffset);
                    activeCell[addressIndex] = entryData.index;
                    self.selectCell(activeCell, { storeOrigin: true });
                    headerPane.scrollTo(entryData.offset);
                }
                break;

            case 'used':
                // search for a target cell
                targetCell = findNextUsedCell(activeCell, direction);

                // fail-safety: if cell address could not be determined or did not change, make a single cell step
                if (!targetCell || _.isEqual(activeCell, targetCell)) {
                    return selectNextCell(direction, 'cell');
                }

                // set selection to the new active cell
                self.selectCell(targetCell, { storeOrigin: true });
                self.scrollToSelectedCell(targetCell);
                break;
            }
        }

        /**
         * Extends the active range in the current selection into the specified
         * direction. If the target column/row is hidden, extends the active
         * range to the next available visible column/row.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         *
         * @param {String} mode
         *  The move mode. Supported values are:
         *  - 'cell':
         *      Expands the active range to the next visible column/row.
         *  - 'page':
         *      Expands the active range by the width/height of the visible
         *      area, and adjusts the scroll position in a way that the range
         *      border remains on the same screen position if possible.
         *  - 'used':
         *      Expands the active range to the next available non-empty cell;
         *      or, if none available, to the first or last column/row in the
         *      sheet.
         */
        function expandToNextCell(direction, mode) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the cell to start movement from
                activeCell = _.clone(_.isArray(selection.originCell) ? selection.originCell : selection.activeCell),
                // the current active range address
                activeRange = selection.ranges[selection.activeRange],
                // the merged range covering the active cell, or the active cell as range object
                mergedRange = mergeCollection.getMergedRange(activeCell) || _.copy({ start: activeCell, end: activeCell }, true),
                // the new active range address
                newRange = null,

                // whether to expand the active range forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to expand the active range horizontally
                columns = (direction === 'left') || (direction === 'right'),

                // the index of the cell address array element in move direction
                addressIndex = columns ? 0 : 1,
                // the column/row collection in the move direction
                collection = columns ? colCollection : rowCollection,
                // the active (focused) grid pane
                gridPane = self.getActiveGridPane(),
                // the header pane in the move direction
                headerPane = self.getActiveHeaderPane(columns),

                // the new column/row entry to move to
                entryData = null;

            // returns the descriptor of a new entry when moving in 'page' mode
            function getPageTargetEntry(index) {

                var // the descriptor for the start entry
                    entryData = collection.getEntry(index),
                    // the position in the middle of the entry
                    entryOffset = entryData.offset + entryData.size / 2,
                    // the size of the visible area, in pixels
                    visibleSize = gridPane.getVisibleRectangle()[columns ? 'width' : 'height'];

                // return the target entry
                return collection.getEntryByOffset(entryOffset + (forward ? visibleSize : -visibleSize) * 0.98, { pixel: true });
            }

            // returns the descriptor of a new entry when moving in 'used' mode
            function getUsedTargetEntry(address) {
                var targetCell = findNextUsedCell(address, direction);
                return targetCell ? collection.getVisibleEntry(targetCell[addressIndex], forward ? 'next' : 'prev') : null;
            }

            // first, try to shrink the range at the correct border according to move direction
            activeCell[addressIndex] = (forward ? activeRange.start : activeRange.end)[addressIndex];
            switch (mode) {
            case 'cell':
                entryData = forward ?
                    collection.getVisibleEntry(activeCell[addressIndex] + 1, 'next') :
                    collection.getVisibleEntry(activeCell[addressIndex] - 1, 'prev');
                break;

            case 'page':
                entryData = getPageTargetEntry(activeCell[addressIndex]);
                break;

            case 'used':
                entryData = getUsedTargetEntry(activeCell);
                break;
            }

            // build the new range address according to the collection entry
            if (entryData && (forward ? (entryData.index <= mergedRange.start[addressIndex]) : (mergedRange.end[addressIndex] <= entryData.index))) {
                newRange = _.copy(activeRange, true);
                (forward ? newRange.start : newRange.end)[addressIndex] = entryData.index;
                // shrink the range until it does not contain any merged ranges partly
                newRange = mergeCollection.shrinkRangeFromMergedRanges(newRange, columns ? 'columns' : 'rows');
            }

            // if shrinking the range is not possible or has failed, expand the range away from active cell
            if (!newRange) {
                activeCell[addressIndex] = (forward ? activeRange.end : activeRange.start)[addressIndex];

                switch (mode) {
                case 'cell':
                    entryData = forward ?
                        collection.getVisibleEntry(activeCell[addressIndex] + 1, 'next') :
                        collection.getVisibleEntry(activeCell[addressIndex] - 1, 'prev');
                    break;

                case 'page':
                    entryData = getPageTargetEntry(activeCell[addressIndex]);
                    break;

                case 'used':
                    entryData = getUsedTargetEntry(activeCell);
                    break;
                }

                // build the new range address according to the collection entry
                if (entryData) {
                    newRange = _.copy(activeRange, true);
                    // expand in move direction, restrict to active cell on the other border
                    (forward ? newRange.end : newRange.start)[addressIndex] = entryData.index;
                    (forward ? newRange.start : newRange.end)[addressIndex] = selection.activeCell[addressIndex];
                    // expand the range until it contains all merged ranges completely
                    newRange = mergeCollection.expandRangeToMergedRanges(newRange);
                }
            }

            // change the active range, if a new range has been found
            if (newRange && !_.isEqual(newRange, activeRange)) {
                self.changeActiveRange(newRange);
            }

            // scroll to target entry (this allows e.g. to scroll inside a large merged range without changing the selection)
            if (entryData) {
                headerPane.scrollToEntry(entryData.index);
            }
        }

        /**
         * Moves the active cell in the current selection into the specified
         * direction. Cycles through all ranges contained in the selection. In
         * case the selection is a single cell, moves it in the entire sheet.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         */
        function moveActiveCell(direction) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // move in entire sheet, if the selection consists of a single cell only
                entireSheet = self.isSingleCellSelection(),
                // whether to move the active cell forwards (right or down)
                forward = (direction === 'right') || (direction === 'down'),
                // whether to move the active cell horizontally
                columns = (direction === 'left') || (direction === 'right'),
                // the index of the cell address array element in primary direction
                index1 = columns ? 0 : 1,
                // the index of the cell address array element in secondary direction
                index2 = 1 - index1,
                // the column/row collection in the primary direction
                collection1 = columns ? colCollection : rowCollection,
                // the column/row collection in the secondary direction
                collection2 = columns ? rowCollection : colCollection,
                // whether no visible cell could be found
                moveSucceeded = false;

            // returns the index of the preceding visible column/row
            function getPrevVisibleIndex(collection, index) {
                var entry = collection.getPrevVisibleEntry(index - 1);
                return entry ? entry.index : -1;
            }

            // returns the index of the following visible column/row
            function getNextVisibleIndex(collection, index) {
                var entry = collection.getNextVisibleEntry(index + 1);
                return entry ? entry.index : (collection.getMaxIndex() + 1);
            }

            // returns whether the passed cell is hidden
            function isCellHidden(address) {
                return !colCollection.isEntryVisible(address[0]) || !rowCollection.isEntryVisible(address[1]) || (!entireSheet && mergeCollection.isHiddenCell(address));
            }

            // moves the active cell to the preceding available position in the selection
            function moveToPrevCell() {

                var // reference to the current active range address
                    activeRange = selection.ranges[selection.activeRange],
                    // reference to the active cell address in the selection object
                    activeCell = selection.activeCell,
                    // the merged range containing the active cell
                    mergedRange = mergeCollection.getMergedRange(activeCell);

                // TODO: skip protected cells (horizontal only)

                // jump to reference cell of a merged range, if located in hidden cell of first column/row
                if (!entireSheet && mergedRange && (activeCell[index1] > mergedRange.start[index1]) && (activeCell[index2] === mergedRange.start[index2])) {
                    selection.activeCell = mergedRange.start;
                    return true;
                }

                // move one column/row backwards (this may leave the active range)
                activeCell[index1] = getPrevVisibleIndex(collection1, mergedRange ? mergedRange.start[index1] : activeCell[index1]);
                if (activeCell[index1] < activeRange.start[index1]) {
                    // do not wrap at sheet borders in single cell selection mode
                    if (entireSheet) { return false; }
                    // out of range: back to last column/row, move to the preceding row/column (this may leave the active range)
                    activeCell[index1] = activeRange.end[index1];
                    // move to start of merged range, if it covers range completely (no cells beside merged range available)
                    if (mergedRange && (mergedRange.start[index1] <= activeRange.start[index1]) && (activeRange.end[index1] <= mergedRange.end[index1])) {
                        activeCell[index2] = getPrevVisibleIndex(collection2, mergedRange.start[index2]);
                    } else {
                        activeCell[index2] = getPrevVisibleIndex(collection2, activeCell[index2]);
                    }
                    if (activeCell[index2] < activeRange.start[index2]) {
                        // out of range: go to last cell in previous range
                        selection.activeRange = (selection.activeRange + selection.ranges.length - 1) % selection.ranges.length;
                        selection.activeCell = _.clone(selection.ranges[selection.activeRange].end);
                    }
                }
                return true;
            }

            // moves the active cell to the next available position in the selection
            function moveToNextCell() {

                var // reference to the current active range address
                    activeRange = selection.ranges[selection.activeRange],
                    // reference to the active cell address in the selection object
                    activeCell = selection.activeCell,
                    // the merged range containing the active cell
                    mergedRange = mergeCollection.getMergedRange(activeCell);

                // TODO: skip protected cells (horizontal only)

                // move one column/row forwards (this may leave the active range)
                activeCell[index1] = getNextVisibleIndex(collection1, mergedRange ? mergedRange.end[index1] : activeCell[index1]);
                if (activeCell[index1] > activeRange.end[index1]) {
                    // do not wrap at sheet borders in single cell selection mode
                    if (entireSheet) { return false; }
                    // out of range: back to first column/row, move to the next row/column (this may leave the active range)
                    activeCell[index1] = activeRange.start[index1];
                    // move to end of merged range, if it covers range completely (no cells beside merged range available)
                    if (mergedRange && (mergedRange.start[index1] <= activeRange.start[index1]) && (activeRange.end[index1] <= mergedRange.end[index1])) {
                        activeCell[index2] = getNextVisibleIndex(collection2, mergedRange.end[index2]);
                    } else {
                        activeCell[index2] = getNextVisibleIndex(collection2, activeCell[index2]);
                    }
                    if (activeCell[index2] > activeRange.end[index2]) {
                        // out of range: go to first cell in next range
                        selection.activeRange = (selection.activeRange + 1) % selection.ranges.length;
                        selection.activeCell = _.clone(selection.ranges[selection.activeRange].start);
                    }
                }
                return true;
            }

            // moves the active cell to the next available visible position
            function moveToVisibleCell() {

                var // the position after moving the active cell th first time
                    firstPos = null,
                    // the merged range containing the initial active cell
                    mergedRange = null,
                    // the column/row collection entry to start in a visible column/row
                    entryData = null;

                // single cell selection: use entire sheet as 'selection range'
                if (entireSheet) {
                    selection.ranges = [model.getSheetRange()];
                    selection.activeRange = 0;
                }

                // leave hidden column/row in move direction to prevent iterating all its (always hidden) cells
                if (!collection2.isEntryVisible(selection.activeCell[index2])) {
                    // try to find a visible column/row inside the merged range
                    if ((mergedRange = mergeCollection.getMergedRange(selection.activeCell))) {
                        entryData = collection2.getVisibleEntry(selection.activeCell[index2], 'nextPrev', (columns ? SheetUtils.getRowInterval : SheetUtils.getColInterval)(mergedRange));
                    }
                    // no merged range found, or completely hidden: try to find a visible entry in the entire sheet
                    if (!entryData) {
                        entryData = collection2.getVisibleEntry(selection.activeCell[index2], 'nextPrev');
                    }
                    // adjust active cell position, if an entry has been found
                    if (entryData) {
                        selection.activeCell[index2] = entryData.index;
                    } else {
                        return false;
                    }
                }

                // move active cell until it is visible
                do {

                    // move once to the previous/next cell
                    if (!(forward ? moveToNextCell() : moveToPrevCell())) {
                        return false;
                    }

                    // prevent endless loop (the entire selection may be located in hidden columns or rows)
                    if (!firstPos) {
                        firstPos = { range: selection.activeRange, cell: _.clone(selection.activeCell) };
                    } else if ((firstPos.range === selection.activeRange) && _.isEqual(firstPos.cell, selection.activeCell)) {
                        return false;
                    }

                // continue until a visible cell has been found
                } while (isCellHidden(selection.activeCell));
                return true;
            }

            // start with the original active cell that was tried to select (it may be hidden by a merged range)
            if (_.isArray(selection.originCell)) {
                selection.activeCell = selection.originCell;
            }

            // try to find a visible cell (may fail, if the entire selection is hidden)
            moveSucceeded = moveToVisibleCell();

            // if the entire selection is invisible (e.g. after hiding it completely when collapsing
            // columns or rows), remove selection and try to move the active cell in the sheet
            if (!moveSucceeded && !entireSheet) {
                entireSheet = true;
                moveSucceeded = moveToVisibleCell();
            }

            // select the cell or set the new selection, scroll to active cell
            if (moveSucceeded) {
                if (entireSheet) {
                    self.selectCell(selection.activeCell, { storeOrigin: true });
                } else {
                    self.setCellSelection(selection);
                }
                // scroll to the effective active cell
                self.scrollToSelectedCell(selection.activeCell);
            }
        }

        /**
         * Expands the selected ranges to entire columns or rows.
         *
         * @param {String} direction
         *  If set to 'columns', the current selection will be expanded to full
         *  columns. If set to 'rows', the current selection will be expanded
         *  to full rows.
         */
        function expandSelectionToSheet(direction) {

            var // current selection in the active sheet
                selection = getActiveSelection(),
                // the index of the cell address array element to be modified
                addressIndex = (direction === 'columns') ? 1 : 0,
                // the last available column/row index in the sheet
                lastIndex = (direction === 'columns') ? model.getMaxRow() : model.getMaxCol();

            // expand all ranges to the sheet borders
            _(selection.ranges).each(function (range, index) {
                range.start[addressIndex] = 0;
                range.end[addressIndex] = lastIndex;
                selection.ranges[index] = mergeCollection.expandRangeToMergedRanges(range);
            });

            self.setCellSelection(selection, { unique: true });
        }

        /**
         * Handles 'keydown' events from all grid panes to adjust the current
         * selection.
         */
        function keyDownHandler(event) {

            // special handling for drawing selection
            if (self.hasDrawingSelection() && !self.isCellEditMode()) {

                // traverse through drawing frames with TAB key
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                    selectNextDrawing(event.shiftKey);
                    return false;
                }

                return;
            }

            // no modifier keys: delete selection, move active cell
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'cell');
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'cell');
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'cell');
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'cell');
                    return false;
                case KeyCodes.PAGE_UP:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'page');
                    return false;
                case KeyCodes.PAGE_DOWN:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'page');
                    return false;
                case KeyCodes.TAB:
                    moveActiveCell(event.shiftKey ? 'left' : 'right');
                    return false;
                case KeyCodes.ENTER:
                    moveActiveCell(event.shiftKey ? 'up' : 'down');
                    return false;
                case KeyCodes.HOME:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)([0, getActiveSelection().activeCell[1]], false);
                    return false;
                case KeyCodes.END:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)([self.getLastUsedCell()[0], getActiveSelection().activeCell[1]], true);
                    return false;
                }
            }

            // additional selection with ALT modifier key
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null, alt: true })) {

                switch (event.keyCode) {
                case KeyCodes.PAGE_UP:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'page');
                    return false;
                case KeyCodes.PAGE_DOWN:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'page');
                    return false;
                }
            }

            // additional selection with CTRL or META modifier key
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null, ctrlOrMeta: true })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('left', 'used');
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('right', 'used');
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('up', 'used');
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? expandToNextCell : selectNextCell)('down', 'used');
                    return false;
                case KeyCodes.HOME:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)([0, 0], false);
                    return false;
                case KeyCodes.END:
                    (event.shiftKey ? expandToVisibleCell : selectVisibleCell)(self.getLastUsedCell(), true);
                    return false;
                }
            }

            // convert selection to entire columns
            if (KeyCodes.matchKeyCode(event, 'SPACE', { ctrlOrMeta: true })) {
                expandSelectionToSheet('columns');
                return false;
            }

            // convert selection to entire rows
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true })) {
                expandSelectionToSheet('rows');
                return false;
            }

            // select entire sheet
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                self.selectRange(model.getSheetRange(), { active: self.getVisibleGridPane('topLeft').getTopLeftAddress() });
                return false;
            }
        }

        /**
         * Keeps the reference of the active sheet model up-to-date.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            drawingCollection = sheetModel.getDrawingCollection();
        }

        /**
         * Initialization of class members.
         */
        function initHandler() {

            // store reference to document model when available
            model = app.getModel();

            // keep the reference of the active sheet model up-to-date
            self.on('change:activesheet', changeActiveSheetHandler);

            // listen to events from corner pane to adjust the selection
            self.getCornerPane().on('select:all', function (event, mode) {

                var // select mode: move active cell the top-left cell in the first visible grid pane
                    // extend mode: do not change the current active cell
                    topLeftAddress = (mode === 'select') ? self.getVisibleGridPane('topLeft').getTopLeftAddress() : getActiveSelection().activeCell;

                self.selectRange(model.getSheetRange(), { active: topLeftAddress });
            });

            // listen to events from all header panes to adjust the selection
            self.iterateHeaderPanes(function (headerPane, paneSide) {

                var // whether the header pane shows column headers
                    columns = PaneUtils.isColumnSide(paneSide),
                    // the column/row index where selection tracking has started
                    startIndex = 0;

                // makes a cell address according to the orientation of the header pane
                function makeCell(index1, index2) {
                    return columns ? [index1, index2] : [index2, index1];
                }

                // makes an adjusted range between start index and passed index
                function makeRange(index) {
                    var selectionInterval = { first: Math.min(startIndex, index), last: Math.max(startIndex, index) };
                    return model[columns ? 'makeColRange' : 'makeRowRange'](selectionInterval);
                }

                headerPane.on({
                    'select:start': function (event, index, mode) {
                        var selection = getActiveSelection({ allowEmpty: true });
                        if ((mode === 'extend') && (selection.ranges.length > 0)) {
                            // modify the active range (from active cell to tracking start index)
                            startIndex = selection.activeCell[columns ? 0 : 1];
                            self.changeActiveRange(makeRange(index));
                        } else {
                            // create/append a new selection range
                            startIndex = index;
                            // set active cell to current first visible cell in the column/row
                            self.selectRange(makeRange(index), {
                                append: mode === 'append',
                                active: makeCell(startIndex, Math.max(0, self.getVisibleHeaderPane(!columns).getFirstVisibleIndex()))
                            });
                        }
                    },
                    'select:move': function (event, index) {
                        self.changeActiveRange(makeRange(index));
                    },
                    'select:end': function (event, index) {
                        self.changeActiveRange(makeRange(index), { unique: true });
                    }
                });
            });

            // listen to events from all grid panes to adjust the selection
            self.iterateGridPanes(function (gridPane) {

                var // the address of the cell where selection tracking has started
                    startAddress = null;

                // makes an adjusted range from the start address to the passed address
                function makeRange(address) {
                    return SheetUtils.adjustRange({ start: _.clone(startAddress), end: _.clone(address) });
                }

                gridPane.on({
                    'select:start': function (event, address, mode) {
                        var selection = getActiveSelection({ allowEmpty: true });
                        if ((mode === 'extend') && (selection.ranges.length > 0)) {
                            // modify the active range (from active cell to tracking start position)
                            startAddress = selection.activeCell;
                            self.changeActiveRange(makeRange(address));
                        } else {
                            // create/append a new selection range
                            startAddress = address;
                            self.selectCell(address, { append: mode === 'append' });
                        }
                    },
                    'select:move': function (event, address) {
                        self.changeActiveRange(makeRange(address));
                    },
                    'select:end': function (event, address) {
                        self.changeActiveRange(makeRange(address), { unique: true });
                    }
                });
            });

            // listen to key events from all panes to adjust the selection
            rootNode.on('keydown', keyDownHandler);
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a callback function that can change the behavior of the
         * application when selecting cell ranges with mouse or keyboard.
         *
         * @param {Function} getHandler
         *  A callback function that will be invoked to receive a different
         *  cell selection to be used for all selection code, especially
         *  changing the selection with keyboard shortcuts. Will be called in
         *  the context of this instance. Must return a selection object, or
         *  null. The latter indicates to use the actual cell selection of the
         *  active sheet.
         *
         * @param {Function} setHandler
         *  A callback function that will be invoked before actually changing
         *  the current cell selection in the active sheet. Will be called in
         *  the context of this instance. Receives the new selection intended
         *  to be set at the active sheet as first parameter. If this function
         *  has processed the selection by itself, it has to return true to
         *  prevent changing the cell selection in the active sheet.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.registerCellSelectionHandlers = function (getHandler, setHandler) {
            getCellSelectionHandler = getHandler;
            setCellSelectionHandler = setHandler;
            return this;
        };

        /**
         * Unregisters the callback functions that have been registered with
         * the method SelectionMixin.registerCellSelectionHandlers().
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.unregisterCellSelectionHandlers = function () {
            getCellSelectionHandler = setCellSelectionHandler = null;
            return this;
        };

        /**
         * Returns an object containing all information about the selection in
         * the active sheet.
         *
         * @returns {Object}
         *  The cell selection data, in the following properties:
         *  - {Array} ranges
         *      The logical addresses of all selected cell ranges.
         *  - {Number} activeRange
         *      Element index of the active range in the 'ranges' property.
         *  - {Number[]} activeCell
         *      The logical address of the active cell in the active range.
         *  - {Array} drawings
         *      The logical positions of all selected drawing frames.
         */
        this.getSelection = function () {
            return sheetModel.getViewAttribute('selection');
        };

        /**
         * Returns all selected cell ranges in the active sheet.
         *
         * @returns {Array}
         *  The selected cell ranges, as array of logical range positions.
         */
        this.getSelectedRanges = function () {
            return this.getSelection().ranges;
        };

        /**
         * Returns the active cell of the selection in the active sheet (the
         * highlighted cell in the active range that will be edited when
         * switching to text edit mode).
         *
         * @returns {Number[]}
         *  The logical position of the active cell.
         */
        this.getActiveCell = function () {
            return this.getSelection().activeCell;
        };

        /**
         * Returns whether the current selection consists of the active cell
         * only, either as single cell, or exactly covering a merged range.
         *
         * @returns {Boolean}
         *  Whether the current selection consists of the active cell only.
         */
        this.isSingleCellSelection = function () {
            var selection = this.getSelection();
            return (selection.ranges.length === 1) && this.isSingleCellInRange(selection.ranges[0]);
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleColumnsSelected = function () {
            return _(this.getSelectedRanges()).any(function (range) { return SheetUtils.getColCount(range) > 1; });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleRowsSelected = function () {
            return _(this.getSelectedRanges()).any(function (range) { return SheetUtils.getRowCount(range) > 1; });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple cells.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple cells.
         */
        this.hasAnyRangeSelected = function () {
            return _(this.getSelectedRanges()).any(function (range) { return SheetUtils.getCellCount(range) > 1; });
        };

        /**
         * Returns whether the current selection contains or partly overlaps at
         * least one merged cell range.
         *
         * @returns {Boolean}
         *  Whether the current selection contains or partly overlaps at least
         *  one merged cell range.
         */
        this.hasMergedRangeSelected = function () {
            return mergeCollection.rangesOverlapAnyMergedRange(this.getSelectedRanges());
        };

        /**
         * Returns, whether a drawing object is selected in the current active
         * sheet.
         *
         * @returns {Boolean}
         *  Whether a drawing object is selected in the active sheet.
         */
        this.hasDrawingSelection = function () {
            return this.getSelectedDrawings().length > 0;
        };

        /**
         * Returns the logical positions of all selected drawing frames in the
         * active sheet.
         *
         * @returns {Array}
         *  The selected drawing frames, as array of logical drawing positions.
         */
        this.getSelectedDrawings = function () {
            return this.getSelection().drawings;
        };

        /**
         * Changes the cell selection in the active sheet, and deselects all
         * drawing frames.
         *
         * @param {Object} selection
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the active cell passed in the selection
         *      is hidden by a merged range, the address of the original active
         *      cell will be stored in the additional selection property
         *      'originCell' which will be included in the selection returned
         *      by the method SelectionMixin.getSelection().
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate ranges will be removed from the
         *      selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.setCellSelection = function (selection, options) {

            var // a merged range hiding the active cell
                mergedRange = mergeCollection.getMergedRange(selection.activeCell, 'hidden');

            // deselect all drawing frames
            selection = _.clone(selection);
            selection.drawings = [];
            delete selection.originCell;

            // adjust active cell if it is hidden by a merged range
            if (mergedRange) {
                if (Utils.getBooleanOption(options, 'storeOrigin', false)) {
                    selection.originCell = selection.activeCell;
                }
                selection.activeCell = mergedRange.start;
            }

            // remove duplicates
            if (Utils.getBooleanOption(options, 'unique', false)) {
                var activeRange = selection.ranges[selection.activeRange];
                selection.ranges = _(selection.ranges).unique(SheetUtils.getRangeName);
                selection.activeRange = Utils.findFirstIndex(selection.ranges, function (range) { return _.isEqual(range, activeRange); });
            }

            // prefer the registered cell selection handler
            if (!_.isFunction(setCellSelectionHandler) || (setCellSelectionHandler.call(self, selection) !== true)) {

                // leave cell in-place edit mode before changing the selection
                self.leaveCellEditMode('cell');
                // commit new selection
                sheetModel.setViewAttribute('selection', selection);
            }

            return this;
        };

        /**
         * Selects a single cell in the active sheet. The cell will become the
         * active cell of the sheet.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be selected.
         *
         * @param {Object} options
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new cell will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the specified cell is hidden by a merged
         *      range, the address of the passed cell will be stored in the
         *      additional selection property 'originCell' which will be
         *      included in the selection returned by the method
         *      SelectionMixin.getSelection().
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate ranges will be removed from the
         *      selection (used when appending a cell to the selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectCell = function (address, options) {
            return this.selectRange({ start: address, end: address }, _({}).extend(options, { active: address }));
        };

        /**
         * Selects a single cell range in the active sheet.
         *
         * @param {Object} range
         *  The logical address of the cell range to be selected.
         *
         * @param {Object} options
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new range will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Number[]} [options.active]
         *      The logical address of the active cell in the selected range.
         *      If omitted, the top-left cell of the range will be activated.
         *      Must be located inside the passed range address.
         *  @param {Boolean} [options.storeOrigin=false]
         *      If set to true, and if the new active cell is hidden by a
         *      merged range, the address of the active cell will be stored in
         *      the additional selection property 'originCell' which will be
         *      included in the selection returned by the method
         *      SelectionMixin.getSelection().
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate ranges will be removed from the
         *      selection (used when appending a range to the selection).
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectRange = function (range, options) {

            var // whether to append the range to the current selection
                append = Utils.getBooleanOption(options, 'append', false),
                // the initial selection to append the new range to
                selection = append ? getActiveSelection({ allowEmpty: true }) : { ranges: [] },
                // the new active cell
                activeCell = Utils.getArrayOption(options, 'active', []);

            // expand range to merged ranges (unless an entire column/row is selected)
            if (!model.isColRange(range) && !model.isRowRange(range)) {
                range = mergeCollection.expandRangeToMergedRanges(range);
            }

            // initialize and set the new selection
            selection.ranges.push(range);
            selection.activeRange = selection.ranges.length - 1;

            // set active cell
            if ((activeCell.length === 2) && SheetUtils.rangeContainsCell(range, activeCell)) {
                selection.activeCell = activeCell;
            } else {
                selection.activeCell = range.start;
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Modifies the position of the current active selection range.
         *
         * @param {Object} range
         *  The new logical address of the active selection range. If the
         *  passed range does not contain the current active cell, the top-left
         *  cell of the range will be activated.
         *
         * @param {Object} options
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.unique=false]
         *      If set to true, duplicate ranges will be removed from the
         *      selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.changeActiveRange = function (range, options) {

            var // the selection to be modified
                selection = getActiveSelection();

            // expand range to merged ranges
            if (!model.isColRange(range) && !model.isRowRange(range)) {
                range = mergeCollection.expandRangeToMergedRanges(range);
            }

            // insert the new active range into the selection object
            selection.ranges[selection.activeRange] = range;
            if (!SheetUtils.rangeContainsCell(range, selection.activeCell)) {
                selection.activeCell = range.start;
            }

            // commit new selection
            return this.setCellSelection(selection, options);
        };

        /**
         * Scrolls the active grid pane to the specified cell. In frozen split
         * view, activates the grid pane that contains the specified cell
         * address, and scrolls it to the cell.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be scrolled to.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.scrollToSelectedCell = function (address) {

            var // the frozen column and row interval
                colInterval = null, rowInterval = null,
                // preferred pane side identifiers
                colSide = null, rowSide = null,
                // the grid pane that will be scrolled
                gridPane = null;

            if (this.hasFrozenSplit()) {
                // select appropriate grid pane according to the passed cell address
                colInterval = sheetModel.getSplitColInterval();
                colSide = (_.isObject(colInterval) && (address[0] <= colInterval.last)) ? 'left' : 'right';
                rowInterval = sheetModel.getSplitRowInterval();
                rowSide = (_.isObject(rowInterval) && (address[1] <= rowInterval.last)) ? 'top' : 'bottom';
                gridPane = this.getVisibleGridPane(PaneUtils.getPanePos(colSide, rowSide)).grabFocus();
            } else {
                gridPane = this.getActiveGridPane();
            }
            gridPane.scrollToCell(address);
            return this;
        };

        /**
         * Selects a single drawing frame in the active sheet.
         *
         * @param {Number[]} position
         *  The logical position of the drawing frame to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.toggle=false]
         *      If set to true, the selection state of the addressed drawing
         *      frame will be toggled. Otherwise, the old drawing selection
         *      will be removed, and the addressed drawing frame will be
         *      selected.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.selectDrawing = function (position, options) {

            var // the current selection to be modified (keep cell selection as it is)
                selection = this.getSelection();

            // first, leave cell in-place edit mode
            this.leaveCellEditMode('cell');

            // construct the new drawing selection
            if (!Utils.getBooleanOption(options, 'toggle', false)) {
                selection.drawings = [position];
            } else if (Utils.spliceValue(selection.drawings, position) === 0) {
                selection.drawings.push(position);
            }

            // commit new selection
            sheetModel.setViewAttribute('selection', selection);
            return this;
        };

        /**
         * Removes selection from all drawing objects and restores the cell
         * selection.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.removeDrawingSelection = function () {

            var // the current selection to be modified (keep cell selection as it is)
                selection = this.getSelection();

            // remove drawing selection
            selection.drawings = [];

            // commit new selection
            sheetModel.setViewAttribute('selection', selection);
            return this;
        };

        /**
         * Scrolls the active grid pane to the specified drawing frame. In
         * frozen split view, scrolls the bottom-right grid pane instead.
         *
         * @param {Number[]} position
         *  The logical position of the drawing frame to be scrolled to.
         *
         * @returns {SelectionMixin}
         *  A reference to this instance.
         */
        this.scrollToSelectedDrawing = function (position) {
            var gridPane = this.hasFrozenSplit() ? this.getVisibleGridPane('bottomRight') : this.getActiveGridPane();
            gridPane.scrollToDrawingFrame(position);
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', initHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = sheetModel = colCollection = rowCollection = mergeCollection = drawingCollection = null;
            getCellSelectionHandler = setCellSelectionHandler = null;
        });

    } // class SelectionMixin

    // exports ================================================================

    return SelectionMixin;

});
