/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/dialogsmixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/drawinglayer/view/imageutil',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Dialogs, ImageUtil, gt) {

    'use strict';

    var // maximum size of an image in any direction, in 1/100 mm
        MAX_IMAGE_SIZE = 21000;

    // mix-in class SpreadsheetDialogsMixin ===================================

    /**
     * Mix-in class for the class SpreadsheetView that adds methods to show
     * modal dialogs.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function SpreadsheetDialogsMixin(app) {

        var // self reference (the spreadsheet view instance)
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Shows a dialog that allows to enter a sheet name, and will invoke an
         * arbitrary callback function for the sheet name. The dialog will be
         * kept open until a valid sheet name has been entered.
         *
         * @param {Function} callback
         *  The callback function invoked when the OK button of the dialog has
         *  been clicked. Receives the current sheet name shown in the dialog.
         *  Must return a Boolean value that states whether the operation using
         *  the sheet name has been finished successfully. In case of an error,
         *  the dialog will be kept open.
         *
         * @param {Object} [options]
         *  A map with additional options for the dialog. The following options
         *  are supported:
         *  @param {String} [options.okLabel]
         *      An alternative label text for the OK button.
         *  @param {String} [options.initialName]
         *      The initial sheet name to be shown when the dialog is opened.
         *      If omitted, the name of the active sheet will be used.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        function showSheetNameDialog(title, callback, options) {

            function showDialog(initialName) {

                var // the Promise object representing the dialog
                    promise = null;

                // close the dialog if document switches to read-only mode
                function changeEditModeHandler(event, editMode) {
                    if (!editMode) { promise.close(); }
                }

                // create the dialog box
                promise = Dialogs.showTextDialog({
                    title: title,
                    okLabel: Utils.getStringOption(options, 'okLabel'),
                    value: initialName,
                    placeholder: gt('Enter sheet name')
                });

                // listen to read-only mode, close dialog automatically
                app.getModel().on('change:editmode', changeEditModeHandler);
                promise.always(function () {
                    app.getModel().off('change:editmode', changeEditModeHandler);
                });

                // show dialog as long as the callback function returns false
                return promise.then(function (sheetName) {
                    return callback.call(self, sheetName) ? sheetName : showDialog(sheetName);
                });
            }

            // show the dialog, return the piped Deferred
            return showDialog(Utils.getStringOption(options, 'initialName', self.getSheetName()));
        }

        // methods ------------------------------------------------------------

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        this.showCopySheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Copy Sheet'), this.copySheet, { okLabel: gt('Copy'), initialName: app.getModel().generateUnusedSheetName() });
        };

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been renamed
         *  successfully; or will rejected, if the dialog has been canceled, or
         *  if the document has switched to read-only mode.
         */
        this.showRenameSheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Rename Sheet'), this.setSheetName, { okLabel: gt('Rename') });
        };

        /**
         * Shows the 'Insert Image' dialog and inserts the resulting image into
         * the active sheet of the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the dialog has been closed and the image has been inserted
         *  into the document.
         */
        this.showInsertImageDialog = function () {

            // show the generic 'Insert Image' dialog
            return ImageUtil.insertImageDialog(app).then(function (imageHolder) {

                var // create the image DOM node, wait for it to load
                    createImagePromise = app.createImageNode(ImageUtil.getFileUrl(app, imageHolder.url));

                // on success, create the drawing object in the document
                createImagePromise.done(function (imgNode) {

                    var // current width of the image, in 1/100 mm
                        width = Utils.convertLengthToHmm(imgNode[0].width, 'px'),
                        // current height of the image, in 1/100 mm
                        height = Utils.convertLengthToHmm(imgNode[0].height, 'px'),
                        // active cell as target position for the image
                        activeCell = self.getActiveCell();

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }

                    self.insertDrawing('image', { drawing: {
                        anchorType: 'oneCell',
                        startCol: activeCell[0],
                        startRow: activeCell[1],
                        width: width,
                        height: height,
                        name: imageHolder.name,
                        imageUrl: imageHolder.url
                    }});

                    app.destroyImageNodes(imgNode);
                });

                // on failure, show a warning message
                createImagePromise.fail(function () {
                    app.rejectEditAttempt('image');
                });

                return createImagePromise;
            });
        };

    } // class SpreadsheetDialogsMixin

    // exports ================================================================

    return SpreadsheetDialogsMixin;

});
