/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/celleditmixin', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // mix-in class CellEditMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * the cell in-place edit mode, and for showing and manipulating
     * highlighted cell ranges based on formula token arrays.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function CellEditMixin(app) {

        var // self reference (spreadsheet view instance)
            self = this,

            // the undo manager of the document model
            undoManager = null,

            // the model instance and collections of the active sheet
            sheetModel = null,

            // the token arrays currently used for highlighted ranges
            currentTokenArrays = [],

            // whether the ranges based on reference tokens will be marked draggable
            draggable = false;

        // private methods ----------------------------------------------------

        /**
         * Handles change events from any of the token arrays, and notifies the
         * own event listeners.
         */
        function changeTokenArrayHandler() {

            var // array of arrays of range info objects from the token arrays
                allRangeInfos = _(currentTokenArrays).invoke('extractRanges', { targetSheet: self.getActiveSheet(), resolveNames: true }),
                // the flattened highlighted range addresses
                highlightRanges = [];

            _(allRangeInfos).each(function (rangeInfos, arrayIndex) {
                _(rangeInfos).each(function (rangeInfo) {
                    highlightRanges.push(_({
                        // the range identifier containing the index of the token array
                        id: arrayIndex + ',' + rangeInfo.index,
                        // ranges of defined names are not draggable
                        draggable: draggable && (rangeInfo.type === 'ref')
                    }).extend(rangeInfo.range));
                });
            });

            // set the highlighted ranges at the model instance of the active sheet
            sheetModel.setViewAttribute('highlightRanges', highlightRanges);
        }

        /**
         * Registers or unregisters the change listener at all current token
         * arrays.
         */
        function registerTokenArrayListeners(type) {
            _(currentTokenArrays).invoke(type, 'triggered', changeTokenArrayHandler);
        }

        /**
         * Keeps the reference of the active sheet model up-to-date.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
        }

        /**
         * Initialization of class members.
         */
        function initHandler() {
            // store reference to document undo manager when available
            undoManager = app.getModel().getUndoManager();
            // keep the reference of the active sheet model up-to-date
            self.on('change:activesheet', changeActiveSheetHandler);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return this.getActiveGridPane().isCellEditMode();
        };

        /**
         * Enters in-place cell edit mode in the active grid pane. See method
         * 'GridPane.enterCellEditMode()' for details.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.enterCellEditMode = function (options) {
            this.getActiveGridPane().enterCellEditMode(options);
            return this;
        };

        /**
         * Leaves in-place cell edit mode and commits the current text. See
         * method 'GridPane.leaveCellEditMode()' for details.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.leaveCellEditMode = function (commitMode) {
            this.getActiveGridPane().leaveCellEditMode(commitMode);
            return this;
        };

        // extended undo/redo -------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * are changed currently.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return this.isCellEditMode() ? this.getActiveGridPane().cellEditValueChanged() : (undoManager.getUndoCount() > 0);
        };

        /**
         * Applies the topmost undo action on the undo stack. In cell in-place
         * edit mode, restores the original contents of the cell.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            if (this.isCellEditMode()) {
                this.getActiveGridPane().restoreOriginalCellEditValue();
                return $.when();
            }
            return undoManager.undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * have been changed, but are not changed anymore (after undo).
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return this.isCellEditMode() ? this.getActiveGridPane().cellEditValueRestored() : (undoManager.getRedoCount() > 0);
        };

        /**
         * Applies the topmost redo action on the redo stack. In cell in-place
         * edit mode, restores the last changed contents of the cell, after
         * these changes have been undone.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            if (this.isCellEditMode()) {
                this.getActiveGridPane().restoreChangedCellEditValue();
                return $.when();
            }
            return undoManager.redo(1);
        };

        // range highlighting -------------------------------------------------

        /**
         * Registers token arrays used to render highlighted cell ranges in the
         * active sheet, and updates the 'highlightRanges' view property of the
         * active sheet whenever the token arrays are changing.
         *
         * @param {Array|TokenArray} tokenArrays
         *  An array of token arrays, or a single token array instance.
         *
         * @param {Boolean} [options]
         *  A map with additional options for this method. The following
         *  options are supported:
         *  @param {Boolean} [options.draggable=false]
         *      If set to true, the cell ranges representing reference tokens
         *      will be marked draggable in the 'highlightRanges' sheet view
         *      attribute.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.startRangeHighlighting = function (tokenArrays, options) {

            // whether the ranges of the reference tokens are draggable
            draggable = Utils.getBooleanOption(options, 'draggable', false);

            // unregister event handlers at old token arrays
            registerTokenArrayListeners('off');

            // store the token arrays
            currentTokenArrays = _.chain(tokenArrays).getArray().clone().value();
            registerTokenArrayListeners('on');

            // notify initial ranges to listeners
            changeTokenArrayHandler();
            return this;
        };

        /**
         * Unregisters all token arrays currently registered.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.endRangeHighlighting = function () {
            return this.startRangeHighlighting([]);
        };

        /**
         * Modifies the cell range address of a single reference token inside a
         * single token array currently registered for range highlighting.
         *
         * @param {String} tokenId
         *  The unique identifier of the reference token to be modified, as
         *  contained in the 'highlightRanges' view attribute.
         *
         * @param {Object} range
         *  The new logical range address to be inserted into the reference
         *  token. The absolute/relative state of the reference components will
         *  not be changed.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.modifyReferenceToken = function (tokenId, range) {

            var // the token array index, and the token index
                indexes = _(tokenId.split(/,/)).map(function (id) { return parseInt(id, 10); }),
                // the reference token to be modified
                refToken = null;

            // modify the reference token, if the passed identifier is valid
            if ((indexes.length === 2) && (0 <= indexes[0]) && (indexes[0] < currentTokenArrays.length)) {
                if ((refToken = currentTokenArrays[indexes[0]].getToken(indexes[1], 'ref'))) {
                    refToken.setRange(range);
                }
            }
            return this;
        };

        // initialization -----------------------------------------------------

        app.on('docs:init', initHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            registerTokenArrayListeners('off');
            currentTokenArrays = null;
        });

    } // class CellEditMixin

    // exports ================================================================

    return CellEditMixin;

});
