/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/headerpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/view/mixin/headertrackingmixin',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, TriggerObject, SheetUtils, PaneUtils, HeaderTrackingMixin, gt) {

    'use strict';

    var // the ratio of the visible pane size to be added before and after
        ADDITIONAL_SIZE_RATIO = 0.5,

        // margin at trailing border of the entire sheet area, in pixels
        SCROLL_AREA_MARGIN = 30,

        // size of the resizer nodes, in pixels
        RESIZER_SIZE = 6,

        // tooltip labels for resizers
        COL_RESIZER_VISIBLE = gt('Drag to change column width'),
        COL_RESIZER_HIDDEN = gt('Drag to show hidden column'),
        ROW_RESIZER_VISIBLE = gt('Drag to change row height'),
        ROW_RESIZER_HIDDEN = gt('Drag to show hidden row');

    // class HeaderPane =======================================================

    /**
     * Represents a single row or column header pane in the spreadsheet view.
     * If the view has been split or frozen at a specific cell position, the
     * view will consist of up to four header panes (left and right column
     * header panes, and top and bottom row header panes).
     *
     * Triggers the following events:
     * - 'change:scrollsize'
     *      After the size of the scrollable area in this header pane has been
     *      changed. Event handlers receive the new scroll area size.
     * - 'change:scrollpos'
     *      After the absolute scroll position of this header pane has been
     *      changed. Event handlers receive the new scroll position.
     * - 'change:interval'
     *      After the column/row interval shown by this header pane has been
     *      changed. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} interval
     *          The new column/row interval, with the zero-based properties
     *          'first' and 'last'.
     *      (3) {Object} position
     *          The position of the new column/row interval, in pixels, in the
     *          properties 'offset' and 'size'.
     *      (4) {Object} [operation]
     *          A description of the performed sheet operation. If omitted, the
     *          header pane has changed its visible interval while scrolling or
     *          zooming. Contains the following properties:
     *          - {String} operation.name
     *              The name of the operation. Must be one of 'insert' (new
     *              columns or rows inserted into the sheet), 'delete' (columns
     *              or rows deleted from the sheet), 'change' (column or row
     *              formatting changed, including size or hidden state), or
     *              'refresh' (layout of all columns/rows has been changed
     *              completely, e.g. after changing the zoom factor of the
     *              sheet).
     *          - {Number} [operation.first]
     *              The zero-based index of the first column/row affected by
     *              the operation. Omitted for the 'refresh' operation.
     *          - {Number} [operation.last]
     *              The zero-based index of the last column/row affected by the
     *              operation. Omitted for the 'refresh' operation.
     *          - {Object} [operation.attrs]
     *              The explicit attributes used for the 'change' operation.
     *              May be undefined even for this operation.
     *          - {Boolean} [operation.rangeBorders=false]
     *              Used for the 'change' operation only. If set to true,
     *              attributes for outer borders will be applied at the outer
     *              borders of the interval only, and attributes for inner
     *              borders will be applied inside the interval. Otherwise, the
     *              outer border attributes will be applied for all borders,
     *              and the inner border attributes will be ignored.
     *          - {Boolean} [operation.visibleBorders=false]
     *              Used for the 'change' operation only. If set to true,
     *              border attributes may be incomplete. Existing border
     *              properties will be applied for visible borders in the
     *              interval only.
     * - 'hide:interval'
     *      After this header pane has been hidden, and the old column/row
     *      interval has been reset to nothing. Listeners will reset or update
     *      additional settings according to the hidden header pane.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the start column/row.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the index of the tracked column/row has changed while
     *      selection tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the current column/row.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} index
     *          The zero-based index of the last column/row.
     * - 'resize:start'
     *      After column/row resize tracking has been started. Receives the
     *      current offset and initial size of the column/row.
     * - 'resize:move'
     *      After the new column/row size has changed while resize tracking is
     *      active. Receives the current offset and size of the column/row.
     * - 'resize:end'
     *      After column/row resize tracking has been finished (either
     *      successfully or not).
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this header pane.
     *
     * @param {String} paneSide
     *  The position of this header pane (one of 'left', 'right', 'top',
     *  'bottom', or 'corner').
     */
    function HeaderPane(app, paneSide) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = null,

            // the active sheet model, and the column/row collection of the active sheet
            sheetModel = null,
            collection = null,

            // whether this header pane contains columns
            columns = PaneUtils.isColumnSide(paneSide),

            // the name of the scroll anchor sheet view attribute
            anchorName = PaneUtils.getScrollAnchorName(paneSide),

            // the container node of this header pane (must be focusable for tracking support)
            rootNode = $('<div>').addClass('header-pane ' + (columns ? 'columns' : 'rows')).attr('tabindex', 0),

            // the content node with the header cell and resizer nodes
            contentNode = $('<div>').addClass('header-content noI18n').appendTo(rootNode),

            // the size of this header pane usable to show column/row header cells
            paneSize = 0,

            // whether this header pane is frozen (not scrollable)
            frozen = false,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenSize = 0,

            // the column/row interval to be displayed in the DOM (null indicates hidden header pane)
            interval = null,

            // the absolute position and size of the column/row interval in the sheet
            position = { offset: 0, size: 0 },

            // the cached number of used columns/rows in the current sheet
            usedCount = 0,

            // the absolute scroll position in the associated grid panes
            scrollPos = 0,

            // the total size of the scroll area
            scrollSize = 0;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        HeaderTrackingMixin.call(this, app, paneSide, contentNode);

        // private methods ----------------------------------------------------

        /**
         * Refreshes the appearance of this header pane according to the
         * position of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = sheetModel.getViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = sheetModel.getViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (view.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // header pane if it is active or in the opposite direction
                focused = (focusPaneSide === paneSide) || (PaneUtils.isColumnSide(focusPaneSide) !== columns);
            } else {
                // a single grid pane is focused
                focused = (PaneUtils.getColPaneSide(focusPanePos) === paneSide) || (PaneUtils.getRowPaneSide(focusPanePos) === paneSide);
            }

            rootNode.toggleClass(Utils.FOCUSED_CLASS, focused);
        }

        /**
         * Redraws the selection highlighting in the header cells of this
         * header pane.
         */
        function renderSelection() {

            var // the merged ranges of selected columns/rows
                selectionIntervals = (columns ? SheetUtils.getColIntervals : SheetUtils.getRowIntervals)(view.getSelectedRanges()),
                // current index into the intervals array
                arrayIndex = 0;

            // find the first selection interval that ends at or before the passed index
            function findSelectionInterval(index) {
                while ((arrayIndex < selectionIntervals.length) && (selectionIntervals[arrayIndex].last < index)) { arrayIndex += 1; }
                return selectionIntervals[arrayIndex];
            }

            // find the first interval, and iterate all cell nodes
            findSelectionInterval(interval.first);
            Utils.iterateSelectedDescendantNodes(contentNode, '.cell', function (cellNode) {
                var index = Utils.getElementAttributeAsInteger(cellNode, 'data-index'),
                    selectionInterval = findSelectionInterval(index);
                $(cellNode).toggleClass(Utils.SELECTED_CLASS, _.isObject(selectionInterval) && (selectionInterval.first <= index));
            }, undefined, { children: true });
        }

        /**
         * Calculates all settings needed to render the column/row header
         * cells.
         *
         * @returns {Array}
         *  An array with descriptor objects for all columns/rows to be
         *  rendered. Each descriptor contains the following properties:
         *  - {Number} index
         *      The zero-based index of the column/row.
         *  - {Number} size
         *      The size of the cell node, in pixels. May be zero, if the
         *      column/row is hidden and precedes a visible column/row.
         *  - {Number} offset
         *      The start position of the cell node, relative to the content
         *      node, in pixels.
         *  - {Number} resizerOffset
         *      The start position of the resizer node, relative to the content
         *      node, in pixels.
         */
        function prepareCellRendering() {

            var // the result array
                entries = [];

            // generate HTML mark-up text for all visible header cells
            collection.iterateVisibleEntries(interval, function (entryData) {

                var // the descriptor for the current entry
                    entry = { index: entryData.index, size: entryData.size };

                // start offset of the cell node and resizer node
                entry.offset = entryData.offset - position.offset;
                entry.resizerOffset = entry.offset + entryData.size;

                // adjust resizer position according to hidden state
                if (entry.size > 0) {
                    entry.resizerOffset -= RESIZER_SIZE / 2;
                } else if (entries.length > 0) {
                    _.last(entries).resizerOffset -= RESIZER_SIZE / 2;
                }

                // store new descriptor in result array
                entries.push(entry);

            }, { hidden: true });

            return entries;
        }

        /**
         * Renders all column header cells in the current column interval.
         */
        function renderColumnCells() {

            var // the HTML mark-up for all visible column header cells
                markup = '',
                // the position and size data for all columns to be rendered
                entries = prepareCellRendering();

            // set font size and line height at content node
            contentNode.css({ fontSize: view.getHeaderFontSize() + 'pt', lineHeight: rootNode.height() + 'px' });

            // generate HTML mark-up text for all column header cells
            _(entries).each(function (entry) {

                var // whether the current column is hidden and precedes a visible column
                    hidden = entry.size === 0,
                    // tool tip for the resizer
                    resizerTooltip = hidden ? COL_RESIZER_HIDDEN : COL_RESIZER_VISIBLE,
                    // column index as element attribute
                    dataAttr = ' data-index="' + entry.index + '"';

                // generate mark-up for the header cell node
                if (hidden) {
                    markup += '<div class="marker" style="left:' + (entry.offset + 1) + 'px;"><div><div></div></div></div>';
                } else {
                    markup += '<div class="cell" style="left:' + entry.offset + 'px;width:' + entry.size + 'px;"' + dataAttr;
                    if (Utils.IE9) { markup += ' unselectable="on"'; } // bug 29905
                    markup += '>' + SheetUtils.getColName(entry.index) + '</div>';
                }

                // generate mark-up for the resizer node
                markup += '<div class="resizer' + (hidden ? ' collapsed' : '') + '" style="left:' + entry.resizerOffset + 'px;width:' + RESIZER_SIZE + 'px;" title="' + resizerTooltip + '"' + dataAttr;
                if (Utils.IE9) { markup += ' unselectable="on"'; } // bug 29905
                markup += '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;

            // render selection again
            renderSelection();
        }

        /**
         * Renders all row header cells in the current row interval.
         */
        function renderRowCells() {

            var // the HTML mark-up for all visible row header cells
                markup = '',
                // the position and size data for all columns to be rendered
                entries = prepareCellRendering();

            // set font size at content node (line height needs to be set per cell node)
            contentNode.css('font-size', view.getHeaderFontSize() + 'pt');

            // generate HTML mark-up text for all column header cells
            _(entries).each(function (entry) {

                var // whether the current row is hidden and precedes a visible row
                    hidden = entry.size === 0,
                    // tool tip for the resizer
                    resizerTooltip = hidden ? ROW_RESIZER_HIDDEN : ROW_RESIZER_VISIBLE,
                    // row index as element attribute
                    dataAttr = ' data-index="' + entry.index + '"';

                // generate mark-up for the header cell node
                if (hidden) {
                    markup += '<div class="marker" style="top:' + (entry.offset + 1) + 'px;"><div><div></div></div></div>';
                } else {
                    markup += '<div class="cell" style="top:' + entry.offset + 'px;height:' + entry.size + 'px;line-height:' + entry.size + 'px;"' + dataAttr;
                    if (Utils.IE9) { markup += ' unselectable="on"'; } // bug 29905
                    markup += '>' + SheetUtils.getRowName(entry.index) + '</div>';
                }

                // generate mark-up for the resizer node
                markup += '<div class="resizer' + (hidden ? ' collapsed' : '') + '" style="top:' + entry.resizerOffset + 'px;height:' + RESIZER_SIZE + 'px;" title="' + resizerTooltip + '"' + dataAttr;
                if (Utils.IE9) { markup += ' unselectable="on"'; } // bug 29905
                markup += '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;

            // render selection again
            renderSelection();
        }

        /**
         * Calculates the effective scroll position from the current scroll
         * anchor view attribute, in pixels.
         */
        function calculateScrollPosition() {
            var scrollAnchor = sheetModel.getViewAttribute(anchorName);
            return Math.max(0, collection.convertScrollAnchorToPixel(scrollAnchor) - hiddenSize);
        }

        /**
         * Updates the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position. Tries to reserve
         * additional space for the full size of this header pane at the
         * trailing border, to be able to scroll page-by-page into the unused
         * area of the sheet.
         */
        function updateScrollAreaSize(newScrollPos) {

            var // the total size of the used area in the sheet, in pixels
                usedSize = collection.getEntry(usedCount).offset,
                // the absolute size of the scrollable area
                newScrollSize = 0;

            // scrollable area includes the used area, and the current (but extended)
            // scroll position (including hidden size to be synchronous with the
            // used size, will be removed below)
            newScrollSize = Math.max(usedSize, newScrollPos + hiddenSize + 2 * paneSize);

            // restrict to the total size of the sheet, plus a little margin, remove
            // the hidden area (the header pane may not start at cell A1 in frozen mode)
            newScrollSize = Math.min(newScrollSize, collection.getTotalSize() + SCROLL_AREA_MARGIN) - hiddenSize;

            // trigger event, if the new scroll area size has changed
            if (scrollSize !== newScrollSize) {
                scrollSize = newScrollSize;
                self.trigger('change:scrollsize', scrollSize);
            }
        }

        /**
         * Updates the relative scroll position and size of the content node,
         * and notifies listeners about the changed scroll position.
         */
        function updateScrollPosition(newScrollPos) {

            var // the relative offset of the content node
                contentOffset = position.offset - newScrollPos - hiddenSize;

            // set the position and size of the content node
            if (columns) {
                contentNode.css({ left: contentOffset, width: position.size });
            } else {
                contentNode.css({ top: contentOffset, height: position.size });
            }

            // cache new scroll position, notify listeners
            if (scrollPos !== newScrollPos) {
                scrollPos = newScrollPos;
                self.trigger('change:scrollpos', scrollPos);
            }
        }

        /**
         * Recalculates the current row/column interval and its position and
         * size in the sheet, and renders all visible header cells and the
         * selection on demand. Triggers a 'change:interval' event, if the
         * visible interval has been changed.
         */
        function updateAndRenderInterval(operation) {

            var // the current effective scroll position
                newScrollPos = calculateScrollPosition(),
                // the absolute position of the first visible pixel
                visibleOffset = newScrollPos + hiddenSize,
                // the absolute offset behind the last visible pixel
                visibleEndOffset = visibleOffset + paneSize,
                // the end offset of the available header cells
                intervalEndOffset = position.offset + position.size,
                // the data of the new first or last column/row
                entryData = null;

            // recalculate the interval on demand (if it is missing, or the
            // visible area is near the borders of the interval; but always if
            // an operation has been applied at the column/row collection)
            if (!interval || _.isObject(operation) ||
                ((position.offset > 0) && (visibleOffset - position.offset < paneSize * ADDITIONAL_SIZE_RATIO / 3)) ||
                ((intervalEndOffset < collection.getTotalSize()) && (intervalEndOffset - visibleEndOffset < paneSize * ADDITIONAL_SIZE_RATIO / 3))
            ) {

                interval = {};

                // start position of the interval (add all leading hidden columns/rows in the sheet)
                entryData = collection.getEntryByOffset(visibleOffset - paneSize * ADDITIONAL_SIZE_RATIO, { pixel: true });
                interval.first = (entryData.offset === 0) ? 0 : entryData.index;
                position.offset = entryData.offset;

                // end position of the interval (add all trailing hidden columns/rows in the sheet)
                entryData = collection.getEntryByOffset(visibleOffset + paneSize * (1 + ADDITIONAL_SIZE_RATIO), { pixel: true });
                interval.last = (entryData.offset + entryData.size >= collection.getTotalSize()) ? collection.getMaxIndex() : entryData.index;
                position.size = entryData.offset + entryData.size - position.offset;

                // render all header cells and the selection
                Utils.log('HeaderPane.updateAndRenderInterval(): paneSide=' + paneSide + ', interval=' + (columns ? SheetUtils.getColIntervalName : SheetUtils.getRowIntervalName)(interval));
                updateScrollPosition(newScrollPos);
                (columns ? renderColumnCells : renderRowCells)();

                // notify listeners
                self.trigger('change:interval', _.clone(interval), operation);
            } else {
                updateScrollPosition(newScrollPos);
            }
        }

        /**
         * Clears the current row/column interval. Triggers a 'hide:interval'
         * event, if the interval was valid and visible before.
         */
        function hideInterval() {
            if (interval) {
                interval = null;
                usedCount = scrollPos = scrollSize = 0;
                self.trigger('hide:interval');
            }
        }

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this header pane is visible.
         */
        function registerEventHandler(source, type, handler) {
            source.on(type, function () {
                if (interval) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Prepares this header pane for changing the active sheet.
         */
        function beforeActiveSheetHandler() {
            // reset the column/row interval, and notify listeners
            hideInterval();
        }

        /**
         * Initializes this header pane after the active sheet has been
         * changed.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();
        }

        /**
         * Checks if the number of used columns/rows in the sheet has been
         * changed, and updates the size of the scroll area.
         */
        function changeLayoutDataHandler() {

            var // the current number of used columns/rows
                newUsedCount = columns ? view.getUsedCols() : view.getUsedRows();

            if (usedCount !== newUsedCount) {
                usedCount = newUsedCount;
                updateScrollAreaSize(scrollPos);
            }
        }

        /**
         * Updates this header pane after specific sheet view attributes have
         * been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }

            // changed zoom factor: recalculate the visible interval
            if (Utils.hasAnyProperty(attributes, ['zoom', anchorName])) {
                updateAndRenderInterval(('zoom' in attributes) ? { name: 'refresh' } : null);
            }

            // repaint the selected header cells
            if ('selection' in attributes) {
                renderSelection();
            }
        }

        /**
         * Handles inserted entries in the column/row collection.
         */
        function insertEntriesHandler(event, changedInterval) {
            updateAndRenderInterval(_({ name: 'insert' }).extend(changedInterval));
        }

        /**
         * Handles deleted entries in the column/row collection.
         */
        function deleteEntriesHandler(event, changedInterval) {
            updateAndRenderInterval(_({ name: 'delete' }).extend(changedInterval));
        }

        /**
         * Handles changed entries in the column/row collection.
         */
        function changeEntriesHandler(event, changedInterval, attributes, options) {
            updateAndRenderInterval(_({ name: 'change', attrs: attributes }).extend(changedInterval, options));
        }

        /**
         * Initialization of class members.
         */
        function initHandler() {

            // the spreadsheet model and view
            view = app.getView();

            // listen to view events
            view.on('before:activesheet', beforeActiveSheetHandler);
            view.on('change:activesheet', changeActiveSheetHandler);
            view.on('change:layoutdata', changeLayoutDataHandler);
            registerEventHandler(view, 'change:sheetviewattributes', changeSheetViewAttributesHandler);
            registerEventHandler(view, columns ? 'insert:columns' : 'insert:rows', insertEntriesHandler);
            registerEventHandler(view, columns ? 'delete:columns' : 'delete:rows', deleteEntriesHandler);
            registerEventHandler(view, columns ? 'change:columns' : 'change:rows', changeEntriesHandler);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the settings and layout of this header pane.
         *
         * @param {Object} settings
         *  The view settings of the pane side represented by this header pane.
         *  Supports the following properties:
         *  @param {Boolean} settings.visible
         *      Whether the header pane is visible.
         *  @param {Number} settings.offset
         *      The absolute offset of the header pane in the view root node,
         *      in pixels.
         *  @param {Number} settings.size
         *      The size of the header pane (width of column panes, height of
         *      row panes), in pixels.
         *  @param {Boolean} settings.frozen
         *      Whether the header pane and the associated grid panes are
         *      frozen (not scrollable in the main direction).
         *  @param {Number} settings.hiddenSize
         *      The total size of the hidden columns/rows in front of the sheet
         *      in frozen view mode.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (settings) {

            // copy passed settings
            paneSize = settings.size;
            frozen = settings.frozen;
            hiddenSize = settings.hiddenSize;

            // initialize according to visibility
            if (settings.visible) {

                // show pane root node and initialize auto-scrolling
                rootNode.show().enableTracking({
                    selector: '.cell, .resizer',
                    autoScroll: settings.frozen ? false : (columns ? 'horizontal' : 'vertical'),
                    borderNode: rootNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 2,
                    maxSpeed: 250
                });

                // position and size
                rootNode.css(columns ?
                    { left: settings.offset, width: paneSize, height: view.getCornerPane().getNode().height() } :
                    { top: settings.offset, width: view.getCornerPane().getNode().width(), height: paneSize }
                );

                // recalculate the interval and header cells
                updateAndRenderInterval();

                // recalculate the scroll size
                updateScrollAreaSize(scrollPos);

                // update CSS classes at root node for focus display
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide().disableTracking();

                // reset the column/row interval, and notify listeners
                hideInterval();
            }

            return this;
        };

        /**
         * Returns the column/row interval currently shown in this header pane.
         *
         * @returns {Object}
         *  The column/row interval of this header pane, in the zero-based
         *  index properties 'first' and 'last'.
         */
        this.getInterval = function () {
            return _.clone(interval);
        };

        /**
         * Returns the offset and size of the entire column/row interval
         * currently covered by this header pane, including the columns/rows
         * around the visible area.
         *
         * @returns {Object}
         *  The offset and size of the column/row interval, in pixels.
         */
        this.getIntervalPosition = function () {
            return _.clone(position);
        };

        /**
         * Returns the offset and size of the visible area currently shown in
         * this header pane.
         *
         * @returns {Object}
         *  The offset and size of the visible area in this header pane, in
         *  pixels.
         */
        this.getVisiblePosition = function () {
            return { offset: hiddenSize + scrollPos, size: paneSize };
        };

        /**
         * Returns the current scroll position of this header pane.
         *
         * @returns {Number}
         *  The current scroll position, in pixels.
         */
        this.getScrollPos = function () {
            return scrollPos;
        };

        /**
         * Returns the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position.
         *
         * @returns {Number}
         *  The size of the scrollable area, in pixels.
         */
        this.getScrollSize = function () {
            return scrollSize;
        };

        /**
         * Returns the zero-based index of the first visible column/row in this
         * header pane. The column/row is considered visible, it at least half
         * of its total size is really visible.
         *
         * @returns {Number}
         *  The zero-based index of the first visible column/row.
         */
        this.getFirstVisibleIndex = function () {

            var // the current scroll anchor
                scrollAnchor = sheetModel.getViewAttribute(anchorName),
                // the layout data of the next column/row
                nextEntry = null;

            // check if the cell is visible less than half, go to next cell in that case
            if ((scrollAnchor.ratio > 0.5) || !collection.isEntryVisible(scrollAnchor.index)) {
                nextEntry = collection.getNextVisibleEntry(scrollAnchor.index + 1);
            }

            return nextEntry ? nextEntry.index : scrollAnchor.index;
        };

        /**
         * Changes the scroll position of this header pane, and triggers a
         * 'change:scrollpos' event that causes all associated grid panes to
         * update their own scroll position, and optionally a debounced
         * 'change:scrollsize' event, if the total size of the scroll area has
         * been changed due to the new scroll position.
         *
         * @param {Number} pos
         *  The new absolute scroll position in the sheet, in pixels.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollTo = app.createDebouncedMethod(function (pos) {

            var // the new scroll position
                newScrollPos = Math.max(0, pos),
                // the offset of the new first visible pixel
                offset = newScrollPos + hiddenSize;

            // do nothing if scroll position did not change
            if (frozen || (scrollPos === newScrollPos)) { return this; }

            // recalculate scroll area size immediately, if new scroll position is outside
            if (newScrollPos > scrollSize - paneSize) {
                updateScrollAreaSize(newScrollPos);
            }

            // update scroll anchor view properties, this will trigger change
            // handler from the view that repositions the content node
            sheetModel.setViewAttribute(anchorName, collection.getScrollAnchorByOffset(offset, { pixel: true }));

            return this;

        }, function () { updateScrollAreaSize(scrollPos); }, { context: this, delay: 250 });

        /**
         * Changes the scroll position of this header pane relative to the
         * current scroll position.
         *
         * @param {Number} diff
         *  The difference for the scroll position, in pixels.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (diff) {
            return this.scrollTo(scrollPos + diff);
        };

        /**
         * Scrolls this header pane to make the specified area visible.
         *
         * @param {Number} offset
         *  The start offset of the area to be made visible, in pixels.
         *
         * @param {Number} size
         *  The size of the area to be made visible, in pixels.
         */
        this.scrollToPosition = function (offset, size) {

            var // the end offset of the area (with little extra space to make selection border visible)
                endOffset = offset + size + 2,
                // the absolute offset and size of the visible area
                visibleOffset = this.getVisiblePosition().offset,
                // the inner pane side (without scroll bars in grid panes)
                innerSize = paneSize - (columns ? Utils.SCROLLBAR_WIDTH : Utils.SCROLLBAR_HEIGHT),
                // whether the leading border is outside the visible area
                leadingOutside = offset < visibleOffset,
                // whether the trailing border is outside the visible area
                trailingOutside = endOffset > visibleOffset + innerSize,
                // new scrolling position
                newScrollPos = scrollPos;

            if (leadingOutside && !trailingOutside) {
                newScrollPos = Math.max(offset, endOffset - innerSize) - hiddenSize;
            } else if (!leadingOutside && trailingOutside) {
                newScrollPos = Math.min(offset, endOffset - innerSize) - hiddenSize;
            }

            // update scroll position
            return this.scrollTo(newScrollPos);
        };

        /**
         * Scrolls this header pane to make the specified column/row visible.
         *
         * @param {Number} index
         *  The zero-based column/row index to be made visible.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scrollToEntry = function (index) {

            var // the position and size of the target column/row
                entryData = collection.getEntry(index);

            // update scroll position
            return this.scrollToPosition(entryData.offset, entryData.size);
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', initHandler);

    } // class HeaderPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: HeaderPane });

});
