/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/view/cellstylepicker',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, IO, EditControls, SheetUtils, CellStylePicker, gt) {

    'use strict';

    var // class name shortcuts
        Button = EditControls.Button,
        RadioGroup = EditControls.RadioGroup,
        RadioList = EditControls.RadioList,

        // predefined number format categories
        NUMBER_FORMAT_CATEGORIES = [
            { value: 'standard',   label: /*#. Number format category in spreadsheets: no special format */ gt('Standard') },
            { value: 'number',     label: /*#. Number format category in spreadsheets: numbers with specific count of decimal places */ gt('Number') },
            { value: 'currency',   label: /*#. Number format category in spreadsheets: number with currency symbol */ gt('Currency') },
            { value: 'date',       label: /*#. Number format category in spreadsheets: date formats */ gt('Date') },
            { value: 'time',       label: /*#. Number format category in spreadsheets: time formats */ gt('Time') },
            { value: 'datetime',   label: /*#. Number format category in spreadsheets: combined date/time formats */ gt('Date and time') },
            { value: 'percent',    label: /*#. Number format category in spreadsheets: numbers with percent sign */ gt('Percentage') },
            { value: 'text',       label: /*#. Number format category in spreadsheets: text only */ gt('Text')},
            { value: 'scientific', label: /*#. Number format category in spreadsheets: scientific notation (e.g. 1.23E+10) */ gt('Scientific') },
            { value: 'fraction',   label: /*#. Number format category in spreadsheets: fractional numbers (e.g. 3 1/4) */ gt('Fraction') },
            { value: 'custom',     label: /*#. Number format category in spreadsheets: all user-defined number formats */ gt('Custom') }
        ];

    // static class SpreadsheetControls =======================================

    /**
     * Additional classes defining specialized GUI controls for the OX
     * Spreadsheet application.
     *
     * @extends EditControls
     */
    var SpreadsheetControls = _.clone(EditControls);

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    SpreadsheetControls.ActiveSheetGroup = RadioGroup.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // get application model and view
            model = app.getModel(),
            view = app.getView(),

            // the scroll button nodes
            prevButton = Utils.createButton({ icon: 'icon-angle-left' }).addClass('scroll-button'),
            nextButton = Utils.createButton({ icon: 'icon-angle-right' }).addClass('scroll-button'),

            // index of the first and last sheet tab button currently visible
            firstIndex = 0,
            lastIndex = 0,

            // whether scroll mode is enabled
            scrollable = false;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'active-sheet-group' });

        // private methods ----------------------------------------------------

        /**
         * Shows all sheet buttons and removes explicit width, and hides the
         * scroll buttons.
         */
        function showAllSheetButtons() {
            self.getOptionButtons().removeClass('hidden').css('width', '');
            prevButton.addClass('hidden');
            nextButton.addClass('hidden').css({ position: '', right: '' });
            scrollable = false;
        }

        /**
         * Shows the sheet buttons starting at the start sheet index currently
         * cached.
         */
        function renderSheetButtons() {

            var // the root node of this group
                groupNode = self.getNode(),
                // the outer width of the group
                width = groupNode[0].clientWidth,
                // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // array index to a sheet button
                buttonIndex = 0,
                // a single sheet button
                sheetButton = null;

            // initialize all sheet buttons, hide scroll buttons
            showAllSheetButtons();

            // check if scroll mode needs to be enabled
            if (width < groupNode[0].scrollWidth) {

                // do not show scroll buttons, if there is only one visible sheet
                if (sheetButtons.length > 1) {

                    // validate scroll index according to available number of sheets
                    firstIndex = Utils.minMax(firstIndex, 0, sheetButtons.length - 1);

                    // show scroll buttons
                    prevButton.removeClass('hidden');
                    nextButton.removeClass('hidden');
                    scrollable = true;

                    // hide leading sheet tabs
                    for (buttonIndex = 0; (buttonIndex < firstIndex) && (width < groupNode[0].scrollWidth); buttonIndex += 1) {
                        sheetButtons.eq(buttonIndex).addClass('hidden');
                    }
                    firstIndex = buttonIndex;

                    // hide trailing sheet tabs
                    for (buttonIndex = sheetButtons.length - 1; (buttonIndex > firstIndex) && (width < groupNode[0].scrollWidth); buttonIndex -= 1) {
                        sheetButtons.eq(buttonIndex).addClass('hidden');
                    }
                    lastIndex = buttonIndex;

                    // enable/disable scroll buttons
                    prevButton.toggleClass(Utils.DISABLED_CLASS, firstIndex === 0);
                    nextButton.toggleClass(Utils.DISABLED_CLASS, lastIndex + 1 === sheetButtons.length);

                } else {
                    // single sheet: initialize indexes
                    firstIndex = lastIndex = 0;
                }

                // shrink single sheet tab button, if still oversized
                if ((firstIndex === lastIndex) && (width < groupNode[0].scrollWidth)) {
                    sheetButton = sheetButtons.eq(firstIndex);
                    sheetButton.width(sheetButton.width() - groupNode[0].scrollWidth + width);
                } else {
                    // add the first hidden sheet button after the last visible, if there is some space left
                    if ((lastIndex + 1 < sheetButtons.length) && (width - (nextButton.position().left + nextButton.outerWidth()) >= 30)) {
                        sheetButton = sheetButtons.eq(lastIndex + 1);
                        sheetButton.removeClass('hidden');
                        sheetButton.width(sheetButton.width() - groupNode[0].scrollWidth + width);
                    }
                    // set fixed position for the right scroll button
                    nextButton.css({ position: 'absolute', right: 0 });
                }
            }
        }

        /**
         * Scrolls to the left to make at least one preceding sheet tab button
         * visible.
         */
        function scrollToPrev() {

            var // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to previous sheets until the index of the last sheet has changed
            while ((firstIndex > 0) && (oldLastIndex === lastIndex)) {
                firstIndex -= 1;
                renderSheetButtons();
            }
        }

        /**
         * Scrolls to the right to make at least one following sheet tab button
         * visible.
         */
        function scrollToNext() {

            var // all sheet tab buttons
                sheetButtons = self.getOptionButtons(),
                // index of the old last visible sheet
                oldLastIndex = lastIndex;

            // move to next sheets until the index of the last sheet has changed
            while ((lastIndex + 1 < sheetButtons.length) && (oldLastIndex === lastIndex)) {
                firstIndex += 1;
                renderSheetButtons();
            }
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(newIndex, options, oldIndex) {

            // do not change anything, if the sheet index did not change
            if (!scrollable || (newIndex === oldIndex)) {
                return;
            }

            // scroll backwards, if the active sheet is located before the first visible
            if (newIndex < firstIndex) {
                firstIndex = newIndex;
                renderSheetButtons();
                return;
            }

            // scroll forwards, if the active sheet is located after the last visible
            while (lastIndex < newIndex) {
                scrollToNext();
            }
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            // create buttons for all visible sheets
            self.clearOptionButtons();
            view.iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, { icon: sheetModel.isProtected() ? 'icon-lock' : null, label: _.noI18n(sheetName), tooltip: _.noI18n(sheetName) });
            });

            // move trailing scroll button to the end
            self.addChildNodes(nextButton);
            self.trigger('refresh:layout');
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to automatic width, shows all sheet tab buttons,
         * and hides the scroll buttons.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setFullWidth = function () {
            this.getNode().css('width', '');
            showAllSheetButtons();
            return this;
        };

        /**
         * Sets this control to a fixed width. Performs further initialization
         * according to the available space: shows/hides the scroll buttons,
         * and draws the available sheet tabs according to the last cached
         * scroll position.
         *
         * @param {Number} width
         *  The new outer width of this control, in pixels.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().width(width);
            renderSheetButtons();
            return this;
        };

        // initialization -----------------------------------------------------

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // rebuild the list of available sheets after it has been changed
        app.on('docs:import:success', function () {
            fillList();
            model.on('change:sheetattributes', fillList); // e.g. sheet protection
            view.on('change:visiblesheets', fillList);
        });

        // add scroll buttons to the group node
        this.addFocusableControl(prevButton).addFocusableControl(nextButton);

        // handle tracking events of scroll buttons
        prevButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', scrollToPrev)
            .on('tracking:end tracking:cancel', function () { view.grabFocus(); });
        nextButton.enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', scrollToNext)
            .on('tracking:end tracking:cancel', function () { view.grabFocus(); });

    }}); // class ActiveSheetGroup

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.ActiveSheetList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: 'icon-reorder',
            tooltip: gt('Select sheet'),
            caret: false,
            preferSide: 'top',
            updateCaptionMode: Utils.getBooleanOption(initOptions, 'showNames', false) ? 'label' : 'none'
        });

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            app.getView().iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton('', index, { icon: sheetModel.isProtected() ? 'icon-lock' : 'docs-empty', label: _.noI18n(sheetName), tooltip: _.noI18n(sheetName) });
            });
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to fixed width mode.
         *
         * @param {Number} width
         *  The new outer width of this control.
         *
         * @returns {ActiveSheetList}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().css({ width: width });
            // prevent wrong position/size of the button element due to padding/margin
            this.getMenuButton().css({ position: 'absolute', left: 0, right: 0, top: 0, bottom: 0 });
            return this;
        };

        // initialization -----------------------------------------------------

        app.on('docs:import:success', function () {
            fillList();
            app.getView().on('change:visiblesheets', fillList);
        });

    }}); // class ActiveSheetList

    // class CellBorderPicker =================================================

    /**
     * The selector for cell borders. Hides specific items in the drop-down
     * menu, according to the current cell selection.
     *
     * @constructor
     *
     * @extends EditControls.BorderPicker
     */
    SpreadsheetControls.CellBorderPicker = EditControls.BorderPicker.extend({ constructor: function (app, initOptions) {

        var // the spreadsheet view instance
            view = app.getView();

        // base constructor ---------------------------------------------------

        EditControls.BorderPicker.call(this, Utils.extendOptions({
            tooltip: gt('Cell borders')
        }, initOptions, {
            showInsideHor: function () { return view.hasMultipleRowsSelected(); },
            showInsideVert: function () { return view.hasMultipleColumnsSelected(); }
        }));

    }}); // class CellBorderPicker

    // class MergePicker ======================================================

    /**
     * The selector for merged ranges, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.MergePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the spreadsheet view instance
            view = app.getView(),

            // whether to show the 'merge' menu item
            showMergeItem = false,
            // whether to show the 'merge horizontal' and 'merge vertical' menu items
            showHVMergeItems = false,
            // whether to show the 'unmerge' menu item
            showUnmergeItem = false;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-merged-cells-on',
            tooltip: gt('Merge or unmerge cells')
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the merged state of the selection.
         */
        function updateHandler() {
            Utils.setControlCaption(self.getCaptionButton(), { icon: showUnmergeItem ? 'docs-merged-cells-off' : 'docs-merged-cells-on' });
        }

        /**
         * Updates the visibility flags of the list items after the selection
         * in the active sheet has been changed, or after the merge collection
         * of the active sheet has changed.
         */
        function updateVisibilityFlags() {

            var // the collection of merged ranges in the active sheet
                mergeCollection = view.getActiveSheetModel().getMergeCollection(),
                // the selected cell ranges
                ranges = view.getSelectedRanges();

            // show the 'merge' list item, if at least one range consists of more than a cell or a merged range
            showMergeItem = _(ranges).any(function (range) {
                var mergedRange = (SheetUtils.getCellCount(range) > 1) ? mergeCollection.getMergedRange(range.start) : null;
                return !mergedRange || !SheetUtils.rangeContainsRange(mergedRange, range);
            });

            // show the 'horizontal'/'vertical' list items, if at least one range consists of multiple columns AND rows
            showHVMergeItems = _(ranges).any(function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]);
            });

            // show the 'unmerge' list item, if at least one range overlaps with a merged range
            showUnmergeItem = mergeCollection.rangesOverlapAnyMergedRange(ranges);
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('', 'merge',      { icon: 'docs-merged-cells-on',         label: gt('Merge cells'),              visible: function () { return showMergeItem; } })
            .createOptionButton('', 'horizontal', { icon: 'docs-merged-cells-horizontal', label: gt('Merge cells horizontally'), visible: function () { return showHVMergeItems; } })
            .createOptionButton('', 'vertical',   { icon: 'docs-merged-cells-vertical',   label: gt('Merge cells vertically'),   visible: function () { return showHVMergeItems; } })
            .createOptionButton('', 'unmerge',    { icon: 'docs-merged-cells-off',        label: gt('Unmerge cells'),            visible: function () { return showUnmergeItem; } });

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // update state flags used for visibility of list items (to keep the 'visible' callback functions simple and fast)
        view.on('change:selection insert:merged delete:merged change:merged', updateVisibilityFlags);

    }}); // class MergePicker

    // class NumberFormatPicker ===============================================

    /**
     * A drop-down list control for choosing number format categories, that
     * will load predefined format codes in the FormatCodePicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the drop-down list.
     *  Supports all options of the base class RadioList.
     */
    SpreadsheetControls.NumberFormatPicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-border',
            tooltip: /*#. in paragraphs and tables cells */ gt('Number Format')
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Insert all supported list entries according to the passed options.
         */
        function fillList() {
            // insert all supported list items
            _(NUMBER_FORMAT_CATEGORIES).each(function (entry) {
                self.createOptionButton('', entry.value, { label: entry.label, dataValue: entry.value });
            });
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {
            // defaults for e.g. not supported number formats (temporary)
            // TODO specify behavior with unsupported number formats
            var newLabel = gt('Unsupported');
            if (_.isString(value) && value.length > 0) {
                var activeEntry = _.find(NUMBER_FORMAT_CATEGORIES, function (elem) {
                    return elem.value === value;
                });
                if (!_.isUndefined(activeEntry)) {
                    newLabel = activeEntry.label;
                }
                // Temporary till we have nice icons
                Utils.setControlCaption(self.getMenuButton(), { label: newLabel });
            }
        }

        // initialization -----------------------------------------------------

        // initialize the drop-down list items
        fillList();

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

    }}); // class NumberFormatPicker

    // class FormatCodePicker =================================================

    /**
     * A drop-down list control for selection of predefined number formats
     * depending to currently selected category in the NumberFormatPicker.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class RadioList.
     */
    SpreadsheetControls.FormatCodePicker = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,
            numberFormats = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Format codes')
        }, initOptions));

        // private methods ----------------------------------------------------
        /**
         * Updates the label of the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {
            var categoryCodes = numberFormats[app.getView().getNumberFormatCategory()];
            if (_.isUndefined(categoryCodes)) return;
            var activeFormatCode = _.find(categoryCodes, function (formatCode) {
                return formatCode.value === value;
            });
            var options = { label: gt('Custom') };
            if (!_.isUndefined(activeFormatCode)) {
                options.label = activeFormatCode.label;
                if (activeFormatCode.red) { options.labelCss = { color: '#ff5555' }; }
            }
            Utils.setControlCaption(self.getMenuButton(), options);
        }

        // methods ------------------------------------------------------------

        /**
         * Refreshes this format code drop-down list after a change in format code category picker
         * @param category
         */
        this.refreshList =  function (category) {
            self.clearMenu();
            // list refresh is not required if standard is chosen.
            if (category.toLowerCase() === 'standard') return;
            _(numberFormats[category]).each(function (code) {
                var labelCss = code.red ? { color: 'red' } : undefined;
                self.createOptionButton('', code.value, { label: code.label, labelCss: labelCss, dataValue: code.value });
            });
        };

        /**
         * Returns the default format code for a selected category.
         * This default code will be set to the active cell
         * when a user sets a new format code category in the picker.
         *
         * @param category
         * @returns {*}
         */
        this.getDefaultCodeForCategory = function (category) {
            if (!category) return null;
            var categoryFormats = numberFormats[category.toLocaleLowerCase()];
            if (_.isUndefined(categoryFormats) || _.isUndefined(categoryFormats[0])) return null;
            return categoryFormats[0].value;
        };

        // initialization -----------------------------------------------------

        // register custom update handler to update button caption
        this.registerUpdateHandler(updateHandler);

        // get localized predefined format codes
        IO.loadResource('io.ox/office/spreadsheet/resource/numberformats').done(function (numberFormatsData) {
            numberFormats = numberFormatsData;
        });

    }}); // class FormatCodePicker

    // class SubtotalList =====================================================

    /**
     * The subtotal 'customizer'. A yet another drop down list for displaying
     * desired subtotal value.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.SubtotalList = RadioList.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the view instance
            view = app.getView();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: gt('Select type of subtotal value')
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            _(app.getView().getSubtotals()).each(function (value, key) {
                self.createOptionButton('', key, getCaptionOption(key, value));
            });
        });

        /**
         * Callback which is called with every controller update (change:selection, change:layoutdata)
         */
        function updateHandler(subtotalType) {

            var // get fresh subtotals
                subtotals = view.getSubtotals();

            // quit if in any case no subtotals is supplied
            if (_.isNull(subtotals)) return;

            // if only 1 cell is selected, quit immediately
            if (subtotals.cells <= 1) {
                self.hide();
                return;
            }

            // if selection only consists of string values, show only cell count and quit
            if (subtotals.numbers === 0) {
                self.clearMenu();
                self.createOptionButton('', 'cells', getCaptionOption('cells', subtotals.cells));
                Utils.setControlCaption(self.getMenuButton(), getCaptionOption('cells', subtotals.cells));
                self.show();
                self.trigger('group:layout');
                return;
            }

            // set requested subtotal type with its properties and show it
            Utils.setControlCaption(self.getMenuButton(), getCaptionOption(subtotalType, subtotals[subtotalType]));
            self.show();
            self.trigger('group:layout');
        }

        // create and return drop-down caption option object according to subtotal type
        function getCaptionOption(type, value) {

            var option = {};

            // show maximal 2 decimal places in the label.
            if (_.isNumber(value)) {
                value = parseFloat(value.toFixed(2));
            }

            switch (type) {
            case 'sum':
                option = { label: gt('Sum: %1$d', _.noI18n(value)), tooltip: gt('Sum of selected cells') };
                break;
            case 'min':
                option = { label: gt('Min: %1$d', _.noI18n(value)), tooltip: gt('Minimum value in selection') };
                break;
            case 'max':
                option = { label: gt('Max: %1$d', _.noI18n(value)), tooltip: gt('Maximum value in selection') };
                break;
            case 'cells':
                option = { label: gt('Count: %1$d', _.noI18n(value)), tooltip: gt('Number of selected cells that contain data') };
                break;
            case 'numbers':
                option = { label: gt('Numerical count: %1$d', _.noI18n(value)), tooltip: gt('Number of selected cells that contain numerical data') };
                break;
            case 'average':
                option = { label: gt('Average: %1$d', _.noI18n(value)), tooltip: gt('Average of selected cells') };
                break;
            default:
                break;
            }
            return option;
        }

        // initialization -----------------------------------------------------

        app.on('docs:import:success', function () {
            fillList();
            view.on('change:layoutdata', fillList);
        });

        this.registerUpdateHandler(updateHandler).hide();

    }}); // class SubtotalList

    // class CellStylePicker ==================================================

    SpreadsheetControls.CellStylePicker = CellStylePicker;

    // class DynamicSplitButton ===============================================

    /**
     * A button to toggle the dynamic split view.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.DynamicSplitButton = Button.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({
            label: /*#. Short (!) label for a button used to split a spreadsheet into 2 or 4 different parts that can be scrolled independently */ gt('Split'),
            tooltip: gt('Split the sheet above and left of the cursor')
        }, initOptions, {
            toggle: true
        }));

    }}); // class DynamicSplitButton

    // class FrozenSplitPicker ================================================

    /**
     * The selector for frozen view panes, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.FrozenSplitPicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: /*#. Short (!) label for a button used to freeze leading columns/rows in a spreadsheet (make them non-scrollable) */ gt('Freeze'),
            tooltip: gt('Freeze the rows above and the columns left of the cursor')
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('', { cols: 0, rows: 1 }, { label: gt('Freeze first row'), tooltip: gt('Freeze the first visible row') })
            .createOptionButton('', { cols: 1, rows: 0 }, { label: gt('Freeze first column'), tooltip: gt('Freeze the first visible column') })
            .createOptionButton('', { cols: 1, rows: 1 }, { label: gt('Freeze first row and column'), tooltip: gt('Freeze the first visible row and the first visible column') });

    }}); // class FrozenSplitPicker

    // exports ================================================================

    return SpreadsheetControls;

});
