/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils',
    ['io.ox/office/tk/utils',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, gt) {

    'use strict';

    var // maximum number of columns that will ever be allowed in a sheet (A to ZZZZ)
        MAX_COL_COUNT = 475254,

        // maximum number of rows that will ever be allowed in a sheet (1 to 99,999,999)
        MAX_ROW_COUNT = 99999999,

        // regular expression pattern matching a valid column name (A to ZZZZ)
        COL_PATTERN = '[A-Z]{1,4}',

        // regular expression pattern matching a row name (1 to 99,999,999)
        ROW_PATTERN = '0*[1-9][0-9]{0,7}',

        // regular expression object matching a column name only
        RE_COL_NAME = new RegExp('^' + COL_PATTERN + '$'),

        // regular expression object matching a row name only
        RE_ROW_NAME = new RegExp('^' + ROW_PATTERN + '$');

    // static class SheetUtils ================================================

    var SheetUtils = {};

    // constants --------------------------------------------------------------

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum width of columns, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_COLUMN_WIDTH = 20000;

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum height of rows, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_ROW_HEIGHT = 20000;

    /**
     * The maximum number of cells to be filled with one range operation.
     */
    SheetUtils.MAX_FILL_CELL_COUNT = 1000;

    /**
     * Minimum allowed zoom factor in the spreadsheet view.
     */
    SheetUtils.MIN_ZOOM = 0.5;

    /**
     * Maximum allowed zoom factor in the spreadsheet view.
     */
    SheetUtils.MAX_ZOOM = 8;

    // methods ----------------------------------------------------------------

    /**
     * Checks the column and row indexes contained in the passed range, and
     * swaps them (in-place) if they are not ordered.
     *
     * @param {Object} range
     *  (in/out) The logical address of a range, that will be adjusted in-place
     *  if required.
     *
     * @returns {Object}
     *  A reference to the passed range.
     */
    SheetUtils.adjustRange = function (range) {
        var tmp = null;
        // sort columns and rows independently
        if (range.start[0] > range.end[0]) { tmp = range.start[0]; range.start[0] = range.end[0]; range.end[0] = tmp; }
        if (range.start[1] > range.end[1]) { tmp = range.start[1]; range.start[1] = range.end[1]; range.end[1] = tmp; }
        return range;
    };

    /**
     * Returns a new range with column and row indexes in the correct order.
     *
     * @param {Object} range
     *  The logical address of a range.
     *
     * @returns {Object}
     *  A new adjusted cell range address.
     */
    SheetUtils.getAdjustedRange = function (range) {
        return {
            start: [Math.min(range.start[0], range.end[0]), Math.min(range.start[1], range.end[1])],
            end: [Math.max(range.start[0], range.end[0]), Math.max(range.start[1], range.end[1])]
        };
    };

    /**
     * Returns the number of columns covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of columns in the passed range.
     */
    SheetUtils.getColCount = function (range) {
        return range.end[0] - range.start[0] + 1;
    };

    /**
     * Returns the number of rows covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of rows in the passed range.
     */
    SheetUtils.getRowCount = function (range) {
        return range.end[1] - range.start[1] + 1;
    };

    /**
     * Returns the number of cells covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of cells in the passed range.
     */
    SheetUtils.getCellCount = function (range) {
        return SheetUtils.getColCount(range) * SheetUtils.getRowCount(range);
    };

    /**
     * Returns whether the passed range contains the specified column index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the column index.
     */
    SheetUtils.rangeContainsCol = function (range, col) {
        return (range.start[0] <= col) && (col <= range.end[0]);
    };

    /**
     * Returns whether the passed range contains the specified row index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the row index.
     */
    SheetUtils.rangeContainsRow = function (range, row) {
        return (range.start[1] <= row) && (row <= range.end[1]);
    };

    /**
     * Returns whether the passed bounding cell range contains the specified
     * cell.
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the cell address.
     */
    SheetUtils.rangeContainsCell = function (boundRange, address) {
        return SheetUtils.rangeContainsCol(boundRange, address[0]) && SheetUtils.rangeContainsRow(boundRange, address[1]);
    };

    /**
     * Returns whether any of the cell ranges in the passed array contains the
     * specified cell.
     *
     * @param {Array} ranges
     *  The logical addresses of the cell ranges to be checked.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether any of the passed cell ranges contains the cell address.
     */
    SheetUtils.rangesContainCell = function (ranges, address) {
        return _(ranges).any(function (range) { return SheetUtils.rangeContainsCell(range, address); });
    };

    /**
     * Returns whether the cell address is one of the outer cells in the passed
     * bounding cell range (a cell located inside the range, but at its left,
     * right, top, or bottom border).
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed cell is one of the outer cells in the range.
     */
    SheetUtils.rangeContainsOuterCell = function (boundRange, address) {
        return (address[0] === boundRange.start[0]) || (address[0] === boundRange.end[0]) || (address[1] === boundRange.start[1]) || (address[1] === boundRange.end[1]);
    };

    /**
     * Returns whether the passed bounding cell range contains another cell
     * range.
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Object} range
     *  The logical cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the other cell range.
     */
    SheetUtils.rangeContainsRange = function (boundRange, range) {
        return SheetUtils.rangeContainsCell(boundRange, range.start) && SheetUtils.rangeContainsCell(boundRange, range.end);
    };

    /**
     * Compares the passed cell addresses.
     *
     * @returns {Number}
     *  - A negative number, if address1 is located before address2 (either in
     *      a preceding row; or in the same row and in a preceding column.
     *  - A positive number, if address1 is located after address2 (either in a
     *      following row; or in the same row and in a following column.
     *  - Zero, if the addresses are equal.
     */
    SheetUtils.compareCells = function (address1, address2) {
        var rowDiff = address1[1] - address2[1];
        return (rowDiff === 0) ? (address1[0] - address2[0]) : rowDiff;
    };

    // range intersections and bounding range ---------------------------------

    /**
     * Returns the unique ranges of the passed range addresses. Does NOT join
     * overlapping or adjacent ranges.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The address of the resulting unique cell ranges.
     */
    SheetUtils.getUniqueRanges = function (ranges) {
        return _.chain(ranges).getArray().unique(SheetUtils.getRangeName).value();
    };

    /**
     * Returns whether the passed ranges overlap with at least one cell.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Boolean}
     *  Whether the passed ranges are overlapping.
     */
    SheetUtils.rangesOverlap = function (range1, range2) {
        return (range1.start[0] <= range2.end[0]) && (range2.start[0] <= range1.end[0]) && (range1.start[1] <= range2.end[1]) && (range2.start[1] <= range1.end[1]);
    };

    /**
     * Returns whether any two ranges in the passed array of cell range
     * addresses overlap each other.
     *
     * @param {Array} ranges
     *  The array of logical range addresses.
     *
     * @returns {Boolean}
     *  Whether any two of the passed ranges are overlapping each other.
     */
    SheetUtils.anyRangesOverlap = function (ranges) {
        // TODO: this has O(n^2) complexity, better algorithm?
        return _(ranges).any(function (range1, index) {
            return _(ranges.slice(index + 1)).any(function (range2) {
                return SheetUtils.rangesOverlap(range1, range2);
            });
        });
    };

    /**
     * Returns the intersecting range of the passed ranges.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Object|Null}
     *  The address of the cell range covered by both passed ranges, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionRange = function (range1, range2) {
        return SheetUtils.rangesOverlap(range1, range2) ? {
            start: [Math.max(range1.start[0], range2.start[0]), Math.max(range1.start[1], range2.start[1])],
            end: [Math.min(range1.end[0], range2.end[0]), Math.min(range1.end[1], range2.end[1])]
        } : null;
    };

    /**
     * Returns the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {Object} boundRange
     *  The logical address of the bounding range.
     *
     * @returns {Array}
     *  The address of the resulting cell ranges covered by the bounding range.
     *  May be an empty array, if the bounding range does not contain any of
     *  the cells in the passed ranges.
     */
    SheetUtils.getIntersectionRanges = function (ranges, boundRange) {
        var result = [];
        _.chain(ranges).getArray().each(function (range) {
            var intersectRange = SheetUtils.getIntersectionRange(range, boundRange);
            if (intersectRange) { result.push(intersectRange); }
        });
        return result;
    };

    /**
     * Iterates the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range. Ranges not contained in the
     * bounding range at all will be skipped in the iteration process.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {Object} boundRange
     *  The logical address of the bounding range.
     *
     * @param {Function} iterator
     *  The iterator function called for all intersection ranges. Receives the
     *  following parameters:
     *  (1) {Object} intersectionRange
     *      The intersection of the original range and the bounding range.
     *  (2) {Object} originalRange
     *      The original range.
     *  (3) {Number} index
     *      The index of the original range in the passed array of ranges.
     *  If the iterator returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [options]
     *  A map with options controlling the behavior of this method. The
     *  following options are supported:
     *  @param {Object} [options.context]
     *      If specified, the iterator will be called with this context
     *      (the symbol 'this' will be bound to the context inside the
     *      iterator function).
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the resulting intersection ranges will be visited
     *      in reversed order.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    SheetUtils.iterateIntersectionRanges = function (ranges, boundRange, iterator, options) {

        var // the calling context for the iterator function
            context = Utils.getOption(options, 'context'),
            // whether to iterate in reversed order
            reverse = Utils.getBooleanOption(options, 'reverse', false);

        return Utils.iterateArray(_.getArray(ranges), function (range, index) {

            var // the intersection between current range and bound range
                intersectRange = SheetUtils.getIntersectionRange(range, boundRange);

            // invoke the iterator, if the intersection is not empty
            return intersectRange ? iterator.call(context, intersectRange, range, index) : undefined;
        }, { reverse: reverse });
    };

    /**
     * Returns the bounding range of the passed cell ranges (the smallest range
     * that contains all passed ranges).
     *
     * @param {Object|Array} [...]
     *  The logical address of a single cell range, or an array with cell range
     *  addresses. The number of parameters that can be passed to this method
     *  is not limited.
     *
     * @returns {Object|Null}
     *  The logical address of the bounding range containing all passed ranges;
     *  or null, if no range address has been passed.
     */
    SheetUtils.getBoundingRange = function () {
        var boundRange = null;
        _(arguments).each(function (ranges) {
            _.chain(ranges).getArray().each(function (range) {
                if (!boundRange) {
                    boundRange = _.copy(range, true);
                } else {
                    boundRange.start[0] = Math.min(boundRange.start[0], range.start[0]);
                    boundRange.start[1] = Math.min(boundRange.start[1], range.start[1]);
                    boundRange.end[0] = Math.max(boundRange.end[0], range.end[0]);
                    boundRange.end[1] = Math.max(boundRange.end[1], range.end[1]);
                }
            });
        });
        return boundRange;
    };

    // column/row intervals ---------------------------------------------------

    /**
     * Returns the number of indexes covered by the passed index interval.
     *
     * @param {Object} interval
     *  A single index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Number}
     *  The number of indexes covered by the passed interval.
     */
    SheetUtils.getIntervalSize = function (interval) {
        return interval.last - interval.first + 1;
    };

    /**
     * Returns whether the passed interval contains the specified index.
     *
     * @param {Object} interval
     *  The index interval, with the zero-based index properties 'first' and
     *  'last'.
     *
     * @param {Number} index
     *  The zero-based index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed interval contains the index.
     */
    SheetUtils.intervalContainsIndex = function (interval, index) {
        return (interval.first <= index) && (index <= interval.last);
    };

    /**
     * Returns whether the passed bounding interval contains another interval.
     *
     * @param {Object} boundInterval
     *  The bounding index interval, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @param {Object} range
     *  The index interval to be checked, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed bounding interval contains the other interval.
     */
    SheetUtils.intervalContainsInterval = function (boundInterval, interval) {
        return (boundInterval.first <= interval.first) && (interval.last <= boundInterval.last);
    };

    /**
     * Returns whether the passed intervals overlap with at least one index.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed intervals are overlapping.
     */
    SheetUtils.intervalsOverlap = function (interval1, interval2) {
        return (interval1.first <= interval2.last) && (interval2.first <= interval1.last);
    };

    /**
     * Returns the intersection of the passed index intervals.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Object|Null}
     *  The intersection interval covered by both passed intervals, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionInterval = function (interval1, interval2) {
        var first = Math.max(interval1.first, interval2.first),
            last = Math.min(interval1.last, interval2.last);
        return (first <= last) ? { first: first, last: last } : null;
    };

    /**
     * Merges the passed column or row intervals.
     */
    function mergeIntervals(sourceIntervals) {

        var // the resulting intervals
            intervals = [];

        // sort the source intervals by start index
        sourceIntervals.sort(function (i1, i2) { return i1.first - i2.first; });

        // merge overlapping intervals
        _(sourceIntervals).each(function (interval) {

            var // last interval in the resulting list
                lastInterval = _.last(intervals);

            if (!lastInterval || (lastInterval.last + 1 < interval.first)) {
                // append a new interval
                intervals.push({ first: interval.first, last: interval.last });
            } else {
                // extend end of last interval
                lastInterval.last = Math.max(lastInterval.last, interval.last);
            }
        });

        return intervals;
    }

    /**
     * Returns the column interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the cell range.
     *
     * @returns {Object}
     *  The column interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getColInterval = function (range) {
        return { first: range.start[0], last: range.end[0] };
    };

    /**
     * Returns the sorted and merged column intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The column intervals covering the passed cell ranges. Each interval
     *  object contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and intervals covering overlapping or adjacent
     *  column ranges have been merged into a single interval object.
     */
    SheetUtils.getColIntervals = function (ranges) {
        return mergeIntervals(_.chain(ranges).getArray().map(SheetUtils.getColInterval).value());
    };

    /**
     * Returns the row interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the cell range.
     *
     * @returns {Object}
     *  The row interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getRowInterval = function (range) {
        return { first: range.start[1], last: range.end[1] };
    };

    /**
     * Returns the sorted and merged row intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The row intervals covering the passed cell ranges. Each interval object
     *  contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and intervals covering overlapping or adjacent
     *  row ranges have been merged into a single interval object.
     */
    SheetUtils.getRowIntervals = function (ranges) {
        return mergeIntervals(_.chain(ranges).getArray().map(SheetUtils.getRowInterval).value());
    };

    /**
     * Builds a logical range address from the passed column and row interval.
     *
     * @param {Object|Number} colInterval
     *  The column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index.
     *
     * @param {Object|Number} rowInterval
     *  The row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index.
     *
     * @returns {Object}
     *  The logical range address built from the passed intervals.
     */
    SheetUtils.makeRangeFromIntervals = function (colInterval, rowInterval) {
        return {
            start: [_.isNumber(colInterval) ? colInterval : colInterval.first, _.isNumber(rowInterval) ? rowInterval : rowInterval.first],
            end: [_.isNumber(colInterval) ? colInterval : colInterval.last, _.isNumber(rowInterval) ? rowInterval : rowInterval.last]
        };
    };

    // unified ranges ---------------------------------------------------------

    /**
     * Divides the passed cell ranges into an array of ordered row intervals
     * which contain the matching parts of the original ranges which do not
     * overlap each other.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single range, or an array of logical range
     *  addresses.
     *
     * @returns {Array}
     *  An array of row band descriptors. Each row band descriptor contains the
     *  following properties:
     *  - {Number} first
     *      The zero-based index of the first row in the row band interval.
     *  - {Number} last
     *      The zero-based index of the last row in the row band interval.
     *  - {Array} intervals
     *      An array with column intervals covering the passed original cell
     *      ranges in this row band.
     */
    SheetUtils.getUnifiedRowBands = function (ranges) {

        var // the row bands containing the matching column intervals
            rowBands = [],
            // the start index in rowBands used to search for matching bands
            startIndex = 0;

        // sort the ranges by start row index
        ranges = _.chain(ranges).getArray().clone().value();
        ranges.sort(function (r1, r2) { return r1.start[1] - r2.start[1]; });

        // build the row bands array
        _(ranges).each(function (range) {

            var // the start and end row index of the range
                firstRow = range.start[1], lastRow = range.end[1],
                // the current row band
                index = 0, rowBand = null;

            // update the row bands containing the range
            for (index = startIndex; index < rowBands.length; index += 1) {
                rowBand = rowBands[index];

                if (rowBand.last < firstRow) {
                    // row band is above the range, ignore it in the next iterations
                    startIndex = index + 1;
                } else if (rowBand.first < firstRow) {
                    // row band needs to be split (range starts inside the row band)
                    rowBands.splice(index, 0, { first: rowBand.first, last: firstRow - 1, ranges: _.clone(rowBand.ranges) });
                    rowBand.first = firstRow;
                    startIndex = index + 1;
                    // next iteration of the for loop will process the current row band again
                } else {

                    if (lastRow < rowBand.last) {
                        // row band needs to be split (range ends inside the row band)
                        rowBands.splice(index + 1, 0, { first: lastRow + 1, last: rowBand.last, ranges: _.clone(rowBand.ranges) });
                        rowBand.last = lastRow;
                    }
                    // row band contains the range completely
                    rowBand.ranges.push(range);
                    // continue with next range, if end of range found
                    if (lastRow === rowBand.last) { return; }
                    // adjust start index in case a new row band needs to be appended
                    firstRow = rowBand.last + 1;
                }
            }

            // no row band found, create and append a new row band
            rowBands.push({ first: firstRow, last: lastRow, ranges: [range] });
        });

        // create the column intervals in all row bands
        _(rowBands).each(function (rowBand) {
            rowBand.intervals = SheetUtils.getColIntervals(rowBand.ranges);
            delete rowBand.ranges;
        });

        return rowBands;
    };

    /**
     * Converts the passed row bands to an array of cell range addresses.
     */
    function convertRowBandsToRanges(rowBands) {

        var // resulting cell ranges
            resultRanges = [],
            // current ranges whose last row ands in the current row band
            bandRanges = [];

        // returns whether the passed range precedes the specified column interval,
        // or whether it starts at the interval, but ends at another column
        function isPrecedingRange(range, interval) {
            return (range.start[0] < interval.first) || ((range.start[0] < interval.first) && (range.end[0] !== interval.last));
        }

        // build ranges from the row bands and their column intervals
        _(rowBands).each(function (rowBand, bandIndex) {

            var // the preceding row band
                prevRowBand = (bandIndex > 0) ? rowBands[bandIndex - 1] : null,
                // array index into the bandRanges array
                rangeIndex = 0;

            // try to extend ranges from the previous row band with ranges from this row band
            if (prevRowBand && (prevRowBand.last + 1 === rowBand.first)) {

                // process all intervals of the current row band, extend ranges or move them to the result array
                _(rowBand.intervals).each(function (interval) {

                    // move all ranges from bandRanges that precede the current interval to the result array
                    while ((rangeIndex < bandRanges.length) && isPrecedingRange(bandRanges[rangeIndex], interval)) {
                        resultRanges = resultRanges.concat(bandRanges.splice(rangeIndex, 1));
                    }

                    // extend the current band range to this row band, if column interval matches
                    var bandRange = bandRanges[rangeIndex];
                    if (bandRange && (bandRange.start[0] === interval.first) && (bandRange.end[0] === interval.last)) {
                        bandRange.end[1] = rowBand.last;
                    } else {
                        bandRanges.splice(rangeIndex, 0, SheetUtils.makeRangeFromIntervals(interval, rowBand));
                    }
                    rangeIndex += 1;
                });

            } else {

                // store all old band ranges in the result array
                resultRanges = resultRanges.concat(bandRanges);
                bandRanges = [];

                // create new ranges for this row band
                _(rowBand.intervals).each(function (interval) {
                    bandRanges.push(SheetUtils.makeRangeFromIntervals(interval, rowBand));
                });
            }
        });

        return resultRanges.concat(bandRanges);
    }

    /**
     * Converts the passed cell ranges to an array of cell ranges that do not
     * overlap each other.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single range, or an array of logical range
     *  addresses.
     *
     * @returns {Array}
     *  An array of cell ranges covering every cell contained in the cell
     *  ranges passed to this function, but that do not overlap each other.
     */
    SheetUtils.getUnifiedRanges = function (ranges) {
        ranges = _.getArray(ranges);
        // at most one range: return directly instead of running the complex row band code
        return (ranges.length <= 1) ? ranges : convertRowBandsToRanges(SheetUtils.getUnifiedRowBands(ranges));
    };

    /**
     * Converts the passed array of logical cell addresses to a unified array
     * of range addresses, trying to join as much cells as possible.
     *
     * @param {Array} addresses
     *  An array of logical cell addresses.
     *
     * @returns {Array}
     *  An array of cell ranges covering every cell contained in the array
     *  passed to this function.
     */
    SheetUtils.joinCellsToRanges = function (addresses) {

        var // the row bands containing column intervals for all existing rows
            rowBands = [];

        // single cell: return directly instead of running the complex row band code
        if (addresses.length === 1) {
            return [{ start: _.clone(addresses[0]), end: _.clone(addresses[0]) }];
        }

        // sort and unify the cell addresses
        addresses = _.clone(addresses);
        addresses.sort(SheetUtils.compareCells);
        addresses = _(addresses).unique(true, SheetUtils.getCellName);

        // create row bands with all rows that contains any cells
        _(addresses).each(function (address) {

            var // the last row band in the array of all row bands
                lastRowBand = _.last(rowBands),
                // the last interval in the current row band
                lastInterval = null;

            // create a new row band on demand
            if (!lastRowBand || (lastRowBand.first !== address[1])) {
                rowBands.push(lastRowBand = { first: address[1], last: address[1], intervals: [] });
            }

            // get the last interval, extend it or create a new interval
            lastInterval =  _.last(lastRowBand.intervals);
            if (lastInterval && (lastInterval.last + 1 === address[0])) {
                lastInterval.last += 1;
            } else {
                lastRowBand.intervals.push({ first: address[0], last: address[0] });
            }
        });

        // convert the row bands to an array of cell ranges
        return convertRowBandsToRanges(rowBands);
    };

    /**
     * Returns a new range with the offset cell address added.
     *
     * @param {Object} range
     *  The logical cell range address as object with 'start' and 'end'
     *  properties, both containing logical cell addresses.
     *
     * @param {Number[]} address
     *  The logical cell address as two-element array, containing the
     *  zero-based column and row indexes of the cell.
     *
     *  @returns {Object}
     *   The new range with the offset added.
     */
    SheetUtils.addOffsetToRange = function (range, address) {
        var newRange = _.copy(range);
        newRange.start[0] += address[0];
        newRange.start[1] += address[1];
        newRange.end[0] += address[0];
        newRange.end[1] += address[1];
        return newRange;
    };

    // generate/parse UI names ------------------------------------------------

    /**
     * Returns the string representation of the passed column index.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column index.
     */
    SheetUtils.getColName = function (col) {
        var name = '';
        while (col >= 0) {
            name = String.fromCharCode(65 + col % 26) + name;
            col = Math.floor(col / 26) - 1;
        }
        return name;
    };

    /**
     * Returns the string representation of the passed column interval.
     *
     * @param {Object|Number} interval
     *  The column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column interval.
     */
    SheetUtils.getColIntervalName = function (interval) {
        return SheetUtils.getColName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getColName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed column intervals.
     *
     * @param {Object|Array} intervals
     *  A single column interval, or an array of column intervals. Each
     *  interval object contains the zero-based index properties 'first' and
     *  'last'.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the column interval names.
     *
     * @returns {String}
     *  The upper-case string representation of the column intervals.
     */
    SheetUtils.getColIntervalsName = function (intervals, separator) {
        return _.chain(intervals).getArray().map(SheetUtils.getColIntervalName).value().join(separator);
    };

    /**
     * Returns the string representation of the passed row index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row index.
     */
    SheetUtils.getRowName = function (row) {
        return String(row + 1);
    };

    /**
     * Returns the string representation of the passed row interval.
     *
     * @param {Object|Number} interval
     *  The row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row interval.
     */
    SheetUtils.getRowIntervalName = function (interval) {
        return SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed row intervals.
     *
     * @param {Object|Array} intervals
     *  A single row interval, or an array of row intervals. Each interval
     *  object contains the zero-based index properties 'first' and 'last'.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the row interval names.
     *
     * @returns {String}
     *  The upper-case string representation of the row intervals.
     */
    SheetUtils.getRowIntervalsName = function (intervals, separator) {
        return _.chain(intervals).getArray().map(SheetUtils.getRowIntervalName).value().join(separator);
    };

    /**
     * Returns the string representation of the passed logical cell address.
     *
     * @param {Number[]} address
     *  The logical cell address as two-element array, containing the
     *  zero-based column and row indexes of the cell.
     *
     * @returns {String}
     *  The string representation of the cell position, in A1 notation.
     */
    SheetUtils.getCellName = function (address) {
        return SheetUtils.getColName(address[0]) + SheetUtils.getRowName(address[1]);
    };

    /**
     * Returns a unique string key for the passed logical cell address which
     * can by used internally e.g. as keys for associative maps. Uses a faster
     * implementation than the method SheetUtils.getCellName().
     *
     * @param {Number[]} address
     *  The logical cell address as two-element array, containing the
     *  zero-based column and row indexes of the cell.
     *
     * @returns {String}
     *  A unique string key for the cell position (the zero-based integral
     *  column and row indexes, separated by a forward slash character).
     */
    SheetUtils.getCellKey = function (address) {
        return address[0] + '/' + address[1];
    };

    /**
     * Returns the logical cell address for the passed unique cell key.
     *
     * @param {String} key
     *  The unique cell key, as returned by the method SheetUtils.getCellKey().
     *
     * @returns {Number[]}
     *  The logical cell address parsed from the passed unique cell key.
     */
    SheetUtils.parseCellKey = function (key) {
        return _(key.split('/')).map(function (index) { return parseInt(index, 10); });
    };

    /**
     * Returns the string representation of the passed logical range address.
     *
     * @param {Object} range
     *  The logical cell range address as object with 'start' and 'end'
     *  properties, both containing logical cell addresses.
     *
     * @returns {String}
     *  The string representation of the cell range, in A1:A1 notation.
     */
    SheetUtils.getRangeName = function (range) {
        return SheetUtils.getCellName(range.start) + ':' + SheetUtils.getCellName(range.end);
    };

    /**
     * Returns the string representation of the passed logical range addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the range names.
     *
     * @returns {String}
     *  The string representation of the cell ranges, in A1:A1 notation.
     */
    SheetUtils.getRangesName = function (ranges, separator) {
        return _.chain(ranges).getArray().map(SheetUtils.getRangeName).value().join(separator);
    };

    /**
     * Returns the column index of the passed column string representation.
     *
     * @param {String} name
     *  The string representation of a column (case-insensitive).
     *
     * @returns {Number}
     *  The zero-based column index, if the passed column name is valid,
     *  otherwise -1.
     */
    SheetUtils.parseColName = function (name) {
        var col = -1;
        name = name.toUpperCase();
        if (RE_COL_NAME.test(name)) {
            for (var ichar = 0; ichar < name.length; ichar += 1) {
                col = (col + 1) * 26 + name.charCodeAt(ichar) - 65;
            }
        }
        return (col < MAX_COL_COUNT) ? col : -1;
    };

    /**
     * Returns the row index of the passed row string representation.
     *
     * @param {String} name
     *  The string representation of a row (one-based).
     *
     * @returns {Number}
     *  The zero-based row index, if the passed row name is valid, otherwise
     *  the value -1.
     */
    SheetUtils.parseRowName = function (name) {
        var row = RE_ROW_NAME.test(name) ? (parseInt(name, 10) - 1) : -1;
        return (row < MAX_ROW_COUNT) ? row : -1;
    };

    /**
     * Generates a localized sheet name with the specified index.
     *
     * @param {Number} index
     *  The number to be inserted into the sheet name.
     *
     * @returns {String}
     *  The generated sheet name (e.g. 'Sheet1', 'Sheet2', etc.).
     */
    SheetUtils.generateSheetName = function (index) {
        return (
            //#. Default sheet names in spreadsheet documents. Should be equal to the sheet
            //#. names used in existing spreadsheet applications (Excel, OpenOffice Calc, ...).
            //#. %1$d is the numeric index in the sheet name (e.g. "Sheet1", "Sheet2", etc.)
            //#. No white-space between the "Sheet" label and sheet index!
            //#, c-format
            gt('Sheet%1$d', _.noI18n(index)));
    };

    // exports ================================================================

    return SheetUtils;

});
