/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tokenarray',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, TriggerObject, SheetUtils) {

    'use strict';

    var // RE pattern for the leading character of a simple sheet name (without apostrophes).
        SHEET_LEADING_CHAR_PATTERN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

        // RE pattern for other characters of a simple sheet name (without apostrophes).
        SHEET_SIMPLE_CHAR_PATTERN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

        // RE pattern for any character of a complex sheet name (with apostrophes).
        SHEET_COMPLEX_CHAR_PATTERN = '[^\\x00-\\x1f\\x80-\\x9f\\[\\]*?:/\\\\]',

        // RE pattern for a simple sheet name.
        SHEET_SIMPLE_PATTERN = SHEET_LEADING_CHAR_PATTERN + SHEET_SIMPLE_CHAR_PATTERN + '*',

        // RE pattern for a complex sheet name.
        SHEET_COMPLEX_PATTERN = SHEET_COMPLEX_CHAR_PATTERN + '+(?:\'\'' + SHEET_COMPLEX_CHAR_PATTERN + '*)*' + SHEET_COMPLEX_CHAR_PATTERN,

        // RE pattern for a complete sheet name used in formulas, e.g. 'Sheet1' in the formula =Sheet1!A1.
        // - Group 1: The name of the sheet (simple sheet name).
        // - Group 2: The name of the sheet (complex sheet name).
        SHEET_REF_PATTERN = '(?:(' + SHEET_SIMPLE_PATTERN + ')|\'(' + SHEET_COMPLEX_PATTERN + ')\')',

        // RE pattern for a sheet range used in formulas, e.g. 'Sheet1:Sheet2' in the formula =SUM(Sheet1:Sheet2!A1).
        // - Group 1: The name of the first sheet (simple sheet names).
        // - Group 2: The name of the second sheet (simple sheet names).
        // - Group 3: The name of the first sheet (complex sheet names).
        // - Group 4: The name of the second sheet (complex sheet names).
        SHEET_RANGE_PATTERN = '(?:(' + SHEET_SIMPLE_PATTERN + '):(' + SHEET_SIMPLE_PATTERN + ')|\'(' + SHEET_COMPLEX_PATTERN + '):(' + SHEET_COMPLEX_PATTERN + ')\')',

        // RE pattern for a column reference (absolute or relative).
        // - Group 1: The absolute marker.
        // - Group 2: The column name.
        COL_REF_PATTERN = '(\\$)?([a-z]+)',

        // RE pattern for a row reference (absolute or relative).
        // - Group 1: The absolute marker.
        // - Group 2: The row name.
        ROW_REF_PATTERN = '(\\$)?([0-9]+)',

        // RE pattern for a 2D cell reference (no leading sheet name).
        // - Group 1: The absolute marker of the column.
        // - Group 2: The column name.
        // - Group 3: The absolute marker of the row.
        // - Group 4: The row name.
        CELL_REF_PATTERN = COL_REF_PATTERN + ROW_REF_PATTERN,

        // RE pattern for a 2D cell range (no leading sheet name).
        // - Group 1: The absolute marker for the first column.
        // - Group 2: The name of the first column.
        // - Group 3: The absolute marker for the first row.
        // - Group 4: The name of the first row.
        // - Group 5: The absolute marker for the second column.
        // - Group 6: The name of the second column.
        // - Group 7: The absolute marker for the second row.
        // - Group 8: The name of the second row.
        CELL_RANGE_PATTERN = CELL_REF_PATTERN + ':' + CELL_REF_PATTERN,

        // RE pattern for a 2D column interval (no leading sheet name).
        // - Group 1: The absolute marker for the first column.
        // - Group 2: The name of the first column.
        // - Group 3: The absolute marker for the second column.
        // - Group 4: The name of the second column.
        COL_RANGE_PATTERN = COL_REF_PATTERN + ':' + COL_REF_PATTERN,

        // RE pattern for a 2D row interval (no leading sheet name).
        // - Group 1: The absolute marker for the first row.
        // - Group 2: The name of the first row.
        // - Group 3: The absolute marker for the second row.
        // - Group 4: The name of the second row.
        ROW_RANGE_PATTERN = ROW_REF_PATTERN + ':' + ROW_REF_PATTERN,

        // RE pattern for the leading character of a name or identifier.
        NAME_LEADING_CHAR_PATTERN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

        // RE pattern for other characters of a name or identifier.
        NAME_INNER_CHAR_PATTERN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

        // RE pattern for a complete name or identifier used in formulas.
        // - Group 1: The entire name.
        NAME_PATTERN = '(' + NAME_LEADING_CHAR_PATTERN + NAME_INNER_CHAR_PATTERN + '*)',

        // RE look-ahead pattern to exclude an opening parenthesis after a name
        // all valid inner characters of names must be excluded too, otherwise
        // the previous groups would match less characters than they should
        NO_PARETHESIS_PATTERN = '(?!\\(|' + NAME_INNER_CHAR_PATTERN + ')';

    // class TokenArray =======================================================

    /**
     * Providing support for parsing localized formula strings and handling of
     * the resulting formula token arrays.
     *
     * Instances of this class trigger the following events:
     * - 'change:tokens'
     *      After the token array has been changed (manually, or by parsing a
     *      formula), or cleared.
     * - 'change:token'
     *      After a single token in the token array has been changed. The event
     *      handlers receive a reference to the changed token, and the array
     *      index of the token.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this formula parser.
     */
    function TokenArray(app) {

        var // self reference
            tokenArray = this,

            // the spreadsheet document model
            model = app.getModel(),

            // the decimal separator character
            DEC = app.getDecimalSeparator(),

            // the list separator character
            SEP = app.getListSeparator(),

            // the translated error code for reference errors
            REF_ERROR = app.getErrorCode('#REF!'),

            // a collection of regular expressions for various formula tokens
            RE = {

                // Number literals: With or without decimal separator, digits preceding
                // and/or following the separator, optional scientific prefix. No
                // leading minus sign, will be parsed as operator, otherwise a formula
                // such as =1-1 becomes an invalid sequence of two numbers.
                NUMBER: new RegExp('^(\\d*\\' + DEC + '\\d+|\\d+\\' + DEC + '?)(e[-+]\\d+)?', 'i'),

                // String literals: Enclosed in double quotes, embedded double quotes
                // are represented by a sequence of two double quote characters.
                STRING: /^("[^"]*")+/,

                // Boolean literals: The localized literals received from server, but
                // not followed by any character valid inside an identifier. This
                // prevents to match 'trueA' or 'false_1' or similar name tokens.
                BOOLEAN: new RegExp('^(' + app.getBooleanLiteral(true) + '|' + app.getBooleanLiteral(false) + ')(?!' + NAME_INNER_CHAR_PATTERN + ')', 'i'),

                // Error code literals: The localized literals received from server.
                ERROR: new RegExp('^(' + app.getErrorCodes().join('|').replace(/([?#])/g, '\\$1') + ')(?!' + NAME_INNER_CHAR_PATTERN + ')', 'i'),

                // Array literals: For now, simply match the entire literal, not any contents.
                ARRAY: /^{[^{]+}/,

                // All unary and binary operators except the list separator.
                OPERATOR: /^(<=|>=|<>|[-+*\/^%&:<>=])/,

                // The list/parameter separator character.
                SEPARATOR: new RegExp('^' + SEP),

                // The opening parenthesis.
                OPEN: /^\(/,

                // The closing parenthesis.
                CLOSE: /^\)/,

                // A cell reference, e.g. A1 or Sheet1!$A$1.
                // - Group 1: The simple sheet name.
                // - Group 2: The complex sheet name.
                // - Group 3: The absolute marker for the column.
                // - Group 4: The column name.
                // - Group 5: The absolute marker for the row.
                // - Group 6: The row name.
                CELL_REF: new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + CELL_REF_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A range reference, e.g. A1:B2 or Sheet1!$A$1:$A$1.
                // - Group 1: The simple sheet name.
                // - Group 2: The complex sheet name.
                // - Group 3: The absolute marker for the first column.
                // - Group 4: The name of the first column.
                // - Group 5: The absolute marker for the first row.
                // - Group 6: The name of the first row.
                // - Group 7: The absolute marker for the second column.
                // - Group 8: The name of the second column.
                // - Group 9: The absolute marker for the second row.
                // - Group 10: The name of the second row.
                RANGE_REF: new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + CELL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A column interval, e.g. A:B or Sheet1!$C:$C (always with colon).
                // - Group 1: The simple sheet name.
                // - Group 2: The complex sheet name.
                // - Group 3: The absolute marker for the first column.
                // - Group 4: The name of the first column.
                // - Group 5: The absolute marker for the second column.
                // - Group 6: The name of the second column.
                COL_INTERVAL: new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + COL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A row interval, e.g. 1:2 or Sheet1!$3:$3 (always with colon).
                // - Group 1: The simple sheet name.
                // - Group 2: The complex sheet name.
                // - Group 3: The absolute marker for the first row.
                // - Group 4: The name of the first row.
                // - Group 5: The absolute marker for the second row.
                // - Group 6: The name of the second row.
                ROW_INTERVAL: new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + ROW_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A 3D cell reference, e.g. Sheet1:Sheet2!$A$1.
                // - Group 1: The simple name of the first sheet.
                // - Group 2: The simple name of the second sheet.
                // - Group 3: The complex name of the first sheet.
                // - Group 4: The complex name of the second sheet.
                // - Group 5: The absolute marker for the column.
                // - Group 6: The column name.
                // - Group 7: The absolute marker for the row.
                // - Group 8: The row name.
                CELL_3D_REF: new RegExp('^' + SHEET_RANGE_PATTERN + '!' + CELL_REF_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A 3D range reference, e.g. Sheet1:Sheet2!$A$1:$A$1.
                // - Group 1: The simple name of the first sheet.
                // - Group 2: The simple name of the second sheet.
                // - Group 3: The complex name of the first sheet.
                // - Group 4: The complex name of the second sheet.
                // - Group 5: The absolute marker for the first column.
                // - Group 6: The name of the first column.
                // - Group 7: The absolute marker for the first row.
                // - Group 8: The name of the first row.
                // - Group 9: The absolute marker for the second column.
                // - Group 10: The name of the second column.
                // - Group 11: The absolute marker for the second row.
                // - Group 12: The name of the second row.
                RANGE_3D_REF: new RegExp('^' + SHEET_RANGE_PATTERN + '!' + CELL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A 3D column interval, Sheet1:Sheet2!$C:$C (always with colon).
                // - Group 1: The simple name of the first sheet.
                // - Group 2: The simple name of the second sheet.
                // - Group 3: The complex name of the first sheet.
                // - Group 4: The complex name of the second sheet.
                // - Group 5: The absolute marker for the first column.
                // - Group 6: The name of the first column.
                // - Group 7: The absolute marker for the second column.
                // - Group 8: The name of the second column.
                COL_3D_INTERVAL: new RegExp('^' + SHEET_RANGE_PATTERN + '!' + COL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // A 3D row interval, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
                // - Group 1: The simple name of the first sheet.
                // - Group 2: The simple name of the second sheet.
                // - Group 3: The complex name of the first sheet.
                // - Group 4: The complex name of the second sheet.
                // - Group 5: The absolute marker for the first row.
                // - Group 6: The name of the first row.
                // - Group 7: The absolute marker for the second row.
                // - Group 8: The name of the second row.
                ROW_3D_INTERVAL: new RegExp('^' + SHEET_RANGE_PATTERN + '!' + ROW_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // Global names with sheet name (defined names, macros, etc.), e.g. Sheet1!my_range.
                // - Group 1: The simple sheet name.
                // - Group 2: The complex sheet name.
                // - Group 3: The name or identifier following the sheet name.
                GLOBAL_NAME: new RegExp('^' + SHEET_REF_PATTERN + '!' + NAME_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // Simple names without sheet name (defined names, function names, macros, etc.), e.g. my_range.
                NAME: new RegExp('^' + NAME_PATTERN + NO_PARETHESIS_PATTERN, 'i'),

                // Any function name (defined names followed by an opening parenthesis)
                FUNCTION: new RegExp('^(' + NAME_PATTERN + ')(?=\\()', 'i'),

                // Any whitespace characters
                WHITESPACE: /^[\s\x00-\x1f\x80-\x9f]+/,

                // Matches a simple sheet name that does not need to be enclosed in apostrophes.
                SIMPLE_SHEET_NAME: new RegExp('^' + SHEET_SIMPLE_PATTERN + '$', 'i')
            },

            // the array of all formula tokens
            tokens = [],

            // current reference sheet
            currRefSheet = -1,

            // current reference address
            currRefAddress = [0, 0];

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // class Token --------------------------------------------------------

        /**
         * Base class for all formula tokens.
         *
         * @constructor
         *
         * @param {String} text
         *  The text representation of the token. Used for example while the
         *  formula is edited and the same token value can be represented by
         *  different text strings, e.g. '1' vs. '1.000'.
         */
        var Token = _.makeExtendable(function (type, text) {

            /**
             * Returns the type identifier of this token.
             */
            this.getType = function () { return type; };

            /**
             * Returns the current text representation of this token.
             */
            this.getText = function () { return text; };

            /**
             * Changes the current text representation of this token, and
             * triggers a 'change'token' event at the token array instance.
             */
            this.setText = function (newText, options) {
                text = newText;
                if (!Utils.getBooleanOption(options, 'silent', false)) {
                    tokenArray.trigger('change:token', this, _(tokens).indexOf(this));
                }
                return this;
            };

            /**
             * Sub classes can overwrite this method to refresh internal data,
             * for example after the sheet collection of the document has been
             * changed.
             *
             * @internal
             */
            this.refresh = function () {};

            /**
             * Sub classes can overwrite this method to relocate the token to a
             * new sheet and cell position.
             *
             * @internal
             */
            this.relocate = function (/*sheets, cols, rows*/) {};

        }); // class Token

        // class ValueToken ---------------------------------------------------

        /**
         * This formula token represents a constant value (a number, a string,
         * a Boolean value, or an error code) in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var ValueToken = Token.extend({ constructor: function (text, value) {

            // base constructor
            Token.call(this, 'value', text);

            /**
             * Returns the current value of this token.
             */
            this.getValue = function () { return value; };

            /**
             * Changes the current value of this token.
             */
            this.setValue = function (newValue, options) {
                value = newValue;
                if (_.isNumber(value)) {
                    this.setText(String(value).replace('.', DEC), options);
                } else if (_.isString(value)) {
                    this.setText('"' + value.replace(/"/g, '""') + '"', options);
                } else if (_.isBoolean(value)) {
                    this.setText(app.getBooleanLiteral(value), options);
                } else {
                    Utils.error('ValueToken.setValue(): unsupported value type');
                    this.setText(app.getErrorCode('#VALUE!'), options);
                }
                return this;
            };

        }}); // class ValueToken

        // class FunctionToken ------------------------------------------------

        /**
         * This formula token represents a function name in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var FunctionToken = Token.extend({ constructor: function (text) {

            // base constructor
            Token.call(this, 'func', text);

            /**
             * Returns the current function name with upper-case characters.
             */
            this.getName = function () { return text.toUpperCase(); };

        }}); // class FunctionToken

        // class RefToken -----------------------------------------------------

        /**
         * This formula token represents a cell reference in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var RefToken = Token.extend({ constructor: function (text, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

            var self = this;

            // base constructor
            Token.call(this, 'ref', text);
            if (!_.isObject(cell2Ref)) { cell2Ref = _.copy(cell1Ref, true); }

            /**
             * Updates the string representation of this reference token,
             * according to the current reference settings.
             */
            function updateTokenText(options) {
                self.setText(generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref), options);
            }

            /**
             * Returns the sheet range referred by this token, in the
             * properties 'sheet1' and 'sheet2'.
             */
            this.getSheetRange = function () {
                var sheet1 = _.isObject(sheet1Ref) ? sheet1Ref.sheet : currRefSheet,
                    sheet2 = _.isObject(sheet2Ref) ? sheet2Ref.sheet : sheet1;
                return { sheet1: Math.min(sheet1, sheet2), sheet2: Math.max(sheet1, sheet2) };
            };

            /**
             * Returns whether this reference token refers to the specified
             * sheet, or refers to a sheet range that contains the specified
             * sheet.
             */
            this.containsSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                return (sheetRange.sheet1 <= sheet) && (sheet <= sheetRange.sheet2);
            };

            /**
             * Returns the logical address of the cell range represented by
             * this token.
             */
            this.getRange = function () {
                return SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: cell2Ref.address });
            };

            /**
             * Changes the current cell range of this token, keeps all absolute
             * flags and sheet references intact.
             */
            this.setRange = function (range, options) {
                cell1Ref.address = _.clone(range.start);
                cell2Ref.address = _.clone(range.end);
                updateTokenText(options);
                return this;
            };

            /**
             * Refreshes the sheet references, after the sheet collection of
             * the document has been changed.
             */
            this.refresh = function () {
                refreshSheetRef(sheet1Ref);
                refreshSheetRef(sheet2Ref);
            };

            /**
             * Relocates this reference token to a new sheet and cell position.
             */
            this.relocate = function (sheets, cols, rows) {

                // relocates the passed cell reference
                function relocateCellRef(cellRef) {
                    if (!cellRef.absCol) { cellRef.address[0] += cols; }
                    if (!cellRef.absRow) { cellRef.address[1] += rows; }
                }

                // TODO: relocate relative sheet references (ODF)

                // relocate both cell references
                relocateCellRef(cell1Ref);
                relocateCellRef(cell2Ref);

                // recalculate and set the text representation
                updateTokenText({ silent: true });
            };

        }}); // class RefToken

        // class NameToken ----------------------------------------------------

        /**
         * This formula token represents a defined name in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var NameToken = Token.extend({ constructor: function (text, name, sheetRef) {

            // base constructor
            Token.call(this, 'name', text);

            /**
             * Returns the current name.
             */
            this.getName = function () { return name; };

            /**
             * Returns the sheet reference descriptor.
             */
            this.getSheetRef = function () { return _.clone(sheetRef); };

            /**
             * Returns the token array with the definition of this name.
             */
            this.getTokenArray = function () {
                // TODO: handle sheet-local names
                return sheetRef ? null : model.getNameCollection().getTokenArray(name);
            };

            /**
             * Refreshes the sheet reference, after the sheet collection of the
             * document has been changed.
             */
            this.refresh = function () {
                refreshSheetRef(sheetRef);
            };

        }}); // class NameToken

        // private methods ----------------------------------------------------

        /**
         * Refreshes the passed sheet reference object. If it contains a
         * literal sheet name, tries to look up the sheet in the document and
         * inserts the index of an existing sheet into the sheet reference.
         */
        function refreshSheetRef(sheetRef) {
            var sheetIndex = 0;
            if (sheetRef && _.isString(sheetRef.sheet)) {
                sheetIndex = model.getSheetIndex(sheetRef.sheet);
                if (sheetIndex >= 0) {
                    sheetRef.sheet = sheetIndex;
                }
            }
        }

        /**
         * Returns the sheet prefix used by cell references and defined names,
         * including the trailing exclamation mark.
         */
        function generateSheetName(sheet1Ref, sheet2Ref) {

            var sheet1Name = null,
                sheet2Name = null,
                simpleNames = false;

            // convert first sheet name
            if (!_.isObject(sheet1Ref)) { return ''; }
            sheet1Name = _.isNumber(sheet1Ref.sheet) ? model.getSheetName(sheet1Ref.sheet) : sheet1Ref.sheet;
            if (!_.isString(sheet1Name)) { return REF_ERROR + '!'; }
            simpleNames = RE.SIMPLE_SHEET_NAME.test(sheet1Name);

            // convert second sheet name
            if (_.isObject(sheet2Ref) && !_.isEqual(sheet1Ref, sheet2Ref)) {
                sheet2Name = _.isNumber(sheet2Ref.sheet) ? model.getSheetName(sheet2Ref.sheet) : sheet2Ref.sheet;
                if (!_.isString(sheet2Name)) { return REF_ERROR + '!'; }
                simpleNames = simpleNames && RE.SIMPLE_SHEET_NAME.test(sheet2Name);
                sheet1Name += ':' + sheet2Name;
            }

            // enclose complex names in apostrophes
            return simpleNames ? (sheet1Name + '!') : ('\'' + sheet1Name.replace(/'/g, '\'\'') + '\'!');
        }

        /**
         * Returns the string representation of the specified reference.
         */
        function generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

            var range = SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: cell2Ref.address }),
                text = generateSheetName(sheet1Ref, sheet2Ref);

            // returns the complete name of the column described by the passed cell reference
            function generateColName(cellRef) {
                return (cellRef.absCol ? '$' : '') + SheetUtils.getColName(cellRef.address[0]);
            }

            // returns the complete name of the row described by the passed cell reference
            function generateRowName(cellRef) {
                return (cellRef.absRow ? '$' : '') + SheetUtils.getRowName(cellRef.address[1]);
            }

            // returns the complete name of the cell described by the passed cell reference
            function generateCellName(cellRef) {
                return generateColName(cellRef) + generateRowName(cellRef);
            }

            if (!model.isValidRange(range)) {
                text += REF_ERROR;
            }

            // generate row interval (preferred over column interval, when entire sheet is selected)
            else if (cell1Ref.absCol && cell2Ref.absCol && model.isRowRange(range)) {
                text += generateRowName(cell1Ref) + ':' + generateRowName(cell2Ref);
            }

            // generate column interval
            else if (cell1Ref.absRow && cell2Ref.absRow && model.isColRange(range)) {
                text += generateColName(cell1Ref) + ':' + generateColName(cell2Ref);
            }

            // generate range or cell address
            else {
                text += generateCellName(cell1Ref);
                if (!_.isEqual(cell1Ref, cell2Ref)) {
                    text += ':' + generateCellName(cell2Ref);
                }
            }

            return text;
        }

        // methods ------------------------------------------------------------

        /**
         * Removes all tokens from this token array.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.clear = function () {
            if (tokens.length > 0) {
                tokens = [];
                this.trigger('change:tokens');
            }
            return this;
        };

        /**
         * Parses the specified formula string.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @param {Number} refSheet
         *  The zero-based index of the sheet that will contain the formula.
         *  Cell references without sheet name will be associated to this
         *  sheet, and defined names without sheet name will be searched in
         *  that sheet first. Can be set to -1 for global formulas not
         *  referring to any specific sheet.
         *
         * @param {Number} refAddress
         *  The logical address of the reference cell used to process relative
         *  cell references in the formula.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.parseFormula = function (formula, refSheet, refAddress) {

            var // the matches of the regular expressions
                matches = null,
                // sheet references, and start and end cell references of cell ranges
                sheet1Ref = null, sheet2Ref = null, cell1Ref = null, cell2Ref = null;

            // returns the sheet index from the passed simple or complex sheet name
            function getSheetRef(simpleSheet, complexSheet) {
                var sheetRef = null,
                    sheetIndex = null;
                if (_.isString(simpleSheet)) {
                    sheetIndex = model.getSheetIndex(simpleSheet);
                    sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : simpleSheet, abs: true };
                } else if (_.isString(complexSheet)) {
                    complexSheet = complexSheet.replace(/''/g, '\'');
                    sheetIndex = model.getSheetIndex(complexSheet);
                    sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : complexSheet, abs: true };
                }
                return _.isObject(sheetRef) ? sheetRef : null;
            }

            // returns a cell reference descriptor
            function getCellRef(absCol, col, absRow, row) {
                var address = [_.isNumber(col) ? col : SheetUtils.parseColName(col), _.isNumber(row) ? row : SheetUtils.parseRowName(row)];
                return model.isValidAddress(address) ? {
                    address: address,
                    absCol: _.isBoolean(absCol) ? absCol : (absCol === '$'),
                    absRow: _.isBoolean(absRow) ? absRow : (absRow === '$')
                } : null;
            }

            // store the new reference sheet and reference address
            currRefSheet = refSheet;
            currRefAddress = refAddress;

            // extract all supported tokens from start of formula string
            tokens = [];
            while (formula.length > 0) {

                // try/finally to be able to continue the while-loop at any place
                try {

                    // unary and binary operators
                    if ((matches = RE.OPERATOR.exec(formula))) {
                        tokens.push(new Token('op', matches[0]));
                        continue;
                    }

                    // list/parameter separator
                    if ((matches = RE.SEPARATOR.exec(formula))) {
                        tokens.push(new Token('sep', matches[0]));
                        continue;
                    }

                    // opening parenthesis
                    if ((matches = RE.OPEN.exec(formula))) {
                        tokens.push(new Token('open', matches[0]));
                        continue;
                    }

                    // closing parenthesis
                    if ((matches = RE.CLOSE.exec(formula))) {
                        tokens.push(new Token('close', matches[0]));
                        continue;
                    }

                    // 3D column intervals (before cell references)
                    if ((matches = RE.COL_3D_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], true, 0);
                        cell2Ref = getCellRef(matches[7], matches[8], true, model.getMaxRow());
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of name and reference, such as name42:Sheet2!$A:$A,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D row intervals
                    if ((matches = RE.ROW_3D_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(true, 0, matches[5], matches[6]);
                        cell2Ref = getCellRef(true, model.getMaxCol(), matches[7], matches[8]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of name and reference, such as name42:Sheet2!$1:$1,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D range references
                    if ((matches = RE.RANGE_3D_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], matches[7], matches[8]);
                        cell2Ref = getCellRef(matches[9], matches[10], matches[11], matches[12]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of cell reference and name, such as Sheet1:Sheet2!$A$1:name42,
                        // or of name and reference, such as name42:Sheet2!$A1:$A1,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D cell references
                    else if ((matches = RE.CELL_3D_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], matches[7], matches[8]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, null, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be the
                        // combination of global name and local name, such as name41:Sheet2!name42,
                        // which has been matched by the current RE for cell references.
                    }

                    // column intervals (before cell references)
                    if ((matches = RE.COL_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], true, 0);
                        cell2Ref = getCellRef(matches[5], matches[6], true, model.getMaxRow());
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                        } else {
                            tokens.push(new Token('bad', matches[0]));
                        }
                        continue;
                    }

                    // row intervals
                    if ((matches = RE.ROW_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(true, 0, matches[3], matches[4]);
                        cell2Ref = getCellRef(true, model.getMaxCol(), matches[5], matches[6]);
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                        } else {
                            tokens.push(new Token('bad', matches[0]));
                        }
                        continue;
                    }

                    // range references
                    if ((matches = RE.RANGE_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], matches[5], matches[6]);
                        cell2Ref = getCellRef(matches[7], matches[8], matches[9], matches[10]);
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of cell reference and name, such as Sheet1!$A$1:name42,
                        // which has been matched by the current RE for range references.
                    }

                    // cell references
                    else if ((matches = RE.CELL_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], matches[5], matches[6]);
                        if (_.isObject(cell1Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, null, sheet1Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be
                        // a local name, such as Sheet1!name42, which has been
                        // matched by the current RE for cell references.
                    }

                    // number literals
                    if ((matches = RE.NUMBER.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], parseFloat(matches[0].replace(DEC, '.'))));
                        continue;
                    }

                    // string literals
                    if ((matches = RE.STRING.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"')));
                        continue;
                    }

                    // Boolean literals
                    if ((matches = RE.BOOLEAN.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], matches[0].toUpperCase() === app.getBooleanLiteral(true)));
                        continue;
                    }

                    // error code literals
                    if ((matches = RE.ERROR.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], matches[0]));
                        continue;
                    }

                    // array literals
                    if ((matches = RE.ARRAY.exec(formula))) {
                        tokens.push(new Token('array', matches[0]));
                        continue;
                    }

                    // global names and other identifiers
                    if ((matches = RE.GLOBAL_NAME.exec(formula))) {
                        tokens.push(new NameToken(matches[0], matches[3], getSheetRef(matches[1], matches[2])));
                        continue;
                    }

                    // function names
                    if ((matches = RE.FUNCTION.exec(formula))) {
                        tokens.push(new FunctionToken(matches[0]));
                        continue;
                    }

                    // local names and other identifiers
                    if ((matches = RE.NAME.exec(formula))) {
                        tokens.push(new NameToken(matches[0], matches[0]));
                        continue;
                    }

                    // whitespace
                    if ((matches = RE.WHITESPACE.exec(formula))) {
                        tokens.push(new Token('ws', matches[0]));
                        continue;
                    }

                    // error, or unsupported tokens: push remaining formula text as 'bad' token
                    matches = [formula];
                    tokens.push(new Token('bad', matches[0]));
                }

                // remove the current match from the formula string
                finally {
                    formula = formula.slice(matches[0].length);
                }
            }

            // post-process the token array
            Utils.iterateArray(tokens, function (token, index) {

                var // a specific token following the current token
                    nextToken = 0,
                    // a specific token preceding the current token
                    prevToken = 0,
                    // text and character index of a space character
                    spaceText = '', spaceIndex = 0;

                // replace combination of <MINUS> <NUMBER> with a negative number, if
                // the minus is the leading token or preceded by an operator or opening parenthesis
                if ((token.getType() === 'op') && (token.getText() === '-') &&
                    (nextToken = tokenArray.getToken(index + 1, 'value')) && _.isNumber(nextToken.getValue()) && (nextToken.getValue() > 0) &&
                    (!(prevToken = tokenArray.getPrevToken(index)) || (/^(op|open)$/.test(prevToken.getType())))
                ) {
                    tokens[index + 1] = new ValueToken('-' + nextToken.getText(), -nextToken.getValue());
                    tokens.splice(index, 1);
                }

                // insert a range intersection operator for a white-space token
                // containing a SPACE character and surrounded by reference/name
                // tokens or other subexpressions in parentheses
                else if ((token.getType() === 'ws') && ((spaceIndex = (spaceText = token.getText()).indexOf(' ')) >= 0) &&
                    (prevToken = tokenArray.getToken(index - 1)) && (/^(ref|name|close)$/.test(prevToken.getType())) &&
                    (nextToken = tokenArray.getToken(index + 1)) && (/^(ref|name|open|func)$/.test(nextToken.getType()))
                ) {
                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < spaceText.length) {
                        tokens.splice(index + 1, 0, new Token('ws', spaceText.slice(spaceIndex + 1)));
                    }
                    // insert an intersection operator for the SPACE character
                    tokens.splice(index + 1, 0, new Token('op', ' '));
                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        tokens[index] = new Token('ws', spaceText.slice(0, spaceIndex));
                    } else {
                        tokens.splice(index, 1);
                    }
                }

            }, { reverse: true });

            // notify listeners
            this.trigger('change:tokens');
            return this;
        };

        /**
         * Relocates this formula to a new reference sheet and address. The
         * string representation of all reference tokens will be adjusted
         * accordingly.
         *
         * @param {Number} refSheet
         *  The zero-based index of the sheet that will contain the formula.
         *  Cell references without sheet name will be associated to this
         *  sheet, and defined names without sheet name will be searched in
         *  that sheet first. Can be set to -1 for global formulas not
         *  referring to any specific sheet.
         *
         * @param {Number} refAddress
         *  The logical address of the reference cell used to process relative
         *  cell references in the formula.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.relocateFormula = function (refSheet, refAddress) {

            var // number of sheets to move the references and names
                sheets = refSheet - currRefSheet,
                // number of columns to move the references
                cols = refAddress[0] - currRefAddress[0],
                // number of rows to move the references
                rows = refAddress[1] - currRefAddress[1];

            if ((sheets !== 0) || (cols !== 0) || (rows !== 0)) {

                // store the new reference sheet and reference address
                currRefSheet = refSheet;
                currRefAddress = _.clone(refAddress);

                // relocate all tokens, notify listeners
                _(tokens).invoke('relocate', sheets, cols, rows);
                this.trigger('change:tokens');
            }

            return this;
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array.
         *
         * @returns {String}
         *  The string representation of the formula.
         */
        this.getFormula = function () {
            return _(tokens).reduce(function (formula, token) { return formula += token.getText(); }, '');
        };

        /**
         * Returns the token at the specified array index.
         *
         * @param {Number} index
         *  The array index of the token to be returned.
         *
         * @param {String} [options.type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The token instance at the specified index; or null if the index is
         *  invalid.
         */
        this.getToken = function (index, type) {
            var token = tokens[index];
            return (token && (!_.isString(type) || (token.getType() === type))) ? token : null;
        };

        /**
         * Returns the token following the token at the specified array index.
         * If that token is a white-space token, it will be skipped, and the
         * token following that white-space token will be returned.
         *
         * @param {Number} index
         *  The array index of the token whose successor will be returned.
         *
         * @param {String} [type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The formula token next to the token at the specified index; or null
         *  if no matching token could be found.
         */
        this.getNextToken = function (index, type) {
            // adjust index, skip a white-space token
            index += this.getToken(index + 1, 'ws') ? 2 : 1;
            // return the token if it matches the specified type
            return this.getToken(index, type);
        };

        /**
         * Returns the token preceding the token at the specified array index.
         * If that token is a white-space token, it will be skipped, and the
         * token preceding that white-space token will be returned.
         *
         * @param {Number} index
         *  The array index of the token whose predecessor will be returned.
         *
         * @param {String} [type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The formula token preceding the token at the specified index; or
         *  null if no matching token could be found.
         */
        this.getPrevToken = function (index, type) {
            // adjust index, skip a white-space token
            index -= this.getToken(index - 1, 'ws') ? 2 : 1;
            // return the token if it matches the specified type
            return this.getToken(index, type);
        };

        /**
         * Invokes the passed iterator function for all tokens contained in
         * this token array.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all tokens. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The current token.
         *  (2) {Number} index
         *      The zero-based token index.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the tokens will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTokens = function (iterator, options) {
            return Utils.iterateArray(tokens, function (token, index) {
                return iterator.call(this, token, index);
            }, options);
        };

        /**
         * Returns the text cursor position of the specified token.
         *
         * @param {Number} index
         *  The array index of the formula token.
         *
         * @returns {Object|Null}
         *  The text cursor position of the token, in the integer properties
         *  'start' and 'end'; or null, if the passed index is invalid.
         */
        this.getTokenPosition = function (index) {

            var // the text cursor position of the token
                position = { start: 0, end: 0 };

            // validate index
            if ((index < 0) || (index >= tokens.length)) { return null; }

            // iterate to the specified token and calculate text position
            for (var currIndex = 0; currIndex <= index; currIndex += 1) {
                position.start = position.end;
                position.end = position.start + tokens[currIndex].getText().length;
            }

            return position;
        };

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.preferOperand=true]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {Object|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position, in the following properties:
         *  - {Token} token
         *      The token instance located at the text cursor position.
         *  - {Number} index
         *      The array index of the token.
         *  - {Number} start
         *      The text position of the first character of the token in the
         *      string representation of this token array.
         *  - {Number} end
         *      The text position behind (!) the last character of the token in
         *      the string representation of this token array.
         *  If the text cursor position is located outside of the formula text,
         *  or at its beginning (without the option 'preferOperand'), this
         *  method will return the value null.
         */
        this.getTokenAtPosition = function (pos, options) {

            var // whether to prefer operand tokens
                preferOperand = Utils.getBooleanOption(options, 'preferOperand', false),
                // the length of the formula text already consumed (including the resulting token)
                length = 0,
                // the array index of the token containing or preceding the text cursor position
                index = 0;

            // returns whether the passed token is an operand token
            function isOperand(token) {
                return (/^(value|array|ref|name|func)$/).test(token.getType());
            }

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && (tokens.length > 0) && isOperand(tokens[0])) {
                return { token: tokens[0], index: 0, start: 0, end: tokens[0].getText().length };
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            index = Utils.findFirstIndex(tokens, function (token) {
                length += token.getText().length;
                return pos <= length;
            });
            if (index < 0) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === length) && (index + 1 < tokens.length) && !isOperand(tokens[index]) && isOperand(tokens[index + 1])) {
                return { token: tokens[index + 1], index: index + 1, start: length, end: length  + tokens[index + 1].getText().length };
            }

            // build and return the complete descriptor for the token
            return { token: tokens[index], index: index, start: length - tokens[index].getText().length, end: length };
        };

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} name
         *      The translated function name.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        this.getFunctionAtPosition = function (pos) {

            var // information about the token at the passed position
                tokenInfo = this.getTokenAtPosition(pos, { preferOperand: true }),
                // current index, current token
                index = 0, token = null,
                // the parameter index and count
                param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        return seps;
                    case 'close':
                        index -= 1;
                        seekToOpeningParenthesis();
                        break;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < tokens.length) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        index += 1;
                        seekToClosingParenthesis();
                        break;
                    case 'close':
                        return seps;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenInfo) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenInfo.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                if ((token = this.getToken(index - 1, 'func'))) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { name: token.getName(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if ((tokenInfo.token.getType() === 'func') && (tokens[tokenInfo.index + 1].getType() === 'open')) {
                index = tokenInfo.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { name: tokenInfo.token.getName(), paramIndex: -1, paramCount: count };
            }

            return null;
        };

        /**
         * Returns the addresses of all cell ranges this token array refers to.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Number} [options.targetSheet]
         *      If specified, the result will contain only ranges contained in
         *      the sheet with this index.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, will return the range addresses referred by all
         *      defined names contained in this token array.
         *
         * @returns {Array}
         *  The logical addresses of all cell ranges contained in this token
         *  array with additional information, in an array of objects with the
         *  following properties:
         *  - {Object} sheet1
         *      The zero-based index of the first sheet in the sheet range
         *      contained by the token.
         *  - {Object} sheet2
         *      The zero-based index of the last sheet in the sheet range
         *      contained by the token.
         *  - {Object} range
         *      The range address represented by a formula token.
         *  - {Number} index
         *      The internal array index of the token containing the range
         *      (may be equal for different objects in the array, for example
         *      for a single defined name referring to multiple cell ranges).
         *  - {String} type
         *      The type of the token containing the range. Will be either
         *      'ref' for a regular reference token, or 'name' for a defined
         *      name that evaluates to one or more ranges in the specified
         *      sheet (only if option 'resolveNames' is set, see above).
         */
        this.extractRanges = function (options) {

            var // the index of the sheet to filter the result
                targetSheet = Utils.getIntegerOption(options, 'targetSheet'),
                // whether to resolve defined name tokens
                resolveNames = Utils.getBooleanOption(options, 'resolveNames', false),
                // the resulting ranges
                result = [];

            function extractReferenceRange(refToken, tokenIndex) {
                if (!_.isNumber(targetSheet) || refToken.containsSheet(targetSheet)) {
                    result.push(_({ range: refToken.getRange(), index: tokenIndex, type: 'ref' }).extend(refToken.getSheetRange()));
                }
            }

            function extractNamedRanges(nameToken, tokenIndex) {
                var tokenArray = nameToken.getTokenArray(),
                    rangeInfos = null;
                if (tokenArray) {
                    rangeInfos = tokenArray.extractRanges(options);
                    _(rangeInfos).each(function (rangeInfo) { _(rangeInfo).extend({ index: tokenIndex, type: 'name' }); });
                    result = result.concat(rangeInfos);
                }
            }

            _(tokens).each(function (token, index) {
                // process each token by its type
                switch (token.getType()) {
                case 'ref':
                    extractReferenceRange(token, index);
                    break;
                case 'name':
                    if (resolveNames) { extractNamedRanges(token, index); }
                    break;
                }
            });

            return result;
        };

        /**
         * Returns the string representation of a cell or range reference to be
         * used in this token array.
         *
         * @param {Object} range
         *  The logical address of the cell range. If the range refers to a
         *  single cell, this method will return a single cell address instead
         *  of a complete range address.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet the returned range reference will
         *  point to. If this value is not equal to the reference sheet of this
         *  token array, the name of the sheet will be inserted into the string
         *  representation returned by this method.
         *
         * @returns {String}
         *  The string representation of the cell or range reference.
         */
        this.buildReference = function (range, sheet) {

            var // whether the range spans entire columns or rows
                colRange = model.isColRange(range),
                rowRange = model.isRowRange(range),
                // the cell references
                cell1Ref = { address: range.start, absCol: rowRange, absRow: colRange && !rowRange },
                cell2Ref = { address: range.end, absCol: rowRange, absRow: colRange && !rowRange },
                // the sheet reference
                sheetRef = (sheet !== currRefSheet) ? { sheet: sheet, abs: true } : null;

            return generateReference(cell1Ref, cell2Ref, sheetRef);
        };

        // initialization -----------------------------------------------------

        // refresh all tokens after the document has been imported and all sheets exist
        if (!app.isImportFinished()) {
            this.listenTo(app, 'docs:import:success', function () {
                _(tokens).invoke('refresh');
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = RE = tokens = null;
        });

    } // class TokenArray

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: TokenArray });

});
