/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/namecollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/model/tokenarray'
    ], function (Utils, ModelObject, TokenArray) {

    'use strict';

    // class NameCollection ===================================================

    /**
     * Collects all global defined names of the spreadsheet document, or the
     * defined names contained in a single sheet of the document.
     *
     * Triggers the following events:
     * - 'insert:name'
     *      After a new defined name has been inserted into the collection.
     *      Event handlers receive the name of the new defined name.
     * - 'delete:name'
     *      After a defined name has been deleted from the collection. Event
     *      handlers receive the name of the deleted defined name.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     */
    function NameCollection(app) {

        var // the definitions of all defined names, mapped by upper-case name
            entries = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private class EntryModel -------------------------------------------

        var EntryModel = ModelObject.extend({ constructor: function (formula) {

            var // the token array representing the definition of this name
                tokenArray = new TokenArray(app);

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            ModelObject.call(this, app);

            // methods  - - - - - - - - - - - - - - - - - - - - - - - - - - - -

            /**
             * Returns the token array representing the formula definition of
             * this defined name.
             *
             * @returns {TokenArray}
             *  The token array containing the name definition.
             */
            this.getTokenArray = function () {
                return tokenArray;
            };

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // TODO: use sheet index for sheet-local names
            tokenArray.parseFormula(formula, -1, [0, 0]);

            // destroy all class members on destruction
            this.registerDestructor(function () {
                tokenArray.destroy();
                tokenArray = null;
            });

        }}); // class EntryModel

        // methods ------------------------------------------------------------

        /**
         * Creates and returns a clone of this collection instance.
         *
         * @returns {NameCollection}
         *  A complete clone of this collection.
         */
        this.clone = function () {
            return new NameCollection(app, entries);
        };

        /**
         * Returns the token array representing the formula definition of the
         * specified defined name.
         *
         * @param {String} name
         *  The name of the new defined name whose token array will be
         *  returned.
         *
         * @returns {TokenArray|Null}
         *  The token array containing the name definition of the specified
         *  name, if existing; otherwise null.
         */
        this.getTokenArray = function (name) {
            var entry = entries[name.toUpperCase()];
            return entry ? entry.getTokenArray() : null;
        };

        /**
         * Creates and stores a new defined name, and triggers an 'insert:name'
         * event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @param {String} formula
         *  The formula expression bound to the defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been created successfully.
         */
        this.insertName = function (name, formula) {

            // validate input parameters (name must not exist)
            name = name.toUpperCase();
            if ((name.length === 0) || (formula.length === 0) || (name in entries)) {
                return false;
            }

            // create a new collection entry and notify listeners
            entries[name] = new EntryModel(formula);
            this.trigger('insert:name', name);
            return true;
        };

        /**
         * Deletes a defined name from this collection, and triggers a
         * 'delete:name' event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been deleted successfully.
         */
        this.deleteName = function (name) {

            // validate input parameters (name must exist)
            name = name.toUpperCase();
            if (!(name in entries)) { return false; }

            // delete the collection entry and notify listeners
            entries[name].destroy();
            delete entries[name];
            this.trigger('delete:name', name);
            return true;
        };

        // initialization -----------------------------------------------------

        // clone collection entries passed as hidden argument to the c'tor (used by the clone() method)
        if (_.isObject(arguments[NameCollection.length])) {
            _(arguments[NameCollection.length]).each(function (entry, name) {
                entries[name] = new EntryModel(entry.getTokenArray().getFormula());
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(entries).invoke('destroy');
            entries = null;
        });

    } // class NameCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: NameCollection });

});
