/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/drawinglayer/model/drawingcollection',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
     'io.ox/office/spreadsheet/model/drawing/chartmodel'
    ], function (Utils, DrawingCollection, SheetUtils, DrawingModelMixin, SheetChartModel) {

    'use strict';

    var // horizontal drawing anchor attribute names
        HORIZONTAL_ATTRIBUTE_NAMES = {
            offset: 'left',
            size: 'width',
            startIndex: 'startCol',
            startOffset: 'startColOffset',
            endIndex: 'endCol',
            endOffset: 'endColOffset'
        },

        // vertical drawing anchor attribute names
        VERTICAL_ATTRIBUTE_NAMES = {
            offset: 'top',
            size: 'height',
            startIndex: 'startRow',
            startOffset: 'startRowOffset',
            endIndex: 'endRow',
            endOffset: 'endRowOffset'
        },

        // maps 'anchorType' attribute values to the internal anchor mode used for anchor attributes
        INTERNAL_ANCHOR_MODES = { twoCell: 'twoCell', oneCell: 'oneCell', absolute: 'absolute', twoCellAsOneCell: 'twoCell', twoCellAsAbsolute: 'twoCell' },

        // maps 'anchorType' attribute values to display anchor mode defining runtime behavior
        DISPLAY_ANCHOR_MODES = { twoCell: 'twoCell', oneCell: 'oneCell', absolute: 'absolute', twoCellAsOneCell: 'oneCell', twoCellAsAbsolute: 'absolute' };

    // class SheetDrawingCollection ===========================================

    /**
     * Represents the drawing collection of a single sheet in a spreadsheet
     * document.
     *
     * @constructor
     *
     * @extends DrawingCollection
     *
     * @param {SpreadsheetApplication} app
     *  The application instance with the document model containing this sheet.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function SheetDrawingCollection(app, sheetModel) {

        var // self reference
            self = this,

            // the document model
            model = app.getModel(),

            // the column and row collections (size and formatting of all columns/rows)
            colCollection = sheetModel.getColCollection(),
            rowCollection = sheetModel.getRowCollection(),

            // maps drawing model indexes (used as logical positions) to drawing models
            drawingModels = [];

        // base constructor ---------------------------------------------------

        DrawingCollection.call(this, app, {
            clientResolveModelHandler: resolveDrawingModel,
            clientResolvePositionHandler: resolveDrawingPosition,
            clientRegisterModelHandler: registerDrawingModel,
            clientSetAttributesHandler: setDrawingAttributesHandler

        });

        // private methods ----------------------------------------------------

        /**
         * Tries to return a drawing model contained in this sheet at the
         * specified logical position.
         *
         * @param {Number[]} position
         *  An arbitrary logical document position.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified logical position, if available;
         *  otherwise null.
         */
        function resolveDrawingModel(position) {
            return ((position.length === 1) && drawingModels[position[0]]) || null;
        }

        /**
         * Tries to return the logical position of the specified drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  An arbitrary drawing model.
         *
         * @param {Number[]|Null}
         *  The logical position of the passed drawing model, if the model is
         *  part of this drawing collection, otherwise null.
         */
        function resolveDrawingPosition(drawingModel) {
            var index = _(drawingModels).indexOf(drawingModel);
            return (index >= 0) ? [index] : null;
        }

        /**
         * Registers a new drawing model that has just been inserted into the
         * drawing collection of this sheet.
         *
         * @param {DrawingModel} drawingModel
         *  The new drawing model.
         *
         * @param {Number[]} position
         *  The logical position of the new drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed logical position is valid, and the drawing model
         *  has been registered successfully.
         */
        function registerDrawingModel(drawingModel, position) {
            if ((position.length === 1) && (0 <= position[0]) && (position[0] <= drawingModels.length)) {
                drawingModels.splice(position[0], 0, drawingModel);
                return true;
            }
            return false;
        }

        /**
         * Returns all available information about the position of a drawing
         * object, according to the passed drawing attributes.
         *
         * @param {Object} anchorAttrs
         *  An attribute map containing drawing anchor attributes.
         *
         * @param {String} anchorMode
         *  The anchor mode that specifies the active anchor attributes in the
         *  passed attribute map. The remaining inactive positioning properties
         *  will be calculated according to these active attributes.
         *
         * @returns {Object}
         *  The result object with position information for the passed drawing
         *  attribute set, in the following properties:
         *  - {Number} leftHmm
         *      The absolute horizontal offset of the drawing object in 1/100
         *      mm, independent from the current sheet zoom factor.
         *  - {Number} topHmm
         *      The absolute vertical offset of the drawing object in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} widthHmm
         *      The width of the drawing object in 1/100 mm, independent from
         *      the current sheet zoom factor.
         *  - {Number} heightHmm
         *      The height of the drawing object in 1/100 mm, independent from
         *      the current sheet zoom factor.
         *  - {Number} left
         *      The absolute horizontal offset of the drawing object in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} top
         *      The absolute vertical offset of the drawing object in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} width
         *      The width of the drawing object in pixels, according to the
         *      current sheet zoom factor.
         *  - {Number} height
         *      The height of the drawing object in pixels, according to the
         *      current sheet zoom factor.
         *  - {Number} startCol
         *      The zero-based index of the column containing the leading
         *      border of the drawing object.
         *  - {Number} startColOffset
         *      The relative offset inside the column containing the leading
         *      border of the drawing object, in 1/100 mm.
         *  - {Number} startRow
         *      The zero-based index of the row containing the top border of
         *      the drawing object.
         *  - {Number} startRowOffset
         *      The relative offset inside the row containing the top border of
         *      the drawing object, in 1/100 mm.
         *  - {Number} endCol
         *      The zero-based index of the column containing the trailing
         *      border of the drawing object.
         *  - {Number} endColOffset
         *      The relative offset inside the column containing the trailing
         *      border of the drawing object, in 1/100 mm.
         *  - {Number} endRow
         *      The zero-based index of the row containing the bottom border of
         *      the drawing object.
         *  - {Number} endRowOffset
         *      The relative offset inside the row containing the bottom border
         *      of the drawing object, in 1/100 mm.
         *  - {Boolean} hiddenCols
         *      Whether the drawing object is located entirely in a range of
         *      hidden columns.
         *  - {Boolean} hiddenRows
         *      Whether the drawing object is located entirely in a range of
         *      hidden rows.
         */
        function getAnchorInformation(anchorAttrs, anchorMode) {

            var // collection entry of the column/row collection
                entryData = null,
                // the resulting position
                position = {};

            // get start position of the drawing object
            switch (anchorMode) {
            case 'twoCell':
            case 'oneCell':
                // left position in 1/100 mm and pixels, start column index and offset
                position.leftHmm = colCollection.getEntryOffsetHmm(anchorAttrs.startCol, anchorAttrs.startColOffset);
                position.left = colCollection.getEntryOffset(anchorAttrs.startCol, anchorAttrs.startColOffset);
                position.startCol = anchorAttrs.startCol;
                position.startColOffset = anchorAttrs.startColOffset;

                // top position in 1/100 mm and pixels, start row index and offset
                position.topHmm = rowCollection.getEntryOffsetHmm(anchorAttrs.startRow, anchorAttrs.startRowOffset);
                position.top = rowCollection.getEntryOffset(anchorAttrs.startRow, anchorAttrs.startRowOffset);
                position.startRow = anchorAttrs.startRow;
                position.startRowOffset = anchorAttrs.startRowOffset;
                break;

            case 'absolute':
                // left position in 1/100 mm and pixels, start column index and offset
                entryData = colCollection.getEntryByOffset(anchorAttrs.left);
                position.leftHmm = entryData.offsetHmm + entryData.relOffsetHmm;
                position.left = entryData.offset + entryData.relOffset;
                position.startCol = entryData.index;
                position.startColOffset = entryData.relOffsetHmm;

                // top position in 1/100 mm and pixels, start row index and offset
                entryData = rowCollection.getEntryByOffset(anchorAttrs.top);
                position.topHmm = entryData.offsetHmm + entryData.relOffsetHmm;
                position.top = entryData.offset + entryData.relOffset;
                position.startRow = entryData.index;
                position.startRowOffset = entryData.relOffsetHmm;
                break;

            default:
                Utils.error('SheetDrawingCollection.getAnchorInformation(): invalid anchor mode "' + anchorMode + '"');
            }

            // get effective size of the drawing object
            switch (anchorMode) {
            case 'twoCell':
                // width in 1/100 mm and pixels; end column index and offset
                position.widthHmm = colCollection.getEntryOffsetHmm(anchorAttrs.endCol, anchorAttrs.endColOffset) - position.leftHmm;
                position.width = colCollection.getEntryOffset(anchorAttrs.endCol, anchorAttrs.endColOffset) - position.left;
                position.endCol = anchorAttrs.endCol;
                position.endColOffset = anchorAttrs.endColOffset;

                // height in 1/100 mm and pixels, end row index and offset
                position.heightHmm = rowCollection.getEntryOffsetHmm(anchorAttrs.endRow, anchorAttrs.endRowOffset) - position.topHmm;
                position.height = rowCollection.getEntryOffset(anchorAttrs.endRow, anchorAttrs.endRowOffset) - position.top;
                position.endRow = anchorAttrs.endRow;
                position.endRowOffset = anchorAttrs.endRowOffset;
                break;

            case 'oneCell':
            case 'absolute':
                // width in 1/100 mm and pixels; end column index and offset
                entryData = colCollection.getEntryByOffset(position.leftHmm + anchorAttrs.width);
                position.widthHmm = entryData.offsetHmm + entryData.relOffsetHmm - position.leftHmm;
                position.width = entryData.offset + entryData.relOffset - position.left;
                position.endCol = entryData.index;
                position.endColOffset = entryData.relOffsetHmm;

                // height in 1/100 mm and pixels, end row index and offset
                entryData = rowCollection.getEntryByOffset(position.topHmm + anchorAttrs.height);
                position.heightHmm = entryData.offsetHmm + entryData.relOffsetHmm - position.topHmm;
                position.height = entryData.offset + entryData.relOffset - position.top;
                position.endRow = entryData.index;
                position.endRowOffset = entryData.relOffsetHmm;
                break;
            }

            // whether the drawing is located entirely in hidden columns or rows
            position.hiddenCols = colCollection.isHiddenInterval({ first: position.startCol, last: position.endCol });
            position.hiddenRows = rowCollection.isHiddenInterval({ first: position.startRow, last: position.endRow });

            return position;
        }

        /**
         * Returns the inactive anchor attributes of the passed attribute set,
         * according to the anchor mode and anchor attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The current values of the inactive anchor attributes, according to
         *  the anchor mode.
         */
        function getInactiveAnchorAttributes(drawingAttrs, options) {

            var // whether to use the display anchor instead of the anchor type
                displayAnchor = Utils.getBooleanOption(options, 'displayAnchor', false),
                // the actual anchor mode used to find inactive anchor attributes
                anchorMode = (displayAnchor ? DISPLAY_ANCHOR_MODES : INTERNAL_ANCHOR_MODES)[drawingAttrs.anchorType],
                // current position of the drawing object
                positionInfo = getAnchorInformation(drawingAttrs, anchorMode);

            switch (anchorMode) {
            case 'twoCell':
                return {
                    left: positionInfo.leftHmm,
                    top: positionInfo.topHmm,
                    width: positionInfo.widthHmm,
                    height: positionInfo.heightHmm
                };

            case 'oneCell':
                return {
                    left: positionInfo.leftHmm,
                    top: positionInfo.topHmm,
                    endCol: positionInfo.endCol,
                    endColOffset: positionInfo.endColOffset,
                    endRow: positionInfo.endRow,
                    endRowOffset: positionInfo.endRowOffset
                };

            case 'absolute':
                return {
                    startCol: positionInfo.startCol,
                    startColOffset: positionInfo.startColOffset,
                    startRow: positionInfo.startRow,
                    startRowOffset: positionInfo.startRowOffset,
                    endCol: positionInfo.endCol,
                    endColOffset: positionInfo.endColOffset,
                    endRow: positionInfo.endRow,
                    endRowOffset: positionInfo.endRowOffset
                };
            }

            return null;
        }

        /**
         * Updates the inactive anchor attributes according to the current
         * anchor mode and anchor attributes.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model whose anchor attributes will be updated.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *  @param {Boolean} [options.notify=false]
         *      If set to true, the changed attributes will be notified with a
         *      'change:attributes' event. Otherwise, the attributes will be
         *      updated silently.
         */
        function updateInactiveAnchorAttributes(drawingModel, options) {

            var // the merged drawing attributes of the drawing model
                drawingAttrs = drawingModel.getMergedAttributes().drawing,
                // the new values of the inactive anchor attributes
                anchorAttrs = getInactiveAnchorAttributes(drawingAttrs, options),
                // the effective notification mode
                notifyMode = Utils.getBooleanOption(options, 'notify', false) ? 'auto' : 'never';

            // set the calculated attributes
            if (anchorAttrs) {
                drawingModel.setAttributes({ drawing: anchorAttrs }, { notify: notifyMode });
            }
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel) {
            // extend the API of the drawing model with sheet-specific methods
            DrawingModelMixin.call(drawingModel, app, sheetModel);
            // initially calculate the inactive anchor attributes according to anchor type
            updateInactiveAnchorAttributes(drawingModel);
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel, position) {
            drawingModels.splice(position[0], 1);
        }

        /**
         * Called after the attributes of a drawing mode have been changed.
         * Recalculates the inactive anchor attributes according to the current
         * display anchor of the drawing model.
         */
        function setDrawingAttributesHandler(drawingModel) {
            updateInactiveAnchorAttributes(drawingModel);
        }

        /**
         * Adjusts the position of the drawing objects, after new columns/rows
         * have been inserted into the sheet.
         */
        function insertInterval(interval, columns) {

            var // the attribute names according to the passed direction
                ATTR_NAMES = columns ? HORIZONTAL_ATTRIBUTE_NAMES : VERTICAL_ATTRIBUTE_NAMES,
                // the maximum column/row
                maxIndex = columns ? model.getMaxCol() : model.getMaxRow(),
                // the number of columns/rows to be inserted
                move = SheetUtils.getIntervalSize(interval),
                // the helper function to convert the drawing range to an interval
                getIntervalFunc = columns ? SheetUtils.getColInterval : SheetUtils.getRowInterval;

            _(drawingModels).each(function (drawingModel) {

                var // the column/row interval covered by the drawing model
                    drawingInterval = getIntervalFunc(drawingModel.getRange()),
                    // the maximum number of columns/rows the drawing can be moved
                    maxMove = Math.min(move, maxIndex - drawingInterval.last),
                    // the new anchor attributes
                    attributes = null;

                // do nothing, if drawing cannot be move anymore
                // TODO: prevent the insert operation completely?
                if (maxMove === 0) { return; }

                // move end position, if the columns/rows are inserted inside the drawing object
                if (interval.first <= drawingInterval.last) {
                    attributes = {};
                    attributes[ATTR_NAMES.endIndex] = drawingInterval.last + maxMove;
                    // move completely, if the columns/rows are inserted before the drawing object
                    if (interval.first <= drawingInterval.first) {
                        attributes[ATTR_NAMES.startIndex] = drawingInterval.first + maxMove;
                    }
                    drawingModel.setAttributes({ drawing: attributes });
                }
            });
        }

        /**
         * Adjusts the position of the drawing objects, after columns/rows have
         * been deleted from the sheet.
         */
        function deleteInterval(interval, columns) {

            var // the attribute names according to the passed direction
                ATTR_NAMES = columns ? HORIZONTAL_ATTRIBUTE_NAMES : VERTICAL_ATTRIBUTE_NAMES,
                // the number of columns/rows to be deleted
                move = SheetUtils.getIntervalSize(interval);

            // iterate in reverse order to be able to delete drawings from the collection
            Utils.iterateArray(drawingModels, function (drawingModel, position) {

                var // the current anchor attributes
                    drawingAttributes = drawingModel.getMergedAttributes().drawing,
                    // the column/row interval covered by the drawing model
                    drawingInterval = { first: drawingAttributes[ATTR_NAMES.startIndex], last: drawingAttributes[ATTR_NAMES.endIndex] },
                    // the new anchor attributes
                    anchorAttributes = null;

                // Delete drawing object with two-cell display anchor mode, if
                // it is contained in the deleted column/row interval, or if it
                // ends at offset zero in the first column/row after the interval.
                if ((DISPLAY_ANCHOR_MODES[drawingAttributes.anchorType] === 'twoCell') &&
                    (interval.first <= drawingInterval.first) &&
                    ((drawingInterval.last <= interval.last) || ((drawingInterval.last + 1 === interval.last) && (drawingAttributes[ATTR_NAMES.endOffset] === 0)))
                ) {
                    self.deleteModel([position]);
                    return;
                }

                // do nothing, if the drawing is located before the deleted interval
                if (drawingInterval.last < interval.first) { return; }

                // Adjust start/end position of drawings overlapping or following
                // the deleted interval. The setDrawingAttributesHandler callback
                // function will care about updating the absolute anchor attributes
                // for twoCell anchor mode, and restoring the cell anchor attributes
                // for onceCell and absolute anchor mode.
                anchorAttributes = {};
                if (interval.last < drawingInterval.last) {
                    // move end position, if the columns/rows are deleted before the end
                    anchorAttributes[ATTR_NAMES.endIndex] = drawingInterval.last - move;
                } else {
                    // cut end of drawing object at the deleted interval
                    anchorAttributes[ATTR_NAMES.endIndex] = interval.first;
                    anchorAttributes[ATTR_NAMES.endOffset] = 0;
                }
                if (interval.last < drawingInterval.first) {
                    // move start position, if the columns/rows are deleted before the start
                    anchorAttributes[ATTR_NAMES.startIndex] = drawingInterval.first - move;
                } else if (interval.first <= drawingInterval.first) {
                    // cut start of drawing object at the deleted interval
                    anchorAttributes[ATTR_NAMES.startIndex] = interval.first;
                    anchorAttributes[ATTR_NAMES.startOffset] = 0;
                }
                drawingModel.setAttributes({ drawing: anchorAttributes });

            }, { reverse: true });
        }

        /**
         * Adjusts the inactive anchor properties of the drawing objects, after
         * columns/rows have been formatted.
         */
        function changeInterval(interval, columns) {

            var // the helper function to convert the drawing range to an interval
                getIntervalFunc = columns ? SheetUtils.getColInterval : SheetUtils.getRowInterval;

            _(drawingModels).each(function (drawingModel) {

                var // the column/row interval covered by the drawing model
                    drawingInterval = getIntervalFunc(drawingModel.getRange());

                if (interval.first <= drawingInterval.last) {
                    updateInactiveAnchorAttributes(drawingModel, { displayAnchor: true, notify: true });
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Creates and returns a clone of this collection instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model that will contain the new cloned collection.
         *
         * @returns {SheetDrawingCollection}
         *  A complete clone of this collection, associated to the specified
         *  sheet model.
         */
        this.clone = function (sheetModel) {
            return new SheetDrawingCollection(app, sheetModel, drawingModels);
        };

        /**
         * Invokes the passed iterator function for all drawing models in this
         * sheet. The drawing models will be visited in order of their logical
         * document positions (Z order).
         *
         * @param {Function} iterator
         *  The iterator function called for all drawing models. Receives the
         *  current drawing model as first parameter, and its logical position
         *  as second parameter. If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the drawing models will be visited in reversed
         *      order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateModelsByPosition = function (iterator, options) {
            return Utils.iterateArray(drawingModels, function (model, index) {
                return iterator.call(this, model, [index]);
            }, options);
        };

        /**
         * Returns the sheet rectangle in pixels covered by a drawing object
         * with the passed formatting attributes.
         *
         * @param {Object} attributes
         *  The merged attribute set of a drawing object.
         *
         * @returns {Object|Null}
         *  The position of the drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'; or null if the drawing object
         *  is located entirely in hidden columns or rows and thus invisible.
         */
        this.getRectangleForAttributes = function (attributes) {

            var // the complete position information object
                positionInfo = getAnchorInformation(attributes.drawing, INTERNAL_ANCHOR_MODES[attributes.drawing.anchorType]);

            // return null for hidden drawing objects
            if (positionInfo.hiddenCols || positionInfo.hiddenRows) {
                return null;
            }

            // keep minimum size of 5 pixels
            if (positionInfo.width < 5) {
                positionInfo.left = Math.max(0, positionInfo.left - Math.round((5 - positionInfo.width) / 2));
                positionInfo.width = 5;
            }
            if (positionInfo.height < 5) {
                positionInfo.top = Math.max(0, positionInfo.top - Math.round((5 - positionInfo.height) / 2));
                positionInfo.height = 5;
            }

            // return the pixel position and size as object
            return { left: positionInfo.left, top: positionInfo.top, width: positionInfo.width, height: positionInfo.height };
        };

        /**
         * Returns the drawing attributes needed to set the position of a
         * drawing object to the passed sheet rectangle.
         *
         * @param {Object} rectangle
         *  The new position for a drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'.
         *
         * @returns {Object}
         *  The attribute set containing the anchor attributes for the new
         *  position of the drawing object.
         */
        this.getAttributesForRectangle = function (rectangle) {

            var // collection entry of the column/row collection
                entryData = null,
                // the new anchor attributes
                attributes = {};

            // horizontal start position
            entryData = colCollection.getEntryByOffset(rectangle.left, { pixel: true });
            attributes.left = entryData.offsetHmm + entryData.relOffsetHmm;
            attributes.startCol = entryData.index;
            attributes.startColOffset = entryData.relOffsetHmm;

            // object width and horizontal end position
            entryData = colCollection.getEntryByOffset(rectangle.left + rectangle.width, { pixel: true });
            attributes.width = entryData.offsetHmm + entryData.relOffsetHmm - attributes.left;
            attributes.endCol = entryData.index;
            attributes.endColOffset = entryData.relOffsetHmm;

            // vertical start position
            entryData = rowCollection.getEntryByOffset(rectangle.top, { pixel: true });
            attributes.top = entryData.offsetHmm + entryData.relOffsetHmm;
            attributes.startRow = entryData.index;
            attributes.startRowOffset = entryData.relOffsetHmm;

            // object height and vertical end position
            entryData = rowCollection.getEntryByOffset(rectangle.top + rectangle.height, { pixel: true });
            attributes.height = entryData.offsetHmm + entryData.relOffsetHmm - attributes.top;
            attributes.endRow = entryData.index;
            attributes.endRowOffset = entryData.relOffsetHmm;

            return { drawing: attributes };
        };

        /**
         * Adds missing generic anchor attributes ('left', 'top', 'width', and
         * 'height') to the passed (incomplete) attribute set of a drawing
         * object.
         *
         * @param {Object} attributes
         *  A drawing attribute set. Must contain a property 'drawing' with
         *  some anchor properties.
         */
        this.addGenericAnchorAttributes = function (attributes) {

            var // the merged attribute set (with default values for all missing attributes)
                mergedAttributes = model.getStyleSheets('drawing').getMergedAttributes(attributes),
                // the inactive anchor attributes
                anchorAttributes = getInactiveAnchorAttributes(mergedAttributes.drawing);

            // add the generic anchor attributes to the passed attribute set
            _(['left', 'top', 'width', 'height']).each(function (attrName) {
                if (attrName in anchorAttributes) {
                    (attributes.drawing || (attributes.drawing = {}))[attrName] = anchorAttributes[attrName];
                }
            });
        };

        // initialization -----------------------------------------------------

        // register own chart model
        this.getModelFactory().registerModelType('chart', SheetChartModel);

        // process inserted and deleted drawing models
        this.on({
            'insert:drawing': insertDrawingHandler,
            'delete:drawing': deleteDrawingHandler
        });

        // adjust position of drawing objects after modifying column collection
        colCollection.on({
            'insert:entries': function (event, interval) { insertInterval(interval, true); },
            'delete:entries': function (event, interval) { deleteInterval(interval, true); },
            'change:entries': function (event, interval) { changeInterval(interval, true); }
        });

        // adjust position of drawing objects after modifying row collection
        rowCollection.on({
            'insert:entries': function (event, interval) { insertInterval(interval, false); },
            'delete:entries': function (event, interval) { deleteInterval(interval, false); },
            'change:entries': function (event, interval) { changeInterval(interval, false); }
        });

        // clone drawing models entries passed as hidden argument to the c'tor (used by the clone() method)
        if (_.isArray(arguments[SheetDrawingCollection.length])) {
            // TODO
        }

    } // class SheetDrawingCollection

    // exports ================================================================

    // derive this class from class DrawingCollection
    return DrawingCollection.extend({ constructor: SheetDrawingCollection });

});
