/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel',
        ['io.ox/office/tk/utils',
         'io.ox/office/drawinglayer/model/drawingmodel',
         'io.ox/office/spreadsheet/utils/sheetutils',
         'io.ox/office/spreadsheet/model/operations',
         'io.ox/office/spreadsheet/model/tokenarray',
         'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
         'io.ox/office/spreadsheet/model/drawing/xychart',
         'io.ox/office/spreadsheet/model/drawing/accentcolorchart',
         'io.ox/office/spreadsheet/model/drawing/axismodel',
         'gettext!io.ox/office/spreadsheet'
        ], function (Utils, DrawingModel, SheetUtils, Operations, TokenArray, ManualColor, XY, AccentColor, AxisModel, gt) {

    'use strict';

    var TYPES = {
        'column standard' : 'column',
        'column clustered' : 'column',
        'column stacked' : 'stackedColumn',
        'column percentStacked' : 'stackedColumn100',
        'bar standard' : 'bar',
        'bar clustered' : 'bar',
        'bar stacked' : 'stackedBar',
        'bar percentStacked' : 'stackedBar100',
        'area standard' : 'area',
        'area clustered' : 'area',
        'area stacked' : 'stackedArea',
        'area percentStacked' : 'stackedArea100',
        'line' : 'line',
        'line standard' : 'line',
        'line stacked' : 'line',
        'line percentStacked' : 'line',
        'spline' : 'line',
        'spline standard' : 'spline',
        'spline stacked' : 'spline',
        'spline percentStacked' : 'spline',
        'scatter' : 'scatter',
        'bubble' : 'bubble',
        'pie' : 'pie',
        'donut' : 'doughnut'
    };

    var MARKER_TYPES = {
        'circle': 'circle',
        'dot': 'circle',
        'square': 'square',
        'triangle' : 'triangle',
        'x' : 'cross',
        'none': 'none'
    };
    var MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    // class ChartModel =======================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     */
    function ChartModel(app, attrs) {
        var self = this;

        var docModel = app.getModel();

        var EMPTYARRAY = [];

        var chartHandler = null;

        var token = null;

        var data = {
            axisX: {},
            axisY: {},
            creditHref: '',
            creditText: '',
            title: {
                text: ''
            },
            data: []
        };
        var axes = {
            x : new AxisModel(app, {}, data.axisX),
            y : new AxisModel(app, {}, data.axisY),
            z : new AxisModel(app, {}, null)
        };

        // base constructor ---------------------------------------------------

        DrawingModel.call(this, app, attrs, { additionalFamilies: ['chart', 'fill'] });

        try {
            var legendPos = self.getMergedAttributes().chart.legendPos;
            switch (legendPos) {
            case 'bottom':
            case 'top':
                data.legend = {
                    verticalAlign: legendPos,
                    horizontalAlign: 'center',
                };
                break;
            case 'left':
            case 'right':
                data.legend = {
                    verticalAlign: 'center',
                    horizontalAlign: legendPos
                };
                break;
            case 'topright':
                data.legend = {
                    horizontalAlign: 'right',
                    verticalAlign: 'top'
                };
                break;
            }
        } catch (e) {
            Utils.warn('error legend pos', e);
        }


        // private methods ------------------------------------------------------------

        function handleChartHandler() {
            var type = self.getChartType().split(' ')[0];

            switch (type) {
            case 'scatter':
            case 'bubble':
                chartHandler = new XY(app, self, data);
                break;
            case 'spline':
            case 'line':
            case 'pie':
            case 'donut':
                chartHandler = new AccentColor(app, self, data);
                break;
            default:
                chartHandler = new ManualColor(app, self, data);
                break;
            }
        }

        function contains(address, sheetName, sheetIndex) {
            var changed = null;
            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var sourcePoints = dataSeries.sourcePoints;

                var formula = getFormula(sourcePoints);

                var rangeInfos = parseFormula(formula, sheetIndex);

                for (var j = 0; j < rangeInfos.length; j++) {
                    if (SheetUtils.rangeContainsCell(rangeInfos[j].range, address)) {
                        dataSeries.needUpdate = true;
                        changed = true;
                        break;
                    }
                }
            }
            return changed;
        }

        function getFormula(sourcePoints) {
            var sep = app.getListSeparator();
            var formula = '';
            for (var key in sourcePoints) {
                var sp = sourcePoints[key];
                if (_.isArray(sp)) {
                    //do nothing
                } else {
                    if (formula.length) {
                        formula += sep;
                    }
                    formula += sp;
                }
            }
            return formula;
        }

        function parseFormula(formula, sheetIndex) {
            if (!token) {
                token = new TokenArray(app);
            }
            var rangeInfos = token.parseFormula(formula).extractRanges({ targetSheet: sheetIndex, resolveNames: true });
            token.clear();
            return rangeInfos;

//            return app.getView().parseFormula(formula).extractRanges({ targetSheet: sheetIndex, resolveNames: true });
        }

        function colRowChangeHandler(event, sheet, interval) {
            switch (event.type) {
            case 'insert:columns':
                checkDataInsideInterval(sheet, interval, 0, +1);
                break;
            case 'delete:columns':
                checkDataInsideInterval(sheet, interval, 0, -1);
                break;
            case 'insert:rows':
                checkDataInsideInterval(sheet, interval, 1, +1);
                break;
            case 'delete:rows':
                checkDataInsideInterval(sheet, interval, 1, -1);
                break;
            }
        }

        function checkDataInsideInterval(sheetModelIndex, interval, addressIndex, plusminus) {
            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var sourcePoints = dataSeries.sourcePoints;
                for (var key in sourcePoints) {
                    checkRangeInsideInterval(sheetModelIndex, sourcePoints, key, interval, addressIndex, plusminus);
                }
            }
        }

        function checkRangeInsideInterval(sheetModelIndex, source, name, interval, addressIndex, plusminus) {
            var formula = source[name];
            if (!formula) {
                return;
            }
            var intervalSheetModelName = docModel.getSheetName(sheetModelIndex);

            var sheetName = null;
            var ios = formula.lastIndexOf('!');
            if (ios > 0) {
                sheetName = formula.substring(0, ios);
            } else {
                var myIndex = self.getPosition()[0];
                sheetName = docModel.getSheetName(myIndex);
            }

            if (sheetName != intervalSheetModelName) {
                return;
            }

            var sourceRange = parseFormula(source[name], sheetModelIndex)[0].range;

            var end = sourceRange.end;
            var start = sourceRange.start;

            var offset = [0, 0];
            offset[addressIndex] = SheetUtils.getIntervalSize(interval);

            var change = null;
            if (end[addressIndex] < interval.first) {
                //no hit
            } else if (start[addressIndex] > interval.last || end[addressIndex] == start[addressIndex]) {
                //move
                change = true;
                end[0] += offset[0] * plusminus;
                end[1] += offset[1] * plusminus;
                start[0] += offset[0] * plusminus;
                start[1] += offset[1] * plusminus;

            } else {
                //scale
                change = true;
                end[0] += offset[0] * plusminus;
                end[1] += offset[1] * plusminus;
            }

            if (change) {
                var newFormula = sheetName + '!' + getCellName(start) + ':' +  getCellName(end);
                source[name] = newFormula;
            }

        }

        function getCellName(address) {
            return '$' + SheetUtils.getColName(address[0]) + '$' + SheetUtils.getRowName(address[1]);
        }

        /**
         * hangs on the global 'docs:update' Event,
         * if any source cell is changed, the correct Series will be marked,
         * so the correct request can be send out, when the chart is visible the nex time
         */
        function updateNotificationHandler(evt) {
            var me = false;
            if (evt.changed) {
                if (evt.changed.type === 'all') {

                    for (var i = 0; i < data.data.length; i++) {
                        var dataSeries = data.data[i];
                        dataSeries.needUpdate = true;
                    }

                    me = true;
                } else {


                    var sheets = evt.changed.sheets;

                    if (sheets) {
                        //update info only addresses

                        for (var key in sheets) {
                            var sheet = sheets[key];

                            var sheetIndex = key | 0;
                            var sheetName = app.getModel().getSheetName(sheetIndex);
                            var tmpCell = [-1, -1];
                            var cells = sheet.cells;
                            for (var c = 0; c < cells.length; c++) {
                                var cell = cells[c];
                                var start = cell.start;
                                var end = cell.end;

                                for (var x = start[0]; x <= end[0]; x++) {
                                    for (var y = start[1]; y <= end[1]; y++) {
                                        tmpCell[0] = x;
                                        tmpCell[1] = y;

                                        if (contains(tmpCell, sheetName, sheetIndex)) {
                                            me = true;
                                        }
                                    }
                                }
                            }
                        }
                    }

                }
            }
            if (me === true) {
                self.refresh();
            }
        }

        function addDataToRequest(list, info, series, index) {
            var inf = {};
            inf.index = index;
            for (var key in series) {
                addDataToList(series, key, list, inf);
            }
            info.push(inf);
        }

        function addDataToList(series, key, list, inf) {
            var entry = series[key];
            if (entry) {
                if (_.isArray(entry)) {
                    var dataSeries = data.data[inf.index];
                    var dataPoints = dataSeries.dataPoints;
                    if (!dataPoints) {
                        dataPoints = [];
                        dataSeries.dataPoints = dataPoints;
                    }
                    if (!dataSeries.name || !dataSeries.name.length) {
                        dataSeries.name = gt('Series %1$d', _.noI18n(inf.index + 1));
                    }

                    var labelName = chartHandler.getLabelName();

                    var name = null;
                    var extra = null;
                    if (key == 'names') {
                        name = labelName;
                        extra = '';
                    } else if (key == 'values') {
                        name = 'y';
                        extra = 0;
                    } else if (key == 'bubbles') {
                        name = 'z';
                        extra = 0;
                    } else if (key == 'title') {
                        dataSeries.name = entry[0] + '';
                        return;
                    }

                    for (var i = 0; i < entry.length; i++) {
                        var d = dataPoints[i];
                        if (!d) {
                            d = {};
                            d.y = 0;
                            d[labelName] = (i + 1) + '';
                            dataPoints[i] = d;
                        }
                        var e = entry[i];
                        if (isNaN(e)) {
                            d[name] = e + extra;
                        } else {
                            d[name] = parseFloat(e) + extra;
                        }
                        d.markerType = dataSeries.marker;
                    }
                } else {
                    var indexOf = list.indexOf(entry);
                    if (indexOf >= 0) {
                        inf[key] = indexOf;
                    } else {
                        inf[key] = list.length;
                        list.push(series[key]);
                    }
                }
            }
        }

        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;
            if (mc.type !== oc.type || mc.group !== oc.group) {
                updateChartType();
            }
            handleChartHandler();
        }

        function getCJSType() {
            var type = self.getChartType();
            return TYPES[type];
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.data.length; i++) {
                var series = data.data[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }

        // public methods -----------------------------------------------------

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var group = attributes.chart.group;
//            if (!group.length) {
//                if (type === 'bar' || type === 'area') {
//                    group = 'clustered';
//                } else if (type === 'line') {
//                    group = 'standard';
//                }
//            }

            if (group.length) {
                return type + ' ' + group;
            } else {
                return type;
            }
        };

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        /**
         * not implemented!
         * @param operation
         */
        this.deleteDataSeries = function (/*operation*/) {
            Utils.warn('chartmodel.deleteDataSeries is not implemented!!!');
        };

        /**
         * refresh checks if something is changed in the series,
         * if true it calls the 'updateview'-request to the server
         */
        this.refresh = function () {


            var gridpane = app.getView().getActiveGridPane();
            if (!gridpane) {
                return;
            }

            var collectToServer = null;
            var collectInfo = null;

            for (var i = 0; i < data.data.length; i++) {

                var d = data.data[i];

                if (d.needUpdate === true) {
                    var sourcePoints = d.sourcePoints;
                    if (!collectToServer) {
                        collectToServer = [];
                        collectInfo = [];
                    }
                    addDataToRequest(collectToServer, collectInfo, sourcePoints, i);
                }

            }

            if (collectToServer !== null) {
                if (collectToServer.length) {
                    Utils.info('Chartmodel has to send requests for missing series data. count = ' + collectToServer.length);
                    var request = app.sendFilterRequest({
                        method: 'POST',
                        params: {
                            action: 'updateview',
                            requestdata: JSON.stringify({
                                areas: collectToServer
                            })
                        }
                    });

                    var labelName = chartHandler.getLabelName();

                    request.then(function (res) {

                        var allAreas = res.areas;

                        for (var s = 0; s < collectInfo.length; s++) {
                            var info = collectInfo[s];
                            var series = info.index;
                            var valuesSource = allAreas[info.values];
                            var titleSource = EMPTYARRAY;
                            if (_.isNumber(info.title)) {
                                titleSource = allAreas[info.title];
                            }
                            var namesSource = EMPTYARRAY;
                            if (_.isNumber(info.names)) {
                                namesSource = allAreas[info.names];
                            }
                            var bubblesSource = EMPTYARRAY;
                            if (_.isNumber(info.bubbles)) {
                                bubblesSource = allAreas[info.bubbles];
                            }

                            var dataSeries = data.data[series];

                            if (titleSource[0]) {
                                dataSeries.name = (titleSource[0].result || 'NOTITLE') + '';
                            } else {
                                dataSeries.name =
                                    //#. A data series in a chart object (a sequence of multiple data points with a title)
                                    //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                                    //#, c-format
                                    gt('Series %1$d', _.noI18n(series + 1));
                            }

                            dataSeries.dataPoints = [];
                            var dataPoints = dataSeries.dataPoints;

                            for (var n = 0; n < valuesSource.length; n++) {
                                var dataPoint = dataPoints[n];
                                if (!dataPoint) {
                                    dataPoint = {};
                                    dataPoints[n] = dataPoint;
                                }
                                dataPoint.y = valuesSource[n].result || 0;

                                if (namesSource[n]) {
                                    dataPoint[labelName] = namesSource[n].result || '';
                                } else {
                                    dataPoint[labelName] = (n + 1) + '';
                                }

                                if (bubblesSource[n]) {
                                    dataPoint.z = bubblesSource[n].result || '';
                                }
                                dataPoint.markerType = dataSeries.marker;
                            }


                            dataSeries.needUpdate = null;


                        }
                        refreshAxis();
                        self.trigger('change:drawing', 'change:series');
                    });
                } else {
                    for (var i = 0; i < data.data.length; i++) {
                        data.data[i].needUpdate = null;
                    }
                    self.trigger('change:drawing', 'change:series');
                }
            }
        };

        /**
         * saves the Operation-Series in the model, but fetches the correct Data only if its visible
         * via trigger the Events 'change:drawing' and 'insert:series' and mark the current series that ists changed
         * @param operation
         *  @param {Object} [options.attrs]
         *   @param {Object} [options.attrs.series]
         *    the content of the series could be a formula {String}, or explicit Values {Array}.
         *    the values can be taken directy. the formula calls a request to server, the next time the chart visible
         *    @param {String|Array} [options.attrs.series.title]
         *    @param {String|Array} [options.attrs.series.names]
         *    @param {String|Array} [options.attrs.series.bubbles]
         *    @param {String|Array} [options.attrs.series.values]
         *
         * @returns {Boolean}
         */
        this.insertDataSeries = function (operation) {
            var attributes = self.getMergedAttributes();

            data.title = ''; //TODO: from where comes the Title?!
            var dataSeries = data.data[operation.series];
            if (!dataSeries) {
                dataSeries = {
                    type: getCJSType()
                };


                if (getCJSType() === 'pie' || getCJSType() === 'doughnut') {
                    if (operation.series > 0) {
                        return;
                    }
                }

                data.data[operation.series] = dataSeries;
            }
            var series = operation.attrs.series;

            if (getCJSType() === 'pie' || getCJSType() === 'doughnut') {
                dataSeries.startAngle = attributes.chart.rotation - 90;
                dataSeries.showInLegend = false;
            } else {
                if (attributes.chart.legendPos === 'off') {
                    dataSeries.showInLegend = false;
                } else {
                    dataSeries.showInLegend = true;
                }
            }

            if (getCJSType() === 'line' || getCJSType() === 'spline') {
                dataSeries.marker = MARKER_TYPES[series.marker];
                if (!dataSeries.marker) {
                    var index = operation.series % MARKER_LIST.length;
                    dataSeries.marker = MARKER_LIST[index];
                }
            }

            chartHandler.handleColor(series, operation.series);

            dataSeries.sourcePoints = series;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';

            this.trigger('change:drawing', 'insert:series', operation.series);

            return true;
        };

        this.setAxisAttributes = function (operation) {
            axes[operation.axis].setAttributes(operation.attrs);
        };

        this.setChartDataSeriesAttributes = function (/*operation*/) {
            Utils.warn('setChartDataSeriesAttributes is not implemented');
            return true;
        };

        /**
         * @returns {TokenArray}
         *  A new formula token array containing the source ranges of all data
         *  series of this chart object.
         */
        this.getTokenArray = function () {
            var content = '';

            if (data.data.length) {
                var sourcePoints = data.data[0].sourcePoints;
                content += getFormula(sourcePoints);

            }
//            var array = token.parseFormula(content);
//            token.clear();
//            return array;

            return docModel.parseFormula(content, this.getSheetModel().getIndex(), [0, 0]);
        };

        this.getNamesToken = function () {
            if (data.data.length) {
                return data.data[0].sourcePoints.names;
            } else {
                return '';
            }
        };

        function getRangeToken(key) {
            if (data.data.length) {
                var startPoints = data.data[0].sourcePoints[key];
                var endPoints = data.data[data.data.length - 1].sourcePoints[key];

                if (!startPoints || !endPoints) {
                    return '';
                }

                if (data.data.length === 1) {
                    return ots(startPoints);
                }

                var start = null;
                var end = null;

                if (_.isArray(startPoints)) {
                    start = ots(startPoints);
                } else {
                    var ios = startPoints.lastIndexOf(':');
                    if (ios >= 0) {
                        start = startPoints.substring(0, ios);
                    } else {
                        start = startPoints;
                    }
                }

                if (_.isArray(endPoints)) {
                    end = ots(endPoints);
                } else {
                    var ioe = endPoints.lastIndexOf(':');
                    if (ioe >= 0) {
                        end = endPoints.substring(ioe + 1, endPoints.length);
                    } else {
                        ioe = endPoints.lastIndexOf('!');
                        if (ioe >= 0) {
                            end = endPoints.substring(ioe + 1, endPoints.length);
                        } else {
                            end = endPoints;
                        }
                    }
                }

                return start + ':' + end;
            } else {
                return '';
            }
        }

        function ots(object) {
            if (_.isArray(object)) {
                var sep = app.getListSeparator();
                return object.join(sep);
            } else {
                return object + '';
            }
        }

        function makeMin(min, max) {
            if (Math.abs(max - min) < 0.1 || (min / max < 5 / 6)) {
                return 0;
            } else {
                return min - ((max - min) / 2);
            }
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function refreshAxis() {
            var type = getCJSType();

            var givenMin = null;
            var givenMax = null;

            var axisInfo = axes.y.getMergedAttributes();
            if (axisInfo) {
                if (axisInfo.min != 'auto') {
                    givenMin = axes.y.min;
                }
                if (axisInfo.max != 'auto') {
                    givenMax = axes.y.max;
                }
            }

            axes.x.refreshInfo();
            axes.y.refreshInfo();



            var yMin = 999999;
            var yMax = -999999;

            if (type.indexOf('100') > 0) {
                var first = data.data[0];
                for (var j = 0; j < first.dataPoints.length; j++) {
                    var y = 0;
                    for (var i = 0; i < data.data.length; i++) {
                        y += data.data[i].dataPoints[j].y;
                    }
                    yMin = Math.min(yMin, y);
                    yMax = Math.max(yMax, y);
                }
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || ((1 - yMin / yMax) * -100);
                    yMax = givenMax || 100;
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || ((1 - yMax / yMin) * +100);
                    yMin = givenMin || -100;
                } else {
                    //Szenario 3
                    yMax =  givenMax || 100;
                    yMin =  givenMin || -100;
                }
            } else {
                for (var i = 0; i < data.data.length; i++) {
                    var s = data.data[i];
                    for (var j = 0; j < s.dataPoints.length; j++) {
                        var y = s.dataPoints[j].y;
                        yMin = Math.min(yMin, y);
                        yMax = Math.max(yMax, y);
                    }
                }
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || makeMin(yMin, yMax);
                    yMax = givenMax || makeMax(yMin, yMax);
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || -makeMin(-yMax, -yMin);
                    yMin = givenMin || -makeMax(-yMax, -yMin);
                } else {
                    //Szenario 3
                    yMax =  givenMax || makeMax(0, yMax);
                    yMin =  givenMin || -makeMax(0, -yMin);
                }
            }


            if (yMin == yMax) {
                yMax = yMin + 1;
            }

            if (yMin < yMax) {
                data.axisY.minimum = yMin;
                data.axisY.maximum = yMax;
                data.axisY.title = '';
            } else {
                data.axisY.title = 'error from ' + yMin + ' to ' + yMax + ' !!!';
            }
        }

        this.getTitlesToken = function () {
            return getRangeToken('title');
        };

        this.getValuesToken = function () {
            return getRangeToken('values');
        };

        this.getBubblesToken = function () {
            return getRangeToken('bubbles');
        };

        this.hasBubbles = function () {
            return self.getChartType() == 'bubble';
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributes);
        this.listenTo(app, 'docs:update', updateNotificationHandler);
        this.listenTo(docModel, 'insert:columns delete:columns insert:rows delete:rows', colRowChangeHandler);

        handleChartHandler();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });
    } // class ChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: ChartModel });

});
