/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/colrowcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/cellmodel'
    ], function (Utils, ModelObject, SheetUtils, CellModel) {

    'use strict';

    // class ColRowCollection =================================================

    /**
     * Collects information about all columns or all rows of a single sheet in
     * a spreadsheet document.
     *
     * Triggers the following events:
     * - 'change:entries'
     *      After the formatting attributes of a column/row interval have been
     *      changed. The event handler receives the index interval of the
     *      changed entries, and the explicit attributes set at the interval
     *      (not the merged attribute sets which may be different in the
     *      various entries of the interval). The attributes parameter may be
     *      missing.
     * - 'insert:entries'
     *      After new columns/rows have been inserted into the sheet. The event
     *      handler receives the index interval of the inserted entries.
     * - 'delete:entries'
     *      After columns/rows have been deleted from the sheet. The event
     *      handler receives the index interval of the deleted entries.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     *
     * @param {Boolean} columns
     *  Whether this collection contains column headers (true) or row headers
     *  (false).
     */
    function ColRowCollection(app, sheetModel, columns) {

        var // self reference
            self = this,

            // the document model
            model = app.getModel(),

            // different property names for column or row properties
            FAMILY_NAME = columns ? 'column' : 'row',
            SIZE_ATTR_NAME = columns ? 'width' : 'height',
            DEFAULT_SIZE_ATTR_NAME = columns ? 'colWidth' : 'rowHeight',

            // sorted sparse array of index ranges (instances of EntryModel)
            entries = [],

            // the largest valid column/row index
            maxIndex = columns ? model.getMaxCol() : model.getMaxRow(),

            // the current zoom factor of the document
            zoom = sheetModel.getViewAttribute('zoom');

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private class EntryModel -------------------------------------------

        var EntryModel = CellModel.extend({ constructor: function (first, last, attributes) {

            // column/row interval covered by this entry
            this.first = first;
            this.last = last;

            // effective position/size of the columns/rows, in 1/100 mm
            this.offsetHmm = 0;
            this.sizeHmm = 0;

            // effective position/size of the columns/rows, in pixels
            this.offset = 0;
            this.size = 0;

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            // do not trigger any events (collection entries are potential mass objects)
            CellModel.call(this, app, attributes, { silent: true, additionalFamilies: FAMILY_NAME });

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // update the initial column/row size of this entry
            this.updateSize();

        }}); // class EntryModel

        // static methods - - - - - - - - - - - - - - - - - - - - - - - - - - -

        /**
         * Returns a clone of the passed collection entry. This entry may even
         * be part of another collection instance (e.g. copy construction).
         */
        EntryModel.cloneEntry = function (sourceEntry) {
            var newEntry = new EntryModel(sourceEntry.first, sourceEntry.last, sourceEntry.getExplicitAttributes());
            newEntry.offsetHmm = sourceEntry.offsetHmm;
            newEntry.offset = sourceEntry.offset;
            return newEntry;
        };

        // methods (prototype, collection entries are mass objects) - - - - - -

        /**
         * Updates the effective pixel size of a single column/row, according
         * to the current formatting attributes of this entry.
         */
        EntryModel.prototype.updateSize = function () {
            var attributes = this.getMergedAttributes()[FAMILY_NAME];
            this.sizeHmm = attributes.visible ? Math.max(0, attributes[SIZE_ATTR_NAME]) : 0;
            this.size = (this.sizeHmm === 0) ? 0 : Math.max(5, convertHmmToPixels(this.sizeHmm));
        };

        /**
         * Returns the absolute offset of the passed index (interpreted
         * relative to the start index of this entry) in 1/100 mm, independent
         * from the current sheet zoom factor.
         */
        EntryModel.prototype.getOffsetHmm = function (index) {
            return this.offsetHmm + this.sizeHmm * index;
        };

        /**
         * Returns the end offset of this entry in 1/100 mm, independent from
         * the current sheet zoom factor.
         */
        EntryModel.prototype.getEndOffsetHmm = function () {
            return this.getOffsetHmm(this.last - this.first + 1);
        };

        /**
         * Returns the absolute offset of the passed index (interpreted
         * relative to the start index of this entry) in pixels, according to
         * the current sheet zoom factor.
         */
        EntryModel.prototype.getOffset = function (index) {
            return this.offset + this.size * index;
        };

        /**
         * Returns the end offset of this entry in pixels, according to the
         * current sheet zoom factor.
         */
        EntryModel.prototype.getEndOffset = function () {
            return this.getOffset(this.last - this.first + 1);
        };

        var // default model for entries missing in the collection
            defaultEntry = new EntryModel(0, maxIndex);

        // private methods ----------------------------------------------------

        /**
         * Converts the passed zoom-independent length in 1/100 mm to a length
         * in pixels according to the current sheet zoom factor.
         */
        function convertHmmToPixels(length) {
            return Utils.convertHmmToLength(length * zoom, 'px', 1);
        }

        /**
         * Converts the passed length in pixels according to the current sheet
         * zoom factor to a zoom-independent length in 1/100 mm.
         */
        function convertPixelsToHmm(length) {
            return Utils.convertLengthToHmm(length / zoom, 'px');
        }

        /**
         * Searches for an entry that contains or follows the passed column/row
         * index.
         *
         * @param {Number} index
         *  The zero-based column/row index. Must be located in the valid range
         *  covered by this collection.
         *
         * @returns {Number}
         *  The array index of the collection entry containing the passed
         *  index. If the index is located in a gap between two entries,
         *  returns the array index of the entry starting after the passed
         *  index. This may also be an array index pointing after the last
         *  existing collection entry.
         */
        function findEntryIndex(index) {
            return _(entries).sortedIndex({ last: index }, 'last');
        }

        /**
         * Updates the dirty entry offsets up to the end of the collection.
         *
         * @param {Number} index
         *  The zero-based column/row index where updating the offsets starts.
         */
        function updateEntryOffsets(index) {

            var // the array index of the first dirty collection entry
                arrayIndex = 0,
                // the last valid collection entry
                prevEntry = null,
                // the current collection entry to be updated
                currEntry = null;

            // update start offsets of all dirty collection entries
            arrayIndex = findEntryIndex(index);
            prevEntry = entries[arrayIndex - 1];
            for (; arrayIndex < entries.length; arrayIndex += 1) {

                // set offset of current entry to the end offset of the previous entry
                currEntry = entries[arrayIndex];
                currEntry.offsetHmm = prevEntry ? prevEntry.getEndOffsetHmm() : 0;
                currEntry.offset = prevEntry ? prevEntry.getEndOffset() : 0;

                // add size of the gap between previous and current entry
                if (!prevEntry || (prevEntry.last + 1 < currEntry.first)) {
                    currEntry.offsetHmm += defaultEntry.sizeHmm * (currEntry.first - (prevEntry ? (prevEntry.last + 1) : 0));
                    currEntry.offset += defaultEntry.size * (currEntry.first - (prevEntry ? (prevEntry.last + 1) : 0));
                }

                prevEntry = currEntry;
            }
        }

        /**
         * Creates a descriptor object for the specified column/row.
         *
         * @param {EntryModel} entry
         *  The collection entry containing the column/row.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object}
         *  A descriptor object containing the following properties:
         *  - {Number} entry.index
         *      The passed column/row index.
         *  - {Number} entry.offsetHmm
         *      The absolute position of the column/row in the entire sheet in
         *      1/100 mm, independent from the current sheet zoom factor.
         *  - {Number} entry.sizeHmm
         *      The effective size (zero for hidden columns/rows) in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} entry.offset
         *      The absolute position of the column/row in pixels, according to
         *      the current sheet zoom factor.
         *  - {Number} entry.size
         *      The effective size (zero for hidden columns/rows) in pixels,
         *      according to the current sheet zoom factor.
         *  - {Object} entry.attributes
         *      The merged attribute set of the column/row.
         *  - {Object} entry.explicit
         *      The explicit attribute set of the column/row.
         */
        function makeDescriptorForEntry(entry, index) {
            return {
                index: index,
                offsetHmm: entry.getOffsetHmm(index - entry.first),
                sizeHmm: entry.sizeHmm,
                offset: entry.getOffset(index - entry.first),
                size: entry.size,
                attributes: entry.getMergedAttributes(),
                explicit: entry.getExplicitAttributes()
            };
        }

        /**
         * Helper function to get the descriptor of a specific column/row,
         * without needing to search for the affected collection entry.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry that contains the passed
         *  index, or that adjoins the gap containing the passed index.
         *
         * @param {Number} index
         *  The index of the column/row whose descriptor will be returned. Must
         *  be contained in the collection entry with the passed index, or must
         *  be contained in the gap between that collection entry and its
         *  predecessor or successor.
         *
         * @returns {Object}
         *  A descriptor object containing the following properties:
         *  - {Number} entry.index
         *      The passed column/row index.
         *  - {Number} entry.offsetHmm
         *      The absolute position of the column/row, in 1/100 mm.
         *  - {Number} entry.sizeHmm
         *      The effective size (zero for hidden columns/rows), in 1/100 mm.
         *  - {Number} entry.offset
         *      The absolute position of the column/row, in pixels.
         *  - {Number} entry.size
         *      The effective size (zero for hidden columns/rows), in pixels.
         *  - {Object} entry.attributes
         *      The merged attribute set of the column/row.
         *  - {Object} entry.explicit
         *      The explicit attribute set of the column/row.
         */
        function makeDescriptor(arrayIndex, index) {

            var // the current collection entry
                entry = null,
                // the generated entry descriptor
                entryData = null;

            // array is empty, or entry points into gap before first entry: use default size
            if ((entries.length === 0) || (arrayIndex < 0) || ((arrayIndex === 0) && (index < entries[0].first))) {
                return makeDescriptorForEntry(defaultEntry, index);
            }

            // index points into a gap between two entries, or to the gap after the last entry
            if ((arrayIndex === entries.length) || (index < entries[arrayIndex].first)) {
                entry = entries[arrayIndex - 1];
                entryData = makeDescriptorForEntry(defaultEntry, index);
                // adjust entry offsets (default entry is always relative to index 0 instead of start index of current gap)
                entryData.offsetHmm = entry.getEndOffsetHmm() + defaultEntry.sizeHmm * (index - entry.last - 1);
                entryData.offset = entry.getEndOffset() + defaultEntry.size * (index - entry.last - 1);
                return entryData;
            }

            // index points into a collection entry
            entry = entries[arrayIndex];
            return makeDescriptorForEntry(entry, index);
        }

        /**
         * Inserts the passed collection entry into the array.
         *
         * @param {Number} arrayIndex
         *  The array index for the new collection entry.
         *
         * @param {EntryModel} entry
         *  The new collection entry.
         */
        function insertEntry(arrayIndex, entry) {
            entries.splice(arrayIndex, 0, entry);
        }

        /**
         * Deletes the specified collection entries from the array.
         *
         * @param {Number} arrayIndex
         *  The array index of the first collection entry to be deleted.
         *
         * @param {Number} [count]
         *  The number of collection entries to be deleted. If omitted, all
         *  collection entries up to the end of the array will be deleted.
         */
        function deleteEntries(arrayIndex, count) {
            // call destructor for all deleted entries
            _(entries.splice(arrayIndex, count)).invoke('destroy');
        }

        /**
         * Splits the collection entry at the passed array index, and inserts
         * the new entry after the existing entry.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry to be split.
         *
         * @param {Number} leadingCount
         *  The new interval size of the existing collection entry. Must be
         *  less than the current interval size of the entry.
         *
         * @returns {EntryModel}
         *  The new collection entry inserted after the existing entry.
         */
        function splitEntry(arrayIndex, leadingCount) {

            var // the existing collection entry
                oldEntry = entries[arrayIndex],
                // the clone of the existing entry
                newEntry = EntryModel.cloneEntry(oldEntry);

            // adjust start and end position of the collection entries
            oldEntry.last = oldEntry.first + leadingCount - 1;
            newEntry.first = oldEntry.first + leadingCount;
            newEntry.offsetHmm = oldEntry.offsetHmm + leadingCount * oldEntry.sizeHmm;
            newEntry.offset = oldEntry.offset + leadingCount * oldEntry.size;

            // insert the new entry into the collection
            insertEntry(arrayIndex + 1, newEntry);
            return newEntry;
        }

        /**
         * Tries to merge the collection entry at the passed array index with
         * its predecessor and/or successor, if these entries are equal. If
         * successful, the preceding and/or following entries will be removed
         * from this collection.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry to be merged with its
         *  predecessor and /or successor. May be an invalid index (no entry
         *  exists in the collection), or may refer to the first or last entry
         *  (no predecessor or successor exists in the collection).
         *
         * @param {Object} [options]
         *  A map with options specifying which entries will be merged with the
         *  specified entry. The following options are supported:
         *  @param {Boolean} [options.prev=false]
         *      If set to true, the entry and its predecessor will be tried to
         *      merge.
         *  @param {Boolean} [options.next=false]
         *      If set to true, the entry and its successor will be tried to
         *      merge.
         *
         * @returns {Object}
         *  An object containing the Boolean flags 'prev' and 'next' specifying
         *  whether the entry has been merged with its predecessor and/or
         *  successor respectively.
         */
        function mergeEntry(arrayIndex, options) {

            var // the result object
                result = {};

            // tries to merge the passed entries, removes the preceding entry
            function tryMerge(arrayIndex) {

                var // the preceding collection entry
                    prevEntry = entries[arrayIndex - 1],
                    // the current collection entry
                    thisEntry = entries[arrayIndex];

                // check that the entries are equal and do not have a gap
                if (prevEntry && thisEntry && (prevEntry.last + 1 === thisEntry.first) && _.isEqual(prevEntry.getExplicitAttributes(), thisEntry.getExplicitAttributes())) {
                    thisEntry.first = prevEntry.first;
                    thisEntry.offsetHmm = prevEntry.offsetHmm;
                    thisEntry.offset = prevEntry.offset;
                    deleteEntries(arrayIndex - 1, 1);
                    return true;
                }
                return false;
            }

            // try to merge the entry with its successor/predecessor
            result.next = Utils.getBooleanOption(options, 'next', false) && tryMerge(arrayIndex + 1);
            result.prev = Utils.getBooleanOption(options, 'prev', false) && tryMerge(arrayIndex);
            return result;
        }

        /**
         * Invokes the passed iterator function for all collection entries that
         * cover the specified column/row interval.
         *
         * @param {Object} interval
         *  The column/row interval to be iterated, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @param {Function} iterator
         *  The iterator function called for every collection entry covered by
         *  the specified interval. Receives the following parameters:
         *  (1) {EntryModel} entry
         *      The current collection entry.
         *  (2) {Number} first
         *      The zero-based index of the first column/row. May be located
         *      inside the passed collection entry, if it is the first entry
         *      visited by this method.
         *  (3) {Number} last
         *      The zero-based index of the last column/row. May be located
         *      inside the passed collection entry, if it is the last entry
         *      visited by this method.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with additional options for this method. The following
         *  options are supported:
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the covered collection entries from last
         *      to first. Must not be used together with 'options.modify'.
         *  @param {Boolean} [options.modify=false]
         *      If set to true, the iterator function wants to modify the
         *      collection entries. Entries not yet existing (gaps between
         *      existing entries) will be created and inserted into the
         *      collection before. Entries not covered completely by the first
         *      or last index of the passed interval will be split before.
         *      After iteration is finished, adjacent entries with equal
         *      formatting attributes will be merged to one entry. Must not be
         *      used together with 'options.reverse'.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateEntries(interval, iterator, options) {

            var // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether the iterator modifies the collection entries
                modify = Utils.getBooleanOption(options, 'modify', false),
                // current column/row index for the next iterator invocation
                index = reverse ? interval.last : interval.first,
                // array index of the next collection entry to be visited
                arrayIndex = findEntryIndex(index),
                // the current collection entry in the loop
                entry = null,
                // whether to break the iteration
                breakLoop = false;

            // invokes the iterator function
            function invokeIterator(entry, other) {
                breakLoop = breakLoop || (iterator.call(self, entry, reverse ? other : index, reverse ? index : other) === Utils.BREAK);
                index = reverse ? (other - 1) : (other + 1);
            }

            // invokes the iterator function for the gap before a collection entry
            function invokeIteratorForGap(other) {

                var // the entry passed to the iterator function
                    entry = null,
                    // helper descriptor to calculate offsets
                    entryData = null;

                if (modify) {
                    // insert new entry into the collection, if iterator wants to modify the collection
                    entry = EntryModel.cloneEntry(defaultEntry);
                } else {
                    // read-only iterator: create a dummy entry that will not be inserted into the collection
                    entry = _.clone(defaultEntry);
                    delete entry.setAttributes;
                    delete entry.setCellAttributes;
                }

                // initialize interval and start offset of the new entry
                entry.first = reverse ? other : index;
                entry.last = reverse ? index : other;
                entryData = makeDescriptor(arrayIndex, entry.first);
                entry.offsetHmm = entryData.offsetHmm;
                entry.offset = entryData.offset;

                // insert new entry into array in modifying mode
                if (modify) {
                    insertEntry(arrayIndex, entry);
                    arrayIndex += 1;
                }

                // invoke the iterator
                invokeIterator(entry, other);

                // Try to merge the new entry with its predecessor (arrayIndex
                // points after the new entry, after merging, it has to be decreased).
                if (modify && mergeEntry(arrayIndex - 1, { prev: true }).prev) {
                    arrayIndex -= 1;
                }
            }

            // invokes the iterator function for an existing collection entry
            function invokeIteratorForEntry(other) {

                // invoke the iterator
                invokeIterator(entries[arrayIndex], other);

                // Try to merge the entry with its predecessor. On success, do
                // not change the array index (it already points to the next
                // entry after merging the entries).
                if (!(modify && mergeEntry(arrayIndex, { prev: true }).prev)) {
                    arrayIndex += reverse ? -1 : 1;
                }
            }

            // modify reversely not implemented
            if (reverse && modify) {
                Utils.error('ColRowCollection.iterateEntries(): modify collection reversely not implemented');
                return Utils.BREAK;
            }

            // split the first collection entry not covered completely by the interval
            if (modify && (arrayIndex < entries.length) && (entries[arrayIndex].first < index)) {
                splitEntry(arrayIndex, index - entries[arrayIndex].first);
                arrayIndex += 1;
            }

            // reverse mode: back to previous entry, if interval end points to gap before current entry
            if (reverse && ((arrayIndex === entries.length) || (interval.last < entries[arrayIndex].first))) {
                arrayIndex -= 1;
            }

            // process all existing collection entries and the gaps covered by the interval
            while (!breakLoop && (0 <= arrayIndex) && (arrayIndex < entries.length) && (reverse ? (interval.first <= entries[arrayIndex].last) : (entries[arrayIndex].first <= interval.last))) {

                // the collection entry to be visited
                entry = entries[arrayIndex];

                // visit the gap between current index and start of the next collection entry
                // (reverse mode: gap between end of the previous collection entry and current index)
                if (reverse ? (entry.last < index) : (index < entry.first)) {
                    invokeIteratorForGap(reverse ? (entry.last + 1) : (entry.first - 1));
                }

                // split the last collection entry not covered completely by the interval
                if (modify && (interval.last < entry.last)) {
                    splitEntry(arrayIndex, interval.last - index + 1);
                }

                // visit the collection entry
                invokeIteratorForEntry(reverse ? Math.max(interval.first, entry.first) : Math.min(interval.last, entry.last));
            }

            // Try to merge last visited entry with its successor (arrayIndex
            // already points to the successor of the last visited entry).
            if (modify) {
                mergeEntry(arrayIndex, { prev: true });
            }

            // visit the gap after the last existing collection entry
            // (reverse mode: before the first existing collection entry)
            if (!breakLoop && (reverse ? (interval.first <= index) : (index <= interval.last))) {
                invokeIteratorForGap(reverse ? interval.first : interval.last);
            }

            return breakLoop ? Utils.BREAK : undefined;
        }

        /**
         * Returns information about the column/row covering the passed offset
         * in the sheet.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in 1/100 mm or pixels, according
         *  to the option 'pixel' (see below).
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  @param {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a descriptor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {Object}
         *  A descriptor object with all properties described for the return
         *  value of the method ColRowCollection.getEntry(), and the additional
         *  properties 'relOffsetHmm' and 'relOffset' (relative offset inside
         *  the column/row). If the passed offset is less than 0, returns
         *  information about the first visible column/row in the sheet; if the
         *  passed offset is greater than the total size of the sheet, returns
         *  information about the last visible column/row in the sheet.
         */
        function getEntryDataByOffset(offset, options) {

            var // whether to use pixels instead of 1/100 mm
                pixel = Utils.getBooleanOption(options, 'pixel', false),
                // the property names for offset and size
                OFFSET_NAME = pixel ? 'offset' : 'offsetHmm',
                SIZE_NAME = pixel ? 'size' : 'sizeHmm',
                // the total size of the sheet
                totalSize = pixel ? self.getTotalSize() : self.getTotalSizeHmm(),
                // the passed offset, restricted to the valid sheet dimension
                currOffset = Utils.minMax(offset, 0, totalSize - 1),
                // the index of the first collection entry after the offset
                arrayIndex = _(entries).sortedIndex(Utils.makeSimpleObject(OFFSET_NAME, currOffset + 1), OFFSET_NAME),
                // the current collection entry
                entry = entries[arrayIndex - 1],
                // the end offset of the previous entry
                endOffset = !entry ? 0 : pixel ? entry.getEndOffset() : entry.getEndOffsetHmm(),
                // the column/row index, relative to the entry or the gap
                relIndex = 0,
                // the resulting entry descriptor
                entryData = null;

            // offset outside sheet limits: always return first/last collection entry if specified
            if (Utils.getBooleanOption(options, 'outerHidden', false)) {
                if (offset < 0) {
                    entryData = self.getEntry(0);
                } else if (offset >= totalSize) {
                    entryData = self.getEntry(maxIndex);
                }
                if (entryData && (entryData.size === 0)) {
                    entryData.relOffset = entryData.relOffsetHmm = 0;
                    return entryData;
                }
            }

            // offset points into the previous entry
            if (entry && (currOffset < endOffset)) {
                relIndex = Math.floor((currOffset - entry[OFFSET_NAME]) / entry[SIZE_NAME]);
                entryData = makeDescriptorForEntry(entry, entry.first + relIndex);
            } else {
                // offset points into the gap before the entry at arrayIndex
                relIndex = Math.floor((currOffset - endOffset) / defaultEntry[SIZE_NAME]);
                entryData = makeDescriptorForEntry(defaultEntry, (entry ? (entry.last + 1) : 0) + relIndex);
                // adjust offsets (default entry is relative to index 0, not to start index of current gap)
                entryData.offsetHmm = (entry ? entry.getEndOffsetHmm() : 0) + defaultEntry.sizeHmm * relIndex;
                entryData.offset = (entry ? entry.getEndOffset() : 0) + defaultEntry.size * relIndex;
            }

            // add relative offset properties
            if (pixel) {
                entryData.relOffset = Utils.minMax(offset - entryData.offset, 0, entryData.size);
                entryData.relOffsetHmm = Utils.minMax(convertPixelsToHmm(entryData.relOffset), 0, entryData.sizeHmm);
            } else {
                entryData.relOffsetHmm = Utils.minMax(offset - entryData.offsetHmm, 0, entryData.sizeHmm);
                entryData.relOffset = Utils.minMax(convertHmmToPixels(entryData.relOffsetHmm), 0, entryData.size);
            }

            return entryData;
        }

        /**
         * Updates the default column/row size according to the current
         * formatting attributes of the sheet model.
         */
        function updateDefaultSize() {

            var // the old default size, in 1/100 mm
                oldDefSizeHmm = defaultEntry.sizeHmm,
                // the old default size, in pixels
                oldDefSize = defaultEntry.size,
                // the merged sheet attributes
                defSizeValue = sheetModel.getMergedAttributes().sheet[DEFAULT_SIZE_ATTR_NAME];

            // set new default size at the default collection entry, calculate effective size
            defaultEntry.setAttributes(Utils.makeSimpleObject(FAMILY_NAME, Utils.makeSimpleObject(SIZE_ATTR_NAME, defSizeValue)));
            defaultEntry.updateSize();

            // do nothing, if the effective size has not changed (check both sizes to catch any rounding errors)
            if ((defaultEntry.sizeHmm !== oldDefSizeHmm) || (defaultEntry.size !== oldDefSize)) {
                updateEntryOffsets(0);
                self.trigger('change:entries', { first: 0, last: maxIndex });
            }
        }

        /**
         * Updates the effective offsets and sizes of all columns/rows in this
         * collection, after the zoom factor of the sheet has been changed.
         */
        function changeViewAttributesHandler(event, attributes) {
            if ('zoom' in attributes) {
                zoom = attributes.zoom;
                _(entries).invoke('updateSize');
                defaultEntry.updateSize();
                updateEntryOffsets(0);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Creates and returns a clone of this collection instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model that will contain the new cloned collection.
         *
         * @returns {ColRowCollection}
         *  A complete clone of this collection, associated to the specified
         *  sheet model.
         */
        this.clone = function (sheetModel) {
            return new ColRowCollection(app, sheetModel, columns, entries);
        };

        /**
         * Returns the maximum valid column/row index available for an entry in
         * this collection.
         *
         * @returns {Number}
         *  The maximum column/row index.
         */
        this.getMaxIndex = function () {
            return maxIndex;
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in 1/100 of millimeters, independent from the current
         * sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in 1/100 of
         *  millimeters.
         */
        this.getTotalSizeHmm = function () {
            // calculate the offset behind the last valid column/row
            return makeDescriptor(entries.length, maxIndex + 1).offsetHmm;
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in pixels according to the current sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in pixels.
         */
        this.getTotalSize = function () {
            // calculate the offset behind the last valid column/row
            return makeDescriptor(entries.length, maxIndex + 1).offset;
        };

        /**
         * Returns the merged default attribute set used for undefined columns
         * or rows.
         *
         * @returns {Object}
         *  The merged default attribute set for undefined columns or rows,
         *  containing the default column/row attributes.
         */
        this.getDefaultAttributes = function () {
            return defaultEntry.getMergedAttributes();
        };

        /**
         * Returns a descriptor object for the column/row with the specified
         * index.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object}
         *  A descriptor object for the specified column/row containing the
         *  following properties:
         *  - {Number} entry.index
         *      The passed column/row index.
         *  - {Number} entry.offsetHmm
         *      The absolute position of the column/row, in 1/100 mm.
         *  - {Number} entry.sizeHmm
         *      The effective size (zero for hidden columns/rows), in 1/100 mm.
         *  - {Number} entry.offset
         *      The absolute position of the column/row, in pixels.
         *  - {Number} entry.size
         *      The effective size (zero for hidden columns/rows), in pixels.
         *  - {Object} entry.attributes
         *      The merged attribute set of the column/row.
         *  - {Object} entry.explicit
         *      The explicit attribute set of the column/row.
         */
        this.getEntry = function (index) {
            return makeDescriptor(findEntryIndex(index), index);
        };

        /**
         * Invokes the passed iterator function for all visible columns/rows
         * covered by the passed index interval.
         *
         * @param {Object} interval
         *  The column/row interval to be iterated, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @param {Function} iterator
         *  The iterator function called for every column/row covered by the
         *  specified interval. Receives an entry descriptor object with all
         *  properties described for the return value of the method
         *  ColRowCollection.getEntry(). If the iterator returns the
         *  Utils.BREAK object, the iteration process will be stopped
         *  immediately.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the columns/rows from last to first.
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all hidden columns/rows that directly precede a
         *      visible column/row will be visited too.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleEntries = function (interval, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit hidden entries
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // the last processed collection entry
                lastEntry = null;

            // visit all collection entries covered by the passed interval
            return iterateEntries(interval, function (entry, first, last) {

                try { // to be able to use a finally block

                    // iterating forward: visit previous hidden entry, if specified
                    if (!reverse && hidden && lastEntry && (entry.size > 0) && (lastEntry.size === 0)) {
                        if (iterator.call(context, makeDescriptorForEntry(lastEntry, first - 1)) === Utils.BREAK) {
                            return Utils.BREAK;
                        }
                    }

                    // iterating backward: visit hidden entry, if specified and last entry was visible
                    if (reverse && hidden && lastEntry && (entry.size === 0) && (lastEntry.size > 0)) {
                        if (iterator.call(context, makeDescriptorForEntry(entry, last)) === Utils.BREAK) {
                            return Utils.BREAK;
                        }
                    }

                    // do not invoke iterator for hidden columns/rows
                    if (entry.size === 0) { return; }

                    // invoke iterator for every column/row covered by the current collection entry
                    return Utils.iterateRange(reverse ? last : first, reverse ? (first - 1) : (last + 1), function (index) {
                        return iterator.call(context, makeDescriptorForEntry(entry, index));
                    }, { step: reverse ? -1 : 1 });

                // always update the variable 'lastEntry', regardless where this method returns
                } finally {
                    lastEntry = entry;
                }

            }, { reverse: reverse });
        };

        /**
         * Returns the number of visible collection entries in the passed
         * interval.
         *
         * @param {Object} interval
         *  The column/row interval, with the zero-based index properties
         *  'first' and 'last'.
         *
         * @returns {Number}
         *  The number of visible collection entries in the passed interval.
         */
        this.getVisibleEntryCount = function (interval) {
            var count = 0;
            iterateEntries(interval, function (entry, first, last) {
                if (entry.size > 0) { count += (last - first + 1); }
            });
            return count;
        };

        /**
         * Returns the indexes of all visible collection entries in the passed
         * interval as plain array.
         *
         * @param {Object} interval
         *  The column/row interval to be processed, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @returns {Number[]}
         *  The indexes of all visible collection entries, in ascending order.
         */
        this.getVisibleEntryIndexes = function (interval) {
            var indexes = [];
            this.iterateVisibleEntries(interval, function (entry) {
                indexes.push(entry.index);
            });
            return indexes;
        };

        /**
         * Returns whether the column/row with the specified index is visible.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Boolean}
         *  Whether the column/row is visible.
         */
        this.isEntryVisible = function (index) {
            return this.getEntry(index).size > 0;
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in 1/100 mm, independent from the current sheet zoom
         * factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in 1/100 mm.
         */
        this.getEntryOffsetHmm = function (index, offsetHmm) {
            var entryData = this.getEntry(index);
            return entryData.offsetHmm + Utils.minMax(offsetHmm, 0, entryData.sizeHmm);
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in pixels, according to the current sheet zoom factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in pixels.
         */
        this.getEntryOffset = function (index, offsetHmm) {
            var entryData = this.getEntry(index);
            return entryData.offset + Utils.minMax(convertHmmToPixels(offsetHmm), 0, entryData.size);
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the first visible column/row that
         * follows the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Object|Null}
         *  A descriptor object with all properties described for the return
         *  value of the method ColRowCollection.getEntry(); or null, if no
         *  more visible columns/rows are available.
         */
        this.getNextVisibleEntry = function (index) {

            var // the resulting entry data
                entryData = null;

            // use iterator to visit the first visible entry following the passed index
            this.iterateVisibleEntries({ first: index, last: maxIndex }, function (entry) {
                entryData = entry;
                return Utils.BREAK;
            });

            return entryData;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the first visible column/row that
         * precedes the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Object|Null}
         *  A descriptor object with all properties described for the return
         *  value of the method ColRowCollection.getEntry(); or null, if no
         *  more visible columns/rows are available.
         */
        this.getPrevVisibleEntry = function (index) {

            var // the resulting entry data
                entryData = null;

            // use iterator to visit the first visible entry following the passed index
            this.iterateVisibleEntries({ first: 0, last: index }, function (entry) {
                entryData = entry;
                return Utils.BREAK;
            }, { reverse: true });

            return entryData;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; optionally about a visible column/row located near
         * the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {String} [method='exact']
         *  Specifies how to look for another column/row, if the specified
         *  column/row is hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified
         *      column/row is hidden, returns null.
         *  - 'next':
         *      Looks for a visible column/row following the passed index.
         *  - 'prev':
         *      Looks for a visible column/row preceding the passed index.
         *  - 'nextPrev':
         *      First, looks for visible column/row following the passed index.
         *      If there is none available, looks for a visible column/row
         *      preceding the passed index.
         *  - 'prevNext':
         *      First, looks for visible column/row preceding the passed index.
         *      If there is none available, looks for a visible column/row
         *      following the passed index.
         *
         * @param {Object} [boundInterval]
         *  If specified, a bounding index interval to restrict the result to.
         *  If no visible entry could be found inside that interval, null will
         *  be returned.
         *
         * @returns {Object|Null}
         *  A descriptor object with all properties described for the return
         *  value of the method ColRowCollection.getEntry(); or null, if no
         *  visible columns/rows are available.
         */
        this.getVisibleEntry = function (index, method, boundInterval) {

            var // the resulting column/row descriptor
                entryData = null;

            function isValidEntryData() {
                return _.isObject(entryData) && (entryData.size > 0) && (!_.isObject(boundInterval) || SheetUtils.intervalContainsIndex(boundInterval, entryData.index));
            }

            switch (method) {
            case 'next':
                entryData = this.getNextVisibleEntry(index);
                break;
            case 'prev':
                entryData = this.getPrevVisibleEntry(index);
                break;
            case 'nextPrev':
                entryData = this.getNextVisibleEntry(index);
                if (!isValidEntryData()) { entryData = this.getPrevVisibleEntry(index); }
                break;
            case 'prevNext':
                entryData = this.getPrevVisibleEntry(index);
                if (!isValidEntryData()) { entryData = this.getNextVisibleEntry(index); }
                break;
            default:
                entryData = this.getEntry(index);
            }

            return isValidEntryData() ? entryData : null;
        };

        /**
         * Returns whether all columns/rows in the passed interval are hidden.
         *
         * @param {Object} interval
         *  The index interval, with zero-based properties 'first' and 'last'.
         *
         * @returns {Boolean}
         *  Whether all columns/rows in the passed interval are hidden.
         */
        this.isHiddenInterval = function (interval) {
            return this.getIntervalPosition(interval).size === 0;
        };

        /**
         * Reduces the passed column/row interval to an interval starting and
         * ending at visible columns/rows.
         *
         * @param {Object} interval
         *  The index interval, with zero-based properties 'first' and 'last'.
         *
         * @returns {Object|Null}
         *  An index interval with 'start' and 'end' properties containing the
         *  indexes of the nearest columns/rows inside the passed interval, if
         *  available. Otherwise, null will be returned.
         */
        this.getVisibleInterval = function (interval) {

            var // find the first and last available entry in the passed interval
                firstEntryData = this.getNextVisibleEntry(interval.first),
                lastEntryData = this.getPrevVisibleEntry(interval.last);

            return firstEntryData && lastEntryData && (firstEntryData.index <= lastEntryData.index) ? { first: firstEntryData.index, last: lastEntryData.index } : null;
        };

        /**
         * Returns information about the column/row covering the passed offset
         * in the sheet in pixels according to the current sheet zoom factor,
         * or in 1/100 of millimeters.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  @param {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a descriptor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {Object}
         *  A descriptor object with all properties described for the return
         *  value of the method ColRowCollection.getEntry(), and the additional
         *  properties 'relOffsetHmm' and 'relOffset' (relative offset inside
         *  the column/row). If the passed offset is less than 0, returns
         *  information about the first visible column/row in the sheet; if the
         *  passed offset is greater than the total size of the sheet, returns
         *  information about the last visible column/row in the sheet (unless
         *  the option 'outerHidden' has been set, see above).
         */
        this.getEntryByOffset = function (offset, options) {
            return getEntryDataByOffset(offset, options);
        };

        /**
         * Converts the passed offset to a scroll anchor position which
         * contains the entry index, and a floating-point ratio describing the
         * exact position inside the entry.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  @param {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a scroll anchor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {Object}
         *  A scroll anchor object with the following properties:
         *  - {Number} index
         *      The zero-based column/row index of the collection entry that
         *      contains the passed offset.
         *  - {Number} ratio
         *      The floating-point ratio inside the collection entry, inside
         *      the interval [0,1).
         */
        this.getScrollAnchorByOffset = function (offset, options) {
            var entryData = this.getEntryByOffset(offset, options);
            return {
                index: entryData.index,
                ratio: (entryData.sizeHmm > 0) ? (entryData.relOffsetHmm / entryData.sizeHmm) : 0
            };
        };

        /**
         * Converts the passed absolute offset in the sheet given in pixels
         * (according to the current sheet zoom factor) to an absolute offset
         * in 1/100 mm (independent from the current sheet zoom factor).
         *
         * @param {Number} offset
         *  An absolute offset in the sheet in pixels (according to the current
         *  sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         */
        this.convertOffsetToHmm = function (offset) {
            var entryData = getEntryDataByOffset(offset, { pixel: true });
            return entryData.offsetHmm + entryData.relOffsetHmm;
        };

        /**
         * Converts the passed absolute offset in the sheet given in 1/100 mm
         * (independent from the current sheet zoom factor) to an absolute
         * offset in pixels (according to the current sheet zoom factor).
         *
         * @param {Number} offsetHmm
         *  An absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in pixels (according to the
         *  current sheet zoom factor).
         */
        this.convertOffsetToPixel = function (offsetHmm) {
            var entryData = getEntryDataByOffset(offsetHmm);
            return entryData.offset + entryData.relOffset;
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in 1/100
         * of millimeters.
         *
         * @param {Object} scrollAnchor
         *  A valid scroll anchor value, with the numeric properties 'index'
         *  and 'ratio'.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in 1/100 mm.
         */
        this.convertScrollAnchorToHmm = function (scrollAnchor) {
            var entryData = this.getEntry(scrollAnchor.index);
            return entryData.offsetHmm + Math.round(entryData.sizeHmm * scrollAnchor.ratio);
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in
         * pixels according to the current sheet zoom factor.
         *
         * @param {Object} scrollAnchor
         *  A valid scroll anchor value, with the numeric properties 'index'
         *  and 'ratio'.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in pixels.
         */
        this.convertScrollAnchorToPixel = function (scrollAnchor) {
            var entryData = this.getEntry(scrollAnchor.index);
            return entryData.offset + Math.round(entryData.size * scrollAnchor.ratio);
        };

        /**
         * Returns the position and size of the specified column/row interval.
         *
         * @param {Object} interval
         *  The index interval of the columns/rows, with the zero-based
         *  properties 'first' and 'last'.
         *
         * @returns {Object}
         *  An object with the following properties:
         *  - {Number} position.offsetHmm
         *      The absolute position of the first column/row in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} position.sizeHmm
         *      The total size of the interval in 1/100 mm, independent form
         *      the current sheet zoom factor.
         *  - {Number} position.offset
         *      The absolute position of the first column/row in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} position.size
         *      The total size of the interval in pixels, according to the
         *      current sheet zoom factor.
         */
        this.getIntervalPosition = function (interval) {

            var // descriptor for the first column/row of the interval
                firstEntryData = this.getEntry(interval.first),
                // descriptor for the first column/row following the interval
                lastEntryData = this.getEntry(interval.last + 1);

            return {
                offsetHmm: firstEntryData.offsetHmm,
                sizeHmm: lastEntryData.offsetHmm - firstEntryData.offsetHmm,
                offset: firstEntryData.offset,
                size: lastEntryData.offset - firstEntryData.offset
            };
        };

        /**
         * Changes the formatting attributes of all entries contained in the
         * passed index interval.
         *
         * @param {Object} interval
         *  The index interval of the collection entries to be changed, in the
         *  zero-based index properties 'first' and 'last'.
         *
         * @param {Object} attributes
         *  The new attributes for the collection entries.
         *
         * @param {Object} [options]
         *  Additional options for border attributes. The following options are
         *  supported:
         *  @param {Boolean} [options.rangeBorders=false]
         *      If set to true, attributes for outer borders will be applied at
         *      the outer borders of the interval only, and attributes for
         *      inner borders will be applied inside the interval. Otherwise,
         *      the outer border attributes will be applied for all borders,
         *      and the inner border attributes will be ignored.
         *  @param {Boolean} [options.visibleBorders=false]
         *      If set to true, border attributes may be incomplete. Existing
         *      border properties will be applied for visible borders in the
         *      interval only.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.setAttributes = function (interval, attributes, options) {

            var // whether any collection entry has been changed
                changed = false,
                // index of the first column/row with dirty pixel offset due to changed sizes
                dirtyIndex = null,
                // range border mode (special handling for inner border attributes)
                rangeBorders = Utils.getBooleanOption(options, 'rangeBorders', false) && _.isObject(attributes.cell),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false);

            // update all entries in the passed interval, move border attributes accordingly
            function updateInterval(interval, innerToLeading, innerToTrailing) {

                var // options for special inner/outer border treatment
                    attributeOptions = { visibleBorders: visibleBorders };

                // add options for inner/outer border treatment
                if (rangeBorders) {
                    attributeOptions.innerLeft = !columns || innerToLeading;
                    attributeOptions.innerRight = !columns || innerToTrailing;
                    attributeOptions.innerTop = columns || innerToLeading;
                    attributeOptions.innerBottom = columns || innerToTrailing;
                }

                // update all entries in the interval
                iterateEntries(interval, function (entry) {

                    var // old size of the entry, in 1/100 mm and in pixels
                        oldSizeHmm = entry.sizeHmm,
                        oldSize = entry.size;

                    // set the passed attributes; post-processing, if any attributes have changed
                    if (entry.setCellAttributes(attributes, attributeOptions)) {
                        changed = true;
                        // update effective size of the collection entry
                        entry.updateSize();
                        // set index of entries with dirty pixel offsets, if size has been changed
                        if (((oldSizeHmm !== entry.sizeHmm) || (oldSize !== entry.size)) && _.isNull(dirtyIndex)) {
                            dirtyIndex = entry.last + 1;
                        }
                    }
                }, { modify: true });
            }

            // update all affected collection entries
            if (rangeBorders && (interval.first < interval.last)) {
                updateInterval({ first: interval.first, last: interval.first }, false, true);
                updateInterval({ first: interval.first + 1, last: interval.last - 1 }, true, true);
                updateInterval({ first: interval.last, last: interval.last }, true, false);
            } else {
                updateInterval(interval, false, false);
            }

            // update pixel offsets of following entries, if the size has been changed
            if (_.isNumber(dirtyIndex)) {
                updateEntryOffsets(dirtyIndex);
            }

            // notify change listeners
            if (changed) {
                this.trigger('change:entries', interval, attributes, options);
            }
            return this;
        };

        /**
         * Inserts new entries (columns or rows) into this collection.
         *
         * @param {Object} interval
         *  The index interval, with the zero-based index properties 'first'
         *  and 'last'. New entries will be inserted before the entry specified
         *  by the property 'first'. The size of the interval specifies the
         *  number of new entries that will be inserted. The new entries will
         *  contain the formatting attributes of the predecessor of the first
         *  existing entry, except that the new entries will always be visible.
         *  If the new entries will be inserted at the beginning of this
         *  collection, they will contain default formatting.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.insertEntries = function (interval) {

            var // the array index of the first entry to be moved
                arrayIndex = findEntryIndex(interval.first),
                // the current collection entry
                entry = entries[arrayIndex],
                // the previous entry used to copy formatting attributes
                prevEntry = null,
                // the explicit attributes to be used for the new entries
                prevAttributes = null,
                // the number of columns/rows to be inserted
                delta = SheetUtils.getIntervalSize(interval);

            // split collection entry if it is not covered completely
            if (entry && (entry.first < interval.first)) {
                splitEntry(arrayIndex, interval.first - entry.first);
                prevEntry = entry;
                arrayIndex += 1;
            } else if ((arrayIndex > 0) && (entries[arrayIndex - 1].last + 1 === interval.first)) {
                // use formatting of previous entry, if it ends exactly before the interval
                prevEntry = entries[arrayIndex - 1];
            }

            // move existing collection entries
            for (var index = arrayIndex; index < entries.length; index += 1) {

                // update index interval of the entry
                entry = entries[index];
                entry.first += delta;
                entry.last += delta;

                // delete following entries moved outside the collection limits
                if (entry.first > maxIndex) {
                    deleteEntries(arrayIndex);
                } else if (entry.last >= maxIndex) {
                    deleteEntries(arrayIndex + 1);
                    entry.last = maxIndex;
                }
            }

            // do not create a new entry, if the preceding column/row has default format
            if (prevEntry) {

                // prepare the explicit attributes to be used for the new entries (always be visible)
                prevAttributes = prevEntry.getExplicitAttributes();
                if ((FAMILY_NAME in prevAttributes) && (prevAttributes[FAMILY_NAME].visible === false)) {
                    delete prevAttributes[FAMILY_NAME].visible;
                }

                // insert a new collection entry, try to merge with adjacent entries
                insertEntry(arrayIndex, new EntryModel(interval.first, interval.last, prevAttributes));
                mergeEntry(arrayIndex, { prev: true, next: true });
            }

            // update pixel offsets of the entries
            updateEntryOffsets(interval.first);

            // notify insert listeners
            this.trigger('insert:entries', interval);
            return this;
        };

        /**
         * Removes existing entries (columns or rows) from this collection.
         *
         * @param {Object} interval
         *  The index interval of the entries to be deleted, in the zero-based
         *  index properties 'first' and 'last'.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.deleteEntries = function (interval) {

            var // the array index of the first entry to be deleted
                arrayIndex = findEntryIndex(interval.first),
                // the current collection entry
                entry = entries[arrayIndex],
                // the number of elements to be deleted from the array
                deleteCount = 0,
                // the number of columns/rows to be removed
                delta = SheetUtils.getIntervalSize(interval);

            // split the part of the entry located before the interval
            if (entry && (entry.first < interval.first)) {
                entry = splitEntry(arrayIndex, interval.first - entry.first);
                arrayIndex += 1;
            }

            // remove and move the remaining entries
            for (var index = arrayIndex; index < entries.length; index += 1) {

                // update number of entries to be deleted, update index interval of moved entries
                entry = entries[index];
                if (entry.last <= interval.last) {
                    deleteCount += 1;
                } else {
                    entry.first = Math.max(interval.first, entry.first - delta);
                    entry.last -= delta;
                }
            }

            // delete the array elements
            if (deleteCount > 0) {
                deleteEntries(arrayIndex, deleteCount);
            }

            // try to merge with previous entry
            mergeEntry(arrayIndex, { prev: true });

            // update pixel offsets of the entries
            updateEntryOffsets(interval.first);

            // notify delete listeners
            this.trigger('delete:entries', interval);
            return this;
        };

        // initialization -----------------------------------------------------

        // update entry sizes, after the default column/row size defined by the sheet has been changed
        sheetModel.on({
            'change:attributes': updateDefaultSize,
            'change:viewattributes': changeViewAttributesHandler
        });
        updateDefaultSize();

        // update entry sizes after the document has been loaded
        if (!app.isImportFinished()) {
            this.listenTo(app, 'docs:import:success', updateDefaultSize);
        }

        // clone collection entries passed as hidden argument to the c'tor (used by the clone() method)
        if (_.isArray(arguments[ColRowCollection.length])) {
            entries = _(arguments[ColRowCollection.length]).map(EntryModel.cloneEntry);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(entries).invoke('destroy');
            defaultEntry.destroy();
            model = sheetModel = entries = defaultEntry = null;
        });

    } // class ColRowCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ColRowCollection });

});
