/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellstylesheets',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/stylesheets'
    ], function (Utils, Color, Border, StyleSheets) {

    'use strict';

    // class CellStyleSheets ==================================================

    /**
     * Contains the cell style sheets of a spreadsheet document.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {SpreadsheetDocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function CellStyleSheets(app, documentStyles) {

        var // self reference
            self = this,

            // the identifier of the built-in 'Hyperlink' style
            hyperlinkStyleId = null;

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'cell', { additionalFamilies: 'character' });

        // private methods ----------------------------------------------------

        /**
         * Creates a new style sheet, if not existing yet. Updates the category
         * and priority of existing and created style sheets.
         */
        function createStyleSheet(name, category, priority, attributes) {

            var // identifier of an existing style sheet with the passed name
                styleId = self.getStyleIdByName(name);

            if (_.isString(styleId)) {
                // update category and priority in existing style sheets
                self.setUICategory(styleId, category);
                self.setUIPriority(styleId, priority);
            } else {
                // create a new (dirty) style sheet, use name as identifier
                self.insertStyleSheet(name, name, null, attributes, {
                    category: category,
                    priority: priority,
                    hidden: category === 'hidden',
                    dirty: true
                });
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Create all missing built-in cell style sheets.
         *
         * @internal
         *  Not intended to be called from any other code than the post-process
         *  handler of the application.
         *
         * @returns {CellStyleSheets}
         *  A reference to this instance.
         */
        this.createMissingStyleSheets = function () {

            function RGB(value)           { return { type: 'rgb',    value: value }; }
            function SCHEME(value)        { return { type: 'scheme', value: value }; }
            function SCHEME_TINT20(value) { return { type: 'scheme', value: value, transformations: [{ type: 'tint', value: 20002 }] }; }
            function SCHEME_TINT40(value) { return { type: 'scheme', value: value, transformations: [{ type: 'tint', value: 40001 }] }; }
            function SCHEME_TINT60(value) { return { type: 'scheme', value: value, transformations: [{ type: 'tint', value: 60003 }] }; }

            var // the identifier of the default cell style
                defId = this.getDefaultStyleId(),
                // the merged attribute set of the default style
                attributes = this.getStyleSheetAttributes(defId),
                // the default font name of the document
                fontName = attributes.character.fontName,

                // predefined border values
                LIGHT_GRAY_BORDER  = { style: 'single', width: 26, color: RGB('B2B2B2') },
                MED_GRAY_BORDER    = { style: 'single', width: 26, color: RGB('7F7F7F') },
                DARK_GRAY_BORDER   = { style: 'single', width: 26, color: RGB('3F3F3F') },
                DOUBLE_GRAY_BORDER = { style: 'double', width: 79, color: RGB('3F3F3F') };

            // standard styles
            createStyleSheet('Standard', 'markup', 1, {});
            createStyleSheet('Positive', 'markup', 2, { character: { fontName: fontName, color: RGB('006100') }, cell: { fillColor: RGB('C6EFCE') } });
            createStyleSheet('Neutral',  'markup', 3, { character: { fontName: fontName, color: RGB('9C6500') }, cell: { fillColor: RGB('FFEB9C') } });
            createStyleSheet('Negative', 'markup', 4, { character: { fontName: fontName, color: RGB('9C0006') }, cell: { fillColor: RGB('FFC7CE') } });
            createStyleSheet('Note',     'markup', 5, { character: { fontName: fontName, color: Color.BLACK }, cell: { fillColor: RGB('FFFFCC'), borderTop: LIGHT_GRAY_BORDER, borderBottom: LIGHT_GRAY_BORDER, borderLeft: LIGHT_GRAY_BORDER, borderRight: LIGHT_GRAY_BORDER } });

            // heading styles
            createStyleSheet('Title',     'headings', 1, { character: { bold: true, fontName: fontName, fontSize: 18, color: SCHEME('dark2') } });
            createStyleSheet('Heading 1', 'headings', 2, { character: { bold: true, fontName: fontName, fontSize: 15, color: SCHEME('dark2') }, cell: { borderBottom: { style: 'single', width: 79, color: SCHEME('accent1') } } });
            createStyleSheet('Heading 2', 'headings', 3, { character: { bold: true, fontName: fontName, fontSize: 13, color: SCHEME('dark2') }, cell: { borderBottom: { style: 'single', width: 79, color: SCHEME_TINT40('accent1') } } });
            createStyleSheet('Heading 3', 'headings', 4, { character: { bold: true, fontName: fontName, fontSize: 11, color: SCHEME('dark2') }, cell: { borderBottom: { style: 'single', width: 53, color: SCHEME_TINT40('accent1') } } });
            createStyleSheet('Heading 4', 'headings', 5, { character: { bold: true, fontName: fontName, fontSize: 11, color: SCHEME('dark2') } });
            createStyleSheet('Total',     'headings', 9, { character: { bold: true, fontName: fontName, fontSize: 11, color: SCHEME('dark1') }, cell: { borderTop: { style: 'single', width: 26, color: SCHEME('accent1') }, borderBottom: { style: 'double', width: 79, color: SCHEME('accent1') } } });

            // other content styles
            createStyleSheet('Output',           'hidden', 1, { character: { fontName: fontName, color: RGB('3F3F3F'), bold: true }, cell: { fillColor: RGB('F2F2F2'), borderTop: DARK_GRAY_BORDER, borderBottom: DARK_GRAY_BORDER, borderLeft: DARK_GRAY_BORDER, borderRight: DARK_GRAY_BORDER } });
            createStyleSheet('Calculation',      'hidden', 2, { character: { fontName: fontName, color: RGB('FA7D00'), bold: true }, cell: { fillColor: RGB('F2F2F2'), borderTop: MED_GRAY_BORDER,  borderBottom: MED_GRAY_BORDER,  borderLeft: MED_GRAY_BORDER,  borderRight: MED_GRAY_BORDER  } });
            createStyleSheet('Input',            'hidden', 3, { character: { fontName: fontName, color: RGB('3F3F76') },             cell: { fillColor: RGB('FFCC99'), borderTop: MED_GRAY_BORDER,  borderBottom: MED_GRAY_BORDER,  borderLeft: MED_GRAY_BORDER,  borderRight: MED_GRAY_BORDER  } });
            createStyleSheet('Explanatory Text', 'hidden', 4, { character: { fontName: fontName, color: RGB('7F7F7F'), italic: true } });
            createStyleSheet('Linked Cell',      'hidden', 5, { character: { fontName: fontName, color: RGB('FA7D00') }, cell: { borderBottom: { style: 'double', width: 17, color: RGB('FA7D00') } } });
            createStyleSheet('Warning Text',     'hidden', 6, { character: { fontName: fontName, color: RGB('FF0000') } });
            createStyleSheet('Check Cell',       'hidden', 7, { character: { fontName: fontName, color: SCHEME('light1'), bold: true }, cell: { fillColor: RGB('A5A5A5'), borderTop: DOUBLE_GRAY_BORDER, borderBottom: DOUBLE_GRAY_BORDER, borderLeft: DOUBLE_GRAY_BORDER, borderRight: DOUBLE_GRAY_BORDER } });

            // accent styles
            _(6).times(function (index) {
                var baseName = 'Accent ' + (index + 1),
                    accentName = 'accent' + (index + 1);
                createStyleSheet('20% - ' + baseName, 'themes', index,      { character: { fontName: fontName, color: SCHEME('dark1') },  cell: { fillColor: SCHEME_TINT20(accentName) } });
                createStyleSheet('40% - ' + baseName, 'themes', index + 6,  { character: { fontName: fontName, color: SCHEME('dark1') },  cell: { fillColor: SCHEME_TINT40(accentName) } });
                createStyleSheet('60% - ' + baseName, 'themes', index + 12, { character: { fontName: fontName, color: SCHEME('light1') }, cell: { fillColor: SCHEME_TINT60(accentName) } });
                createStyleSheet(baseName,            'themes', index + 18, { character: { fontName: fontName, color: SCHEME('light1') }, cell: { fillColor: SCHEME(accentName) } });
            });

            // hyperlink style
            createStyleSheet('Hyperlink', 'hidden', 99, { character: { fontName: fontName, color: SCHEME('hyperlink'), underline: true } });
            hyperlinkStyleId = self.getStyleIdByName('Hyperlink');

            return this;
        };

        /**
         * Returns the identifier of the built-in 'Hyperlink' cell style sheet.
         *
         * @returns {String}
         *  The identifier of the built-in 'Hyperlink' cell style sheet.
         */
        this.getHyperlinkStyleId = function () {
            return hyperlinkStyleId;
        };

    } // class CellStyleSheets

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: CellStyleSheets });

});
