/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/drawingcontroller',
    ['io.ox/office/tk/config',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/chartcreator',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/spreadsheet'
    ], function (Config, Operations, Controls, ChartCreator, Label, Utils, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        TextField = Controls.TextField;

    // class DrawingController =================================================

    /**
     * DrawingController (will) encapsulate(s) all Drawing-Features of the Controller and of the View (Sidepane-toolbox)
     * For now Chart are part of Drawings!
     */
    function DrawingController(app) {

        var self = this,
            model = null,
            view = null;

        // private methods ----------------------------------------------------

        /**
         * initToolBox initiates the view components
         * -delete Drawing
         * -change Chart Type
         */
        function initToolBox() {

            var drawingbox = view.createToolBox('drawing', { label: /*#. drawing objects: images, diagrams, ... */ gt('Drawing'), visible: 'document/editable/drawing' });
            drawingbox.addRightTab();
            drawingbox.addGroup('drawing/delete', new Button(Controls.DELETE_DRAWING_OPTIONS));

            if (Config.isDebug()) {

                var chartbox = view.createToolBox('chart', { label: /*#. drawing objects: images, diagrams, ... */ gt('Chart'), visible: 'drawing/chart' });

                chartbox.addGroup('drawing/charttype', new Controls.ChartTypePicker());

                addLabeled(chartbox, 'drawing/chartnames', gt('Names'));
                addLabeled(chartbox, 'drawing/charttitles', gt('Titles'));
                addLabeled(chartbox, 'drawing/chartvalues', gt('Values'));
                addLabeled(chartbox, 'drawing/chartbubbles', gt('Sizes'), 'drawing/chartBubbles');
            }
        }

        function addLabeled(toolBox, key, label, visible) {
            toolBox.newLine();
            toolBox.addPrivateGroup(new Label({ tooltip: label, label: label, width: 60, classes: 'no-borders' }));
            toolBox.addGap(1);
            toolBox.addGroup(key, new TextField({ tooltip: label, select: true, width: 160, visible: visible }));
        }

        /**
         * changeSelection
         * if only one Drawing is selected and this is a Chart, the function highlights its source-cells via TokenArray
         */
        var changeSelection = (function () {

            var // the token arrays containing the highlighted ranges of the selected drawing object
                tokenArrays = [];

            return function (evt, selection) {

                if (tokenArrays.length > 0) {
                    view.endRangeHighlighting();
                    _(tokenArrays).invoke('destroy');
                    tokenArrays = [];
                }

                if (selection.drawings.length === 1) {

                    var drawingCollection = view.getActiveSheetModel().getDrawingCollection(),
                        chartModel = drawingCollection.findModel(selection.drawings[0], { type: 'chart' });

                    if (chartModel) {
                        tokenArrays.push(chartModel.getTokenArray());
                    }
                }

                if (tokenArrays.length > 0) {
                    view.startRangeHighlighting(tokenArrays/*, {draggable: true}*/);
                }
            };
        }());

        /**
         * @return the ID of the current selected Drawing
         */
        function getDrawingIndices() {
            var selection = view.getSelection();
            if (selection.drawings.length == 1) {
                return selection.drawings[0];
            }
        }

        /**
         * @return the current selected Drawing
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingIndices();
            if (drawingIndices) {
                var drawingCollection = view.getActiveSheetModel().getDrawingCollection();
                return drawingCollection.findModel(drawingIndices, { type: type });
            }
        }

        /**
         * setDrawingAttributes generates the Operation for changing Attributes of Drawings and Charts
         */
        function setDrawingAttributes(attrs) {
            var generator = model.createOperationsGenerator();
            var position = getDrawingIndices();
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, view.getActiveSheet(), position, { attrs: attrs });
            model.applyOperations(generator.getOperations());
        }

        /**
         * generates an insert-Chart-Op and Insert-Chart-DataSeries chosen by
         * current Selection
         *
         * @returns {jQuery.Deferred}
         *  a deferred object, which is already resolved
         */
        function insertChart() {
            return new ChartCreator(app).deferred();
        }

        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['image/insert/dialog'] = {
                parent: 'document/editable',
                set: function () {
                    return view.showInsertImageDialog();
                },
                // allow insertion of drawings only if sheet is not protected
                enable: function () {
                    return !view.getActiveSheetModel().isProtected();
                },
                focus: 'wait' // wait for the dialog before returning focus to application
            };

            defs['chart/insert'] = {
                parent: 'document/editable',
                set: insertChart
            };

            defs['document/editable/drawing'] = {
                parent: 'document/editable',
                enable: function () {
                    return view.hasDrawingSelection() && !view.getActiveSheetModel().isProtected();
                }
            };

            defs.drawing = {
                parent: 'document/editable/drawing',
                get: getDrawingModel
            };

            defs['drawing/delete'] = {
                parent: 'document/editable/drawing',
                set: function () {
                    view.deleteDrawings();
                }
            };

            defs['drawing/chart'] = {
                parent: 'document/editable/drawing',
                enable: function () {
                    return _.isObject(this.getValue());
                },
                get: function () {
                    return getDrawingModel('chart');
                }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) {
                    return chartModel ? chartModel.getChartType() : null;
                },
                set: function (type) {
                    var types = type.split(' ');
                    var change = {};
                    change.type = types[0] + '2d';
                    if (types.length > 1) {
                        change.chartGroup = types[1];
                    } else {
                        change.chartGroup = '';
                    }
                    setDrawingAttributes({ chart: change });
                }
            };

            addChartDef('names');
            addChartDef('titles');
            addChartDef('values');
            addChartDef('bubbles', true);

            function addChartDef(name, enable) {
                var fnName = 'get' + Utils.capitalize(name) + 'Token';
                var def = {
                    parent: 'drawing/chart',
                    get: function (chartModel) {
                        return chartModel ? chartModel[fnName]() : null;
                    }
//                        ,
//                                set: function (type) {
//                                    setDrawingAttributes({ chart: { type: type + '2d' } });
//                                }
                };

                var enableName = 'has' + Utils.capitalize(name);
                def.enable = function () {
                    var chartModel = getDrawingModel('chart');
                    if (chartModel) {
                        if (enable) {
                            return chartModel[enableName]();
                        } else {
                            return true;
                        }
                    } else {
                        return false;
                    }
                };
                defs['drawing/chart' + name] = def;
            }

            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        //View and Listener initialization
        app.on('docs:init', function () {
            model = app.getModel();
            view = app.getView();

            initToolBox();
            view.on('change:selection', changeSelection);
        });

    } // class DrawingController

    // exports ================================================================

    return DrawingController;

});
