/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/model',
     'io.ox/office/spreadsheet/view/view',
     'io.ox/office/spreadsheet/app/controller',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, IO, EditApplication, SheetUtils, SpreadsheetModel, SpreadsheetView, SpreadsheetController, gt) {

    'use strict';

    var // default settings for the localeData object
        DEFAULT_LOCALE_DATA = {
            functions: {},
            errorCodes: { '#DIV/0!': '#DIV/0!', '#N/A': '#N/A', '#NAME?': '#NAME?', '#NULL!': '#NULL!', '#NUM!': '#NUM!', '#REF!': '#REF!', '#VALUE!': '#VALUE!' },
            decSeparator: '.',
            listSeparator: ',',
            trueLiteral: 'TRUE',
            falseLiteral: 'FALSE'
        };

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} [appOptions]
     *  A map of static application options, that have been passed to the
     *  static method BaseApplication.createLauncher().
     *
     * @param {Object} [launchOptions]
     *  A map of options to control the properties of the application. Supports
     *  all options supported by the base class EditApplication.
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (appOptions, launchOptions) {

        var // self reference
            self = this,

            // the locale data (translated function names etc.)
            localeData = _.copy(DEFAULT_LOCALE_DATA, true),

            // the function help resource
            functionHelp = {};

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, appOptions, launchOptions, {
            newDocumentParams: { initial_sheetname: SheetUtils.generateSheetName(1) },
            preProcessHandler: preProcessDocument,
            preProcessProgressSize: 0.1,
            postProcessHandler: postProcessDocument,
            postProcessProgressSize: 0,
            importFailedHandler: prepareInvalidDocument,
            flushHandler: flushHandler,
            realTimeDelay: 10
        });

        // private methods ----------------------------------------------------

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been prepared successfully; or rejected on any error.
         */
        function preProcessDocument() {

            var // load function help data asynchronously
                functionHelpPromise = IO.loadResource('io.ox/office/spreadsheet/resource/functions');

            // store function help data, but ignore the success/failure result
            // (application may run even without function help)
            functionHelpPromise.done(function (data) { functionHelp = data; });

            // load the locale data for the current UI language from the server
            return self.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'query',
                    requestdata: JSON.stringify({
                        type: 'localeData',
                        locale: Utils.LOCALE
                    })
                }
            })
            .then(function (result) {

                // extract locale data settings from response data object
                _(localeData.functions).extend(Utils.getObjectOption(result, 'functions'));
                _(localeData.errorCodes).extend(Utils.getObjectOption(result, 'errorCodes'));
                localeData.decSeparator = Utils.getStringOption(result, 'decSeparator', localeData.decSeparator)[0];
                localeData.listSeparator = Utils.getStringOption(result, 'listSeparator', (localeData.decSeparator === ',') ? ';' : ',');
                localeData.trueLiteral = Utils.getStringOption(result, 'trueLiteral', localeData.trueLiteral).toUpperCase();
                localeData.falseLiteral = Utils.getStringOption(result, 'falseLiteral', localeData.falseLiteral).toUpperCase();
                localeData.engineVersion = Utils.getStringOption(result, 'engineVersion');
                Utils.info('SpreadsheetApplication.preProcessDocument(): new locale data:', localeData);

                // check that the translated function names exist in the function help
                functionHelpPromise.done(function (data) {
                    var supportedFuncs = _(localeData.functions).values(),
                        annotatedFuncs = _(data).keys(),
                        missingFuncs = _(supportedFuncs).difference(annotatedFuncs),
                        unknownFuncs = _(annotatedFuncs).difference(supportedFuncs);
                    if (missingFuncs.length > 0) {
                        missingFuncs.sort();
                        Utils.error('SpreadsheetApplication.preProcessDocument(): missing help texts for functions:', missingFuncs);
                    }
                    if (unknownFuncs.length > 0) {
                        unknownFuncs.sort();
                        Utils.warn('SpreadsheetApplication.preProcessDocument(): help available for unknown functions:', unknownFuncs);
                    }
                });

            }, function () {
                Utils.error('SpreadsheetApplication.preProcessDocument(): could not download locale data for "' + Utils.LOCALE + '"');
                return $.Deferred().reject({ headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') });
            });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {
            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            return (self.getModel().getSheetCount() === 0) ? $.Deferred().reject() : self.getModel().postProcessDocument();
        }

        /**
         * Will be called when importing the document fails for any reason.
         */
        function prepareInvalidDocument() {
            // document must contain at least one sheet, to prevent if-else
            // statements for empty documents at thousand places in the view code
            self.getModel().prepareInvalidDocument();
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, or sending as mail.
         */
        function flushHandler() {
            return self.getView().prepareFlush();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the sorted list with the names of all supported functions
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported functions.
         */
        this.getFunctionNames = function () {
            return _.chain(localeData.functions).values().sort().value();
        };

        /**
         * Returns the name of the specified function, translated for the
         * current UI language.
         *
         * @param {String} englishFuncName
         *  The English name of the function.
         *
         * @returns {String}
         *  The translated name of the specified function if available,
         *  otherwise the passed function name.
         */
        this.getFunctionName = function (englishFuncName) {
            englishFuncName = englishFuncName.toUpperCase();
            return (englishFuncName in localeData.functions) ? localeData.functions[englishFuncName] : englishFuncName;
        };

        /**
         * Returns a help text descriptor for the specified translated function
         * name.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {Object|Null}
         *  A help text descriptor if available, with the following properties:
         *  - {String} description
         *      A text describing the entire formula.
         *  - {String[]} params
         *      An array with the names of all function parameters. Functions
         *      with trailing repeating parameters will be mentioned only once
         *      in this array.
         *  - {String[]} paramshelp
         *      An array describing each parameter more detailed. This array
         *      MUST have the same length as the 'params' array.
         *  - {Number} [varargs]
         *      If contained, the zero-based index of the first parameter that
         *      can be used repeatedly. MUST be less than the length of the
         *      'params' array. If this index does not point to the last array
         *      element, the function expects pairs or even larger tuples of
         *      repeating parameters.
         */
        this.getFunctionHelp = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in functionHelp) ? functionHelp[localFuncName] : null;
        };

        /**
         * Returns the sorted list with the names of all supported error codes
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported error codes.
         */
        this.getErrorCodes = function () {
            return _.chain(localeData.errorCodes).values().sort().value();
        };

        /**
         * Returns the name of the specified error code, translated for the
         * current UI language.
         *
         * @param {String} errorCode
         *  The English name of the error code, e.g. '#REF!', '#NUM!' etc.
         *
         * @returns {String}
         *  The translated name of the specified error code if available,
         *  otherwise the unmodified passed string.
         */
        this.getErrorCode = function (errorCode) {
            errorCode = errorCode.toUpperCase();
            return (errorCode in localeData.errorCodes) ? localeData.errorCodes[errorCode] : errorCode;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @returns {String}
         *  The decimal separator for the current UI language.
         */
        this.getDecimalSeparator = function () {
            return localeData.decSeparator;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getListSeparator = function () {
            return localeData.listSeparator;
        };

        /**
         * Returns the string literal of the specified Boolean value for the
         * current UI language.
         *
         * @param {Boolean} value
         *  The Boolean value.
         *
         * @returns {String}
         *  The string literal of the specified Boolean value.
         */
        this.getBooleanLiteral = function (value) {
            return localeData[value ? 'trueLiteral' : 'falseLiteral'];
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return localeData.engineVersion;
        };

    }}); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'icon-table', search: true });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
