/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/editframework/view/hyperlinkutil',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'gettext!io.ox/office/text'
    ], function (Utils, Dialogs, gt) {

    'use strict';

    var PROTS = new RegExp('^(http://|https://|ftp://|mailto:)');

    // templates for web/ftp identifications
    var TXTMS = new RegExp('^(www\\.|ftp\\.)');

    var HyperlinkUtil = {};

    // methods ----------------------------------------------------------------

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {String} text
     *  The optional text which represents the URL
     *
     * @param {String} url
     *  An optional URL which is set for the supplied text
     *
     * @param {BaseApplication} app
     *  The calling app
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text.
     */
    HyperlinkUtil.showDialog = function (text, url, app) {
        return HyperlinkUtil.showHyperlinkDialog(
                app, {
                    title: gt('Insert/Edit Hyperlink'),
                    valueURL: url,
                    placeholderURL: gt('Enter URL'),
                    valueText: text,
                    placeholderText: gt('Enter visible text'),
                    okLabel: gt('Insert')
                }
         );
    };

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {BaseApplication} app
     *  The calling application.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      If specified, the title of the dialog window that will be shown in
     *      a larger font.
     *  @param {String} [options.valueText='']
     *      The initial value of the text field.
     *  @param {String} [options.placeholderText='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {String} [options.valueURL='']
     *      The initial value of the URL field.
     *  @param {String} [options.placeholderURL='']
     *      The place-holder text that will be shown in the empty URL field.
     *  @param {String} [options.okLabel=gt('OK')]
     *      The label of the primary button that triggers the intended action
     *      by resolving the promise object returned by this method.
     *  @param {String} [options.cancelLabel=gt('Cancel')]
     *      The label of the Cancel button that rejects the promise object
     *      returned by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button or the remove button have been activated, or rejected if the
     *  dialog has been canceled. The done handlers registered at the promise
     *  object will receive a object containing the text and URL entered by the
     *  user. The object contains null for text and the URL if remove has been
     *  clicked.
     */
    HyperlinkUtil.showHyperlinkDialog = function (app, options) {

        var model = app.getModel();

        var // the text input fields
            inputurlid = _.uniqueId('url'),
            inputtextid = _.uniqueId('text'),
            inputUrl = $('<input>', {
                    value: Utils.getStringOption(options, 'valueURL', '')
                }).addClass('span12 nice-input').attr({'data-property': 'url', tabindex: 0, id: inputurlid, 'data-default-enter': true, role: 'textbox', 'aria-labelledby': 'label-url-02'}),

            inputText = $('<input>', {
                    value: Utils.getStringOption(options, 'valueText', ''),
                    placeholder: Utils.getStringOption(options, 'placeholderURL', '')
                }).addClass('span12 nice-input').attr({'data-property': 'text', tabindex: 0, id: inputtextid, 'data-default-enter': true, role: 'textbox', 'aria-labelledby': 'label-text-01'}),
            // the dialog object
            dialog = Dialogs.createDialog(Utils.extendOptions(options, { enter: Dialogs.defaultKeyEnterHandler }))
            .append(
                $('<div>').addClass('row-fluid').css({'margin-top': '10px'}).append(
                    $('<div>').addClass('control-group').css({'margin-bottom': '0px'}).append(
                        $('<label>').addClass('control-label').attr({'for': inputtextid, 'id': 'label-text-01'}).text(gt('Text:')),
                        $('<div>').addClass('controls').css({'margin-right': '10px'}).append(inputText))
                    )
                )
            .append(
                $('<div>').addClass('row-fluid').css({'margin-top': '10px'}).append(
                    $('<div>').addClass('control-group').css({'margin-bottom': '0px'}).append(
                        $('<label>').addClass('control-label').attr({'for': inputurlid, 'id': 'label-url-02'}).text(gt('URL:')),
                        $('<div>').addClass('controls').css({'margin-right': '10px'}).append(inputUrl))
                    )
                ),

            // the result deferred
            def = $.Deferred(),
            insertButton,
            removeButton = null;

        // special handling for iPad to make nice-input and nice-input:focus work
        if (_.browser.iOS && _.browser.WebKit) {
            inputUrl.css('-webkit-appearance', 'none');
            inputText.css('-webkit-appearance', 'none');
        }

        // add OK and Cancel buttons & remove button to remove hyperlink
        dialog.addPrimaryButton('ok', Utils.getStringOption(options, 'okLabel', gt('OK')), null, { tabIndex: 0 });
        dialog.addButton('cancel', Utils.getStringOption(options, 'cancelLabel', gt('Cancel')), null, { tabIndex: 0 });
        // only add remove button if we already have a set URL
        if (Utils.getStringOption(options, 'valueURL', '').length > 0) {
            dialog.addDangerButton('remove', Utils.getStringOption(options, 'removeLabel', gt('Remove')), null, { tabIndex: 0 });
            removeButton = dialog.getFooter().children('.btn-danger');
        }
        insertButton = dialog.getFooter().children('.btn-primary');

        function checkTextFields() {
            var textUrl = inputUrl.val(),
                text = inputText.val();

            if (!model.getEditMode()) {
                // disable ok and delete button if we don't have edit rights
                insertButton.attr({'disabled': 'disabled', 'aria-disabled': true, 'aria-label': gt('Insert')});
                if (removeButton) {
                    removeButton.attr({'disabled': 'disabled', 'aria-disabled': false, 'aria-label': gt('Insert')});
                }
            } else if ((text && text.length > 0) && (textUrl && textUrl.length > 0)) {
                insertButton.removeAttr('disabled').attr('aria-disabled', false);
            } else {
                insertButton.attr({'disabled': 'disabled', 'aria-disabled': true, 'aria-label': gt('Insert')});
            }
        }

        function editModeChangeHandler() { checkTextFields(); }

        // register a change handlers to toggle the insert button
        inputUrl.bind('input', checkTextFields);
        inputText.bind('input', checkTextFields);

        // register handler to initially focus the url or text input
        dialog.getPopup().on('shown', function () {

            if (Utils.getStringOption(options, 'valueText', '').length > 0)
                dialog.getBody().find('[data-property="url"]').focus();
            else
                dialog.getBody().find('[data-property="text"]').focus();
            checkTextFields();
        });

        // disable the ok button if the dialog is already opened but we lost the edit rights.
        model.on('change:editmode', editModeChangeHandler);

        // show the dialog and register listeners for the results
        dialog.show().done(function (action, data, node) {
            if (action === 'ok') {
                var text = $(node).find('[data-property="text"]').val();
                var url = $.trim($(node).find('[data-property="url"]').val());

                // add http: as default if protocol is missing
                var protIndex = url.indexOf(':');
                if (protIndex === -1) {
                    url = 'http://' + url;
                } else {
                    if (!HyperlinkUtil.isLegalUrl(url)) {
                        Utils.warn('you tried to add illegal code ' + url);
                        app.getView().yell('warning', gt('The code is an invalid hyperlink'));
                        url = null;
                    }
                }
                def.resolve({ text: text, url: url });
            } else if (action === 'remove') {
                def.resolve({ text: null, url: null });
            } else {
                def.reject();
            }
            model.off('change:editmode', editModeChangeHandler);
        });

        return def.promise();
    };


    HyperlinkUtil.isLegalUrl = function (url) {
        var good = PROTS.exec(url.toLowerCase());
        if (good) {
            return true;
        } else {
            return null;
        }
    };

    /**
     * create a jQuery Popup node with link, edit and remove function
     *
     * @param {BaseApplication} app
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     *
     * @returns {jQuery Element}
     */

    HyperlinkUtil.createPopupNode = function (app) {
        var hyperlinkPopup = $('<div>', { contenteditable: false })
            .addClass('inline-popup hyperlink f6-target').css('display', 'none')
            .append(
                $('<a>').addClass('textwrap').attr({ href: '', rel: 'noreferrer', target: '_blank', tabindex: 0 }),
                $('<br/>'),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Edit')).click(function () { app.getController().executeItem('character/hyperlink/dialog'); return false; }),
                $('<span>').text(' | '),
                $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Remove')).click(function () { app.getController().executeItem('character/hyperlink/remove'); return false; }));

        if (Modernizr.touch) {
            // some CSS changes for touch devices
            hyperlinkPopup.addClass('touch');
            hyperlinkPopup.find('a.popupbutton').addClass('touch');
        }

        //hyperlinkPopup.bind('touchstart mousedown', function () {
        //    return false;
        //});


        return hyperlinkPopup;

    };

    HyperlinkUtil.updatePopup = function (url, popup) {
        var link = popup.find('a.textwrap');
        //var link = $('a.textwrap', hyperlinkPopup[0]);
        link.text(HyperlinkUtil.limitHyperlinkText(url));
        if (!url || url.charAt(0) === '#') {
            link.attr({
                title: ''
            });
            link.css({
                color: '',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        } else if (HyperlinkUtil.isLegalUrl(url)) {
            link.css({
                color: '',
                cursor: '',
                textDecoration: ''
            });
            link.attr({
                href: url,
                title: url
            });
        } else {
            link.attr({
                title: 'Javascript Code is not allowed'
            });
            link.css({
                color: 'red',
                cursor: 'default',
                textDecoration: 'none'
            });
            link.removeAttr('href');
        }
    };

    /**
     * adds listener for the popupNode for changing the Editmode
     *
     * @param {SpreadsheetModel} model
     *  The application is necessary for the listeners of edit and remove
     *  (character/hyperlink/dialog and character/hyperlink/remove)
     *
     * @param {jQuery Element} popupNode
     *  @see HyperlinkUtil.createPopupNode
     *
     * @param {Function} listener
     *  an extra listener when Editmode is changed
     */

    HyperlinkUtil.addEditModeListener = function (model, popupNode, listener) {
        var updateEditMode = function () {
            var readOnly = model.getEditMode() !== true;
            popupNode.find('br').toggle(!readOnly);
            popupNode.find('span').toggle(!readOnly);
            popupNode.find('a.popupbutton').toggle(!readOnly);
            listener();
        };
        model.on('change:editmode', updateEditMode);


        // must be called to set the initial state correctly
        updateEditMode();
    };

    /**
     * Shortens a hyperlink to a maximal number of
     * characters.
     *
     * @param text {String} Hyperlink to shorten
     */
    HyperlinkUtil.limitHyperlinkText = function (text) {

        var maxChars = 255,
            result = text;

        if (_.isString(text) && text.length > maxChars) {
            var length = text.length,
                start = text.slice(0, Math.round(maxChars / 2)),
                end = text.slice(length - Math.round(maxChars / 2));

            result = start + '...' + end;
        }

        return result;
    };


    /**
     * checks if popupnode is already created in given parent
     *
     * @param {jQuery Element} parent
     *
     */

    HyperlinkUtil.noPopup = function (parent) {
        if (parent.children('.inline-popup.hyperlink').length === 0) {
            return true;
        }
    };

    /**
     *  Checks for a text which defines a hyperlink
     *  e.g. http://www.open-xchange.com
     *
     *  @param {String} text
     *
     *  @returns {String} if an hyperlink is found
     */

    HyperlinkUtil.checkForHyperlink = function (text) {
        var result = null;

        var content = PROTS.exec(text);

        if (content) {
            var foundProtocol = content[1];
            if (foundProtocol !== 'mailto:') {
                var index = text.indexOf('//');

                // At least one character after the protocol must be there
                if ((index + 2) < text.length && HyperlinkUtil.checkURL(text)) {
                    // create result with correct Position objects
                    result = text;
                }
            } else if (foundProtocol === 'mailto:') {
                var atIndex = text.indexOf('@');
                var dotIndex = text.lastIndexOf('.');

                // mailto: needs a '@' and '.' char and at least one char after the dot
                // we also use the regexp to check for other problems
                if ((atIndex > ('mailto:'.length + 1)) &&
                    (dotIndex > atIndex) && (dotIndex < (text.length - 1)) &&
                    HyperlinkUtil.checkURL(text)) {

                    // create result with correct Position objects
                    result = text;
                }
            }
        } else {
            //some magic auto detection for text without protocol

            content = TXTMS.exec(text);

            if (content) {
                var textTemplate = content[1];
                if (textTemplate === 'www.') {
                    // http://
                    if (text.length > 'www.'.length) {
                        result = 'http://' + text;
                    }
                } else if (textTemplate === 'ftp.') {
                    // ftp://
                    if (text.length > 'ftp.'.length) {
                        result = 'ftp://' + text;
                    }
                }
            }
            else {
                // some more magic for a mail address
                if (HyperlinkUtil.checkMailAddress(text)) {
                    result = 'mailto:' + text;
                }
            }
        }
        return result;
    };


    /**
     * Checks an URL based on some basic rules. This is not a fully compliant
     * URL check.
     *
     * @param {String} url
     *
     * @returns {Boolean}
     *  true if the URL seems to be correct otherwise false
     */
    HyperlinkUtil.checkURL = function (url) {
        return (/^([a-z]([a-z]|\d|\+|-|\.)*):(\/\/(((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:)*@)?((\[(|(v[\da-f]{1,}\.(([a-z]|\d|-|\.|_|~)|[!\$&'\(\)\*\+,;=]|:)+))\])|((\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5]))|(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=])*)(:\d*)?)(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*|(\/((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)?)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)){0})(\?((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|[\uE000-\uF8FF]|\/|\?)*)?(\#((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|\/|\?)*)?$/i.test(url));
    };

    /**
     * Checks an mailaddress based on some basic rules. This is not a fully compliant
     * mailaddress check.
     *
     * @param {String} mailAddress
     *
     * @returns {Boolean}
     *  true if the URL seems to be correct otherwise false
     */
    HyperlinkUtil.checkMailAddress = function (mailAddress) {
        return (/^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,4}$/i.test(mailAddress));
    };

    // exports ================================================================

    return HyperlinkUtil;

});
