/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editsidepane',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/sidepane',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/editframework/view/editcontrols',
     'gettext!io.ox/office/editframework'
    ], function (Utils, SidePane, ToolBox, EditControls, gt) {

    'use strict';

    var // class name shortcuts
        Button = EditControls.Button,
        TextField = EditControls.TextField;

    // class SidePane =========================================================

    /**
     * @constructor
     *
     * @extends SidePane
     *
     * @param {EditApplication} app
     *  The application containing this side pane.
     */
    function EditSidePane(app) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        SidePane.call(this, app, 'side', { position: 'right' });

        // private methods ----------------------------------------------------

        /**
         * Initializes the upper tool box.
         */
        function initializeToolBox() {

            var // the button groups that will be shown/hidden dynamically
                acquireButton = new EditControls.AcquireEditButton(app),
                undoButton = new Button(EditControls.UNDO_OPTIONS),
                redoButton = new Button(EditControls.REDO_OPTIONS);

            function updateVisibility(editMode) {
                acquireButton.getNode().closest('.group-container').toggleClass('hidden', editMode !== false);
                undoButton.getNode().closest('.group-container').toggleClass('hidden', editMode !== true);
            }

            // create a fixed tool box with common controls for all edit applications
            self.addViewComponent(new ToolBox(app, 'main', { fixed: 'top' })
                .addGroup('view/sidepane', new Button(EditControls.HIDE_SIDEPANE_OPTIONS))
                .addRightTab()
                .addGroup('document/acquireedit', acquireButton)
                .addGap()
                .addGroup('document/undo', undoButton)
                .addGroup('document/redo', redoButton)
                .addGap()
                .addGroup('app/quit', new Button(EditControls.QUIT_OPTIONS))
                .newLine()
                .addGroup('document/rename', new TextField({ classes: 'no-borders document-name', tooltip: gt('Rename document'), select: true, fullWidth: true }))
            );

            // show/hide controls according to edit mode
            app.getModel().on('change:editmode', function (event, editMode) {
                updateVisibility(editMode);
            });

            // initially, hide all controls
            updateVisibility(null);
        }

        /**
         * Updates the side pane width to keep the inner width of the
         * scrollable area fixed.
         */
        function refreshLayout() {

            var // the root node of the side pane
                paneNode = self.getNode(),
                // the scrollable container node in the side pane (as DOM node)
                scrollableNode = self.getScrollableNode()[0],
                // the current client width of the side pane
                clientWidth = scrollableNode.clientWidth;

            if (!self.isReallyVisible()) { return; }

            // update width of the side pane to get the fixed client width
            // accept that client width may be slightly larger (2 pixels) to
            // prevent jumping between expected width +1/-1..
            if ((clientWidth > 0) && ((clientWidth < SidePane.DEFAULT_WIDTH) || (clientWidth - 2 > SidePane.DEFAULT_WIDTH))) {
                paneNode.width(paneNode.width() + SidePane.DEFAULT_WIDTH - clientWidth);

                // re-check client width, it may still be one pixel too small
                if (Math.floor(scrollableNode.clientWidth) < SidePane.DEFAULT_WIDTH) {
                    paneNode.width(paneNode.width() + 1);
                }

                // Refresh all view panes again (width of side pane has been changed).
                // It is important to keep this call inside the if() condition to
                // prevent endless loop (this method will be called from
                // 'refresh:layout' events triggered by BaseView.refreshPaneLayout()).
                app.getView().refreshPaneLayout();
            }
        }

        // initialization -----------------------------------------------------

        // create a fixed tool box with common controls for all edit applications
        initializeToolBox();

        // listen to refresh events of the base class
        this.on('refresh:layout', refreshLayout);

    } // class EditSidePane

    // exports ================================================================

    // derive this class from class SidePane
    return SidePane.extend({ constructor: EditSidePane });

});
