/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/stylesheets',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/editframework/model/format/documentstyles'
    ], function (Utils, ModelObject, DocumentStyles) {

    'use strict';

    // private static functions ===============================================

    /**
     * Returns the attribute map stored in the passed DOM element.
     *
     * @param {jQuery} element
     *  The DOM element, as jQuery object.
     *
     * @param {Object} [initOptions]
     * A map with additional options controlling the behavior of this method.
     * the following options are supported:
     *  @param {String} [initOptions.family]
     *      If specified, extracts the attributes of a specific attribute
     *      family from the attribute map. Otherwise, returns the complete map
     *      object with all attributes mapped by their family.
     *  @param {Boolean} [initOptions.clone=false]
     *      If set to true, the returned attribute map will be a clone of the
     *      original map.
     *
     * @returns {Object}
     *  The attribute map if existing, otherwise an empty object.
     */
    function getElementAttributes(element, initOptions) {

        var // the original and complete attribute map
            attributes = element.data('attributes'),
            // the attribute family to be extracted from the complete map
            family = Utils.getStringOption(initOptions, 'family'),
            // whether to clone the resulting object
            clone = Utils.getBooleanOption(initOptions, 'clone', false);

        // reduce to selected family
        if (_.isObject(attributes) && _.isString(family)) {
            attributes = (family in attributes) ? attributes[family] : undefined;
        }

        // return attributes directly or as a deep clone
        return _.isObject(attributes) ? ((clone === true) ? _.copy(attributes, true) : attributes) : {};
    }

    // class StyleSheets ======================================================

    /**
     * Container for hierarchical style sheets of a specific attribute family.
     * Implements indirect element formatting via style sheets and direct
     * element formatting via explicit attribute maps.
     *
     * Attribute definitions that have been registered with the method
     * 'DocumentStyles.registerAttributeDefinitions()' may contain the
     * following additional properties:
     *  - {Function} [definition.format]
     *      A function that applies the passed attribute value to a DOM element
     *      (usually its CSS formatting). Will be called in the context of the
     *      style sheet container instance. The function receives the DOM
     *      element as jQuery object in the first parameter, and the attribute
     *      value in the second parameter. An alternative way to update the
     *      element formatting using a complete map of all attribute values is
     *      to specify a global format handler (see options below).
     *  - {Function|Boolean} [definition.preview]
     *      A function that applies the passed attribute value to a DOM element
     *      used to show a preview of a style sheet in the GUI. Will be called
     *      in the context of the style sheet container instance. The function
     *      receives the DOM element as jQuery object in the first parameter,
     *      and the attribute value in the second parameter. An alternative way
     *      to update the preview element formatting using a complete map of
     *      all attribute values is to specify a global preview handler (see
     *      options below). If the boolean value 'true' will be set instead of
     *      a function, the registered 'format' method will be called instead.
     *
     * Triggers the following events:
     * - 'insert:stylesheet': After a new style sheet has been inserted into
     *      this container. The event handler receives the identifier of the
     *      new style sheet.
     * - 'delete:stylesheet': After a style sheet has been deleted from this
     *      container. The event handler receives the identifier of the deleted
     *      style sheet.
     * - 'change:stylesheet': After the formatting attributes of a style sheet
     *      in this container have been changed. The event handler receives the
     *      identifier of the changed style sheet.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document, that will own this style sheet container
     *  instance.
     *
     * @param {String} styleFamily
     *  The main attribute family represented by the style sheets contained by
     *  instances of the derived container class. The style sheets and the DOM
     *  elements referring to the style sheets must support all attributes of
     *  this attribute family.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the behavior of the style sheet container.
     *  All callback functions will be called in the context of this style
     *  sheet container instance. The following options are supported:
     *  @param {Boolean} [initOptions.styleSheetSupport=true]
     *      If set to false, this container is not allowed to contain style
     *      sheets. It can only be used to format DOM elements with explicit
     *      formatting attributes. DEPRECATED!
     *  @param {String|Array} [initOptions.additionalFamilies]
     *      The name of an additional attribute family, or an array with names
     *      of additional attribute families supported by the style sheets of
     *      this container instance.
     *  @param {Object} [initOptions.parentResolvers]
     *      The parent style families whose associated style sheets can contain
     *      attributes of the family supported by this style sheet container
     *      class. The DOM elements referring to the style sheets of the
     *      specified style families must be ancestors of the DOM elements
     *      referring to the style sheets of this container. The passed object
     *      maps the attribute family to an ancestor element resolver function.
     *      The function receives the descendant DOM element as jQuery object
     *      in the first parameter, and returns the ancestor element of that
     *      DOM element which is associated to the parent style family used as
     *      map key.
     *  @param {Function} [initOptions.formatHandler]
     *      A callback function that will be called for every DOM element whose
     *      attributes have been changed, and that will format the DOM element
     *      according to the new attribute values. In difference to the
     *      individual formatter methods specified in the definitions for
     *      single attributes, this handler will be called once for a DOM
     *      element regardless of the number of changed attributes. Receives
     *      the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the format handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a Deferred object that will be resolved after execution.
     *  @param {Function} [initOptions.previewHandler]
     *      A callback function that will be called if a DOM element used in
     *      the GUI to show a preview of a style sheet will be formatted. In
     *      difference to the individual previewer methods specified in the
     *      definitions for single attributes, this handler will be called once
     *      for the DOM element regardless of the number of changed attributes.
     *      Receives the following parameters:
     *      (1) {jQuery} element
     *          The element whose attributes have been changed.
     *      (2) {Object} mergedAttributes
     *          The effective attribute values merged from style sheets and
     *          explicit attributes, as map of attribute maps (name/value
     *          pairs), keyed by attribute family.
     *      (3) {Boolean} async
     *          If set to true, the preview handler may execute asynchronously
     *          in a browser timeout loop. In this case, the handler MUST
     *          return a Deferred object that will be resolved after execution.
     *  @param {Function} [initOptions.styleAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      complete 'attributes' object of a style sheet. The attributes
     *      returned by this function may depend on a source element (e.g. on
     *      the position of this source element in its parents). The function
     *      receives the following parameters:
     *      (1) {Object} styleAttributes
     *          The complete original 'attributes' object of the style sheet.
     *          MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element referring to the style sheet.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     *  @param {Function} [initOptions.elementAttributesResolver]
     *      A function that extracts and returns specific attributes from the
     *      explicit attribute map of an element. The attributes returned by
     *      this function may depend on a source element (e.g. on the position
     *      of this source element in its parents). The function receives the
     *      following parameters:
     *      (1) {Object} elementAttributes
     *          The original explicit element attributes of the style sheet, as
     *          map of attribute maps (name/value pairs), keyed by attribute
     *          family. MUST NOT be changed by the callback function!
     *      (2) {jQuery} element
     *          The element whose attributes have been passed in the first
     *          parameter.
     *      (3) {jQuery} [sourceNode]
     *          The descendant source node.
     */
    function StyleSheets(app, documentStyles, styleFamily, initOptions) {

        var // self reference
            self = this,

            // style sheets, mapped by identifier
            styleSheets = {},

            // the names of all supported attribute families
            supportedFamilies = null,

            // identifier of the default style sheet
            defaultStyleId = null,

            // whether the default style is also hidden
            isDefaultStyleHidden = false,

            // whether this container supports style sheets
            styleSheetSupport = Utils.getFunctionOption(initOptions, 'styleSheetSupport', true),

            // the resolver functions for DOM ancestor elements, keyed by attribute family
            parentResolvers = Utils.getObjectOption(initOptions, 'parentResolvers', {}),

            // formatting handler, called after attributes of a model element have been changed
            formatHandler = Utils.getFunctionOption(initOptions, 'formatHandler'),

            // preview handler, called after attributes of a GUI preview element have been changed
            previewHandler = Utils.getFunctionOption(initOptions, 'previewHandler'),

            // custom resolver for style attributes depending on a context element
            styleAttributesResolver = Utils.getFunctionOption(initOptions, 'styleAttributesResolver'),

            // custom resolver for explicit element attributes depending on a context element
            elementAttributesResolver = Utils.getFunctionOption(initOptions, 'elementAttributesResolver');

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed style sheet is a descendant of the other
         * passed style sheet.
         */
        function isDescendantStyleSheet(styleSheet, ancestorStyleSheet) {
            while (styleSheet) {
                if (styleSheet === ancestorStyleSheet) { return true; }
                styleSheet = styleSheets[styleSheet.parentId];
            }
            return false;
        }

        /**
         * Restricts the passed attribute set to the attribute families
         * supported by the style sheet container.
         *
         * @param {Object} mergedAttributes
         *  The attribute set whose member field not representing a supported
         *  attribute map will be removed.
         */
        function restrictToSupportedFamilies(mergedAttributes) {
            // remove attribute maps of unsupported families
            _(mergedAttributes).each(function (unused, family) {
                if (!_(supportedFamilies).contains(family)) {
                    delete mergedAttributes[family];
                }
            });
        }

        /**
         * Returns the complete attributes of an ancestor of the passed
         * element, if a parent style family has been registered and its
         * element resolver function returns a valid ancestor element, and
         * extends missing attributes by their current default values.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element whose ancestor will be searched and queried for its
         *  attributes. If this object is a jQuery collection, uses the first
         *  DOM node it contains. If missing, returns just the current default
         *  values of all supported attribute families.
         *
         * @returns {Object}
         *  The formatting attributes of an ancestor of the passed element, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family. If no ancestor element has been found, returns the current
         *  default values for all supported attribute families.
         */
        function resolveBaseAttributes(element) {

            var // passed element, as jQuery object
                $element = $(element),
                // the matching parent element, its style family, the style sheet container, and the attributes
                parentElement = null, parentFamily = null, parentStyleSheets = null, parentAttributes = null,
                // the resulting merged attributes of the ancestor element
                mergedAttributes = {};

            // collect attributes from ancestor element if specified (only one parent style family must match at a time)
            if ($element.length > 0) {

                // find a matching ancestor element and its style family (only one parent must match at a time)
                _(parentResolvers).each(function (elementResolver, family) {
                    if ($(parentElement).length === 0) {
                        parentElement = elementResolver.call(self, $element);
                        parentFamily = family;
                    }
                });

                // add the element attributes of the ancestor element (only the supported attribute families)
                if ($(parentElement).length > 0) {
                    parentStyleSheets = documentStyles.getStyleSheets(parentFamily);
                    parentAttributes = parentStyleSheets.getElementAttributes(parentElement, { special: true, sourceNode: $element });
                    documentStyles.extendAttributes(mergedAttributes, parentAttributes);
                }
            }

            // remove attribute maps of unsupported families (this also removes the style identifier of the parent element)
            restrictToSupportedFamilies(mergedAttributes);

            // add missing default attribute values of supported families not found in the parent element
            _(supportedFamilies).each(function (family) {
                if (!(family in mergedAttributes)) {
                    _(mergedAttributes).extend(documentStyles.getDefaultAttributes(family));
                }
            });

            return mergedAttributes;
        }

        /**
         * Returns the attributes from the specified style sheet and its parent
         * style sheets. Does not add default attribute values, or attributes
         * from ancestors of the passed element.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {HTMLElement|jQuery} [element]
         *  An element referring to a style sheet in this container whose
         *  attributes will be extracted. Will be passed to a custom style
         *  attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  style attributes resolver (see the option 'styleAttributesResolver'
         *  passed to the constructor). If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The formatting attributes contained in the style sheet and its
         *  ancestor style sheets, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveStyleSheetAttributes(styleId, element, sourceNode) {

            var // passed element, as jQuery object
                $element = $(element),
                // passed source node, as jQuery object
                $sourceNode = $(sourceNode),
                // the resulting merged attributes of the style sheet and its ancestors
                mergedAttributes = {};

            // collects style sheet attributes recursively through parents
            function collectStyleAttributes(styleSheet) {

                // exit recursive call chain if no more parent styles are available
                if (!styleSheet) { return; }

                // call recursively to get the attributes of the parent style sheets
                collectStyleAttributes(styleSheets[styleSheet.parentId]);

                // add all attributes of the current style sheet, mapped directly by attribute family
                documentStyles.extendAttributes(mergedAttributes, styleSheet.attributes);

                // try user-defined resolver for style attributes mapped in non-standard structures
                if (_.isFunction(styleAttributesResolver) && ($element.length > 0)) {
                    documentStyles.extendAttributes(mergedAttributes, styleAttributesResolver.call(self, styleSheet.attributes, $element, $sourceNode));
                }
            }

            // fall-back to default style sheet if passed identifier is invalid
            if (!(styleId in styleSheets) && (defaultStyleId in styleSheets)) {
                styleId = defaultStyleId;
            }

            // collect attributes from the style sheet and its parents
            collectStyleAttributes(styleSheets[styleId]);

            // remove attribute maps of unsupported families
            restrictToSupportedFamilies(mergedAttributes);

            // add resulting style sheet identifier to the attributes
            mergedAttributes.styleId = styleId;

            return mergedAttributes;
        }

        /**
         * Returns the explicit attributes from the specified element. Does not
         * add default attribute values. Uses a custom element attributes
         * resolver function if specified in the constructor of this style
         * sheet container.
         *
         * @param {HTMLElement|jQuery} [element]
         *  The element whose explicit attributes will be extracted. Will be
         *  passed to a custom attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @param {HTMLElement|jQuery} [sourceNode]
         *  The source DOM node corresponding to a child attribute family that
         *  has initiated the call to this method. Will be passed to a custom
         *  element attributes resolver (see the option
         *  'elementAttributesResolver' passed to the constructor). If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {Object}
         *  The explicit formatting attributes contained in the element, as map
         *  of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         */
        function resolveElementAttributes(element, sourceNode) {

            var // passed element, as jQuery object
                $element = $(element),
                // the resulting merged attributes of the element
                mergedAttributes = getElementAttributes($element);

            // call custom element attribute resolver
            if (_.isFunction(elementAttributesResolver)) {
                mergedAttributes = elementAttributesResolver.call(self, mergedAttributes, $element, $(sourceNode));
            }

            return mergedAttributes;
        }

        /**
         * Filters the passed attribute set by supported attributes in-place.
         */
        function filterSupportedAttributes(mergedAttributes) {

            _(mergedAttributes).each(function (attributeValues, family) {

                var // attribute definitions of the current family
                    definitions = documentStyles.getAttributeDefinitions(family);

                if (definitions) {
                    _(attributeValues).each(function (value, name)  {
                        if (!DocumentStyles.isRegisteredAttribute(definitions, name, initOptions)) {
                            delete attributeValues[name];
                        }
                    });
                } else if (!styleSheetSupport || (family !== 'styleId')) {
                    delete mergedAttributes[family];
                }
            });

            return mergedAttributes;
        }

        /**
         * Updates the element formatting according to the passed attributes.
         *
         * @param {jQuery} element
         *  The element whose formatting will be updated, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set with all attribute values merged from
         *  style sheet and explicit attributes.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  registered format or preview handler has finished. If the
         *  'options.async' options is set, and the format handler supports
         *  asynchronous execution, the Promise will be resolved sometime in
         *  the future. Otherwise, the returned Promise is already resolved
         *  when this method returns.
         */
        function updateElementFormatting(element, mergedAttributes, options) {

            var // definitions of own attributes
                definitions = documentStyles.getAttributeDefinitions(styleFamily),
                // whether to render a GUI preview element
                preview = Utils.getBooleanOption(options, 'preview', false),
                // whether to call the real format handler or the preview handler
                currentFormatHandler = preview ? previewHandler : formatHandler,
                // the result Deferred object returned from the format or preview handler
                resultDef = null;

            if (!preview) {
                self.DBG_COUNT = (self.DBG_COUNT || 0) + 1;
            }

            // call single format handlers for all attributes of the own style family
            _(mergedAttributes[styleFamily]).each(function (value, name) {
                var formatMethod = null;
                if (name in definitions) {
                    formatMethod = Utils.getFunctionOption(definitions[name], 'format');
                    if (preview && (Utils.getBooleanOption(definitions[name], 'preview', false) !== true)) {
                        formatMethod = Utils.getFunctionOption(definitions[name], 'preview');
                    }
                    if (_.isFunction(formatMethod)) {
                        formatMethod.call(self, element, value);
                    }
                }
            });

            // call format handler taking all attributes at once
            if (_.isFunction(currentFormatHandler)) {
                try {
                    resultDef = currentFormatHandler.call(self, element, mergedAttributes, Utils.getBooleanOption(options, 'async', false));
                } catch (ex) {
                    Utils.exception(ex);
                    resultDef = $.Deferred().reject();
                }
            }

            // wrap resultDef in $.when() to convert plain values to a Deferred object
            return $.when(resultDef).promise();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the style family name of this style sheet container.
         */
        this.getStyleFamily = function () {
            return styleFamily;
        };

        /**
         * Returns an array with the names of all attribute families supported
         * by style sheets contained in this instance, including the main style
         * family.
         *
         * @returns {Array}
         *  A string array containing the supported attribute families.
         */
        this.getSupportedFamilies = function () {
            return supportedFamilies;
        };

        /**
         * Returns the identifier of the default style sheet.
         *
         * @returns {String|Null}
         *  The identifier of the default style sheet.
         */
        this.getDefaultStyleId = function () {
            return defaultStyleId;
        };

        /**
         * Inserts a new style sheet into this container. An existing style
         * sheet with the specified identifier will be replaced.
         *
         * @param {String} styleId
         *  The unique identifier of of the new style sheet.
         *
         * @param {String} name
         *  The user-defined name of of the new style sheet.
         *
         * @param {String|Null} parentId
         *  The identifier of of the parent style sheet the new style sheet
         *  will derive undefined attributes from.
         *
         * @param {Object} attributes
         *  The formatting attributes contained in the new style sheet. The
         *  structure of this object is dependent on the style family of this
         *  container.
         *
         * @param {Object} [options]
         *  A map of options to control the behavior of the new style sheet.
         *  The following options are supported:
         *  @param {Boolean} [options.hidden=false]
         *      Determines if the style should be displayed in the user
         *      interface.
         *  @param {String} [options.category]
         *      If specified, the name of a category that can be used to group
         *      style sheets into several sections in the user interface.
         *  @param {Number} [options.priority=0]
         *      The sorting priority of the style (the lower the value the
         *      higher the priority).
         *  @param {Boolean} [options.defStyle=false]
         *      True, if the new style sheet is the default style sheet of this
         *      style sheet container. The default style will be used for all
         *      elements without explicit style sheet. Only the first style
         *      sheet that will be inserted with this flag will be permanently
         *      registered as default style sheet.
         *  @param {Boolean} [options.dirty=false]
         *      True, if the new style sheet has been added manually and not
         *      by loading a document. Therefore this style sheet needs to be
         *      saved as soon as it will be used.
         *
         * @returns {Boolean}
         *  Whether creating the new style sheet was successful.
         */
        this.insertStyleSheet = function (styleId, name, parentId, attributes, options) {

            var // the new style sheet instance
                styleSheet = null;

            // check that style sheets are enabled at all
            if (!styleSheetSupport) {
                Utils.error('StyleSheets.insertStyleSheet(): creating style sheets not allowed for style family "' + styleFamily + '"');
                return false;
            }

            if (!_.isString(styleId) || (styleId.length === 0)) {
                Utils.error('StyleSheets.insertStyleSheet(): missing style sheet identifier');
                return false;
            }

            // TODO: disallow multiple insertion? (predefined style sheets may be created on demand without checking existence)
            if (styleId in styleSheets) {
                Utils.warn('StyleSheets.insertStyleSheet(): style sheet "' + styleId + '" exists already');
            }

            // get or create a style sheet object, set identifier and user-defined name of the style sheet
            styleSheet = styleSheets[styleId] || (styleSheets[styleId] = {});
            styleSheet.id = styleId;
            styleSheet.name = name || styleId;

            // set parent of the style sheet, check for cyclic references
            styleSheet.parentId = parentId;
            if (isDescendantStyleSheet(styleSheets[styleSheet.parentId], styleSheet)) {
                Utils.warn('StyleSheets.insertStyleSheet(): cyclic reference, cannot set style sheet parent "' + parentId + '"');
                styleSheet.parentId = null;
            }

            // set style sheet options
            styleSheet.hidden = Utils.getBooleanOption(options, 'hidden', false);
            styleSheet.category = Utils.getStringOption(options, 'category', styleSheet.category);
            styleSheet.priority = Utils.getIntegerOption(options, 'priority', 0);
            styleSheet.dirty = Utils.getBooleanOption(options, 'dirty', false);

            // setting a following visible paragraph style as default style after a hidden default style (Fix for 27716)
            if ((isDefaultStyleHidden) && (!styleSheet.hidden)) {
                defaultStyleId = styleId;
                isDefaultStyleHidden = false;
            }

            // set default style sheet
            if (Utils.getBooleanOption(options, 'defStyle', false)) {
                if (_.isNull(defaultStyleId)) {
                    defaultStyleId = styleId;
                    if (styleSheet.hidden) {
                        isDefaultStyleHidden = true;
                    }
                } else if (defaultStyleId !== styleId) {
                    Utils.warn('StyleSheets.insertStyleSheet(): multiple default style sheets');
                }
            }

            // store a deep clone of the passed attributes
            styleSheet.attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};

            // notify listeners
            this.trigger('insert:stylesheet', styleId);
            return true;
        };

        /**
         * Removes an existing style sheet from this container.
         *
         * @param {String} styleId
         *  The unique identifier of of the style sheet to be removed.
         *
         * @returns {Boolean}
         *  Whether removing the style sheet was successful.
         */
        this.deleteStyleSheet = function (styleId) {

            if (!_.isString(styleId) || !(styleId in styleSheets)) {
                Utils.error('StyleSheets.deleteStyleSheet(): style sheet "' + styleId + '" does not exist');
                return false;
            }

            if (styleId === defaultStyleId) {
                Utils.error('StyleSheets.deleteStyleSheet(): cannot remove default style sheet');
                return false;
            }

            // update parent of all style sheets referring to the removed style sheet
            _(styleSheets).each(function (childSheet) {
                if (styleId === childSheet.parentId) {
                    childSheet.parentId = styleSheets[styleId].parentId;
                }
            });

            // remove style sheet from map
            delete styleSheets[styleId];

            // notify listeners
            this.trigger('delete:stylesheet', styleId);
            return true;
        };

        /**
         * Returns whether this container contains the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed
         *  identifier.
         */
        this.containsStyleSheet = function (styleId) {
            return styleId in styleSheets;
        };

        /**
         * Returns whether this container contains a style sheet with the
         * specified name (not identifier).
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {Boolean}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.containsStyleSheetByName = function (styleName) {
            styleName = styleName.toLowerCase();
            return _(styleSheets).any(function (styleSheet) {
                return styleSheet.name.toLowerCase() === styleName;
            });
        };

        /**
         * Returns the style identifier of the first style sheet found with the
         * specified name.
         *
         * @param {String} styleName
         *  The name of the style sheet.
         *
         * @returns {String|Null}
         *  Whether this instance contains a style sheet with the passed name.
         */
        this.getStyleIdByName = function (styleName) {
            var resultId = null;
            styleName = styleName.toLowerCase();
            _(styleSheets).any(function (styleSheet, styleId) {
                if (styleSheet.name.toLowerCase() === styleName) {
                    resultId = styleId;
                    return true;
                }
            });
            return resultId;
        };

        /**
         * Returns the names of all style sheets in a map, keyed by their
         * unique identifiers.
         *
         * @returns {Object}
         *  A map with all style sheet names, keyed by style sheet identifiers.
         */
        this.getStyleSheetNames = function () {
            var names = {};
            _(styleSheets).each(function (styleSheet, styleId) {
                if (!styleSheet.hidden) {
                    names[styleId] = styleSheet.name;
                }
            });
            return names;
        };

        /**
         * Returns the merged attributes of the own style family from the
         * specified style sheet and its parent style sheets.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Object}
         *  The formatting attributes contained in the style sheet and its
         *  parent style sheets in this container up to the map of default
         *  attributes, as map of name/value pairs.
         */
        this.getStyleSheetAttributes = function (styleId) {

            var // start with attribute default values
                mergedAttributes = resolveBaseAttributes();

            // extend with attributes of specified style sheet
            return documentStyles.extendAttributes(mergedAttributes, resolveStyleSheetAttributes(styleId));
        };

        this.getUICategory = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].category : null;
        };

        this.setUICategory = function (styleId, category) {
            var styleSheet = styleSheets[styleId];
            if (styleSheet) {
                styleSheet.category = category;
            }
            return this;
        };

        /**
         * Returns the UI priority for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Number}
         *  The UI priority. If the style sheet does not exist, returns 0.
         */
        this.getUIPriority = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].priority : 0;
        };

        this.setUIPriority = function (styleId, prioritiy) {
            var styleSheet = styleSheets[styleId];
            if (styleSheet) {
                styleSheet.priority = prioritiy;
            }
            return this;
        };

        /**
         * Returns the identifier of the parent style sheet for the specified
         * style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The parent id of the style sheet; or null, if no parent exists.
         */
        this.getParentId = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].parentId : null;
        };

        /**
         * Returns the hidden state of the style sheet or null if the specified
         * style sheet doesn't exists.
         *
         * @param {String} styleId
         *
         * @returns {Boolean|Null}
         *  The hidden state of the style sheet; or null, if the style sheet
         *  does not exist.
         */
        this.isHidden = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].hidden : null;
        };

        /**
         * Returns the user defined name for the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {String|Null}
         *  The user defined name of the style sheet; or null, if the style
         *  sheet does not exist.
         */
        this.getName = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].name : null;
        };

        /**
         * Return whether the specified style sheet is dirty.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Boolean|Null}
         *  The dirty state of the style sheet; or null, if the style sheet
         *  does not exist.
         */
        this.isDirty = function (styleId) {
            return (styleId in styleSheets) ? styleSheets[styleId].dirty : null;
        };

        /**
         * Changes the dirty state of the specified style sheet.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @param {Boolean} dirty
         */
        this.setDirty = function (styleId, dirty) {
            if (styleId in styleSheets) {
                styleSheets[styleId].dirty = dirty;
            }
        };

        /**
         * Returns the complete 'attributes' object of the specified style
         * sheet as-is, without resolving the parent style sheets, or
         * converting to the attributes of a specific attribute family.
         *
         * @param {String} styleId
         *  The unique identifier of the style sheet.
         *
         * @returns {Object}
         *  The complete 'attributes' object contained in the style sheet, as a
         *  deep clone.
         */
        this.getStyleSheetAttributeMap = function (styleId) {
            return (styleId in styleSheets) ? _.copy(styleSheets[styleId].attributes, true) : {};
        };

        /**
         * Builds an attribute map containing all formatting attributes of the
         * own style family, and of the additional attribute families supported
         * by the style sheets in this container, set to the null value.
         * Additionally, the property 'styleId' set to null will be added to
         * the attribute set.
         *
         * @returns {Object}
         *  An attribute set with null values for all attributes registered for
         *  the supported attribute families.
         */
        this.buildNullAttributes = function () {
            var attributes = documentStyles.buildNullAttributes(supportedFamilies);
            attributes.styleId = null;
            return attributes;
        };

        /**
         * Returns the merged values of the formatting attributes from a style
         * sheet and explicit attributes.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set containing an optional style sheet
         *  identifier in the property 'styleId', and explicit attributes.
         *
         * @param {Object} [options]
         *  A map of options controlling the operation. Supports the following
         *  options:
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this container.
         */
        this.getMergedAttributes = function (attributes, options) {

            var // the identifier of the style sheet
                styleId = Utils.getStringOption(attributes, 'styleId', null),
                // resulting merged attributes (start with defaults and style sheet attributes)
                mergedAttributes = this.getStyleSheetAttributes(styleId);

            // add the passed explicit attributes (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            documentStyles.extendAttributes(mergedAttributes, attributes, options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes
            return filterSupportedAttributes(mergedAttributes);
        };

        /**
         * Returns the values of the formatting attributes in the specified DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be returned. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  A map of options controlling the operation. Supports the following
         *  options:
         *  @param {HTMLElement|jQuery} [options.sourceNode]
         *      A descendant of the passed element associated to a child
         *      attribute family. Will be passed to a style attribute resolver
         *      callback function where it might be needed to resolve the
         *      correct attributes according to the position of this source
         *      node.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result map.
         *
         * @returns {Object}
         *  A complete merged attribute set with the values of all formatting
         *  attributes supported by this container.
         */
        this.getElementAttributes = function (element, options) {

            var // the descendant source node
                sourceNode = Utils.getOption(options, 'sourceNode'),
                // the identifier of the style sheet referred by the element
                styleId = StyleSheets.getElementStyleId(element),
                // resulting merged attributes (start with defaults, parent element, and style sheet attributes)
                mergedAttributes = resolveBaseAttributes(element);

            // add attributes of the style sheet and its parents
            documentStyles.extendAttributes(mergedAttributes, resolveStyleSheetAttributes(styleId, element, sourceNode));

            // add the explicit attributes of the element (protect effective style sheet identifier)
            styleId = mergedAttributes.styleId;
            documentStyles.extendAttributes(mergedAttributes, resolveElementAttributes(element, sourceNode), options);
            mergedAttributes.styleId = styleId;

            // filter by supported attributes
            return filterSupportedAttributes(mergedAttributes);
        };

        /**
         * Changes specific formatting attributes in the specified DOM element.
         *
         * @param {HTMLElement|jQuery} element
         *  The element whose attributes will be changed. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with all formatting attributes to be
         *  changed. To clear an explicit attribute value (thus defaulting to
         *  the current style sheet), the respective value in this attribute
         *  set has to be set explicitly to the null value. Attributes missing
         *  in this attribute set will not be modified at the element.
         *
         * @param {Object} [options]
         *  A map of options controlling the operation. Supports the following
         *  options:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, explicit element attributes that are equal to
         *      the attributes of the current style sheet will be removed from
         *      the element.
         *  @param {Boolean} [options.special=false]
         *      If set to true, allows to change special attributes (attributes
         *      that are marked with the 'special' flag in the attribute
         *      definitions passed to the constructor).
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Function} [options.changeListener]
         *      If specified, will be called if the attributes of the element
         *      have been changed. Will be called in the context of this style
         *      sheet container instance. Receives the passed element as first
         *      parameter, the old explicit attribute set as second parameter,
         *      and the new explicit attribute set as third parameter.
         *
         * @returns {StyleSheets}
         *  A reference to this style sheets container.
         */
        this.setElementAttributes = function (element, attributes, options) {

            var // whether to remove element attributes equal to style attributes
                clear = Utils.getBooleanOption(options, 'clear', false),
                // whether a preview element is rendered
                preview = Utils.getBooleanOption(options, 'preview', false),
                // change listener notified for changed attributes
                changeListener = Utils.getFunctionOption(options, 'changeListener'),
                // new style sheet identifier
                styleId = attributes.styleId,

                // the element, as jQuery object
                $element = $(element),
                // the existing explicit element attributes
                oldElementAttributes = getElementAttributes($element),
                // new explicit element attributes (clone, there may be multiple elements pointing to the same data object)
                newElementAttributes = _.copy(oldElementAttributes, true),
                // merged attribute values from style sheets and explicit attributes
                mergedAttributes = resolveBaseAttributes(element);

            // add/remove new style sheet identifier, or get current style sheet identifier
            if (_.isString(styleId)) {
                newElementAttributes.styleId = styleId;
            } else if (_.isNull(styleId)) {
                delete newElementAttributes.styleId;
            } else {
                styleId = StyleSheets.getElementStyleId(element);
            }

            // collect all attributes of the new or current style sheet, and its parents
            documentStyles.extendAttributes(mergedAttributes, resolveStyleSheetAttributes(styleId, element));

            // add or remove the passed explicit attributes
            attributes = _.copy(attributes, true);
            restrictToSupportedFamilies(attributes);
            _(attributes).each(function (attributeValues, family) {

                var // definitions of own attributes
                    definitions = documentStyles.getAttributeDefinitions(family),
                    // passed attribute values
                    attributeValues = attributes[family],
                    // add an attribute map for the current family
                    elementAttributeValues = null;

                if (_.isObject(attributeValues)) {

                    // add an attribute map for the current family
                    elementAttributeValues = newElementAttributes[family] || (newElementAttributes[family] = {});

                    // update the attribute map with the passed attributes
                    _(attributeValues).each(function (value, name) {
                        if (DocumentStyles.isRegisteredAttribute(definitions, name, options)) {
                            // check whether to clear the attribute
                            if (_.isNull(value) || (clear && _.isEqual(mergedAttributes[family][name], value))) {
                                delete elementAttributeValues[name];
                            } else {
                                elementAttributeValues[name] = value;
                            }
                        }
                    });

                    // remove empty attribute value maps completely
                    if (_.isEmpty(elementAttributeValues)) {
                        delete newElementAttributes[family];
                    }
                }
            });

            // check if any attributes have been changed
            if (!_.isEqual(oldElementAttributes, newElementAttributes)) {

                // write back new explicit attributes to the element
                $element.data('attributes', newElementAttributes);

                // Do really update the element formatting only if 'preview' is true or after the import of the
                // document is completed.
                if (preview || app.isImportFinished()) {
                    // merge explicit attributes into style attributes, and update element formatting
                    documentStyles.extendAttributes(mergedAttributes, resolveElementAttributes(element), options);
                    updateElementFormatting($element, mergedAttributes, { preview: preview });
                }

                // call the passed change listener
                if (_.isFunction(changeListener)) {
                    changeListener.call(this, element, oldElementAttributes, newElementAttributes);
                }
            }

            return this;
        };

        /**
         * Updates the CSS formatting in the specified DOM element, according
         * to its current attribute and style settings.
         *
         * @param {HTMLElement|jQuery} element
         *  The element to be updated. If this object is a jQuery collection,
         *  uses the first DOM node it contains.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Object} [options.baseAttributes]
         *      An attribute set with all attribute values of the ancestor of
         *      the passed element, merged from style sheet and explicit
         *      attributes, keyed by attribute family. Used internally while
         *      updating the formatting of all child elements of a node.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, a GUI element used to preview the formatting of
         *      a style sheet will be rendered. Executes the 'preview' callback
         *      functions instead of the regular 'format' callback functions
         *      registered in the attribute definitions, and calls the preview
         *      handler instead of the format handler registered at this class
         *      instance.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the registered format or preview handler will
         *      be asked to execute asynchronously in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  registered format or preview handler has finished. If the
         *  'options.async' options is set, and the format handler supports
         *  asynchronous execution, the Promise will be resolved sometime in
         *  the future. Otherwise, the returned Promise is already resolved
         *  when this method returns.
         */
        this.updateElementFormatting = function (element, options) {

            var // the identifier of the style sheet referred by the element
                styleId = StyleSheets.getElementStyleId(element),
                // precalculated base attributes passed to this method
                baseAttributes = Utils.getObjectOption(options, 'baseAttributes'),
                // the merged attributes of the passed element
                mergedAttributes = null;

            if (_.isObject(baseAttributes)) {
                mergedAttributes = documentStyles.getDefaultAttributes(styleFamily);
                documentStyles.extendAttributes(mergedAttributes, baseAttributes);
            } else {
                mergedAttributes = resolveBaseAttributes(element);
            }

            // add attributes of the style sheet and its parents, and the explicit attributes of the element
            documentStyles.extendAttributes(mergedAttributes, resolveStyleSheetAttributes(styleId, element));
            documentStyles.extendAttributes(mergedAttributes, resolveElementAttributes(element));

            // update the formatting of the element
            return updateElementFormatting($(element), mergedAttributes, options);
        };

        // initialization -----------------------------------------------------

        // add additional attribute families passed in the options
        supportedFamilies = Utils.getStringOption(initOptions, 'additionalFamilies') || Utils.getArrayOption(initOptions, 'additionalFamilies', []);
        supportedFamilies = _.getArray(supportedFamilies);
        supportedFamilies.push(styleFamily);
        supportedFamilies = _.unique(supportedFamilies);

        // forward other public methods to methods of DocumentStyles
        _(['getCssFontFamily', 'getCssColor', 'getCssTextColor', 'getCssBorderAttributes', 'getCssBorder']).each(function (method) {
            this[method] = _.bind(documentStyles[method], documentStyles);
        }, this);

    } // class StyleSheets

    // static methods ---------------------------------------------------------

    /**
     * Returns the explicit attributes stored in the passed element node.
     *
     * @param {HTMLElement|jQuery} element
     *  The element whose attributes will be returned. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {String} [family]
     *  If specified, extracts the attributes of a specific attribute family
     *  from the attribute map. Otherwise, returns the complete map object with
     *  all attributes mapped by their family.
     *
     * @returns {Object}
     *  The explicit attributes contained in the passed element, as a deep
     *  clone of the original attribute map.
     */
    StyleSheets.getExplicitAttributes = function (element, family) {
        return getElementAttributes($(element), { family: family, clone: true });
    };

    /**
     * Returns the identifier of the style sheet referred by the passed
     * element.
     *
     * @param {HTMLElement|jQuery} element
     *  The element whose style sheet identifier will be returned. If this
     *  object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {String|Null}
     *  The style sheet identifier at the passed element.
     */
    StyleSheets.getElementStyleId = function (element) {
        var styleId = getElementAttributes($(element)).styleId;
        return _.isString(styleId) ? styleId : null;
    };

    /**
     * Returns whether the passed elements contain equal formatting attributes.
     *
     * @param {HTMLElement|jQuery} element1
     *  The first element whose formatting attributes will be compared with the
     *  attributes of the other passed element. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} element2
     *  The second element whose formatting attributes will be compared with
     *  the attributes of the other passed element. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether both elements contain equal formatting attributes.
     */
    StyleSheets.hasEqualElementAttributes = function (element1, element2) {
        return _.isEqual(getElementAttributes($(element1)), getElementAttributes($(element2)));
    };

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: StyleSheets });

});
