/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2012 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/editframework/app/mailactions',
    ['io.ox/core/extensions',
     'io.ox/core/extPatterns/links',
     'io.ox/core/notifications',
     'io.ox/mail/api',
     'io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/extensionregistry',
     'gettext!io.ox/office/files'
    ], function (ext, links, CoreNotification, MailApi, Utils, ExtensionRegistry, gt) {

    'use strict';

    // static class MailActions ===============================================

    var MailActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Mail application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {FileExtensions} fileExtensions
     *  A singleton object containing configuration settings of all file
     *  extensions supported by this application instance.
     *
     */
    MailActions.create = function (appBaseName) {

        var // the module name of the edit application
            ACTION_POINT = 'io.ox/mail/office/' + appBaseName,
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(options) {
            ox.launch(MODULE_NAME + '/main', options);
        }

        // initialization -----------------------------------------------------

        // Copy mail attachment to Files and edit
        new links.Action(ACTION_POINT + 'edit-as-new', {
            id: 'save',
            capabilities: 'infostore',
            requires: function (e) {
                return e.collection.has('one') && ExtensionRegistry.isEditable(e.context.filename, MODULE_NAME);
            },
            multiple: function (list) {
                MailApi.saveAttachments(list)
                    .done(function (result) {
                        CoreNotification.yell('success', gt('Document has been stored in your default folder as "%1$s".', _.noI18n(result[0].data.filename)));
                        launchApplication({ action: 'load', file: result[0].data });
                    })
                    .fail(CoreNotification.yell);
            }
        });

        // An action link to edit files attached to an e-mail
        ext.point('io.ox/mail/attachment/links').extend(new links.Link({
            id: appBaseName + '_edit',
            index: 450,
            label: gt('Edit as new'),
            ref: ACTION_POINT + 'edit-as-new'
        }));
    };

    // exports ================================================================

    return MailActions;

});
