/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/model/modelobject',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/object/baseobject'
    ], function (Events, Utils, BaseObject) {

    'use strict';

    // class ModelObject ======================================================

    /**
     * An abstract model object that allows to trigger change events to event
     * listeners. As long as the application still imports the document, no
     * events will be triggered to the event listeners.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends Events
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {Object} [initOptions]
     *  A map with additional options for this instance. The following options
     *  are supported:
     *  @param {Boolean} [initOptions.silent=false]
     *      If set to true, this instance will never trigger any change events.
     *      Thus, specific sub classes may chose to be always silent although
     *      their base class derives from this class.
     */
    function ModelObject(app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);
        if (!Utils.getBooleanOption(initOptions, 'silent', false)) {
            Events.extend(this);
        }

        // initialization -----------------------------------------------------

        if (!this.events) {
            // add dummy trigger() method for silent objects
            this.trigger = function () { return this; };
        } else if (!app.isImportFinished()) {
            // start triggering events after the document has been imported
            this.listenTo(app, 'docs:import:success', function () {
                self.trigger = self.events.trigger;
            });
            this.trigger = function () { return this; };
        }

        if (this.events) {
            this.registerDestructor(_.bind(this.events.destroy, this.events));
        }

    } // class ModelObject

    // exports ================================================================

    return _.makeExtendable(ModelObject);

});
