/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/config',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mixedborder',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/utils/clipboard',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/view/cellcollection',
     'io.ox/office/spreadsheet/view/statusbar',
     'io.ox/office/spreadsheet/view/gridpane',
     'io.ox/office/spreadsheet/view/headerpane',
     'io.ox/office/spreadsheet/view/cornerpane',
     'io.ox/office/spreadsheet/view/controls',
     'io.ox/office/spreadsheet/view/mixin/selectionmixin',
     'io.ox/office/spreadsheet/view/mixin/celleditmixin',
     'io.ox/office/spreadsheet/view/mixin/dialogsmixin',
     'gettext!io.ox/office/spreadsheet',
     'less!io.ox/office/spreadsheet/view/style.less'
    ], function (Utils, KeyCodes, Config, Pane, ToolBox, Border, MixedBorder, EditView, SheetUtils, PaneUtils, Clipboard, Operations, CellCollection, StatusBar, GridPane, HeaderPane, CornerPane, Controls, SelectionMixin, CellEditMixin, DialogsMixin, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button,
        Label = Controls.Label,
        UnitField = Controls.UnitField,
        RadioList = Controls.RadioList,

        // all available zoom factors
        ZOOM_FACTORS = [0.5, 0.75, 1, 1.25, 1.5, 1.75, 2, 2.5, 3, 4, 5, 6, 8],

        // the size of the freeze separator nodes, in pixels
        FREEZE_SIZE = 1,

        // the size of the split separator nodes, in pixels
        SPLIT_SIZE = 2,

        // the highlighted inner size of the split tracking node, in pixels
        TRACKING_SIZE = 4,

        // the additional margin of the split tracking nodes, in pixels
        TRACKING_MARGIN = 3,

        // the position offset of tracking nodes compared to split lines
        TRACKING_OFFSET = (TRACKING_SIZE - SPLIT_SIZE) / 2 + TRACKING_MARGIN,

        // the minimum size of grid/header panes, in pixels
        MIN_PANE_SIZE = 10,

        // default settings for a pane side
        DEFAULT_PANE_SIDE_SETTINGS = { visible: true, offset: 0, size: 0, frozen: false, showOppositeScroll: false, hiddenSize: 0 },

        // dummy sheet layout data, used before first view update message for a sheet
        DEFAULT_SHEET_DATA = { sheet: -1, usedCols: 0, usedRows: 0 },

        // default layout data for mixed border attributes
        DEFAULT_BORDER_DATA = {
            borderTop: {},
            borderBottom: {},
            borderLeft: {},
            borderRight: {}
        },

        // default layout data for subtotal results
        DEFAULT_SUBTOTAL_DATA = {
            cells: 0,
            numbers: 0,
            sum: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY,
            average: Number.NaN
        },

        // maps border names to the names of the opposite borders and relative cell position
        OPPOSITE_BORDERS = {
            borderTop:    { borderName: 'borderBottom', cols:  0, rows: -1 },
            borderBottom: { borderName: 'borderTop',    cols:  0, rows:  1 },
            borderLeft:   { borderName: 'borderRight',  cols: -1, rows:  0 },
            borderRight:  { borderName: 'borderLeft',   cols:  1, rows:  0 }
        },

        // default options for unit fields (column width and row height)
        DEFAULT_UNIT_FIELD_OPTIONS = {
            width: 70,
            css: { textAlign: 'right' },
            min: 0,
            precision: 100,
            smallStep: 100,
            largeStep: 500,
            roundStep: true
        },

        // the maximum number of returned search result hit
        MAX_SEARCH_RESULT_COUNT = 100,

        // clipboard client and server id to identify the copy data when pasting.
        clipboardId = { client: null, server: null };

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'change:visiblesheets'
     *      After the collection of visible sheets has been changed.
     * - 'before:activesheet'
     *      Before another sheet will be activated. Event handlers receive the
     *      zero-based index of the old active sheet in the document.
     * - 'change:activesheet'
     *      After another sheet has been activated. Event handlers receive the
     *      zero-based index of the new active sheet in the document, and the
     *      sheet model instance of the new active sheet.
     * - 'change:layoutdata'
     *      After the selection layout data has been updated, either directly
     *      or from the response data of a server view update.
     * - 'change:sheetviewattributes'
     *      After the view attributes of the active sheet have been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes of the active sheet with their new values.
     * - 'change:selection'
     *      Will be triggered directly after the 'change:sheetviewattributes'
     *      event caused by a changed selection, for convenience. Event
     *      handlers receive the new selection.
     * - 'celledit:enter', 'celledit:change', 'celledit:leave'
     *      After entering or leaving the cell in-place edit mode, or changing
     *      text while edit mode is active in any grid pane. Event handlers
     *      receive the position identifier of the grid pane.
     * - 'insert:columns', 'delete:columns', 'change:columns'
     *      The column collection in the active sheet has been changed. See
     *      class 'ColRowCollection' for details.
     * - 'insert:rows', 'delete:rows', 'change:rows'
     *      The row collection in the active sheet has been changed. See class
     *      'ColRowCollection' for details.
     * - 'insert:merged', 'delete:merged', 'change:merged'
     *      The merge collection in the active sheet has been changed. See
     *      class 'MergeCollection' for details.
     * - 'insert:drawing', 'delete:drawing', 'change:drawing'
     *      The collection of drawing objects in the active sheet has been
     *      changed. See class 'DrawingCollection' for details.
     *
     * @constructor
     *
     * @extends EditView
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this spreadsheet view.
     */
    function SpreadsheetView(app) {

        var // self reference
            self = this,

            // the scrollable root DOM node containing the headers and panes
            rootNode = $('<div>').addClass('abs pane-root'),

            // the top-left corner pane
            cornerPane = null,

            // settings for all pane sides, mapped by pane side identifier
            paneSideSettings = {},

            // the row/column header panes, mapped by position keys
            headerPanes = {},

            // cell collections for all grid panes
            cellCollections = {},

            // the grid panes, mapped by position keys
            gridPanes = {},

            // the status bar below the grid panes
            statusBar = null,

            // the status label at the bottom border
            statusLabel = new Controls.StatusLabel(app, { type: 'info', fadeOut: true }),

            // the split line separating the left and right panes
            colSplitLine = $('<div>').addClass('split-line columns'),

            // the split line separating the top and bottom panes
            rowSplitLine = $('<div>').addClass('split-line rows'),

            // the split tracking node separating left and right panes
            colSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking columns')
                .css({ width: TRACKING_SIZE + 'px', padding: '0 ' + TRACKING_MARGIN + 'px' }),

            // the split tracking node separating top and bottom panes
            rowSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking rows')
                .css({ height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px 0' }),

            // the split tracking node covering the intersection of the other tracking points
            centerSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking columns rows')
                .css({ width: TRACKING_SIZE + 'px', height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px' }),

            // all split tracking nodes, as jQuery collection
            allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode),

            // tracking overlay nodes for column/row resizing
            resizeOverlayNode = $('<div>')
                .addClass('abs resize-tracking')
                .append($('<div>').addClass('abs leading'), $('<div>').addClass('abs trailing')),

            // drop-down menu control for format codes
            formatCodePicker = null,

            // the collections of visible and hidden sheets
            visibleSheets = [],
            hiddenSheets = [],

            // the spreadsheet document model, and the style sheet containers
            model = null,
            documentStyles = null,
            cellStyles = null,

            // the sheet model instance and collections of the active sheet
            activeSheetModel = null,
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            drawingCollection = null,

            // settings of the active sheet (index and size of used area)
            sheetSettings = _.copy(DEFAULT_SHEET_DATA, true),

            // the contents and formatting of the selection and active cell
            selectionSettings = null,

            // search results
            searchLayoutData = {},

            activeSearchQuery = '',

            // default current selected search occurrence index in the iteration
            searchResultPointer = -1,

            replaceAction = false,

            // format painter
            painterAttrs = {},
            stateChange = false,
            isHyper = true,

            //autoSum
            sumFormula = [];

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            deferredInitHandler: deferredInitHandler,
            grabFocusHandler: grabFocusHandler,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, top: 20, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        SelectionMixin.call(this, app, rootNode);
        CellEditMixin.call(this, app, rootNode);
        DialogsMixin.call(this, app, rootNode);

        // private methods ----------------------------------------------------

        /**
         * Invokes the specified method at all cell collections that are
         * currently contained by visible grid panes.
         */
        function invokeCellCollectionMethod(method) {

            var // the arguments passed to the cell collection method
                methodArgs = _.toArray(arguments).slice(1),
                // the results of the invocations
                results = [];

            _(cellCollections).each(function (cellCollection, panePos) {
                if (gridPanes[panePos].hasLayerRange()) {
                    results.push(cellCollection[method].apply(cellCollection, methodArgs));
                }
            });
            return results;
        }

        /**
         * Tries to find a cell collection that contains the cell range
         * completely.
         *
         * @param {Object} range
         *  The logical address of the cell range.
         *
         * @returns {CellCollection|Null}
         *  The visible cell collection containing the range. If no collection
         *  has been found, null will be returned.
         */
        function getVisibleCellCollection(range) {
            return _(cellCollections).find(function (cellCollection, panePos) {
                return gridPanes[panePos].hasLayerRange() && cellCollection.containsRange(range);
            });
        }

        /**
         * Calls the passed iterator function for all visible cells in the
         * passed cell ranges. For each cell range, a cell collection will be
         * searched containing the range completely. If no cell collection
         * contains a specific range, it will be skipped in the iteration
         * process, or optionally the iteration will be stopped.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times. See the method
         *  'CellCollection.iterateCellsInRanges()' for details about the
         *  parameters passed to the iterator.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method.
         *  Supports all options also supported by the method
         *  'CellCollection.iterateCellsInRanges()'. Additionally, the
         *  following options are supported:
         *  @param {Object} [options.skipMissingRanges=false]
         *      If set to true, range not covered by any of the existing cell
         *      collections will be skipped silently. By default, the iteration
         *      process will be stopped in this case (this method returns the
         *      Utils.BREAK object immediately).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, or if a range has been
         *  found without corresponding cell collection, otherwise undefined.
         */
        function iterateCellsInRanges(ranges, iterator, options) {

            var // whether to skip ranges not contained by any collection
                skipMissingRanges = Utils.getBooleanOption(options, 'skipMissingRanges', false);

            // process each range for its own
            return Utils.iterateArray(_.getArray(ranges), function (range) {
                var cellCollection = getVisibleCellCollection(range);
                return cellCollection ? cellCollection.iterateCellsInRanges(range, iterator, options) : skipMissingRanges ? undefined : Utils.BREAK;
            });
        }

        /**
         * Imports the passed settings for the active cell (in the property
         * 'active'), the borders of the selection (in the property 'borders'),
         * and the subtotal results (in the property 'subtotals').
         */
        function importSelectionSettings(settings) {

            // clear the old selection settings
            selectionSettings = {};

            // import active cell
            if (_.isObject(settings.active)) {
                var attrs = Utils.getObjectOption(settings.active, 'attrs', null);
                selectionSettings.active = CellCollection.extractCellData(settings.active);
                if (_.isObject(attrs)) {
                    selectionSettings.active.attributes = cellStyles.getMergedAttributes(attrs);
                    selectionSettings.active.explicit = _.copy(attrs, true);
                }
            } else {
                selectionSettings.active = _.copy(CellCollection.DEFAULT_CELL_DATA, true);
            }

            // fall-back to default settings for cell attributes
            if (!('attributes' in selectionSettings.active)) {
                selectionSettings.active.attributes = cellCollections.bottomRight.getDefaultAttributes();
                selectionSettings.active.explicit = {};
            }

            // import mixed border attributes
            selectionSettings.borders = _.copy(DEFAULT_BORDER_DATA, true);
            if (_.isObject(settings.borders)) {
                _(selectionSettings.borders).extend(settings.borders);
                // delete inner borders for specific selections
                if (!self.hasMultipleColumnsSelected()) { delete selectionSettings.borders.borderInsideVert; }
                if (!self.hasMultipleRowsSelected()) { delete selectionSettings.borders.borderInsideHor; }
            }

            // import subtotal results
            selectionSettings.subtotals = _.copy(DEFAULT_SUBTOTAL_DATA, true);
            if (_.isObject(settings.subtotals)) {
                _(selectionSettings.subtotals).extend(settings.subtotals);
                selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;
            }

            // import selection lock data
            selectionSettings.locked = null;
            if (_.isBoolean(settings.locked)) {
                selectionSettings.locked = settings.locked;
            }

        }

        /**
         * Updates the selection part of the view layout data from the current
         * row, column, and cell collections.
         */
        var updateSelectionSettings = app.createDebouncedMethod($.noop, function () {
            var // the current cell selection
                selection = self.getSelection(),
                // the address of the active cell in the passed selection
                activeCell = selection.activeCell,
                // collection entry of the active cell
                activeCellEntry = self.getCellEntry(activeCell),
                // object to collect and sort all border types
                allBorders = {},
                // result of the range iterator (Utils.BREAK if any range is not contained completely in a cell collection)
                result = null,
                // the info if current selection is editable
                selectionLocked = false;

            // iterate all ranges, bail out if a range not covered by any cell collection has been found
            result = iterateCellsInRanges(selection.ranges, function (cellData, originalRange) {

                var // cloning the global OPPOSITE_BORDERS for local modifications
                    oppositeBorders = _.copy(OPPOSITE_BORDERS, true),
                    // checking the cellData for a merged range
                    mergedRange = mergeCollection.getMergedRange(cellData.address),
                    // additional columns of a merged cell
                    additionalCols = 0,
                    // additional rows of a merged cell
                    additionalRows = 0;

                if (mergedRange) {
                    if (_.isEqual(mergedRange.start, cellData.address)) {
                        // modifying opposite borders for merged cells
                        oppositeBorders.borderRight.cols = SheetUtils.getColCount(mergedRange);
                        oppositeBorders.borderBottom.rows = SheetUtils.getRowCount(mergedRange);
                        additionalCols = oppositeBorders.borderRight.cols - 1;
                        additionalRows = oppositeBorders.borderBottom.rows - 1;
                    } else {
                        return; // ignoring hidden cells inside merged ranges
                    }
                }

                // process all outer borders of the cell
                _(oppositeBorders).each(function (OPPOSITE_BORDER, borderName) {
                    var // cell data of the preceding/next cell (taking care of merged cells)
                        nextCellData = self.getCellEntry([cellData.address[0] + OPPOSITE_BORDER.cols, cellData.address[1] + OPPOSITE_BORDER.rows]),
                        // the border value of the current and adjacent cell
                        thisBorder = cellData.attributes.cell[borderName],
                        nextBorder = nextCellData ? nextCellData.attributes.cell[OPPOSITE_BORDER.borderName] : null,
                        // whether the borders are visible
                        thisBorderVisible = Border.isVisibleBorder(thisBorder),
                        nextBorderVisible = Border.isVisibleBorder(nextBorder),
                        // the target border name for the border collection (inner or outer borders)
                        targetBorderName = null,
                        targetBorders = null;

                    // get the target border name for the border collection
                    switch (borderName) {
                    case 'borderLeft':
                        targetBorderName = (cellData.address[0] === originalRange.start[0]) ? 'borderLeft' : null;
                        break;
                    case 'borderRight':
                        targetBorderName = (cellData.address[0] + additionalCols === originalRange.end[0]) ? 'borderRight' : 'borderInsideVert';
                        break;
                    case 'borderTop':
                        targetBorderName = (cellData.address[1] === originalRange.start[1]) ? 'borderTop' : null;
                        break;
                    case 'borderBottom':
                        targetBorderName = (cellData.address[1] + additionalRows === originalRange.end[1]) ? 'borderBottom' : 'borderInsideHor';
                        break;
                    }

                    // skip inner border, if already processed by preceding cell
                    if (!_.isString(targetBorderName)) { return; }
                    targetBorders = allBorders[targetBorderName] || (allBorders[targetBorderName] = []);

                    // insert all visible borders into the border collection
                    if (thisBorderVisible) { targetBorders.push(thisBorder); }
                    if (nextBorderVisible) { targetBorders.push(nextBorder); }

                    // ignore invisible border, if the other border is visible
                    if (!thisBorderVisible && !nextBorderVisible) { targetBorders.push(Border.NONE); }
                });
                // This selection is editable only if all its cells are editable, and sheet is not protected
                if (activeSheetModel.isProtected() && !cellData.attributes.cell.unlocked) { selectionLocked = true; }
            });

            // not all cells available in the collections, request new data from
            // server, keep old selection settings until that data has been
            // received to prevent flickering icons in the side pane
            if (result === Utils.BREAK) {
                requestSelectionUpdate(selection);
                return;
            }

            // initialize the selection settings object with default data
            importSelectionSettings({});

            // add data of active cell from cell collection
            if (activeCellEntry) {
                selectionSettings.active = activeCellEntry;
            }

            // all border information collected and sorted in allBorders
            _(allBorders).each(function (borders, borderName) {
                selectionSettings.borders[borderName] = MixedBorder.mixBorders(borders);
            });

            // iterate unified ranges to calculate subtotals (no check for result needed, all cells are available)
            iterateCellsInRanges(SheetUtils.getUnifiedRanges(selection.ranges), function (cellEntry) {
                if (!CellCollection.isBlank(cellEntry)) {
                    selectionSettings.subtotals.cells += 1;
                    if (CellCollection.isNumber(cellEntry) && _.isFinite(cellEntry.result)) {
                        selectionSettings.subtotals.numbers += 1;
                        selectionSettings.subtotals.sum += cellEntry.result;
                        selectionSettings.subtotals.min = Math.min(selectionSettings.subtotals.min, cellEntry.result);
                        selectionSettings.subtotals.max = Math.max(selectionSettings.subtotals.max, cellEntry.result);
                    }
                }
            }, { existing: true });
            selectionSettings.subtotals.average = selectionSettings.subtotals.sum / selectionSettings.subtotals.numbers;

            // add selection locked attribute
            selectionSettings.locked = selectionLocked;

            // notify listeners
            self.trigger('change:layoutdata');

        }, { delay: 200 });

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            Utils.log('SpreadsheetView.requestSelectionUpdate(): sheet=' + sheetSettings.sheet);
            self.requestUpdate({ selection: { ranges: selection.ranges, active: selection.activeCell } });
        }

        /**
         * Triggers a 'before:activesheet' event in preparation to change the
         * current active sheet.
         */
        function prepareSetActiveSheet() {
            // do nothing if no sheet is currently active (initially after import)
            if (sheetSettings.sheet >= 0) {
                self.leaveCellEditMode('cell');
                self.trigger('before:activesheet', sheetSettings.sheet);
            }
        }

        /**
         * Activates the specified sheet, and triggers a 'change:activesheet'
         * event with the new sheet index.
         *
         * @param {Number} sheet
         *  The zero-based index of the new active sheet.
         */
        function setActiveSheet(sheet) {

            // initialize the sheet layout data
            sheetSettings = Utils.extendOptions(DEFAULT_SHEET_DATA, { sheet: sheet });

            // get model instance and collections of the new active sheet
            activeSheetModel = model.getSheetModel(sheet);
            colCollection = activeSheetModel.getColCollection();
            rowCollection = activeSheetModel.getRowCollection();
            mergeCollection = activeSheetModel.getMergeCollection();
            drawingCollection = activeSheetModel.getDrawingCollection();

            // notify listeners
            self.trigger('change:activesheet', sheet, activeSheetModel);
            self.trigger('change:selection', self.getSelection());

            // refresh the layout of the header and grid panes, send initial update request
            initializePanes();
            self.requestUpdate({ cells: model.getSheetRange() });

            // set cursor to a visible cell, if the sheet has not been displayed yet
            if (!activeSheetModel.getViewAttribute('visited')) {
                // deferred as a workaround for missing notification after import
                app.executeDelayed(function () {
                    self.selectCell(self.getVisibleCell([0, 0], 'next') || [0, 0]);
                    activeSheetModel.setViewAttribute('visited', true);
                });
            }
        }

        /**
         * Updates the collection of visible sheets, and triggers a
         * 'change:visiblesheets' event.
         */
        function updateVisibleSheets() {

            visibleSheets = [];
            hiddenSheets = [];

            // collect all supported and visible sheets
            model.iterateSheetModels(function (sheetModel, sheet, name) {

                // ignore unsupported sheet types
                if (!/^(worksheet)$/.test(sheetModel.getType())) { return; }

                // collect visible and hidden sheets, count hidden sheets
                if (sheetModel.getMergedAttributes().sheet.visible) {
                    visibleSheets.push({ sheet: sheet, name: name });
                } else {
                    hiddenSheets.push({ sheet: sheet, name: name });
                }
            });

            // notify listeners
            self.trigger('change:visiblesheets');
        }

        /**
         * Returns the array index of the specified sheet in the collection of
         * all visible sheets.
         */
        function getVisibleIndex(sheet) {
            var arrayIndex = _(visibleSheets).sortedIndex({ sheet: sheet }, 'sheet');
            return ((0 <= arrayIndex) && (arrayIndex < visibleSheets.length) && (visibleSheets[arrayIndex].sheet === sheet)) ? arrayIndex : -1;
        }

        /**
         * Handles changed view attributes in the active sheet. Shows a changed
         * zoom factor in the status label and redraws the view with the new
         * zoom factor, and triggers change events for the attributes.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // changed selection: update selection data (cell contents, subtotals, etc.) and global appearance
            if ('selection' in attributes) {
                updateSelectionSettings();
                rootNode.toggleClass('drawing-selection', attributes.selection.drawings.length > 0);
            }

            // changed zoom factor: show the current zoom in the status label
            if ('zoom' in attributes) {
                _.defer(function () {
                    self.showStatusLabel(
                        //#. %1$d is the current zoom factor, in percent
                        //#, c-format
                        gt('Zoom: %1$d%', _.noI18n(Math.round(attributes.zoom * 100)))
                    );
                });
            }

            // update header and grid panes, after split/freeze settings have changed
            if (Utils.hasProperty(attributes, /^(zoom$|split)/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed column attributes in the active sheet. Updates the
         * size of frozen panes after the effective width of the columns has
         * changed.
         */
        function changeColumnsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('column' in attributes) && Utils.hasProperty(attributes.column, /^(width|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Handles changed row attributes in the active sheet. Updates the size
         * of frozen panes after the effective height of the rows has changed.
         */
        function changeRowsHandler(event, interval, attributes) {
            if (self.hasFrozenSplit() && ('row' in attributes) && Utils.hasProperty(attributes.row, /^(height|visible)$/)) {
                initializePanes();
            }
        }

        /**
         * Prepares insertion of a sheet into the document.
         */
        function beforeInsertSheetHandler(event, sheet) {
            // active sheet will change, if a sheet will be inserted before it
            if (sheet <= sheetSettings.sheet) {
                prepareSetActiveSheet();
            }
        }

        /**
         * Finalizes insertion of a sheet into the document.
         */
        function afterInsertSheetHandler(event, sheet) {

            var // the model of the new sheet
                sheetModel = model.getSheetModel(sheet);

            // registers an event handler that will be called while the sheet is active
            function handleActiveSheetEvent(eventSource, eventType, options) {

                var // the local event handler called before the event will be triggered
                    handlerFunc = Utils.getFunctionOption(options, 'handler'),
                    // the type of the event to be triggered
                    triggerType = Utils.getStringOption(options, 'trigger');

                // handle specified event of the event source object, do nothing if the sheet is not active
                eventSource.on(eventType, function () {
                    if (sheetModel === activeSheetModel) {
                        if (_.isFunction(handlerFunc)) {
                            handlerFunc.apply(self, _.toArray(arguments));
                        }
                        if (_.isString(triggerType)) {
                            self.trigger.apply(self, [triggerType].concat(_.toArray(arguments).slice(1)));
                        }
                    }
                });
            }

            // listen to selection changes in the sheet, notify listeners of the view
            handleActiveSheetEvent(sheetModel, 'change:viewattributes', { handler: changeSheetViewAttributesHandler, trigger: 'change:sheetviewattributes' });

            // listen to changes in the column collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getColCollection(), 'insert:entries', { trigger: 'insert:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'delete:entries', { trigger: 'delete:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'change:entries', { handler: changeColumnsHandler, trigger: 'change:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the row collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'insert:entries', { trigger: 'insert:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'delete:entries', { trigger: 'delete:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'change:entries', { handler: changeRowsHandler, trigger: 'change:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the merge collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'insert:merged', { trigger: 'insert:merged' });
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'delete:merged', { trigger: 'delete:merged' });
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'change:merged', { trigger: 'change:merged' });
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the drawing collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'insert:drawing', { trigger: 'insert:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'delete:drawing', { trigger: 'delete:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'change:drawing', { trigger: 'change:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'triggered', { handler: updateSelectionSettings });

            // copy current zoom from active sheet to new sheet
            if (activeSheetModel) {
                sheetModel.setViewAttribute('zoom', activeSheetModel.getViewAttribute('zoom'));
            }

            // update visible sheets collection, if the visibility of the sheet changes
            sheetModel.on('change:attributes', function (event2, newAttributes, oldAttributes) {
                var activeSheet = self.getActiveSheet({ visible: true });
                if (newAttributes.sheet.visible !== oldAttributes.sheet.visible) {
                    updateVisibleSheets();
                    // activate another sheet, if the active sheet has been hidden
                    if ((sheetModel === activeSheetModel) && !newAttributes.sheet.visible && (visibleSheets.length > 0)) {
                        activeSheet = Utils.minMax(activeSheet, 0, visibleSheets.length - 1);
                        self.setActiveSheet(activeSheet, { visible: true });
                    }
                }
            });

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet <= sheetSettings.sheet) {
                setActiveSheet(sheetSettings.sheet + 1);
            }
        }

        /**
         * Prepares deletion of a sheet from the document.
         */
        function beforeDeleteSheetHandler(event, sheet) {
            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.leaveCellEditMode('discard');
            }
            // active sheet will change, if a sheet before will be deleted, or itself
            if (sheet <= sheetSettings.sheet) {
                prepareSetActiveSheet();
            }
        }

        /**
         * Finalizes deletion of a sheet from the document.
         */
        function afterDeleteSheetHandler(event, sheet) {

            // update the collection of visible sheets
            updateVisibleSheets();

            // update active sheet index
            if (sheet < sheetSettings.sheet) {
                // sheet before active sheet deleted: decrease index of active sheet
                setActiveSheet(sheetSettings.sheet - 1);
            } else if (sheet === sheetSettings.sheet) {
                // active sheet deleted: keep index of active sheet, unless it was the last sheet
                setActiveSheet(Math.min(sheetSettings.sheet, model.getSheetCount() - 1));
            }
        }

        /**
         * Prepares moving a sheet inside the document.
         */
        function beforeMoveSheetHandler() {
            // Bug 31772: cancel cell edit mode to prevent generating additional operations
            if (activeSheetModel) {
                self.leaveCellEditMode('discard');
            }
        }

        /**
         * Finalizes moving a sheet inside the document.
         */
        function afterMoveSheetHandler(evt, from, to) {
            var currentSheet = self.getActiveSheet();
            // change the active index if the active sheet is the moved one
            if (currentSheet === from) {
                prepareSetActiveSheet();
                setActiveSheet(to);
            }

            // update the collection of visible sheets
            updateVisibleSheets();
        }

        /**
         * Handles a renamed sheet.
         */
        function renameSheetHandler() {
            // update the collection of visible sheets
            updateVisibleSheets();
        }

        /**
         * Shows a notification message, if renaming or copying a sheet has
         * failed. Returns a Boolean value reflecting the passed result error
         * code.
         *
         * @param {String} result
         *  The result code received from the document model after a sheet has
         *  been tried to rename or to copy.
         *
         * @returns {Boolean}
         *  Whether the passed result is NOT an error code (success).
         */
        function yellSheetNameError(result) {

            // show an appropriate notification for the result code
            switch (result) {
            case 'empty':
                self.yell('info', gt('The sheet name must not be empty. Please enter a name.'));
                break;
            case 'invalid':
                self.yell('info', gt('This sheet name contains invalid characters. Please ensure that the name does not contain / \\ * ? : [ or ], and does neither start nor end with an apostrophe.'));
                break;
            case 'used':
                self.yell('info', gt('This sheet name is already used. Please enter another name.'));
                break;
            }

            return result.length === 0;
        }

        /**
         * Handles 'update' push messages. If the message data contains a list
         * of changed cells for the active sheet, updates the visible cells in
         * all grid panes.
         *
         * @param {Object} data
         *  The message data of the 'update' push message.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // map of changed data per sheet
                sheets = null,
                // map of changed data of the current sheet
                sheetData = null,
                // changed cell ranges (array of ranges, or single range address)
                ranges = null;

            // update message may not contain any change data
            if (!_.isObject(changedData)) { return; }
            Utils.info('SpreadsheetView.updateNotificationHandler(): notification received:', changedData);

            // process by update type (specific 'cells', or 'all')
            switch (Utils.getStringOption(changedData, 'type')) {
            case 'cells':
                if ((sheets = Utils.getObjectOption(changedData, 'sheets'))) {
                    if ((sheetData = Utils.getObjectOption(sheets, sheetSettings.sheet))) {

                        // extract the addresses of the changed cell ranges
                        if (_.isArray(sheetData.cells) && (sheetData.cells.length > 0)) {
                            ranges = sheetData.cells;
                            _(ranges).each(function (range) { if (!range.end) { range.end = range.start; } });
                        } else {
                            Utils.warn('SpreadsheetView.updateNotificationHandler(): no changed cells');
                        }
                    }
                } else {
                    Utils.warn('SpreadsheetView.updateNotificationHandler(): no sheets defined');
                }
                break;
            case 'all':
                ranges = model.getSheetRange();
                break;
            default:
                Utils.error('SpreadsheetView.updateNotificationHandler(): invalid type in cell update data');
                return;
            }

            // trigger forced update of all grid panes in the active sheet
            if (_.isObject(ranges)) {
                self.requestUpdate({ cells: ranges });
            }

            // update selection layout data (will request selection update from
            // server, if selection is not completely visible)
            updateSelectionSettings();

            // throw away search results if sheet is altered in any way
            var isSearchActive = app.getWindow().search.active;
            if (isSearchActive && !replaceAction) {
                self.searchClear();
            }

        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            var // the event source node
                sourceNode = $(this),
                // whether column split tracking and/or row split tracking is active
                colSplit = sourceNode.hasClass('columns'),
                rowSplit = sourceNode.hasClass('rows'),
                // minimum and maximum position of split lines
                minLeft = cornerPane.getNode().width(),
                maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - SPLIT_SIZE,
                minTop = cornerPane.getNode().height(),
                maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getTrackingSplitLeft() {
                var offset = paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0);
                return (offset - MIN_PANE_SIZE < minLeft) ? minLeft : (offset + MIN_PANE_SIZE > maxLeft) ? maxLeft : offset;
            }

            // returns the Y position of a split line according to the current event
            function getTrackingSplitTop() {
                var offset = paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0);
                return (offset - MIN_PANE_SIZE < minTop) ? minTop : (offset + MIN_PANE_SIZE > maxTop) ? maxTop : offset;
            }

            function updateNodeOffset(node, propName, offset, min, max) {
                node.toggleClass('collapsed', (offset <= min) || (offset >= max))
                    .add(centerSplitTrackingNode)
                    .css(propName, (offset - TRACKING_OFFSET) + 'px');
            }

            function finalizeTracking() {
                allSplitTrackingNodes.removeClass('tracking-active');
                self.grabFocus();
            }

            switch (event.type) {

            case 'tracking:start':
                colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                break;

            case 'tracking:move':
                if (colSplit) { updateNodeOffset(colSplitTrackingNode, 'left', getTrackingSplitLeft(), minLeft, maxLeft); }
                if (rowSplit) { updateNodeOffset(rowSplitTrackingNode, 'top', getTrackingSplitTop(), minTop, maxTop); }
                break;

            case 'tracking:end':
                var splitWidth = null, splitHeight = null;
                if (colSplit) {
                    splitWidth = getTrackingSplitLeft();
                    splitWidth = ((minLeft < splitWidth) && (splitWidth < maxLeft)) ? activeSheetModel.convertPixelToHmm(splitWidth - cornerPane.getNode().width()) : 0;
                } else {
                    splitWidth = activeSheetModel.getSplitWidthHmm();
                }
                if (rowSplit) {
                    splitHeight = getTrackingSplitTop();
                    splitHeight = ((minTop < splitHeight) || (splitHeight < maxTop)) ? activeSheetModel.convertPixelToHmm(splitHeight - cornerPane.getNode().height()) : 0;
                } else {
                    splitHeight = activeSheetModel.getSplitHeightHmm();
                }
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight);
                finalizeTracking();
                break;

            case 'tracking:cancel':
                initializePanes();
                finalizeTracking();
                break;
            }
        }

        /**
         * Handles 'keydown' events from the view root node.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (self.isCellEditMode()) { return; }

            // special handling for drawing selection
            if (self.hasDrawingSelection()) {

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                    if (self.requireEditMode()) { self.deleteDrawings(); }
                    return false;
                }

                return;
            }

            // enter cell edit mode with current cell contents
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                self.enterCellEditMode();
                return false;
            }

            // enter cell edit mode with empty text area
            if (KeyCodes.matchKeyCode(event, 'BACKSPACE')) {
                self.enterCellEditMode({ text: '' });
                return false;
            }

            // clear contents (not formatting) of all sells in the selection
            if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                if (self.requireEditMode()) { self.fillCellRanges(null); }
                return false;
            }
        }

        /**
         * Handles 'keypress' events from the view root node. Starts the cell
         * in-place edit mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {

            var // the initial text to be inserted into the text area
                initialText = null;

            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, or if drawings are selected
            if (self.isCellEditMode() || self.hasDrawingSelection()) { return; }

            // ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            if (((event.charCode > 32) && ((event.ctrlKey || event.metaKey) === event.altKey)) || ((event.charCode === 32) && KeyCodes.matchModifierKeys(event))) {

                // build string from character code
                initialText = String.fromCharCode(event.charCode);

                // percentage number format: add percent sign to number-like first character
                if ((selectionSettings.active.format.cat === 'percent') && (/^[-0-9]$/.test(initialText) || (initialText === app.getDecimalSeparator()))) {
                    initialText += '%';
                }

                // start cell edit mode, and drop the key event
                self.enterCellEditMode({ quick: true, text: initialText, pos: 1 });
                return false;
            }
        }

        /**
         * configure global search for OX Spreadsheet
         */
        function initSearch() {

            // helper variables
            var win = app.getWindow(),
                controller = app.getController(),
                // the ox search panel
                searchDiv = win.nodes.search.attr('tabindex', 0).css('outline', 'none'),
                searchForm = searchDiv.find('form'),
                searchInput = searchForm.find('input[name="query"]'),
                searchButton = searchForm.find('button[data-action="search"]'),
                // additional controls for search & replace
                nextButton = $('<button>')
                    .addClass('btn margin-right')
                    .append(Utils.createIcon('icon-chevron-right'))
                    .attr('tabindex', 0),
                previousButton = $('<button>')
                    .addClass('btn')
                    .append(Utils.createIcon('icon-chevron-left'))
                    .css({'border-radius': '4px 0 0 4px'})
                    .attr('tabindex', 0),
                replaceInput = $('<input>', { type: 'text', name: 'replace' })
                    .attr({placeholder: gt('Replace with ...'), tabindex: 0 })
                    .addClass('search-query input-large'),
                replaceButton = $('<button>')
                    .addClass('btn')
                    .css({'border-radius': '0'})
                    .attr('tabindex', 0)
                    .text(gt('Replace')),
                replaceAllButton = $('<button>')
                    .addClass('btn')
                    .attr('tabindex', 0)
                    .text(/*#. in search&replace: replace all occurrences */ gt('Replace all')),
                clearReplaceInputIcon = Utils.createIcon('icon-remove clear-query');

            // add next/prev search hit,replace input, replace and replace all buttons
            searchForm.find('div.input-append').append(
                $('<div>').addClass('btn-group').css('vertical-align', 'top').append(previousButton, nextButton),
                $('<label>').addClass('search-query-container').append(replaceInput, clearReplaceInputIcon),
                replaceButton,
                replaceAllButton
             );

            // init replacement
            win.search.replacement = '';
            win.search.replacementInput = replaceInput;

            // Handles the 'search' event from the search panel.
            function searchHandler() {
                // do a new search if its not done yet or  if search query has changed.
                if (_.isUndefined(searchLayoutData.sheets) || activeSearchQuery !== win.search.query) {
                    controller.executeItem('document/search/search', win.search.query);
                } else {
                    // if there are search results, iterate them with the enter key.
                    self.searchIterate();
                }
            }

            // when search bar is opened
            function searchOpenHandler() {
                _.defer(self.refreshPaneLayout);
            }

            // when search bar is closed
            function searchCloseHandler() {
                controller.executeItem('document/search/close');
                _.defer(self.refreshPaneLayout);
            }

            // Handles the 'orientationchange' event of mobile devices
            function orientationChangeHandler(event) {
                // use event.orientation and media queries for the orientation detection.
                // 'window.orientation' depends on where the device defines it's top
                // and therefore says nothing reliable about the orientation.
                if (event && event.orientation === 'landscape') {
                    setLargeSearchPane();
                } else if (event && event.orientation === 'portrait') {
                    setSmallSearchPane();
                } else if (Modernizr.mq('(orientation: landscape)')) {
                    setLargeSearchPane();
                } else if (Modernizr.mq('(orientation: portrait)')) {
                    setSmallSearchPane();
                }
            }

            // Handles tab and esc key events
            function keyHandler(event) {
                var items, focus, index, down;

                if (event.keyCode === KeyCodes.TAB) {
                    event.preventDefault();

                    items = $(this).find('[tabindex]');
                    focus = $(document.activeElement);
                    down = ((event.keyCode === KeyCodes.TAB) && !event.shiftKey);

                    index = (items.index(focus) || 0);
                    if (!down) {
                        index--;
                    } else {
                        index++;
                    }

                    if (index >= items.length) {
                        index = 0;
                    } else if (index < 0) {
                        index = items.length - 1;
                    }
                    items[index].focus();
                    return false;

                } else if (event.keyCode === KeyCodes.ESCAPE) {
                    win.search.close();
                }
            }

            // Handles the 'change' event of the search input.
            // Stops the propagation of the 'change' event to all other attached handlers
            // if the focus stays inside the search panel.
            // With that moving the focus out of the search input e.g. via tab
            // doesn't trigger the search and so navigation to other search panel controls
            // is possible.
            function searchInputChangeHandler(event) {
                event.stopImmediatePropagation();
                // if we stop the event propagation we need to update the query here
                win.search.query = win.search.getQuery();
                return false;
            }

            function setLargeSearchPane() {
                searchForm.find('input').addClass('input-large').removeClass('input-medium');
            }

            function setSmallSearchPane() {
                searchForm.find('input').addClass('input-medium').removeClass('input-large');
            }

            function editmodeChangeHandler(event, editMode) {
                replaceButton.toggleClass('disabled', !editMode);
                replaceAllButton.toggleClass('disabled', !editMode);
            }

            // set the tool tips
            Utils.setControlTooltip(searchInput, gt('Find text'), 'bottom');
            Utils.setControlTooltip(searchInput.siblings('i'), gt('Clear'), 'bottom');
            Utils.setControlTooltip(searchForm.find('button[data-action=search]'), gt('Start search'), 'bottom');
            Utils.setControlTooltip(previousButton, gt('Select previous search result'), 'bottom');
            Utils.setControlTooltip(nextButton, gt('Select next search result'), 'bottom');
            Utils.setControlTooltip(replaceInput, gt('Replacement text'), 'bottom');
            Utils.setControlTooltip(replaceButton, gt('Replace selected search result and select the next result'), 'bottom');
            Utils.setControlTooltip(replaceAllButton, gt('Replace all search results'), 'bottom');
            Utils.setControlTooltip(searchForm.find('a.close'), gt('Close search'), 'bottom');

            // accessibility addition
            searchForm.addClass('f6-target');

            // event handling
            searchInput.on('keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'ENTER')) {
                    // prevent that enter key causing lost of focus
                    event.preventDefault();
                    // do the search
                    searchHandler();
                }
            });
            searchButton.on('click', searchHandler);
            previousButton.on('click', function () { controller.executeItem('document/search/previousResult'); });
            nextButton.on('click', function () { controller.executeItem('document/search/nextResult'); });
            replaceButton.on('click', function () { controller.executeItem('document/search/replaceSelected'); });
            replaceAllButton.on('click', function () { controller.executeItem('document/search/replaceAll'); });
            searchDiv.on('keydown', keyHandler);
            replaceInput.on('change', function () { win.search.replacement = Utils.cleanString(this.value || ''); });
            clearReplaceInputIcon.on('click', function () { replaceInput.val(''); win.search.replacement = ''; });

            // TODO: We should make the search able to handle tab between multiple controls in the search bar,
            // registering as first handler to the 'change' event is just an interim solution.
            Utils.bindAsFirstHandler(searchInput, 'change', searchInputChangeHandler);

            // subscribe the events handlers above
            win.on('search:open', searchOpenHandler)
                .on('search:close', searchCloseHandler);

            // handle edit mode change situation
            model.on('change:editmode', editmodeChangeHandler);

            // when scrolling on mobile devices we need to
            // place the search & replace into the view port
            if (Modernizr.touch) {
                win.on('search:open', searchOpenHandler);
                $(window).on('orientationchange', orientationChangeHandler);
                orientationChangeHandler();
            }
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            // store reference to document model and the style sheet containers
            model = app.getModel();
            documentStyles = model.getDocumentStyles();
            cellStyles = documentStyles.getStyleSheets('cell');

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // set CSS class for formatting dependent on edit mode (e.g. mouse pointers)
            model.on('change:editmode', function (event, editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // listen for sheet attributes changes, deactivate column/row resizing if necessary
            model.on('change:sheetattributes', function () {
                rootNode.toggleClass('protected-sheet', activeSheetModel.isProtected());
            });

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            _(gridPanes).each(function (gridPane) { rootNode.append(gridPane.getNode()); });
            _(headerPanes).each(function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLine, rowSplitLine, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // enable tracking, register event handlers
            allSplitTrackingNodes
                .attr('tabindex', 1) // make focusable to prevent immediate 'tracking:cancel' event
                .enableTracking()
                .on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

            // handle generic key events
            rootNode.on({ keydown: keyDownHandler, keypress: keyPressHandler });

            // update cell collections after generating and applying internal 'setCellContents' operations
            model.registerCellOperationHandler(Operations.SET_CELL_CONTENTS, function (sheetModel, start, operation) {
                if (!_.isArray(operation.contents) || !_(operation.contents).all(_.isArray)) { return false; }
                if (sheetModel === activeSheetModel) {
                    invokeCellCollectionMethod('setCellContents', start, operation);
                }
                return true;
            });

            // update cell collections after generating and applying internal 'fillCellRange' operations
            model.registerRangeOperationHandler(Operations.FILL_CELL_RANGE, function (sheetModel, range, operation) {
                if (sheetModel === activeSheetModel) {
                    invokeCellCollectionMethod('fillCellRange', range, operation);
                }
                return true;
            });

            // update cell collections after generating and applying internal 'clearCellRange' operations
            model.registerRangeOperationHandler(Operations.CLEAR_CELL_RANGE, function (sheetModel, range) {
                if (sheetModel === activeSheetModel) {
                    invokeCellCollectionMethod('clearCellRange', range);
                }
                return true;
            });

            // update cell collections after generating and applying internal 'autoFill' operations
            model.registerRangeOperationHandler(Operations.AUTO_FILL, function (sheetModel, range, operation) {
                if (sheetModel === activeSheetModel) {
                    invokeCellCollectionMethod('autoFill', range, operation);
                }
                return true;
            });

            // handle sheet events
            model.on({
                'insert:sheet:before': beforeInsertSheetHandler,
                'insert:sheet:after': afterInsertSheetHandler,
                'delete:sheet:before': beforeDeleteSheetHandler,
                'delete:sheet:after': afterDeleteSheetHandler,
                'move:sheet:before': beforeMoveSheetHandler,
                'move:sheet:after': afterMoveSheetHandler,
                'rename:sheet': renameSheetHandler
            });

            // application notifies changed contents/results of cells
            app.on('docs:update', updateNotificationHandler);

            // activate a sheet (TODO: use document view settings)
            app.on('docs:import:after', function () {
                self.setActiveSheet(0, { visible: true }).grabFocus();
            });

            // update position of grid panes, when general view layout changes
            app.on('docs:import:success', function () {
                self.on('refresh:layout', initializePanes);
            });

            // create the status bar
            self.addPane(statusBar = new StatusBar(app));
            app.on('docs:state:error', function () { statusBar.hide(); });

            // trigger a change:selection event for convenience
            self.on('change:sheetviewattributes', function (event, attributes) {
                if ('selection' in attributes) {
                    self.trigger('change:selection', attributes.selection);
                }
            });

            self.on('change:activesheet', function (event, sheet, sheetModel) {
                rootNode.toggleClass('protected-sheet', sheetModel.isProtected());
            });

            // process events triggered by header panes
            _(headerPanes).each(function (headerPane, paneSide) {

                var columns = PaneUtils.isColumnSide(paneSide),
                    offsetAttr = columns ? 'left' : 'top',
                    sizeAttr = columns ? 'width' : 'height',
                    showResizeTimer = null;

                function showResizerOverlayNode(offset, size) {
                    resizeOverlayNode.attr('data-orientation', columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                    // Delay displaying the overlay nodes, otherwise they cover
                    // the resizer drag nodes of the header pane which will
                    // interfere with double click detection.
                    showResizeTimer = app.executeDelayed(function () {
                        rootNode.addClass('tracking-active');
                    }, { delay: 200 });
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisiblePosition().offset;
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    showResizeTimer.abort();
                    showResizeTimer = null;
                    rootNode.removeClass('tracking-active');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                // visualize resize tracking from header panes
                headerPane.on({
                    'resize:start': function (event, offset, size) {
                        self.leaveCellEditMode('cell');
                        showResizerOverlayNode(offset, size);
                    },
                    'resize:move': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                    }
                });
            });

            // process events triggered by grid panes
            _(gridPanes).each(function (gridPane, panePos) {

                // finish format painter after selection tracking
                gridPane.on('select:end', function (event, address, apply) {
                    if (apply && self.applyPainter()) {
                        self.applyPainter(false);
                    }
                });

                // forward cell edit mode in any grid pane to view listeners
                gridPane.on('celledit:enter celledit:change celledit:leave', function (event) {
                    self.trigger(event.type, panePos);
                });
            });

            // initialize selection settings, used until first real update
            importSelectionSettings({});

            // log current selection in debug mode
            if (Config.isDebug()) {

                // create the output nodes in the debug pane
                self.addDebugInfoNode('version', { header: 'application', tooltip: 'Calc Engine version' })
                    .addDebugInfoHeader('selection', { tooltip: 'Current sheet selection' })
                    .addDebugInfoNode('sheet', { tooltip: 'Active (visible) sheet' })
                    .addDebugInfoNode('ranges', { tooltip: 'Selected cell ranges' })
                    .addDebugInfoNode('active', { tooltip: 'Active cell in selected ranges' })
                    .addDebugInfoNode('draw', { tooltip: 'Selected drawing objects' })
                    .addDebugInfoHeader('formatting', { tooltip: 'Formatting of the active cell' })
                    .addDebugInfoNode('cell', { tooltip: 'Explicit cell attributes of active cell' })
                    .addDebugInfoNode('char', { tooltip: 'Explicit character attributes of active cell' })
                    .addDebugInfoNode('style', { tooltip: 'Cell style sheet of active cell' })
                    .addDebugInfoNode('format', { tooltip: 'Number format settings of active cell' });

                // log the version of the Calc Engine
                app.on('docs:import:after', function () {
                    self.setDebugInfoText('application', 'version', app.getEngineVersion());
                });

                // log information about the active sheet
                self.on('change:layoutdata change:activesheet', function () {
                    var used = ((self.getUsedCols() > 0) && (self.getUsedRows() > 0)) ? SheetUtils.getRangeName({ start: [0, 0], end: [self.getUsedCols() - 1, self.getUsedRows() - 1] }) : '<empty>';
                    self.setDebugInfoText('selection', 'sheet', 'index=' + sheetSettings.sheet + ', name="' + self.getSheetName() + '", used=' + used);
                });

                // log all selection events
                self.on('change:selection', function (event, selection) {
                    self.setDebugInfoText('selection', 'ranges', 'count=' + selection.ranges.length + ', ranges=[' + _(selection.ranges).map(SheetUtils.getRangeName).join() + ']')
                        .setDebugInfoText('selection', 'active', 'range=' + SheetUtils.getRangeName(selection.ranges[selection.activeRange]) + ', cell=' + SheetUtils.getCellName(selection.activeCell) + (selection.originCell ? (', origin=' + SheetUtils.getCellName(selection.originCell)) : ''))
                        .setDebugInfoText('selection', 'draw', 'count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + (selection.drawings).map(JSON.stringify).join() + ']') : '<none>'));
                });

                // log explicit formatting of the active cell
                self.on('change:layoutdata', function () {
                    var attributes = selectionSettings.active.explicit;
                    function stringify(attrs) { return JSON.stringify(attrs || {}).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7'); }
                    self.setDebugInfoText('formatting', 'cell', stringify(attributes.cell))
                        .setDebugInfoText('formatting', 'char', stringify(attributes.character))
                        .setDebugInfoText('formatting', 'style', _.isString(attributes.styleId) ? ('id="' + attributes.styleId + '", name="' + cellStyles.getName(attributes.styleId) + '"') : '')
                        .setDebugInfoText('formatting', 'format', stringify(selectionSettings.active.format));
                });
            }

            // init global search for spreadsheet
            initSearch();
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            self.createToolBox('format', { label: gt('Font'), visible: 'view/editable/cell' })
                .addGroup('character/fontname', new Controls.FontFamilyPicker(app, { width: 117 }))
                .addRightTab()
                .addGroup('character/fontsize', new Controls.FontHeightPicker({ width: 47 }))
                .newLine()
                .addGroup('character/bold',      new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic',    new Button(Controls.ITALIC_OPTIONS))
                .addGroup('character/underline', new Button(Controls.UNDERLINE_OPTIONS))
                .addGroup('character/strike',    new Button(Controls.STRIKEOUT_OPTIONS))
                .addRightTab()
                .addGroup('character/color', new Controls.ColorPicker(app, 'text', { icon: 'docs-font-color', tooltip: gt('Text color') }));

            self.createToolBox('alignment', { label: gt('Cell Format'), visible: 'view/editable/cell' })
                .addGroup('cell/stylesheet', new Controls.CellStylePicker(app, { fullWidth: true }))
                .newLine()
                .addGroup('cell/alignhor', new RadioList({ icon: 'docs-para-align-left', tooltip: /*#. text alignment in cells */ gt('Horizontal alignment'), updateCaptionMode: 'icon' })
                    .createOptionButton('', 'auto',    { icon: 'docs-cell-h-align-auto',    label: /*#. text alignment in cells */ gt('Automatic') })
                    .createOptionButton('', 'left',    { icon: 'docs-cell-h-align-left',    label: /*#. text alignment in cells */ gt('Left') })
                    .createOptionButton('', 'center',  { icon: 'docs-cell-h-align-center',  label: /*#. text alignment in cells */ gt('Center') })
                    .createOptionButton('', 'right',   { icon: 'docs-cell-h-align-right',   label: /*#. text alignment in cells */ gt('Right') })
                    .createOptionButton('', 'justify', { icon: 'docs-cell-h-align-justify', label: /*#. text alignment in cells */ gt('Justify') })
                )
                .addGap()
                .addGroup('cell/alignvert', new RadioList({ icon: 'docs-cell-vertical-bottom', tooltip: /*#. text alignment in cells */ gt('Vertical alignment'), updateCaptionMode: 'icon' })
                    .createOptionButton('', 'top',     { icon: 'docs-cell-v-align-top',     label: /*#. vertical text alignment in cells */ gt('Top') })
                    .createOptionButton('', 'middle',  { icon: 'docs-cell-v-align-middle',  label: /*#. vertical text alignment in cells */ gt('Middle') })
                    .createOptionButton('', 'bottom',  { icon: 'docs-cell-v-align-bottom',  label: /*#. vertical text alignment in cells */ gt('Bottom') })
                    .createOptionButton('', 'justify', { icon: 'docs-cell-v-align-justify', label: /*#. vertical text alignment in cells */ gt('Justify') })
                )
                .addGap()
                .addGroup('cell/merge', new Controls.MergePicker(app))
                .addRightTab()
                .addGroup('cell/linebreak', new Button({ icon: 'docs-insert-linebreak', tooltip: gt('Automatic word wrap'), toggle: true, visibility: 'editmode' }))
                .newLine()
                .addGroup('cell/fillcolor', new Controls.ColorPicker(app, 'fill', { icon: 'docs-cell-fill-color', tooltip: gt('Fill color') }))
                .addGap()
                .addGroup('cell/border/mode', new Controls.CellBorderPicker(app))
                .addGap()
                .addGroup('format/painter', new Button({ icon: 'docs-format-painter', tooltip: /*#. copy formatting from one location and apply it to another */ gt('Format painter'), toggle: true, visibility: 'editmode' }))
                .addRightTab()
                .addGroup('cell/resetAttributes', new Button(Controls.CLEAR_FORMAT_OPTIONS));

            self.createToolBox('borders', { label: gt('Borders'), visible: 'cell/border/value' })
                .addGroup('cell/border/style', new Controls.BorderStylePicker({ width: 96 }))
                .addGap()
                .addGroup('cell/border/width', new Controls.CellBorderWidthPicker({ width: 35 }))
                .addGap()
                .addGroup('cell/border/color', new Controls.ColorPicker(app, 'line', { icon: 'docs-cell-fill-color', tooltip: gt('Border color') }));

            self.createToolBox('numberformat', { label: gt('Number Format'), visible: 'view/editable/cell' })
                .addGroup('cell/numberformat/category', new Controls.NumberFormatPicker(app, { width: 100 }))
                .addRightTab()
                .addGroup('cell/numberformat/formatcodeselector', formatCodePicker = new Controls.FormatCodePicker(app, { width: 115 }));

            self.createToolBox('rowscols', { label: gt('Rows and Columns'), visible: 'view/editable/cell' })
                .addGroup('row/insert', new Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }))
                .addGroup('row/delete', new Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'docs-table-resize-row', tooltip: gt('Row height'), classes: 'no-borders' }))
                .addGap(1)
                .addGroup('row/height', new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Row height'), max: SheetUtils.MAX_ROW_HEIGHT })))
                .newLine()
                .addGroup('column/insert', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('column/delete', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'docs-table-resize-column', tooltip: gt('Column width'), classes: 'no-borders' }))
                .addGap(1)
                .addGroup('column/width',  new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Column width'), max: SheetUtils.MAX_COLUMN_WIDTH })));

            self.createToolBox('insert', { label: gt('Insert'), visible: 'view/editable/cell' })
                .addGroup('math/operators', new Button({ icon: 'docs-auto-sum', tooltip: gt('Sum automatically') }))
                .addGap()
                .addGroup('character/hyperlink/dialog', new Button({ icon: 'docs-hyperlink', tooltip: gt('Insert/Edit hyperlink') }))
                .addGap()
                .addGroup('image/insert/dialog', new Button({ icon: 'docs-image-insert', tooltip: gt('Insert image') }))
                .addGap();
                /*.addGroup('chart/insert', new Button({ icon: 'icon-bar-chart', tooltip: gt('Insert Chart') }));*/
            self.createToolBox('view', { label: /*#. Name for a toolbox that contains GUI elements to change view settings for the current sheet (split/freeze etc.) */ gt('View') })
                .addGroup('view/split/dynamic', new Controls.DynamicSplitButton({ width: 100 }))
                .addRightTab()
                .addGroup('view/split/frozen', new Controls.FrozenSplitPicker({ width: 91 }));

            self.getOverlayToolBox()
                .addGroup('character/bold',   new Button(Controls.BOLD_OPTIONS))
                .addGroup('character/italic', new Button(Controls.ITALIC_OPTIONS));

            // create the bottom overlay pane
            self.addPane(new Pane(app, 'overlaybottom', { position: 'bottom', classes: 'inline right', overlay: true, transparent: true, hoverEffect: true })
                .addViewComponent(new ToolBox(app, 'overlaylabels', { landmark: false }).addPrivateGroup(statusLabel))
            );

			// drawing is moved to DrawingController

            if (Config.isDebug()) {
                self.createDebugToolBox();

                self.debugRequestUpdate = function (type) {
                    switch (type) {
                    case 'view':
                        updateNotificationHandler({ changed: { type: 'all' } });
                        break;
                    case 'cells':
                        updateNotificationHandler({ changed: { type: 'cells' }, sheets: Utils.makeSimpleObject(sheetSettings.sheet, { cells: self.getSelectedRanges() }) });
                        break;
                    case 'selection':
                        requestSelectionUpdate(self.getSelection());
                        break;
                    }
                };

                self.debugInsertDrawing = function () {
                    var generator = model.createOperationsGenerator(),
                        index = drawingCollection.getModelCount(),
                        selection = _(self.getSelection()).extend({ drawings: [] });
                    _(self.getSelectedRanges()).each(function (range) {
                        var attrs = { drawing: {
                            name: 'Diagram ' + (index + 1),
                            description: 'Very, very, very, very, very, very, very, very, very long description for this diagram.',
                            startCol: range.start[0],
                            startColOffset: 100,
                            startRow: range.start[1],
                            startRowOffset: 100,
                            endCol: range.end[0],
                            endColOffset: 2400,
                            endRow: range.end[1],
                            endRowOffset: 400
                        } };
                        generator.generateDrawingOperation(Operations.INSERT_DRAWING, sheetSettings.sheet, [index], { type: 'diagram', attrs: attrs });
                        selection.drawings.push([index]);
                        index += 1;
                    });
                    model.applyOperations(generator.getOperations());
                    activeSheetModel.setViewAttribute('selection', selection);
                };

                self.getDebugActionMenu()
                    .addSeparator()
                    .addGroup('debug/view/update', new Button({ icon: 'icon-refresh', label: _.noI18n('Request full view update'),              value: 'view' }))
                    .addGroup('debug/view/update', new Button({ icon: 'icon-refresh', label: _.noI18n('Request view update of selected cells'), value: 'cells' }))
                    .addGroup('debug/view/update', new Button({ icon: 'icon-refresh', label: _.noI18n('Request update of selection data'),      value: 'selection' }))
                    .addSeparator()
                    .addGroup('debug/insert/drawing', new Button({ icon: 'icon-sitemap',   label: _.noI18n('Insert a generic drawing object') }));
            }
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            // prevent JS errors when importing the document fails, and the view does not have activated a sheet yet
            if (activeSheetModel) {
                self.getActiveGridPane().grabFocus();
            }
        }

        /**
         * Refreshes the layout of all visible panes and tracking nodes.
         */
        function initializePanes() {

            var // the size of the header corner node (and thus of the row/column header nodes)
                headerWidth = 0, headerHeight = 0,
                // whether frozen split mode is active
                frozenSplit = self.hasFrozenSplit(),
                // whether dynamic split mode is really active
                dynamicSplit = !frozenSplit && self.hasSplit(),
                // the size of the split lines
                splitLineSize = frozenSplit ? FREEZE_SIZE : SPLIT_SIZE,
                // whether the split lines are visible
                colSplit = false, rowSplit = false,
                // start position of the split lines
                splitLineLeft = activeSheetModel.getSplitWidth(),
                splitLineTop = activeSheetModel.getSplitHeight();

            // expand dynamic splits to minimum split size
            if (dynamicSplit) {
                if (splitLineLeft > 0) { splitLineLeft = Math.max(splitLineLeft, MIN_PANE_SIZE); }
                if (splitLineTop > 0) { splitLineTop = Math.max(splitLineTop, MIN_PANE_SIZE); }
            }

            // whether the left and top panes are visible
            paneSideSettings.left.visible = (dynamicSplit || frozenSplit) && (splitLineLeft > 0);
            paneSideSettings.top.visible = (dynamicSplit || frozenSplit) && (splitLineTop > 0);

            // calculate current size of header nodes
            // TODO: use current maximum row index visible in the panes
            cornerPane.initializePaneLayout(model.getMaxRow());
            headerWidth = cornerPane.getNode().width();
            headerHeight = cornerPane.getNode().height();

            // calculate inner width of left panes
            if (paneSideSettings.left.visible) {
                paneSideSettings.left.offset = headerWidth;
                paneSideSettings.left.size = splitLineLeft;
                paneSideSettings.left.hiddenSize = frozenSplit ? colCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorLeft')) : 0;
            } else {
                _(paneSideSettings.left).extend({ offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate inner height of top panes
            if (paneSideSettings.top.visible) {
                paneSideSettings.top.offset = headerHeight;
                paneSideSettings.top.size = splitLineTop;
                paneSideSettings.top.hiddenSize = frozenSplit ? rowCollection.convertScrollAnchorToPixel(activeSheetModel.getViewAttribute('anchorTop')) : 0;
            } else {
                _(paneSideSettings.top).extend({ offset: 0, size: 0, hiddenSize: 0 });
            }

            // calculate effective position of split lines
            splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
            splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;

            // determine whether right and bottom panes are visible (must have enough room in split and frozen mode)
            paneSideSettings.right.visible = !paneSideSettings.left.visible || (splitLineLeft + splitLineSize + MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
            paneSideSettings.bottom.visible = !paneSideSettings.top.visible || (splitLineTop + splitLineSize + MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

            // visibility of the split lines
            colSplit = paneSideSettings.left.visible && paneSideSettings.right.visible;
            rowSplit = paneSideSettings.top.visible && paneSideSettings.bottom.visible;

            // calculate the resulting grid pane positions and sizes
            if (paneSideSettings.right.visible) {
                paneSideSettings.right.offset = colSplit ? (splitLineLeft + splitLineSize) : headerWidth;
                paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                paneSideSettings.right.hiddenSize = frozenSplit ? (paneSideSettings.left.hiddenSize + paneSideSettings.left.size) : 0;
            } else if (paneSideSettings.left.visible) {
                paneSideSettings.left.size = rootNode.width() - headerWidth;
            }
            if (paneSideSettings.bottom.visible) {
                paneSideSettings.bottom.offset = rowSplit ? (splitLineTop + splitLineSize) : headerHeight;
                paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                paneSideSettings.bottom.hiddenSize = frozenSplit ? (paneSideSettings.top.hiddenSize + paneSideSettings.top.size) : 0;
            } else if (paneSideSettings.top.visible) {
                paneSideSettings.top.size = rootNode.height() - headerHeight;
            }

            // set frozen mode (left/top panes are not scrollable in frozen mode in their
            // own direction, e.g. left frozen panes are not scrollable to left/right)
            paneSideSettings.left.frozen = frozenSplit;
            paneSideSettings.right.frozen = false;
            paneSideSettings.top.frozen = frozenSplit;
            paneSideSettings.bottom.frozen = false;

            // set up scroll bar visibility in the opposite direction (e.g. left/right
            // scroll bars of top panes are hidden, if bottom panes are visible)
            paneSideSettings.left.showOppositeScroll = !paneSideSettings.right.visible;
            paneSideSettings.right.showOppositeScroll = true;
            paneSideSettings.top.showOppositeScroll = !paneSideSettings.bottom.visible;
            paneSideSettings.bottom.showOppositeScroll = true;

            // initialize the header panes
            _(headerPanes).each(function (headerPane, paneSide) {
                headerPane.initializePaneLayout(paneSideSettings[paneSide]);
            });

            // initialize the grid panes
            _(gridPanes).each(function (gridPane, panePos) {
                gridPane.initializePaneLayout(paneSideSettings[PaneUtils.getColPaneSide(panePos)], paneSideSettings[PaneUtils.getRowPaneSide(panePos)]);
            });

            // visibility and position of the split lines
            colSplitLine.toggle(colSplit).css({ left: splitLineLeft, width: splitLineSize });
            rowSplitLine.toggle(rowSplit).css({ top: splitLineTop, height: splitLineSize });

            // visibility and position of the split tracking nodes
            colSplitTrackingNode.toggle(dynamicSplit && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
            rowSplitTrackingNode.toggle(dynamicSplit && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
            centerSplitTrackingNode.toggle(dynamicSplit && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPanes[panePos];
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getColPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (!paneSideSettings[PaneUtils.getRowPaneSide(panePos)].visible) {
                panePos = PaneUtils.getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPanes[panePos];
        };

        /**
         * Returns the grid pane that is currently focused.
         *
         * @returns {GridPane}
         *  The focused grid pane instance.
         */
        this.getActiveGridPane = function () {
            return this.getVisibleGridPane(activeSheetModel.getViewAttribute('activePane'));
        };

        /**
         * Invokes the passed iterator function for all grid panes contained in
         * this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all grid pane instances. Receives
         *  the following parameters:
         *  (1) {GridPane} gridPane
         *      The current grid pane instance.
         *  (2) {String} panePos
         *      The position identifier of the grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGridPanes = function (iterator, options) {
            return _(gridPanes).any(function (gridPane, panePos) {
                return iterator.call(this, gridPane, panePos) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPanes[paneSide];
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getColPaneSide(panePos)];
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPanes[PaneUtils.getRowPaneSide(panePos)];
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible column header pane (left
         *  or right); otherwise returns the first visible row header pane (top
         *  or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            return columns ?
                (paneSideSettings.left.visible ? headerPanes.left : headerPanes.right) :
                (paneSideSettings.top.visible ? headerPanes.top : headerPanes.bottom);
        };

        /**
         * Returns a header pane associated to the grid pane that is currently
         * focused.
         *
         * @param {Boolean} columns
         *  If set to true, returns the column header pane (left or right);
         *  otherwise returns the row header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  A header pane instance associated to the grid pane that is
         *  currently focused.
         */
        this.getActiveHeaderPane = function (columns) {
            var panePos = activeSheetModel.getViewAttribute('activePane');
            return columns ? this.getColHeaderPane(panePos) : this.getRowHeaderPane(panePos);
        };

        /**
         * Invokes the passed iterator function for all header panes contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all header pane instances.
         *  Receives the following parameters:
         *  (1) {HeaderPane} headerPane
         *      The current header pane instance.
         *  (2) {String} paneSide
         *      The identifier of the pane side ('left', 'top', etc.).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateHeaderPanes = function (iterator, options) {
            return _(headerPanes).any(function (headerPane, paneSide) {
                return iterator.call(this, headerPane, paneSide) === Utils.BREAK;
            }, Utils.getOption(options, 'context')) ? Utils.BREAK : undefined;
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Collects several update requests sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.requestUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // counter for running requests (next free index)
                nextRequestIndex = 0,
                // minimum request index valid for evaluation (ignore requests after switching to another sheet)
                requiredRequestIndex = 0;

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {

                // do not request anything in error mode
                if (app.getState() === 'error') { return self; }

                // initialize request data, if missing or for another sheet
                if (!requestData || (requestData.sheet !== sheetSettings.sheet)) {
                    requestData = { sheet: sheetSettings.sheet, locale: Utils.LOCALE };
                }

                // extend the array of cell ranges
                if (_.isObject(requestProps.cells)) {
                    requestProps.cells = _.getArray(requestProps.cells);
                    if (_.isArray(requestData.cells)) {
                        requestProps.cells = requestData.cells.concat(requestProps.cells);
                    }
                }

                // insert the passed request data
                _(requestData).extend(requestProps);

                return self;
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // local reference to the current request dats
                    localRequestData = requestData,
                    // current request index
                    localRequestIndex = nextRequestIndex,
                    // start time of this request
                    localStartTime = _.now(),
                    // the new server request
                    def = null;

                if (!localRequestData || (app.getState() === 'error')) { return; }
                requestData = null;

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // restrict requested ranges to visible area of all current grid panes
                if (_.isArray(localRequestData.cells)) {

                    var // the (unified) visible ranges in all grid panes
                        boundRanges = SheetUtils.getUnifiedRanges(invokeCellCollectionMethod('getRange')),
                        // an array of cell range arrays containing the intersection ranges for each bounding range
                        intersections = _(boundRanges).map(function (boundRange) { return SheetUtils.getIntersectionRanges(localRequestData.cells, boundRange); });

                    // flatten the intersection arrays, finally unify the resulting range array
                    localRequestData.cells = SheetUtils.getUnifiedRanges(_(intersections).flatten(true));
                }

                // immediately delete the request data (new updates may be requested in the meantime)
                localRequestIndex = nextRequestIndex;
                nextRequestIndex += 1;
                Utils.info('SpreadsheetView.executeUpdateRequest(): sending update request #' + (localRequestIndex + 1) + ':', localRequestData);

                // wait for pending actions (bug 30044, must wait for the
                // 'insertSheet' operation before requesting data for the sheet)
                def = runningDef = app.saveChanges().then(function () {
                    return app.sendFilterRequest({
                        method: 'POST',
                        params: {
                            action: 'updateview',
                            requestdata: JSON.stringify(localRequestData)
                        }
                    });
                });

                def.fail(function () {
                    Utils.error('SpreadsheetView.executeUpdateRequest(): update request #' + (localRequestIndex + 1) + ' failed');
                })
                .done(function (layoutData) {

                    Utils.info('SpreadsheetView.executeUpdateRequest(): update response #' + (localRequestIndex + 1) + ' received (' + (_.now() - localStartTime) + 'ms):', layoutData);
                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet)) {
                        Utils.error('SpreadsheetView.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (localRequestIndex < requiredRequestIndex) { return; }

                    // copy sheet layout data to internal class member
                    sheetSettings = _.copy(layoutData.sheet, true);

                    // if this update request is triggered by the search function
                    if (_.isObject(localRequestData.search) && !_.isObject(layoutData.found)) {
                        // be silent if this is a search result refresh request.
                        // a search result refresh request is fired if a local/remote user changes the document while searching.
                        if (!localRequestData.search.refresh) {
                            self.yell('info', /*#. Search&Replace: used when the searched text has not been found */ gt('There are no cells matching "%1$s".', localRequestData.search.searchText));
                        }
                    }

                    // if there are search results, save to sheetSettings
                    if (_.isObject(layoutData.found)) {
                        // save search result
                        searchLayoutData = _.copy(layoutData.found, true);
                        // jump to first search result only if this is a new search.
                        if (!replaceAction) {
                            // iterate initially to first result
                            self.searchIterate();
                        }
                    }

                    // initialize the cell collections for all grid panes
                    if (_.isArray(layoutData.cells)) {
                        invokeCellCollectionMethod('importCellEntries', localRequestData.cells, layoutData.cells);
                    }

                    // copy layout data for the current selection
                    if (_.isObject(layoutData.selection)) {
                        importSelectionSettings(layoutData.selection);
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        self.trigger('change:layoutdata remote:layoutdata');
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        self.trigger('change:layoutdata remote:layoutdata');
                    }
                });
            }

            // delete the current cached request data when the active sheet changes
            self.on('before:activesheet', function () {
                requestData = null;
                requiredRequestIndex = nextRequestIndex;
            });

            // create and return the debounced method SpreadsheetView.requestUpdate()
            return app.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 100, maxDelay: 250 });

        }()); // SpreadsheetView.requestUpdate()

        /**
         * Additional processing before the document will be flushed for
         * download, print etc. Leaves in-place cell edit mode and commits the
         * current text.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  preparations for flushing the document have been finished.
         */
        this.prepareFlush = function () {
            this.leaveCellEditMode('cell');
            return $.when();
        };

        /**
         * Tries to initiate an edit action. if the document is in read-only
         * mode, a notification will be shown to the user.
         *
         * @returns {Boolean}
         *  Whether modifying the document is posssible.
         */
        this.requireEditMode = function () {
            var editMode = model.getEditMode();
            if (!editMode) { app.rejectEditAttempt(); }
            return editMode;
        };

        /**
         * Shows the passed text in the status label control at the bottom
         * border of the application pane.
         *
         * @param {String} text
         *  The text to be shown in the status label.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.showStatusLabel = function (text) {
            statusLabel.setValue(text);
            return this;
        };

        /**
         * Returns the current zoom factor of the active sheet.
         *
         * @returns {Number}
         *  The current zoom factor, as floating-point number. The value 1
         *  represents a zoom of 100%.
         */
        this.getZoom = function () {
            return activeSheetModel.getViewAttribute('zoom');
        };

        /**
         * Decreases the current zoom factor of all sheets in the document.
         */
        this.decZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoom = Utils.findLast(ZOOM_FACTORS, function (zoom) { return zoom < currZoom; });

            // set new zoom at all sheets
            if (_.isNumber(prevZoom)) {
                model.iterateSheetModels(function (sheetModel) {
                    sheetModel.setViewAttribute('zoom', prevZoom);
                });
            }
            return this;
        };

        /**
         * Increases the current zoom factor of all sheets in the document.
         */
        this.incZoom = function () {

            var // current zoom factor
                currZoom = this.getZoom(),
                // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoom = _(ZOOM_FACTORS).find(function (zoom) { return zoom > currZoom; });

            // set new zoom at all sheets
            if (_.isNumber(nextZoom)) {
                model.iterateSheetModels(function (sheetModel) {
                    sheetModel.setViewAttribute('zoom', nextZoom);
                });
            }
            return this;
        };

        /**
         * Calculates the effective font size of the passed font size,
         * according to the current zoom factor.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @returns {Number}
         *  The effective font size, in points.
         */
        this.getEffectiveFontSize = function (fontSize) {
            return Utils.round(fontSize * this.getZoom(), 0.1);
        };

        /**
         * Returns the effective font size for the current zoom factor to be
         * used for the column and row labels in all header panes.
         *
         * @returns {Number}
         *  The effective font size for all header panes, in points.
         */
        this.getHeaderFontSize = function () {
            return Utils.minMax(Utils.round(5 * this.getZoom() + 6, 0.1), 8, 30);
        };

        // cells and ranges ---------------------------------------------------

        /**
         * Returns whether a range consists of a single cell, or whether it is
         * a merged range.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return (SheetUtils.getCellCount(range) === 1) || mergeCollection.isMergedRange(range);
        };

        /**
         * Returns the address of a visible cell located as close as possible
         * to the passed cell address.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {String} method
         *  Specifies how to look for another cell, if the specified cell is
         *  hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified cell is
         *      hidden, returns null.
         *  - 'next':
         *      Looks for a visible column and/or row following the cell.
         *  - 'prev':
         *      Looks for a visible column and/or row preceding the cell.
         *  - 'nextPrev':
         *      First, looks for visible column and/or row following the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *  - 'prevNext':
         *      First, looks for visible column and/or row preceding the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *
         * @returns {Number[]|Null}
         *  The address of a visible cell near the passed cell address; or null
         *  if all columns or all rows are hidden.
         */
        this.getVisibleCell = function (address, method) {

            var // collection entries of visible column/row
                colEntry = colCollection.getVisibleEntry(address[0], method),
                rowEntry = rowCollection.getVisibleEntry(address[1], method);

            return (colEntry && rowEntry) ? [colEntry.index, rowEntry.index] : null;
        };

        /**
         * Returns the layout data of the specified cell. Tries to receive the
         * layout data from all cell collections.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object|Null}
         *  The collection entry with the cell layout data, if available;
         *  otherwise null.
         */
        this.getCellEntry = function (address) {
            return cellCollections.bottomRight.getCellEntry(address) ||
                cellCollections.bottomLeft.getCellEntry(address) ||
                cellCollections.topRight.getCellEntry(address) ||
                cellCollections.topLeft.getCellEntry(address);
        };

        /**
         * Returns the subtotal results in the current selection.
         *
         * @returns {Object}
         *  The current subtotal results in the current selection, in the
         *  properties 'sum', 'min', 'max', 'numbers', 'cells', and 'average'.
         */
        this.getSubtotals = function () {
            return _.clone(selectionSettings.subtotals);
        };

        /**
         * Returns whether the passed ranges are editable, whereas editable
         * means that no locked cells are found in the ranges, and the sheet
         * where these ranges are, is not protected.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the
         *  passed ranges are editable, and rejected if they are not.
         */
        this.areRangesEditable = function (ranges) {

            var // the resulting Deferred object
                def = null,
                // whether all cells in all ranges are editable (not locked)
                rangesEditable = true,
                // result of the cell range iteration
                result = null;

            // return resolved Deferred object and quit if sheet is not protected at all
            if (!activeSheetModel.isProtected()) {
                return $.when();
            }

            // Visit all cells in the given ranges. If any locked cell has been found,
            // set the 'rangesEditable' flag to false and exit the iterator. If any range
            // is not contained in the cell collections, Utils.BREAK will be returned by
            // iterateCellsInRanges() itself while 'rangesEditable' is still true.
            result = iterateCellsInRanges(ranges, function (cellData) {
                if (!cellData.attributes.cell.unlocked) {
                    rangesEditable = false;
                    return Utils.BREAK;
                }
            });

            // if 'rangesEditable' is false, a locked cell has been found
            if (!rangesEditable) {
                return $.Deferred().reject();
            }

            // if iterator did not return Utils.BREAK, selection is completely
            // covered by the cell collections, and none of the cells is locked
            if (result !== Utils.BREAK) {
                return $.when();
            }

            // iterator has returned Utils.BREAK without any known locked cell:
            // selection contains ranges that are not covered by the cell collections,
            // send server request to detect the locked state of the selected cells
            ranges = _.getArray(ranges);
            def = app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'updateview',
                    requestdata: JSON.stringify({
                        sheet: sheetSettings.sheet,
                        locale: Utils.LOCALE,
                        selection: { ranges: ranges, active: ranges[0].start }
                    })
                }
            });

            // reject Deferred object if selection contains a locked cell
            return def.then(function (response) {
                return response.selection.locked ? $.Deferred().reject() : null;
            });
        };

        /**
         * Returns if the selection is locked (there is at least one locked cell
         * in the selection).
         *
         * @returns {Object}
         */
        this.isSelectionLocked = function () {
            return selectionSettings.locked;
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the active sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the active sheet.
         */
        this.hasSplit = function () {
            return activeSheetModel.hasSplit();
        };

        /**
         * Returns the current view split mode of the active sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  view is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return activeSheetModel.getSplitMode();
        };

        /**
         * Returns whether the view is currently split, and dynamic split mode
         * is activated. Returns also true, if the split view is currently
         * frozen, but will thaw to dynamic split mode.
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return activeSheetModel.hasDynamicSplit();
        };

        /**
         * Returns whether the view is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return activeSheetModel.hasFrozenSplit();
        };

        /**
         * Toggles the dynamic view split mode according to the current view
         * split mode, and the current selection. Enabling dynamic split mode
         * in a frozen split view will thaw the frozen panes. Otherwise,
         * enabling the dynamic split mode will split the view at the position
         * left of and above the active cell. Disabling the dynamic split mode
         * will remove the split also if the view is in frozen split mode.
         *
         * @param {Boolean} state
         *  The new state of the dynamic view split mode.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the position of the active cell in the sheet
                cellRectangle = null,
                // the visible area of the active grid pane that will be split
                visibleRectangle = null,
                // the new split positions
                splitWidth = 0, splitHeight = 0,
                // additional scroll anchor attributes
                attributes = null;

            // disable dynamic or frozen split mode
            if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert frozen split to dynamic split
            else if (state && this.hasFrozenSplit()) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // set dynamic split at active cell
            else if (state && this.hasSplit) {

                // additional scroll anchor attributes
                attributes = {};

                // get position of active cell and visible area of the grid pane
                cellRectangle = activeSheetModel.getCellRectangle(address, { expandMerged: true });
                visibleRectangle = this.getActiveGridPane().getVisibleRectangle();

                // do not split vertically, if the active cell is shown at the left border
                splitWidth = cellRectangle.left - visibleRectangle.left;
                if ((-cellRectangle.width < splitWidth) && (splitWidth < MIN_PANE_SIZE)) {
                    splitWidth = 0;
                }

                // do not split horizontally, if the active cell is shown at the top border
                splitHeight = cellRectangle.top - visibleRectangle.top;
                if ((-cellRectangle.height < splitHeight) && (splitHeight < MIN_PANE_SIZE)) {
                    splitHeight = 0;
                }

                // split in the middle of the grid pane, if active cell is outside the visible area
                if (((splitWidth === 0) && (splitHeight === 0)) ||
                    (splitWidth < 0) || (splitWidth > visibleRectangle.width - MIN_PANE_SIZE) ||
                    (splitHeight < 0) || (splitHeight > visibleRectangle.height - MIN_PANE_SIZE)
                ) {
                    splitWidth = Math.floor(visibleRectangle.width / 2);
                    splitHeight = Math.floor(visibleRectangle.height / 2);
                    attributes.anchorRight = colCollection.getScrollAnchorByOffset(visibleRectangle.left + splitWidth, { pixel: true });
                    attributes.anchorBottom = rowCollection.getScrollAnchorByOffset(visibleRectangle.top + splitHeight, { pixel: true });
                    attributes.activePane = 'topLeft';
                } else {
                    if (splitWidth > 0) { attributes.anchorRight = { index: address[0], ratio: 0 }; }
                    if (splitHeight > 0) { attributes.anchorBottom = { index: address[1], ratio: 0 }; }
                }

                // activate the dynamic split view
                splitWidth = activeSheetModel.convertPixelToHmm(splitWidth);
                splitHeight = activeSheetModel.convertPixelToHmm(splitHeight);
                activeSheetModel.setDynamicSplit(splitWidth, splitHeight, attributes);
            }

            return this;
        };

        /**
         * Toggles the frozen view split mode according to the current view
         * split mode, and the current selection. Enabling frozen split mode
         * in a dynamic split view will freeze the panes with their current
         * size (split mode 'frozenSplit'). Otherwise, enabling the frozen
         * split mode will split the view at the position left of and above the
         * active cell. Disabling the frozen split mode will either return to
         * dynamic split mode (if split mode was 'frozenSplit'), or will remove
         * the frozen split at all (if split mode was 'frozen').
         *
         * @param {Boolean} state
         *  The new state of the frozen view split mode.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (state) {

            var // whether the view is currently split
                hasSplit = this.hasSplit(),
                // address of the active cell
                address = this.getActiveCell(),
                // the current scroll position
                colAnchor = null, rowAnchor = null,
                // the new frozen column and row interval
                colInterval = null, rowInterval = null;

            // convert frozen split to dynamic split
            if (!state && hasSplit && (this.getSplitMode() === 'frozenSplit')) {
                activeSheetModel.setDynamicSplit(activeSheetModel.getSplitWidthHmm(), activeSheetModel.getSplitHeightHmm());
            }

            // disable split mode completely
            else if (!state && hasSplit) {
                activeSheetModel.clearSplit();
            }

            // convert dynamic split to frozen split
            else if (state && hasSplit && !this.hasFrozenSplit()) {
                activeSheetModel.setFrozenSplit(activeSheetModel.getSplitColInterval(), activeSheetModel.getSplitRowInterval());
            }

            // enable frozen split mode
            else if (state && !hasSplit) {

                // calculate frozen column interval (must result in at least one frozen column)
                colAnchor = activeSheetModel.getViewAttribute('anchorRight');
                colInterval = { first: (colAnchor.ratio < 0.5) ? colAnchor.index : (colAnchor.index + 1), last: address[0] - 1 };
                if (colInterval.first > colInterval.last) { colInterval = null; }

                // calculate frozen row interval (must result in at least one frozen row)
                rowAnchor = activeSheetModel.getViewAttribute('anchorBottom');
                rowInterval = { first: (rowAnchor.ratio < 0.5) ? rowAnchor.index : (rowAnchor.index + 1), last: address[1] - 1 };
                if (rowInterval.first > rowInterval.last) { rowInterval = null; }

                // activate the frozen split view
                activeSheetModel.setFrozenSplit(colInterval, rowInterval);
            }

            return this;
        };

        /**
         * Freezes the passed number of columns and rows in the active sheet,
         * regardless of the current split settings.
         *
         * @param {Number} cols
         *  The number of columns to freeze. The value zero will not freeze any
         *  column.
         *
         * @param {Number} rows
         *  The number of rows to freeze. The value zero will not freeze any
         *  row.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setFixedFrozenSplit = function (cols, rows) {

            var // the scroll anchors for the first visible column/row
                colAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasColSplit() ? 'anchorLeft' : 'anchorRight'),
                rowAnchor = activeSheetModel.getViewAttribute(activeSheetModel.hasRowSplit() ? 'anchorTop' : 'anchorBottom'),
                // the first visible column and row in the split view
                col = ((colAnchor.index < model.getMaxCol()) && (colAnchor.ratio > 0.5)) ? (colAnchor.index + 1) : colAnchor.index,
                row = ((rowAnchor.index < model.getMaxRow()) && (rowAnchor.ratio > 0.5)) ? (rowAnchor.index + 1) : rowAnchor.index;

            // reduce start indexes to be able to freeze the passed number of columns/rows
            col = Math.max(0, Math.min(col, model.getMaxCol() - cols + 1));
            row = Math.max(0, Math.min(row, model.getMaxRow() - rows + 1));

            // activate the frozen split view
            activeSheetModel.setFrozenSplit((cols > 0) ? { first: col, last: col + cols - 1 } : null, (rows > 0) ? { first: row, last: row + rows - 1 } : null);

            return this;
        };

        // sheet operations ---------------------------------------------------

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view. In case of an error, shows a notification message.
         *
         * @returns {Boolean}
         *  Whether the new sheet has been inserted successfully.
         */
        this.insertSheet = function () {

            var // try to insert a new sheet
                result = model.insertSheet(sheetSettings.sheet + 1, model.generateUnusedSheetName());

            // insert the new sheet into the document, and activate the new sheet
            if (result.length === 0) {
                this.activateNextSheet();
                return true;
            }

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only.
         *
         * @returns {Boolean}
         *  Whether the sheet has been deleted successfully.
         */
        this.deleteSheet = function () {
            // handlers of the delete events will update active sheet
            return model.deleteSheet(sheetSettings.sheet);
        };

        /**
         * Moves the active sheet in the spreadsheet document to a new
         * position.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (toSheet) {
            // TODO: activate the sheet at its new position
            return model.moveSheet(sheetSettings.sheet, toSheet);
        };

        /**
         * Copies the active sheet and inserts the copy with the provided sheet
         * name behind the active sheet. In case of an error, shows a
         * notification message.
         *
         * @param {String} sheetName
         *  The name of the new sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been copied successfully.
         */
        this.copySheet = function (sheetName) {

            var // try to copy the sheet
                result = model.copySheet(sheetSettings.sheet, sheetSettings.sheet + 1, sheetName);

            // activate the new sheet
            if (result.length === 0) {
                this.activateNextSheet();
                return true;
            }

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Returns the name of the active sheet.
         *
         * @returns {String}
         *  The name of the active sheet.
         */
        this.getSheetName = function () {
            return model.getSheetName(sheetSettings.sheet);
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         * In case of an error, shows a notification message.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {Boolean}
         *  Whether the sheet has been renamed successfully.
         */
        this.setSheetName = function (sheetName) {

            var // try to rename the sheet
                result = model.setSheetName(sheetSettings.sheet, sheetName);

            // show notification message on error
            return yellSheetNameError(result);
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributes = function () {
            return activeSheetModel.getMergedAttributes();
        };

        /**
         * Changes the formatting attributes of the active sheet.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set with the new attributes for the
         *  active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setSheetAttributes = function (attributes) {

            var // the generator for the operation
                generator = model.createOperationsGenerator();

            // leave cell edit mode before hiding the sheet
            if (_.isObject(attributes.sheet) && (attributes.sheet.visible === false)) {
                this.leaveCellEditMode('cell');
            }

            generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, sheetSettings.sheet, { attrs: attributes });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Returns the number of visible sheets in this spreadsheet view.
         *
         * @returns {Number}
         *  The number of visible sheets in the document.
         */
        this.getVisibleSheetCount = function () {
            return visibleSheets.length;
        };

        /**
         * Returns the number of hidden sheets in this spreadsheet view that
         * can be made visible.
         *
         * @returns {Number}
         *  The number of hidden sheets in the document. Does NOT include the
         *  sheets with unsupported types that will always be hidden.
         */
        this.getHiddenSheetCount = function () {
            return hiddenSheets.length;
        };

        /**
         * Makes all hidden sheets in the view visible.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.showAllSheets = function () {

            var // the generator for the operation
                generator = model.createOperationsGenerator(),
                // the attribute set to be inserted into the operations
                operationOptions = { attrs: { sheet: { visible: true } } };

            _(hiddenSheets).each(function (sheetInfo) {
                generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, sheetInfo.sheet, operationOptions);
            });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Invokes the passed iterator function for all visible sheet contained
         * in this view.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all visible sheets. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} index
         *      The zero-based sheet index in the collection of all visible
         *      sheets.
         *  (3) {String} sheetName
         *      The name of the sheet.
         *  (4) {Number} sheet
         *      The zero-based sheet index in the document model.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleSheets = function (iterator, options) {
            return Utils.iterateArray(visibleSheets, function (sheetInfo, index) {
                return iterator.call(this, model.getSheetModel(sheetInfo.sheet), index, sheetInfo.name, sheetInfo.sheet);
            }, options);
        };

        /**
         * Returns the model instance of the active sheet.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getActiveSheetModel = function () {
            return activeSheetModel;
        };

        /**
         * Returns the zero-based index of the active sheet currently displayed
         * in this spreadsheet view. Optionally, the index of the active sheet
         * inside the collection of all visible sheets can be returned.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the index of the active sheet inside the
         *      collection of all visible sheets will be returned instead of
         *      the physical sheet index.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet, depending on the passed
         *  options. Will return -1, if the index in the collection of visible
         *  sheets is requested, but the active sheet is currently hidden.
         */
        this.getActiveSheet = function (options) {
            // look up the index of the active sheet in the collection of visible sheets,
            // or return the plain sheet index of the active sheet
            return Utils.getBooleanOption(options, 'visible', false) ? getVisibleIndex(sheetSettings.sheet) : sheetSettings.sheet;
        };

        /**
         * Selects a sheet in this spreadsheet view instance.
         *
         * @param {Number} index
         *  The zero-based index of the new active sheet. If the specified
         *  sheet is currently hidden, it will not be activated.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, the passed sheet index will be interpreted as
         *      index into the collection of all visible sheets, instead of the
         *      physical sheet index.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet, options) {

            // get sheet index from the collection of visible sheets
            if (Utils.getBooleanOption(options, 'visible', false)) {
                sheet = ((0 <= sheet) && (sheet < visibleSheets.length)) ? visibleSheets[sheet].sheet : -1;
            }

            // do nothing, if sheet index does not change, or if the sheet is hidden
            if ((sheetSettings.sheet !== sheet) && (getVisibleIndex(sheet) >= 0)) {
                prepareSetActiveSheet();
                setActiveSheet(sheet);
            }

            return this;
        };

        /**
         * Activates the preceding visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activatePreviousSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) - 1, { visible: true });
        };

        /**
         * Activates the next visible sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.activateNextSheet = function () {
            return this.setActiveSheet(this.getActiveSheet({ visible: true }) + 1, { visible: true });
        };

        /**
         * Returns the number of columns in the used area of the active sheet.
         *
         * @returns {Number}
         *  The number of columns in the used area of the active sheet.
         */
        this.getUsedCols = function () {
            return sheetSettings.usedCols;
        };

        /**
         * Returns the number of rows in the used area of the active sheet.
         *
         * @returns {Number}
         *  The number of rows in the used area of the active sheet.
         */
        this.getUsedRows = function () {
            return sheetSettings.usedRows;
        };

        /**
         * Returns the address of the last used cell in the active sheet. If
         * the sheet is completely empty, returns the address [0, 0].
         *
         * @returns {Number[]}
         *  The address of the last used cell in the active sheet.
         */
        this.getLastUsedCell = function () {
            var col = this.getUsedCols() - 1, row = this.getUsedRows() - 1;
            return ((col >= 0) && (row >= 0)) ? [col, row] : [0, 0];
        };
        // column operations --------------------------------------------------

        /**
         * Returns the column attributes of the column the active cell is
         * currently located in.
         *
         * @returns {Object}
         *  The column attribute map of the active cell.
         */
        this.getColumnAttributes = function () {
            return colCollection.getEntry(this.getActiveCell()[0]).attributes.column;
        };

        /**
         * Returns whether additional columns can be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the
         *  current selection and the number of used columns in the sheet.
         */
        this.columnsInsertable = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges()),
                // the number of inserted columns
                count = Utils.getSum(intervals, SheetUtils.getIntervalSize);

            // the number of used columns, and the number of inserted columns must not exceed the total number of columns in the sheet
            return (this.getUsedCols() + count) <= (model.getMaxCol() + 1);
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertColumns = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            activeSheetModel.insertColumns(intervals);
            return this;
        };

        /**
         * Returns whether the selected columns can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected columns can be deleted.
         */
        this.columnsDeleteable = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges()),
                // the number of deleted columns
                count = Utils.getSum(intervals, SheetUtils.getIntervalSize);

            // prevent deleting all columns (entire sheet) at once
            return (0 < count) && (count <= model.getMaxCol());
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteColumns = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            activeSheetModel.deleteColumns(intervals);
            return this;
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customWidth' column attribute.
         *  @param {Number} [options.target]
         *      The target column to be changed. If specified, the column width
         *      will be set to that column only. If the current selection
         *      contains any ranges covering this column completely, the column
         *      height will be set to all columns contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnWidth = function (width, options) {

            var // the selected ranges
                ranges = this.getSelectedRanges(),
                // the target column to be changed
                targetCol = Utils.getIntegerOption(options, 'target');

            // use column ranges in selection, if target column has been passed
            if (_.isNumber(targetCol)) {
                // filter column ranges in selection
                ranges = _(ranges).filter(function (range) { return model.isColRange(range); });
                // if the target column is not contained in the selection, change that column only
                if (!_(ranges).any(function (range) { return SheetUtils.rangeContainsCol(range, targetCol); })) {
                    ranges = model.makeColRange(targetCol);
                }
            }

            activeSheetModel.setColumnWidth(SheetUtils.getColIntervals(ranges), width, options);
            return this;
        };

        /**
         * Sets optimal column width based od content of cells.
         *
         * @param {Number} targetCol
         *  The target column to be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         *
         */
        this.setOptimalColumnWidth = function (targetCol) {

            var
                selectedRanges = this.getSelectedRanges(),
                visibleRanges,
                cellWidth = 0,
                address = [],
                newCellWidth = 0,
                cellData,
                additive,
                mergedHorizontaly,
                mergedVerticaly,
                uniqCellWidth = [],
                uniqColsIndexHolder = [];

            _(gridPanes).each(function (gridPane, panePos) {
                if (gridPane.hasLayerRange()) {
                    if (_.isNumber(targetCol)) {
                        // filter column ranges in selection containing the target column
                        visibleRanges = _(selectedRanges).filter(function (range) {
                            return model.isColRange(range) && SheetUtils.rangeContainsCol(range, targetCol);
                        });
                        if (visibleRanges.length === 0) {
                            visibleRanges = SheetUtils.getIntersectionRanges(model.makeColRange(targetCol), cellCollections[panePos].getRange());
                        } else {
                            visibleRanges = SheetUtils.getIntersectionRanges(selectedRanges, cellCollections[panePos].getRange());
                        }
                    }
                    _(visibleRanges).each(function (visibleRange) {
                        for (var i = visibleRange.start[0]; i < visibleRange.end[0] + 1; i++) {
                            uniqCellWidth[i] = 0;
                            for (var j = visibleRange.start[1]; j < visibleRange.end[1] + 1; j++) {
                                address = [i, j];
                                //procces only if cell is not merged
                                mergedHorizontaly = gridPane.getCellNode(address).attr('data-col-span');
                                mergedVerticaly = gridPane.getCellNode(address).attr('data-row-span');
                                if (!mergedHorizontaly && !mergedVerticaly) {
                                    newCellWidth = parseInt(gridPane.getCellNode(address).attr('data-optimal-width'), 10);
                                    if (_.isNaN(newCellWidth)) { newCellWidth = 0; }
                                    cellData = cellCollections[panePos].getCellEntry(address);
                                    //determine padding for justify or regular word wrap
                                    if (!_.isNull(cellData) && cellData.attributes.cell.alignHor === 'justify') {
                                        additive = 4;
                                    } else {
                                        additive = 6;
                                    }
                                    if (newCellWidth > 0 && newCellWidth + additive > cellWidth) {
                                        cellWidth = newCellWidth + additive;
                                    }
                                    newCellWidth = 0;
                                }
                            }
                            if (_.indexOf(uniqColsIndexHolder, i) < 0) {
                                uniqColsIndexHolder.push(i); // storing only unique indexes, without duplicates
                            }
                            if (cellWidth > uniqCellWidth[i]) {
                                uniqCellWidth[i] = cellWidth; // storing unique data for cell, not to be overriden by other gridpane in iteration (split mode)
                            }
                            cellWidth = 0;
                        } // end of for i
                    });
                }
            });
            for (var m = 0; m < uniqColsIndexHolder.length; m++) {
                if (uniqCellWidth[uniqColsIndexHolder[m]] > 0) {
                    activeSheetModel.setColumnWidth({ first: uniqColsIndexHolder[m], last: uniqColsIndexHolder[m] }, Utils.convertLengthToHmm(uniqCellWidth[uniqColsIndexHolder[m]] / self.getZoom(), 'px'), { custom: false });
                } else {
                    //if empty column, reset width to default
                    activeSheetModel.setColumnWidth({ first: uniqColsIndexHolder[m], last: uniqColsIndexHolder[m] }, colCollection.getDefaultAttributes().column.width, { custom: false });
                }
            }

            return this;
        };

        // row operations -----------------------------------------------------

        /**
         * Returns the row attributes of the row the active cell is currently
         * located in.
         *
         * @returns {Object}
         *  The row attribute map of the active cell.
         */
        this.getRowAttributes = function () {
            return rowCollection.getEntry(this.getActiveCell()[1]).attributes.row;
        };

        /**
         * Returns whether additional rows scan be inserted into the sheet,
         * based on the current selection.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the current
         *  selection and the number of used rows in the sheet.
         */
        this.rowsInsertable = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges()),
                // the number of inserted rows
                count = Utils.getSum(intervals, SheetUtils.getIntervalSize);

            // the number of used rows, and the number of inserted rows must not exceed the total number of rows in the sheet
            return (this.getUsedRows() + count) <= (model.getMaxRow() + 1);
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertRows = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            activeSheetModel.insertRows(intervals);
            return this;
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted.
         */
        this.rowsDeleteable = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges()),
                // the number of deleted rows
                count = Utils.getSum(intervals, SheetUtils.getIntervalSize);

            // prevent deleting all rows (entire sheet) at once
            return (0 < count) && (count <= model.getMaxRow());
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteRows = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            activeSheetModel.deleteRows(intervals);
            return this;
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.custom=true]
         *      The new value of the 'customHeight' row attribute.
         *  @param {Number} [options.target]
         *      The target row to be changed. If specified, the row height will
         *      be set to that row only. If the current selection contains any
         *      ranges covering this row completely, the row height will be set
         *      to all rows contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowHeight = function (height, options) {

            var // the selected ranges
                ranges = this.getSelectedRanges(),
                // the target row to be changed
                targetRow = Utils.getIntegerOption(options, 'target');

            // use row ranges in selection, if target row has been passed
            if (_.isNumber(targetRow)) {
                // filter row ranges in selection
                ranges = _(ranges).filter(function (range) { return model.isRowRange(range); });
                // if the target row is not contained in the selection, change that row only
                if (!_(ranges).any(function (range) { return SheetUtils.rangeContainsRow(range, targetRow); })) {
                    ranges = model.makeRowRange(targetRow);
                }
            }

            activeSheetModel.setRowHeight(SheetUtils.getRowIntervals(ranges), height, options);
            return this;
        };

        /**
         * Sets optimal row height based od content of cells.
         *
         * @param {Number} targetRow
         *  The target row to be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         *
         */
        this.setOptimalRowHeight = function (targetRow) {

            var
                selectedRanges = this.getSelectedRanges(),
                visibleRanges,
                cellData = 0,
                address = [],
                newCellData = 0,
                newCellDataSpan,
                mergedHorizontaly,
                mergedVerticaly,
                colVisibleRangeStart,
                colVisibleRangeEnd,
                rowVisibleRangeStart,
                rowVisibleRangeEnd,
                uniqCellHeight = [],
                uniqRowsIndexHolder = [];

            _(gridPanes).each(function (gridPane, panePos) {
                if (gridPane.hasLayerRange()) {
                    if (_.isNumber(targetRow)) {
                        // filter column ranges in selection containing the target column
                        visibleRanges = _(selectedRanges).filter(function (range) {
                            return model.isRowRange(range) && SheetUtils.rangeContainsRow(range, targetRow);
                        });
                        if (visibleRanges.length === 0) {
                            visibleRanges = SheetUtils.getIntersectionRanges(model.makeRowRange(targetRow), cellCollections[panePos].getRange());
                        } else {
                            visibleRanges = SheetUtils.getIntersectionRanges(selectedRanges, cellCollections[panePos].getRange());
                        }
                    }
                    _(visibleRanges).each(function (visibleRange) {
                        rowVisibleRangeStart = visibleRange.start[1];
                        rowVisibleRangeEnd = visibleRange.end[1] + 1;
                        colVisibleRangeStart = visibleRange.start[0];
                        colVisibleRangeEnd = visibleRange.end[0] + 1;

                        for (var i = rowVisibleRangeStart; i < rowVisibleRangeEnd; i += 1) {
                            uniqCellHeight[i] = 0;
                            for (var j = colVisibleRangeStart; j < colVisibleRangeEnd; j += 1) {
                                address = [j, i];
                                //procces only if cell is not merged
                                mergedHorizontaly = gridPane.getCellNode(address).attr('data-col-span');
                                mergedVerticaly = gridPane.getCellNode(address).attr('data-row-span');
                                if (!mergedHorizontaly && !mergedVerticaly) {
                                    newCellDataSpan = gridPane.getCellNode(address).find('span');
                                    //if we caught more than one span inside cell
                                    for (var k = 0; k < newCellDataSpan.length; k += 1) {
                                        newCellData += $(newCellDataSpan[k]).height();
                                    }
                                    if (newCellData > cellData) {
                                        cellData = newCellData;
                                    }
                                    newCellData = 0;
                                }
                            }
                            if (_.indexOf(uniqRowsIndexHolder, i) < 0) {
                                uniqRowsIndexHolder.push(i); // storing only unique indexes, without duplicates
                            }
                            if (cellData > uniqCellHeight[i]) {
                                uniqCellHeight[i] = cellData; // storing unique data for cell, not to be overriden by other gridpane in iteration (split mode)
                            }
                            cellData = 0;
                        } //end of for i
                    });
                } // end of if hasLayerRange()
            });
            for (var m = 0; m < uniqRowsIndexHolder.length; m++) {
                if (uniqCellHeight[uniqRowsIndexHolder[m]] > 0) {
                    activeSheetModel.setRowHeight({ first: uniqRowsIndexHolder[m], last: uniqRowsIndexHolder[m] }, Utils.convertLengthToHmm((uniqCellHeight[uniqRowsIndexHolder[m]] + 2) / self.getZoom(), 'px'), { custom: false });
                } else {
                    //if empty row, reset height to default
                    activeSheetModel.setRowHeight({ first: uniqRowsIndexHolder[m], last: uniqRowsIndexHolder[m] }, rowCollection.getDefaultAttributes().row.height, { custom: false });
                }
            }

            return this;
        };

        /**
         * Sets optimal row height when word wrap or inline cell editing applied
         *
         * @param {Boolean} fontChange
         *  Flag containing information if font size or font style are changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         *
         */
        this.setOptimalRowHeightWrap = function (fontChange) {

            var selectedRanges = this.getSelectedRanges(),
                self = this,
                visibleRanges = [],
                intersectionRange,
                uniqVisibleRange = {},
                cellData,
                createdNodes = [],
                createdNode,
                selectedIntervals,
                range,
                endInterval,
                mergedHorizontaly,
                mergedVerticaly,
                tempStorageNode = $('#io-ox-office-temp'),
                textWrap;

            _(gridPanes).each(function (gridPane, panePos) {
                if (gridPane.hasLayerRange()) {
                    selectedIntervals = SheetUtils.getRowIntervals(selectedRanges);
                    range = cellCollections[panePos].getRange();

                    _(selectedIntervals).each(function (selectedInterval) {
                        endInterval = selectedInterval.last + 1;
                        //if all rows are selected, or rows that are going outside visible area, they are ignored for calculation
                        for (var k = selectedInterval.first; k < (endInterval > range.end[1] ? range.end[1] : endInterval); k++) {
                            intersectionRange = SheetUtils.getIntersectionRange(model.makeRowRange(k), cellCollections[panePos].getRange());
                            if (!_.isNull(intersectionRange)) { //if grid pane doesnt contain selected range #31041
                                visibleRanges.push(intersectionRange);
                            }
                        }
                    });

                    if (visibleRanges.length > 0) { // also for #31041
                        _(visibleRanges).each(function (visibleRange) {
                            for (var i = visibleRange.start[1]; i < visibleRange.end[1] + 1; i++) {
                                //if for this row user set custom height, ignore changes
                                if (rowCollection.getEntry(i).attributes.row.customHeight === false) {
                                    createdNode = tempStorageNode.find('.resizeHelper.id' + i);
                                    if (createdNode.length === 0) {
                                        createdNode = $('<div>').addClass('resizeHelper id' + i).css({
                                            'position': 'absolute',
                                            'visibility': 'hidden',
                                            height: 'auto',
                                            width: '400%'
                                        });
                                        createdNodes.push(createdNode);
                                        self.insertTemporaryNode(createdNode);
                                    }
                                    for (var j = visibleRange.start[0]; j < visibleRange.end[0] + 1; j++) {
                                        //procces only if cell is not merged
                                        mergedHorizontaly = gridPane.getCellNode([j, i]).attr('data-col-span');
                                        mergedVerticaly = gridPane.getCellNode([j, i]).attr('data-row-span');
                                        if ((typeof mergedHorizontaly === 'undefined' || mergedHorizontaly === false) && (typeof mergedVerticaly === 'undefined' || mergedVerticaly === false)) {
                                            cellData = cellCollections[panePos].getCellEntry([j, i]);
                                            if (!_.isNull(cellData)) {
                                                if (cellData.display.length !== 0) {
                                                    textWrap = CellCollection.isWrappedText(cellData);
                                                    if (createdNode.find('.name' + cellData.address[0] + '-' + cellData.address[1]).length === 0) {
                                                        createdNode.append(
                                                            $('<div>').addClass('name' + cellData.address[0] + '-' + cellData.address[1]).css({
                                                                'padding': '0px 2px',
                                                                'font-family': documentStyles.getCssFontFamily(cellData.attributes.character.fontName),
                                                                'font-size': cellData.attributes.character.fontSize * self.getZoom() + 'pt',
                                                                'white-space': textWrap ? 'pre-wrap' : 'normal',
                                                                'width': textWrap ? (gridPane.getCellNode([j, i]).width() - 4) : 'auto',
                                                                'word-wrap': textWrap ? 'break-word' : 'normal',
                                                                'float': 'left',
                                                                'line-height': 'normal'
                                                            }).append($('<span>').text(cellData.display))
                                                        );
                                                    }
                                                } else if (fontChange || !_.isUndefined(cellData.attributes.character.fontSize)) {
                                                    //if its empty cell, but font size have changed
                                                    if (tempStorageNode.find('.name' + cellData.address[0] + '-' + cellData.address[1]).length === 0) {
                                                        createdNode.append(
                                                            $('<div>').addClass('name' + cellData.address[0] + '-' + cellData.address[1]).css({
                                                                'height': cellData.attributes.character.fontSize * self.getZoom() + 'pt',
                                                                'padding': '2px 0px',
                                                                'float': 'left'
                                                            })
                                                        );
                                                    }
                                                }
                                            } //end of if (!_.isNull(cellData))
                                        } // end of if merged
                                    } // end of for j
                                }
                            } // end of for i
                        });
                    }
                    if (visibleRanges.length > 0) {
                        _(visibleRanges).each(function (visibleRange) {
                            //create unique visible range from set of visible ranges from gridPanes
                            if (_.isUndefined(uniqVisibleRange.start) && _.isUndefined(uniqVisibleRange.end)) {
                                uniqVisibleRange.start = visibleRange.start[1];
                                uniqVisibleRange.end = visibleRange.end[1];
                            } else {
                                if (visibleRange.start[1] < uniqVisibleRange.start) {
                                    uniqVisibleRange.start = visibleRange.start[1];
                                }
                                if (visibleRange.end[1] > uniqVisibleRange.end) {
                                    uniqVisibleRange.end = visibleRange.end[1];
                                }
                            }
                        });
                    }
                }
            }); // end of gridPanes each
            for (var i = uniqVisibleRange.start; i < uniqVisibleRange.end + 1; i++) {
                if (tempStorageNode.find('.resizeHelper.id' + i).height() > 0) {
                    activeSheetModel.setRowHeight({ first: i, last: i }, Utils.convertLengthToHmm((tempStorageNode.find('.resizeHelper.id' + i).height() + 2) / self.getZoom(), 'px'), { custom : false });
                }
            }
            _(createdNodes).invoke('remove');

            return this;
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the value and formatting of the current active cell.
         *
         * @returns {Object}
         *  The cell value and formatting, in the following properties:
         *  - {String} display
         *      The formatted display string.
         *  - {Null|Number|Boolean|String} result
         *      The typed result value of the cell, or the formula result.
         *  - {String|Null} formula
         *      The formula definition, if the current cell is a formula cell.
         *  - {Object} attributes
         *      All formatting attributes of the cell, mapped by cell attribute
         *      family ('cell' and 'character'), and the identifier of the cell
         *      style sheet in the 'styleId' string property.
         *  - {Object} explicit
         *      The explicit formatting attributes of the cell, mapped by cell
         *      attribute family ('cell' and 'character'), and optionally the
         *      identifier of the cell style sheet.
         */
        this.getCellContents = function () {
            return _.clone(selectionSettings.active);
        };

        /**
         * Sets the contents of the active cell, and updates the view according
         * to the new cell value and formatting.
         *
         * @param {String} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellContents = function (value, attributes, options) {
            if (self.isSelectionLocked()) {
                self.yell('Protected cells cannot be modified.');
                return this;
            }
            activeSheetModel.setSingleCellContents(this.getActiveCell(), value, attributes, options);
            return this;
        };

        /**
         * Fills all cell ranges in the current selection with the same value
         * and formatting, and updates the view according to the new cell value
         * and formatting.
         *
         * @param {String} [value]
         *  The new value for all cells in the current selection. If omitted,
         *  only the cell formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operations, set to the name of the current UI language.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.fillCellRanges = function (value, attributes, options) {

            // quit if current selection contains locked cells
            if (self.isSelectionLocked()) {
                self.yell('info', gt('Protected cells cannot be modified.'));
                return this;
            }

            var // the result of the fill operation
                result = activeSheetModel.fillCellRanges(this.getSelectedRanges(), value, attributes, options),
                changedFontSize = null,
                changedStyle = null,
                //trigger resize row function only on certain actions:
                //changing font size, font style, justify, word wrap,
                resizableEventTriggers = false,
                fontChange = false; //flag for resizing empty cell on font or style change

            if (result === 'overflow') {
                this.yell('info', gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)));
                return this;
            }

            if (_.isObject(attributes)) {
                changedStyle = attributes.styleId;

                if (_.isObject(attributes.character)) {
                    changedFontSize = attributes.character.fontSize;
                }
                if (_.isObject(attributes.cell) && Utils.hasAnyProperty(attributes.cell, ['alignHor', 'wrapText'])) {
                    resizableEventTriggers = true;
                }
            }
            if (changedFontSize || changedStyle) {
                fontChange = true;
                resizableEventTriggers = true;
            }
            if (resizableEventTriggers) {
                this.setOptimalRowHeightWrap(fontChange);
            }
            //for clear formating - reset height to default
            if (_.isUndefined(attributes)) {
                this.setOptimalRowHeightWrap(true);
            }

            return this;
        };

        /**
         * Deletes the values and formatting attributes from all cell ranges in
         * the current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearCellRanges = function () {

            // quit if current selection contains locked cells
            if (self.isSelectionLocked()) {
                self.yell('info', gt('Protected cells cannot be modified.'));
                return this;
            }

            // generate and apply the clear operations
            activeSheetModel.clearCellRanges(this.getSelectedRanges());

            // update the automatic row heights
            this.setOptimalRowHeightWrap(true);

            return this;
        };

        /**
         * Support for the auto fill operation to fill a selected range with predefined values.
         *
         * @param {Object} autoFillRange
         *  The logical address of the cell range describing the auto fill range.
         *
         * @param {Object} startRange
         *  The logical address of the cell range describing the range, when the auto fill
         *  modification was started.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when given ranges are editable,
         *  and will be rejected if not.
         */
        this.autoFill = function (autoFillRange, startRange) {

            // do autoFill only after we are sure we know if the range is editable or not
            return self.areRangesEditable(autoFillRange).done(function () {
                var // the generator for the operations
                    generator = model.createOperationsGenerator(),
                // additional options for the operations
                    operationOptions = null,
                // the positiion of the upper left or bottom right corner of the auto fill range, that is not part of the selection
                    target = null;

                if (! _.isEqual(autoFillRange.end, startRange.end)) {
                    target = _.clone(autoFillRange.end);
                } else if (! _.isEqual(autoFillRange.start, startRange.start)) {
                    // modifying only one number of the logical end position
                    target = _.clone(autoFillRange.end);

                    if (autoFillRange.start[0] === startRange.start[0]) {
                        target[1] = autoFillRange.start[1];
                    } else if (autoFillRange.start[1] === startRange.start[1]) {
                        target[0] = autoFillRange.start[0];
                    }
                }

                // do nothing, if selection did not change
                if (! target) { return; }

                operationOptions = { target: target };

                // create and send the 'autofill' operation
                generator.generateRangeOperation(Operations.AUTO_FILL, sheetSettings.sheet, startRange, operationOptions);

                model.applyOperations(generator.getOperations());

            }).fail(function () {
                self.yell('info', gt('Protected cells cannot be modified.'));
            });
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { cell: Utils.makeSimpleObject(name, value) });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  all attributes set explicitly will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillCellRanges(undefined, { character: Utils.makeSimpleObject(name, value) });
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearAttributes = function () {
            return this.fillCellRanges(undefined, undefined, { clear: true });
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MixedBorder.getBorderMode(selectionSettings.borders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MixedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setBorderMode = function (borderMode) {
            var borderAttributes = MixedBorder.getBorderAttributes(borderMode, selectionSettings.borders);
            activeSheetModel.setBorderAttributes(self.getSelectedRanges(), borderAttributes);
            return this;
        };

        /**
         * Returns the merged border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the merged borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.) in the current
         *  selection. Each border attribute value contains the properties
         *  'style', 'width', and 'color'. If the border property is equal in
         *  all affected cells, it will be set as property value in the border
         *  object, otherwise the property will be set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectionSettings.borders, true);
        };

        /**
         * Changes the specified border properties (line style, line color, or
         * line width) of all visible cell borders in the current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.changeVisibleBorders = function (border) {
            activeSheetModel.changeVisibleBorders(this.getSelectedRanges(), border);
            return this;
        };

        /**
         * Merges or unmerges all cells of the current selection.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the selection.
         *  - 'toggle': Will be mapped to 'merge', if the selection does not
         *      cover any merged ranges, otherwise will be mapped to 'unmerge'.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.mergeRanges = function (type) {

            var // the selected ranges
                ranges = this.getSelectedRanges(),
                // the result of the operation
                result = null;

            // toggle merge state if specified
            if (type === 'toggle') {
                type = mergeCollection.rangesOverlapAnyMergedRange(ranges) ? 'unmerge' : 'merge';
            }

            // send the 'mergeCells' operations
            result = activeSheetModel.mergeRanges(ranges, type, self.getUsedCols(), self.getUsedRows());

            // yell errors
            if (result === 'overlap') {
                this.yell('info', gt('Overlapping ranges cannot be merged.'));
            }

            return this;
        };

        // drawing operations -------------------------------------------------

        /**
         * Inserts a new drawing object into the active sheet, and selects it
         * afterwards.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing in the sheet.
         *
         * @param {Function} [generatorCallback]
         *  A callback function that will be invoked after the initial
         *  'insertDrawing' operation has been generated. Allows to create
         *  additional operations for the drawing object, before all these
         *  operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SpreadsheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of the active sheet that will contain the
         *      drawing object.
         *  (3) {Array} position
         *      The logical position of the new drawing object in the sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertDrawing = function (type, attributes, generatorCallback) {

            var // the logical position of the new drawing object
                position = [drawingCollection.getModelCount()],
                // create the new drawing object
                result = activeSheetModel.insertDrawing(position, type, attributes, generatorCallback);

            // select the new drawing object
            if (result.length === 0) {
                this.selectDrawing(position);
            } else {
                this.removeDrawingSelection();
            }

            return this;
        };

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteDrawings = function () {

            if (activeSheetModel.isProtected()) {
                self.yell('info', gt('Objects on a protected sheet cannot be deleted.'));
                return this;
            }

            var // the selected drawing objects
                positions = this.getSelectedDrawings();

            // back to cell selection, delete all drawings previously selected
            this.removeDrawingSelection();
            activeSheetModel.deleteDrawings(positions);

            return this;
        };

        /**
         * Setting drawing attributes.
         *
         * @param {Object} [attributes]
         *  A drawing attribute set, keyed by the supported attribute family
         *  'drawing'. Omitted attributes will not be changed. Attributes set
         *  to the value null will be removed from the drawing.
         *
         * @param {Array} [pos]
         *  An array with at least one logical drawing address. If not
         *  specified, the selected drawings are used.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setDrawingAttributes = function (attributes, pos) {

            var // the selected drawing objects
                drawings = pos || this.getSelectedDrawings(),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send the 'setDrawingAttributes' operations
            _(drawings).each(function (position) {
                generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, sheetSettings.sheet, position, { attrs: attributes });
            });
            model.applyOperations(generator.getOperations());

            return this;
        };

        // search operations --------------------------------------------------

        /**
         *  Triggers a search: Build a search object and trigger a requestUpdate with it.
         *
         *  @param {String} searchQuery
         *  The string to be searched.
         */
        this.search = function (searchQuery, options) {

            // quit if search bar is maybe opened but user are searching yet
            if (searchQuery.length === 0) {return; }

            // reset current search result pointer and range pointer before every search
            searchResultPointer = -1;
            activeSearchQuery = searchQuery;
            self.searchClear();

            var isRefresh = Utils.getBooleanOption(options, 'refresh', false);

            // build request props
            var requestProps = {};

            // The search starts from the selected cell per default.
            // This way, user will always get the closest result hit to his/her eye focus.
            var currentActiveCell = self.getActiveCell();

            requestProps.search = {
                searchText : searchQuery,
                start: currentActiveCell,
                limit: MAX_SEARCH_RESULT_COUNT,
                refresh: isRefresh
            };

            // request the search results with an view update
            self.requestUpdate(requestProps);
        };

        /**
         * Iterate search result items.
         *
         * @param {String} direction
         * The iterate direction. 2 values possible: 'next' or 'prev'
         */
        this.searchIterate = function (direction) {

            // if there are no results or if user wants to search a new query, do a search refresh implicitely and quit
            if (!_.isObject(searchLayoutData.sheets) || activeSearchQuery !== app.getWindow().search.query) {
                if (app.getState() !== 'sending') self.search(app.getWindow().search.query);
                return;
            }

            // if direction is not passed, set next as default
            direction = direction || 'next';

            var currentSheetResult = searchLayoutData.sheets[sheetSettings.sheet];

            // do nothing, if user switched to other sheets after search.
            if (_.isUndefined(currentSheetResult)) return;

            // get result array size
            var searchResultSize = currentSheetResult.length;

            // init next iteration address
            var nextHitAddress = [];

            // set ring structure for next and prev actions
            switch (direction)
            {
            case 'prev':
                if (searchResultPointer > 0) {
                    searchResultPointer--;
                } else {
                    searchResultPointer = searchResultSize - 1;
                }
                break;
            case 'next':
                if (searchResultPointer < (searchResultSize - 1)) {
                    searchResultPointer++;
                } else {
                    searchResultPointer = 0;
                }
                break;
            default:
                return;
            }

            // get next iteration item
            nextHitAddress = currentSheetResult[searchResultPointer];

            // go to the grid area and select that cell
            this.getActiveGridPane().scrollToCell(nextHitAddress);

            // and finally, go to that next iteration cell
            this.selectCell(nextHitAddress);

        };

        /**
         * Replaces the value of the current (single) selection with the replacement value.
         *
         * @param {Object} option.all = false
         * option.all = true -> do a replace all.
         */
        this.searchReplace = function (option) {

            if (activeSheetModel.isProtected()) {
                self.yell('info', gt('You cannot use this command on a protected sheet.'));
                return;
            }

            // get fresh search query and replacement
            var win = app.getWindow(),
                searchQuery = win.search.getQuery(),
                replacementText = win.search.replacement;

            // Similar like native indexOf, but only for cell addresses.
            function indexOfAddress(addressArray, address) {
                var resultIndex = -1;
                _.each(addressArray, function (elem, index) {
                    if (address[0] === elem[0] && address[1] === elem[1]) {
                        resultIndex = index;
                    }
                });
                return resultIndex;
            }

            // 'fly' to cell: scroll and select desired cell.
            function goToCell(address) {
                if (!model.isValidAddress(address)) return;
                self.getActiveGridPane().scrollToCell(address);
                self.selectCell(address);
            }

            // do nothing and return immediately if search query is empty
            if (searchQuery === '') {
                self.yell('info', gt('Nothing to replace.'));
                return;
            }

            // defaults to false-> a single replacement
            var replaceAll = Utils.getBooleanOption(option, 'all', false);

            // initiate a replace call to calcengine.
            if (replaceAll) {
                var
                def = null,
                localStartTime = _.now(),
                localRequestData = {
                    sheet: sheetSettings.sheet,
                    searchText: searchQuery,
                    replaceText: replacementText
                };

                def = app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'replaceAll',
                        requestdata: JSON.stringify(localRequestData)
                    }
                });
                def.fail(function () {
                    Utils.error('SpreadsheetView.executeReplace(): replace request # failed');
                })
                .done(function (result) {
                    Utils.info('SpreadsheetView.executeReplace(): replace response # received (' + (_.now() - localStartTime) + 'ms)', result);
                    var replacedCellsCount = 0;
                    var changedRanges = [];

                    if (_.isEmpty(result.changed.sheets)) {
                        self.yell('info', gt('Nothing to replace.'));
                        return;
                    }

                    changedRanges = result.changed.sheets[sheetSettings.sheet].cells;

                    _.each(changedRanges, function (elem) {
                        if (_.isUndefined(elem.end)) elem.end = elem.start;
                        replacedCellsCount = replacedCellsCount + SheetUtils.getCellCount(elem);
                    });

                    var countLabel = gt.format(
                        //#. %1$d is the number of rows in an oversized text table
                        //#, c-format
                        gt.ngettext('Successfully replaced %1$d cell.', 'Successfully replaced %1$d cells.', replacedCellsCount),
                        _.noI18n(replacedCellsCount)
                    );

                    self.yell('info', countLabel);
                });

            } else { // single replacement

                var activeResult = searchLayoutData.sheets;

                // do a initial search if not done yet, and additionally everytime user changes the query.
                // Don't search while app is sending data to prevent invalid old results.
                if (_.isUndefined(activeResult) || activeSearchQuery !== app.getWindow().search.query) {
                    if (app.getState() !== 'sending') self.search(searchQuery);
                    return;
                }

                // fly to the search results if user are not on result cells
                if (indexOfAddress(activeResult[sheetSettings.sheet], self.getActiveCell()) === -1) {
                    goToCell(activeResult[sheetSettings.sheet][0]);
                }

                var currentCellContent = self.getCellContents(),
                    oldCellValue = '',
                    newCellValue = '';

                // generate replacement value:
                // if the current search result hit is a formula cell, check equality and replace the formula.
                // TODO handle number format special replacements like Date, Currency etc.
                if (_.isString(currentCellContent.formula)) {
                    newCellValue = currentCellContent.formula.replace(searchQuery, replacementText, 'gi');
                    oldCellValue = currentCellContent.formula;
                } else { // if cell is a normal value cell, replace the display property.
                    newCellValue = currentCellContent.display.replace(searchQuery, replacementText, 'gi');
                    oldCellValue = currentCellContent.display;
                }

                // and finally only do single replace operation if we have a new value
                if (oldCellValue !== newCellValue) {
                    replaceAction = true;
                    self.setCellContents(newCellValue, undefined, { parse: true });
                    var activeResultArray = activeResult[sheetSettings.sheet],
                        activeCell = self.getActiveCell(),
                        activeCellIndex = 0;
                    // take out the cell to be replaced out of the result array, and save its index to
                    // do a immediate replacement at that address.
                    activeResult[sheetSettings.sheet] = _.reject(activeResultArray, function (element, index) {
                        if (element[0] === activeCell[0] && element[1] === activeCell[1]) {
                            activeCellIndex = index;
                            return true;
                        }
                        return false;
                    });

                    var nextCell = activeResult[sheetSettings.sheet][activeCellIndex];
                    if (!_.isUndefined(nextCell)) {
                        // iterate to next cell in the search result
                        goToCell(nextCell);
                    } else {
                        //we're at the end of the results array. Check if we still have results to iterate.
                        if (activeResult[sheetSettings.sheet].length === 0) {
                            searchLayoutData.sheets = undefined;
                            self.yell('info', gt('Nothing else to replace.'));
                        } else {
                            // iterate back at the beginning of the search result
                            goToCell(activeResult[sheetSettings.sheet][0]);
                        }
                    }
                }
            }
        };

        /**
         * Clears local search result array and optionally search/replace input fields.
         *
         * @param {Object} [options.clearInputs=false] - Clears search/replace input fields if set to true.
         *
         */
        this.searchClear = function (options) {
            searchLayoutData = [];
            var win = app.getWindow();
            if (Utils.getBooleanOption(options, 'clearInputs', false)) {
                win.search.clear();
                win.search.replacementInput.val('');
            }
            replaceAction = false;
        };

        /**
         * Sum automatically button
         */
        this.insertAutoSum = function () {

            //variable declarations
            var activeCellEntry,
                selection = self.getSelection(),
                getRangeName = SheetUtils.getRangeName(selection.ranges[0]),

                startColumn = selection.ranges[0].start[0],
                endColumn = selection.ranges[0].end[0],
                startRow = selection.ranges[0].start[1],
                endRow = selection.ranges[0].end[1],

                activeRow = selection.ranges[0].start[1],
                activeCellColumn = selection.activeCell[0],
                activeCellRow = selection.activeCell[1],

                rangeHighlightColumn,
                rangeHighlightRow,
                eachRange,
                formula,
                lastIsEmptyColumn,
                lastIsEmptyRow,
                allEmptyColumn,
                allEmptyRow,
                column,
                row,
                nextRow,
                nextColumn,
                i, j, j2,

                cellLogicalStartName = SheetUtils.getCellName(selection.ranges[0].start),
                cellLogicalEndName = SheetUtils.getCellName([selection.ranges[0].end[0] - 1, selection.ranges[0].end[1]]),
                cellLogicalEndName2 = SheetUtils.getCellName([selection.ranges[0].end[0], selection.ranges[0].end[1] - 1]),

                rangeColumnBack = cellLogicalStartName + ':' + cellLogicalEndName, //if last cell in range is empty in row -> write sum inside (don't go into new column)
                rangeRowBack = cellLogicalStartName + ':' + cellLogicalEndName2;

            /** active cell is selected only, no cell-range */
            if ((selection.ranges[0].start[0] === selection.ranges[0].end[0]) && (selection.ranges[0].start[1] === selection.ranges[0].end[1])) {

                //ajax call: client -> calcengine
                app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'query',
                        requestdata: JSON.stringify({
                            type: 'autoSum',
                            locale: Utils.LOCALE,
                            sheet: sheetSettings.sheet,
                            selectionStart: [selection.ranges[0].start[0], selection.ranges[0].start[1]],
                            selectionEnd: [selection.ranges[0].end[0], selection.ranges[0].end[1]],
                            activeCell: [selection.activeCell[0], selection.activeCell[1]]
                        })
                    }
                })
                .done(function (response) { //response - calcengine returned data
                    //show sum-range and on press-enter show sum-result
                    self.enterCellEditMode({ text: response.formula[0].value });
                })
                .fail(function () {
                    Utils.error('SpreadshetView.initLocaleData(): could not download locale data for "' + Utils.LOCALE + '"');
                });
            }
            /** multiple columns/rows are selected */
            else if ((selection.ranges[0].start[0] !== selection.ranges[0].end[0]) && (selection.ranges[0].start[1] !== selection.ranges[0].end[1])) {
                //get range name of the first column
                eachRange = SheetUtils.getRangeName({ start: selection.ranges[0].start, end: [selection.ranges[0].start[0], selection.ranges[0].end[1]] });
                formula = '=SUM(' + eachRange + ')';
                var targetRanges = [{start: [selection.ranges[0].start[0], selection.ranges[0].end[1] + 1], end: [selection.ranges[0].end[0], selection.ranges[0].end[1] + 1]}];
                self.areRangesEditable(targetRanges).done(function () {
                    //write sum for every column seperately in a new row
                    activeSheetModel.fillCellRanges(targetRanges, formula, null, { ref: [selection.ranges[0].start[0], selection.ranges[0].end[1] + 1]});
                })
                .fail(function () {
                    self.yell('info', gt('Protected cells cannot be modified.'));
                });
            }
            else {
                /** cell-range is selected (one column only): top->down, down->up */
                if (selection.ranges[0].start[0] === selection.ranges[0].end[0]) {

                    lastIsEmptyColumn = null;
                    allEmptyRow = true;

                    //loop to see if selected cell range (column) contains numbers
                    for (i = 0; i <= (endRow - startRow); i++) {
                    //loop in rows (down, up)

                        row = endRow - i;
                        activeCellEntry = self.getCellEntry([startColumn, row]);

                        if (CellCollection.isBlank(activeCellEntry)) {
                            //do nothing
                        }
                        else {
                            allEmptyRow = false;
                        }
                    }
                    if (allEmptyRow) {

                      //ajax call: client -> calcengine
                        app.sendFilterRequest({
                            method: 'POST',
                            params: {
                                action: 'query',
                                requestdata: JSON.stringify({
                                    type: 'autoSum',
                                    locale: Utils.LOCALE,
                                    sheet: sheetSettings.sheet,
                                    selectionStart: [selection.ranges[0].start[0], selection.ranges[0].start[1]],
                                    selectionEnd: [selection.ranges[0].end[0], selection.ranges[0].end[1]],
                                    activeCell: [selection.activeCell[0], selection.activeCell[1]]
                                })
                            }
                        })
                        .done(function (response) { //response - calcengine returned data

                            sumFormula = response;

                            for (i = 0; i <= (endRow - startRow); i++) {
                                //loop in rows (down, up).

                                activeCellRow =  startRow + i;

                                //select active cell
                                self.selectCell([startColumn, activeCellRow]);
                                //write sum into active cell
                                self.setCellContents(sumFormula.formula[i].value, sumFormula.formula[i].attrs);

                            }//end-of for()
                        })
                        .fail(function () {
                            Utils.error('SpreadshetView.initLocaleData(): could not download locale data for "' + Utils.LOCALE + '"');
                        });
                    }//end-of allEmptyRow
                    else {
                        activeCellEntry = self.getCellEntry([startColumn, endRow]);

                        //check if last cell in column is empty
                        lastIsEmptyColumn = CellCollection.isBlank(activeCellEntry);
                        if (lastIsEmptyColumn) { //last cell in cell range column is empty -> write sum inside
                            getRangeName = rangeRowBack; //new range with one column less
                            self.selectCell(selection.ranges[0].end);
                        }
                        else {//write sum into first empty row (in same column) that comes next.

                            for (j = 1; j < 1000; j++) {

                                nextRow = endRow + j;
                                activeCellEntry = self.getCellEntry([startColumn, nextRow]);

                                if (CellCollection.isBlank(activeCellEntry)) {
                                    self.selectCell([selection.ranges[0].end[0], selection.ranges[0].end[1] + j]);
                                    break;
                                }
                            }//end-of for()
                        }
                        self.setCellContents('=SUM(' + getRangeName + ')');

                        //highlight range
                        rangeHighlightColumn = {start: selection.ranges[0].start, end: [selection.ranges[0].end[0], selection.ranges[0].end[1] + 1]};
                        this.selectRange(rangeHighlightColumn);
                    }
                }
                /** cell-range is selected (one row only): left->right, right->left */
                else if (selection.ranges[0].start[1] === selection.ranges[0].end[1]) {

                    allEmptyColumn = true;
                    lastIsEmptyRow = null;

                    //loop to see if selected cell range (row) contains numbers
                    for (i = 0; i <= (endColumn - startColumn); i++) {
                    //loop in columns (right, left)

                        column = endColumn - i;
                        activeCellEntry = self.getCellEntry([column, activeRow]);

                        if (CellCollection.isBlank(activeCellEntry)) {
                            //do nothing
                        }
                        else {
                            allEmptyColumn = false;
                        }
                    }
                    if (allEmptyColumn) {

                        //ajax call: client -> calcengine
                        app.sendFilterRequest({
                            method: 'POST',
                            params: {
                                action: 'query',
                                requestdata: JSON.stringify({
                                    type: 'autoSum',
                                    locale: Utils.LOCALE,
                                    sheet: sheetSettings.sheet,
                                    selectionStart: [selection.ranges[0].start[0], selection.ranges[0].start[1]],
                                    selectionEnd: [selection.ranges[0].end[0], selection.ranges[0].end[1]],
                                    activeCell: [selection.activeCell[0], selection.activeCell[1]]
                                })
                            }
                        })
                        .done(function (response) { //response - calcengine returned data

                            sumFormula = response;

                            for (i = 0; i <= (endColumn - startColumn); i++) {
                                //loop in columns (right, left).

                                activeCellColumn =  startColumn + i;

                                //select active cell
                                self.selectCell([activeCellColumn, activeCellRow]);
                                //write sum into active cell
                                self.setCellContents(sumFormula.formula[i].value, sumFormula.formula[i].attrs);

                            }//end-of for()
                        })
                        .fail(function () {
                            Utils.error('SpreadshetView.initLocaleData(): could not download locale data for "' + Utils.LOCALE + '"');
                        });
                    }//end-of allEmpty
                    else { //cell range contains numbers -> sum numbers in row only

                        activeCellEntry = self.getCellEntry([endColumn, activeRow]);

                        //check is last cell in range is empty
                        lastIsEmptyRow = CellCollection.isBlank(activeCellEntry);
                        if (lastIsEmptyRow) { //if last cell in cell range row is empty -> write sum inside
                            getRangeName = rangeColumnBack; //new range with one column less
                            self.selectCell(selection.ranges[0].end);
                        }
                        else {//write sum into first empty column (in same row) that comes next.

                            for (j2 = 1; j2 < 1000; j2++) {

                                nextColumn = endColumn + j2;
                                activeCellEntry = self.getCellEntry([nextColumn, activeRow]);

                                if (CellCollection.isBlank(activeCellEntry)) {
                                    self.selectCell([selection.ranges[0].end[0] + j2, selection.ranges[0].end[1]]);
                                    break;
                                }
                            }//end-of for()
                        }
                        self.setCellContents('=SUM(' + getRangeName + ')');

                        //highlight range
                        rangeHighlightRow = {start: selection.ranges[0].start, end: [selection.ranges[0].end[0] + 1, selection.ranges[0].end[1]]};
                        this.selectRange(rangeHighlightRow);

                    }//end-of else
                }
            } //end-of  cell range is selected (one row only): left->right, right->left
        };/** end-of Sum automatically button */

        // returns the current number format category of the current active cell in the current selection
        this.getNumberFormatCategory = function () {
            return selectionSettings.active.format.cat;
        };

        /**
         * Sets selected number format category for the current active cell in the current selection,
         * and implicitly set the first format code of the predefined template of this category to the cell.
         */
        this.setNumberFormatCategory = function (category) {
            var defaultCodeForCategory = formatCodePicker.getDefaultCodeForCategory(category);
            if (category === 'custom' || _.isNull(defaultCodeForCategory)) return;
            self.setCellAttribute('numberFormat', { code: formatCodePicker.getDefaultCodeForCategory(category) });
        };

        // returns the format code of the current active cell in the current selection
        this.getNumberFormatCode = function () {
            return selectionSettings.active.attributes.cell.numberFormat.code || 'General';
        };

        // tells the format code dropdown to refresh its list
        this.refreshFormatCodeList = function (category) {
            formatCodePicker.refreshList(category);
        };

        // helper to automatically remove highlighted button - format painter
        this.applyPainterState = function () {
            if (_.isEmpty(painterAttrs)) {
                return false;
            }
        };

        // format painter
        this.applyPainter = function (state) {
            if (state) {
                //global variable
                painterAttrs = this.getCellContents();
            }
            else {
                if (stateChange) { //global variable
                    if (model.getEditMode()) { //check whether the document can be modified by the current user
                        // do applyPaint only after we are sure we know if the range is editable or not
                        self.areRangesEditable(self.getSelectedRanges()).done(function () {
                            if (painterAttrs.explicit.hasOwnProperty('styleId')) { //check for "Cell style" and remove attr hyperlink
                                if (painterAttrs.explicit.styleId === cellStyles.getHyperlinkStyleId()) {
                                    isHyper = false;
                                }
                            }
                            //check for "Insert/Edit hyperlink" and remove attr hyperlink
                            if (_.isObject(painterAttrs.explicit.character)) {

                                if (_.isObject(painterAttrs.explicit.character.color)) {
                                    if (painterAttrs.explicit.character.color.hasOwnProperty('value')) {
                                        if (painterAttrs.explicit.character.color.value === 'hyperlink') {
                                            isHyper = false;
                                        }
                                        else isHyper = true;
                                    }
                                }
                            }
                            if (isHyper) {
                                if (_.isObject(painterAttrs.explicit.character)) {
                                    if (painterAttrs.explicit.character.hasOwnProperty('url')) {
                                        delete painterAttrs.explicit.character.url;
                                        self.fillCellRanges(undefined, painterAttrs.explicit, { clear: true });
                                        isHyper = true;
                                    }
                                    self.fillCellRanges(undefined, painterAttrs.explicit, { clear: true });
                                    isHyper = true;
                                }
                                else {
                                    self.fillCellRanges(undefined, painterAttrs.explicit, { clear: true });
                                    isHyper = true;
                                }
                            }
                        }).fail(function () {
                            self.yell('info', gt('Protected cells cannot be modified.'));
                        });
                    }
                    painterAttrs = {};
                }
            }
            stateChange = state;
            isHyper = true;
            return state;
        };//end-of format painter

        /**
         * Creates a client clipboard ID and resets the server clipboard ID.
         *
         * @returns {String}
         *  The client clipboard ID.
         */
        this.createClientClipboardId = function () {
            clipboardId = {client: Clipboard.createClientClipboardId(), server: null};
            return clipboardId.client;
        };

        /**
         * Returns the server clipboard ID if the client clipboard ID matches
         * the given one, otherwise null.
         *
         * @param {String} clientClipboardId
         *  The client clipboard ID.
         *
         * @returns {String|Null}
         *  The corresponding server clipboard ID.
         */
        this.getServerClipboardId = function (clientClipboardId) {
            return (clipboardId.client === clientClipboardId) ? clipboardId.server : null;
        };

        /**
         * Initiates the server side clipboard copy action.
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the copy action request.
         */
        this.clipboardServerCopy = function (selection) {

            var activeRange = selection.ranges[selection.activeRange],
                localRequestData = {
                    sheet: sheetSettings.sheet,
                    selection: {sheet: sheetSettings.sheet, start: activeRange.start, end: activeRange.end}
                };

            return app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'copy',
                    requestdata: JSON.stringify(localRequestData)
                }
            })
            .done(function (result) {
                // store server clipboard id
                clipboardId.server = result.clipboardHandle;
                Utils.info('clipboard - copy - response received - server id: ' + result.clipboardHandle);
            })
            .fail(function () {
                Utils.error('clipboard - copy - request failed');
            });
        };

        /**
         * Initiates the server side clipboard paste action.
         *
         * @param {Object} selection
         *  The current selection.
         *
         * @param {String} serverClipboardId
         *  The server clipboard id that identifies the data on the server side.
         *
         * @returns {jQuery.Promise}
         *  The Promise of the paste action request.
         */
        this.clipboardServerPaste = function (selection, serverClipboardId) {

            var activeRange = selection.ranges[selection.activeRange],
                localRequestData = {
                    sheet: sheetSettings.sheet,
                    selection: {sheet: sheetSettings.sheet, start: activeRange.start, end: activeRange.end},
                    clipboardHandle: serverClipboardId
                };

            return app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'paste',
                    requestdata: JSON.stringify(localRequestData)
                }
            })
            .done(function () {
                Utils.info('clipboard - paste - response received');
            })
            .fail(function () {
                Utils.error('clipboard - paste - request failed');
            });
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // create the header pane instances, initialize settings for pane sides
        _(['left', 'right', 'top', 'bottom']).each(function (paneSide) {
            paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
            headerPanes[paneSide] = new HeaderPane(app, paneSide);
        });

        // create the cell collections and grid pane instances
        _(['topLeft', 'topRight', 'bottomLeft', 'bottomRight']).each(function (panePos) {
            cellCollections[panePos] = new CellCollection(app);
            cellCollections[panePos].on('triggered', updateSelectionSettings);
            gridPanes[panePos] = new GridPane(app, panePos, cellCollections[panePos]);
        });

        // create the corner pane instance
        cornerPane = new CornerPane(app);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(gridPanes).invoke('destroy');
            _(headerPanes).invoke('destroy');
            _(cellCollections).invoke('destroy');
            cornerPane.destroy();
            gridPanes = headerPanes = cellCollections = cornerPane = null;
        });

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
