/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statusbar',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/component',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Pane, Component, Controls, gt) {

    'use strict';

    var // class name shortcuts
        Button = Controls.Button;

    // class StatusBar ========================================================

    /**
     * @constructor
     *
     * @extends Pane
     */
    function StatusBar(app) {

        var // self reference
            self = this,

            // get application model and view
            view = app.getView(),

            // the sheet selector controls
            sheetGroup = null,
            subtotalsTypeGroup = null,
            largeSheetList = null,
            smallSheetList = null,

            // the view component for all sheet controls
            sheetComponent = null,

            // the view component for the zoom buttons
            zoomComponent = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, app, 'statusbar', { position: 'bottom', classes: 'status-bar inline' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the sheet selectors according to the current
         * width of the sheet component.
         */
        function refreshSheetComponent() {

            var // the root node of this component
                rootNode = sheetComponent.getNode(),
                // the current width of the sheet component
                width = rootNode.width();

            // calculate and set remaining space for the specified sheet selector
            function setRemainingWidth(group) {
                var remainingWidth = group.getNode().outerWidth() + width - rootNode[0].scrollWidth;
                group.setWidth(remainingWidth);
            }

            // hide the sheet list selectors, show the sheet tab selector with automatic width
            rootNode.width(1);
            largeSheetList.hide();
            smallSheetList.hide();
            sheetGroup.show().setFullWidth();

            // check whether ALL sheet tabs fit into the available space
            if (width < rootNode[0].scrollWidth) {
                if (width < 300) {
                    // show large sheet drop-down list for small devices
                    sheetGroup.hide();
                    largeSheetList.show().setWidth(width);
                    // calculate and set remaining space for the sheet drop-down button
                    setRemainingWidth(largeSheetList);
                } else {
                    // show additional sheet selector controls
                    smallSheetList.show();
                    // calculate and set remaining space for the sheet tabs
                    setRemainingWidth(sheetGroup);
                }
            }

            // reset sheet component to its current width
            rootNode.width(width);
        }

        /**
         * Refreshes the layout of the status bar after its size has been
         * changed.
         */
        function refreshLayout() {

            var // the available inner width in the status bar
                availableWidth = self.getNode().width();

            // hide subtotal label if not enough space available
            if (availableWidth < 380) {
                subtotalsTypeGroup.hide();
            } else {
                subtotalsTypeGroup.refresh();
            }

            // reduce available width by the size of the zoom component
            availableWidth -= zoomComponent.getNode().outerWidth(true);

            // set new width to the sheet component and refresh all controls
            sheetComponent.getNode().width(availableWidth);
            refreshSheetComponent();
        }

        // initialization -----------------------------------------------------

        this.addViewComponent(sheetComponent = new Component(app, 'sheets')
            .addGroup('view/sheet/active', largeSheetList = new Controls.ActiveSheetList(app, { showNames: true }))
            .addGroup('view/sheet/active', smallSheetList = new Controls.ActiveSheetList(app))
            .addGroup('view/sheet/active', sheetGroup = new Controls.ActiveSheetGroup(app))
            .addGroup('sheet/insert', new Button({ icon: 'icon-plus', tooltip: gt('Insert sheet') }))
            .addPrivateGroup(new Controls.PopupMenu(app, 'sheetactions', { icon: 'icon-ellipsis-vertical', tooltip: gt('More sheet operations'), caret: false, position: 'top' })
                .addGroup('sheet/rename/dialog', new Button({ label: gt('Rename sheet') }))
//bug 31560: disabled in 7.4.2                .addGroup('sheet/copy/dialog', new Button({ label: gt('Copy sheet') }))
                .addGroup('sheet/delete', new Button({ label: gt('Delete sheet') }))
                .addGroup('sheet/visible', new Button({ label: gt('Hide sheet'), value: false }))
                .addSeparator()
                .addGroup('sheet/reorder/dialog', new Button({ label: gt('Reorder sheets') }))
                .addGroup('sheet/showall', new Button({ label: gt('Show all hidden sheets') }))
            )
        );

        this.addViewComponent(zoomComponent = new Component(app, 'zoom')
            .addGroup('view/subtotals/type', subtotalsTypeGroup = new Controls.SubtotalList(app, { position: 'top' }))
            .addGroup('view/zoom/dec', new Button(Controls.ZOOMOUT_OPTIONS))
            .addGroup('view/zoom/inc', new Button(Controls.ZOOMIN_OPTIONS))
        );

        // refresh layout after status bar changes its size
        view.on('refresh:layout', refreshLayout);
        zoomComponent.on('component:layout', refreshLayout);

        // refresh sheet component after sheet tabs have been inserted, deleted, or changed
        sheetGroup.on('refresh:layout', refreshSheetComponent);

    } // class StatusBar

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: StatusBar });

});
