/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/dialogsmixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/drawinglayer/view/imageutil',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Dialogs, ImageUtil, gt) {

    'use strict';

    var // maximum size of an image in any direction, in 1/100 mm
        MAX_IMAGE_SIZE = 21000;

    // mix-in class SpreadsheetDialogsMixin ===================================

    /**
     * Mix-in class for the class SpreadsheetView that adds methods to show
     * modal dialogs.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function SpreadsheetDialogsMixin(app) {

        var // self reference (the spreadsheet view instance)
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Shows a dialog that allows to enter a sheet name, and will invoke an
         * arbitrary callback function for the sheet name. The dialog will be
         * kept open until a valid sheet name has been entered.
         *
         * @param {Function} callback
         *  The callback function invoked when the OK button of the dialog has
         *  been clicked. Receives the current sheet name shown in the dialog.
         *  Must return a Boolean value that states whether the operation using
         *  the sheet name has been finished successfully. In case of an error,
         *  the dialog will be kept open.
         *
         * @param {Object} [options]
         *  A map with additional options for the dialog. The following options
         *  are supported:
         *  @param {String} [options.okLabel]
         *      An alternative label text for the OK button.
         *  @param {String} [options.initialName]
         *      The initial sheet name to be shown when the dialog is opened.
         *      If omitted, the name of the active sheet will be used.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        function showSheetNameDialog(title, callback, options) {

            function showDialog(initialName) {

                var // the Promise object representing the dialog
                    promise = null;

                // close the dialog if document switches to read-only mode
                function changeEditModeHandler(event, editMode) {
                    if (!editMode) { promise.close(); }
                }

                // create the dialog box
                promise = Dialogs.showTextDialog({
                    title: title,
                    okLabel: Utils.getStringOption(options, 'okLabel'),
                    value: initialName,
                    placeholder: gt('Enter sheet name')
                });

                // listen to read-only mode, close dialog automatically
                app.getModel().on('change:editmode', changeEditModeHandler);
                promise.always(function () {
                    app.getModel().off('change:editmode', changeEditModeHandler);
                });

                // show dialog as long as the callback function returns false
                return promise.then(function (sheetName) {
                    return callback.call(self, sheetName) ? sheetName : showDialog(sheetName);
                });
            }

            // show the dialog, return the piped Deferred
            return showDialog(Utils.getStringOption(options, 'initialName', self.getSheetName()));
        }

        // methods ------------------------------------------------------------

        /**
         * Shows a dialog that allows to enter a name of a new sheet copied
         * from the active sheet. The dialog will be kept open until a valid
         * sheet name has been entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been copied
         *  successfully; or will be rejected, if the dialog has been canceled,
         *  or if the document has switched to read-only mode.
         */
        this.showCopySheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Copy Sheet'), this.copySheet, { okLabel: gt('Copy'), initialName: app.getModel().generateUnusedSheetName() });
        };

        /**
         * Shows a dialog that allows to enter a new name for the active sheet.
         * The dialog will be kept open until a valid sheet name has been
         * entered.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object representing the dialog. Will be
         *  resolved with the new sheet name, after the sheet has been renamed
         *  successfully; or will rejected, if the dialog has been canceled, or
         *  if the document has switched to read-only mode.
         */
        this.showRenameSheetDialog = function () {
            // the callback function will be called in the context of this view instance
            return showSheetNameDialog(gt('Rename Sheet'), this.setSheetName, { okLabel: gt('Rename') });
        };

        /**
         * Shows the 'Insert Image' dialog and inserts the resulting image into
         * the active sheet of the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the dialog has been closed and the image has been inserted
         *  into the document.
         */
        this.showInsertImageDialog = function () {

            // show the generic 'Insert Image' dialog
            return ImageUtil.insertImageDialog(app).then(function (imageHolder) {

                var // create the image DOM node, wait for it to load
                    createImagePromise = app.createImageNode(ImageUtil.getFileUrl(app, imageHolder.url));

                // on success, create the drawing object in the document
                createImagePromise.done(function (imgNode) {

                    var // current width of the image, in 1/100 mm
                        width = Utils.convertLengthToHmm(imgNode[0].width, 'px'),
                        // current height of the image, in 1/100 mm
                        height = Utils.convertLengthToHmm(imgNode[0].height, 'px'),
                        // active cell as target position for the image
                        activeCell = self.getActiveCell();

                    // restrict size to fixed limit
                    if (width > height) {
                        if (width > MAX_IMAGE_SIZE) {
                            height = Math.round(height * MAX_IMAGE_SIZE / width);
                            width = MAX_IMAGE_SIZE;
                        }
                    } else {
                        if (height > MAX_IMAGE_SIZE) {
                            width = Math.round(width * MAX_IMAGE_SIZE / height);
                            height = MAX_IMAGE_SIZE;
                        }
                    }

                    self.insertDrawing('image', { drawing: {
                        anchorType: 'oneCell',
                        startCol: activeCell[0],
                        startRow: activeCell[1],
                        width: width,
                        height: height,
                        name: imageHolder.name,
                        imageUrl: imageHolder.url
                    }});

                    app.destroyImageNodes(imgNode);
                });

                // on failure, show a warning message
                createImagePromise.fail(function () {
                    app.rejectEditAttempt('image');
                });

                return createImagePromise;
            });
        };

        /**
         * Shows the 'Reorder sheets' dialog and calls the 'Move Sheet' Operations
         * after the dialog is closed successfully
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after the
         *  dialog has been closed
         */
        this.showReorderSheetsDialog = function () {
            var model = app.getModel();
            var view = app.getView();


            var SHEETINDEX = 'sheet-index';
            var names = model.getSheetNames();
            var currentSheet = view.getActiveSheet();
            var currentChosen = null;

            var collectOps = [];

            var dHolder = $('<div>');
            dHolder.addClass('dialog-ordersheets-holder');

            var list = $('<div>');
            list.attr('role', 'tablist');
            list.addClass('dialog-ordersheets-list');

            function checkScrollOutside() {
                var pHeight = dHolder.height();
                var cHeight = currentChosen.height();
                var pos = currentChosen.position();
                if (pos.top < 0) {
                    Utils.scrollToChildNode(dHolder, currentChosen);
                } else if (pos.top + cHeight > pHeight) {
                    Utils.scrollToChildNode(dHolder, currentChosen);
                }
            }

            function checkButtonsActive() {
                up.removeClass('disabled');
                up.css({ pointerEvents: 'auto'});
                up.attr('tabindex', 0);
                down.removeClass('disabled');
                down.css({ pointerEvents: 'auto'});
                down.attr('tabindex', 0);

                if (!currentChosen.prev('*').length) {
                    up.addClass('disabled');
                    up.css({ pointerEvents: 'none'});
                    up.attr('tabindex', null);
                } else if (!currentChosen.next('*').length) {
                    down.addClass('disabled');
                    down.css({ pointerEvents: 'none'});
                    down.attr('tabindex', null);
                }
            }

            function selectButton(evt) {
                currentChosen.removeClass('btn-primary');
                currentChosen = $(evt.currentTarget);
                currentChosen.addClass('btn-primary');
                checkButtonsActive();
            }

            _.each(names, function (name, index) {
                var sheetModel = model.getSheetModel(index);
                if (!sheetModel.getMergedAttributes().sheet.visible) {
                    return;
                }
                var button = $('<button>');
                button.text(name);
                button.attr('role', 'button');
                button.attr('tabindex', 0);
                button.attr('title', name);
                button.attr('data-toggle', 'tab');
                button.attr(SHEETINDEX, index);

                button.addClass('btn');
                button.addClass('dialog-list-item');
                button.on('click', selectButton);
                if (index === currentSheet) {
                    button.addClass('btn-primary');
                    currentChosen = button;
                }
                list.append(button);

            });


            var control = $('<div>');
            control.addClass('dialog-ordersheets-control');

            var up = $.button({
                click: function () {
                    var index = currentChosen.attr(SHEETINDEX) | 0;
                    var other = currentChosen.prev('*');
                    if (other.length > 0) {
                        var otherIndex = other.attr(SHEETINDEX) | 0;
                        currentChosen.insertBefore(other);
                        collectOps.push({
                            from: index,
                            to: otherIndex
                        });
                        currentChosen.attr(SHEETINDEX, otherIndex);
                        other.attr(SHEETINDEX, index);
                        checkButtonsActive();
                    }
                    checkScrollOutside();
                }
            });
            up.addClass('btn');
            up.append(Utils.createIcon('icon-arrow-up'));
            up.css({ marginTop: '-3px', marginBottom: '5px', display: 'block'});
            up.attr('tabindex', 2);

            var down = $.button({
                data: { dir: 1 },
                click: function () {
                    var index = currentChosen.attr(SHEETINDEX) | 0;
                    var other = currentChosen.next('*');
                    if (other.length > 0) {
                        var otherIndex = other.attr(SHEETINDEX) | 0;
                        currentChosen.insertAfter(other);
                        collectOps.push({
                            from: index,
                            to: otherIndex
                        });
                        currentChosen.attr(SHEETINDEX, otherIndex);
                        other.attr(SHEETINDEX, index);
                        checkButtonsActive();
                    }
                    checkScrollOutside();
                }
            });
            down.addClass('btn');
            down.append(Utils.createIcon('icon-arrow-down'));
            down.css({ display: 'block'});
            down.attr('tabindex', 2);

            control.append(up);
            control.append(down);

            var dialog = Dialogs.createDialog({ width: 300, 'data-default-enter': true, title: gt('Reorder sheets') });
            dialog.addPrimaryButton('ok', gt('OK'), null, { tabIndex: 1 });
            dialog.addButton('cancel', gt('Cancel'), null, { tabIndex: 1 });

            var cnt = dialog.getContentNode();
            cnt.css({
                paddingRight: '2px',
                paddingTop: 0,
                paddingBottom: 0
            });

            dHolder.append(list);

            dialog.append(dHolder);
            dialog.append(control);

            checkButtonsActive();

            function changeEditModeHandler(event, editMode) {
                if (!editMode) { dialog.close(); }
            }



            var promise = dialog.show();

            app.getModel().on('change:editmode', changeEditModeHandler);
            promise.always(function () {
                app.getModel().off('change:editmode', changeEditModeHandler);
            });

            checkScrollOutside();

            return promise.done(function (reason) {
                if (reason === 'ok') {
                    model.moveSheets(collectOps);
                }
            });

        };

    } // class SpreadsheetDialogsMixin

    // exports ================================================================

    return SpreadsheetDialogsMixin;

});
