/**
 * All content on this website (including text, images, source code and any
 * other original works), unless otherwise noted, is licensed under a Creative
 * Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2013 Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/hyperlink',
    ['io.ox/office/editframework/view/hyperlinkutil'], function (HyperlinkUtil) {

    'use strict';

    /**
     * Hyperlink is inited by Gridpane,
     * here all listeners a set to the current gridpane an view
     *
     * @param {Baseapplication} app
     *
     * @param {Jquery.div} cellLayerNode
     * is the lowest div, before the cells begin,
     * so observer can always find the correct div for current Cell
     *
     */
    function Hyperlink(app, gridPane) {

        var hyperlinkPopup = HyperlinkUtil.createPopupNode(app),
            cellCollection = gridPane.getCellCollection();

        hyperlinkPopup.css('margin-top', '3px');
        hyperlinkPopup.css('min-width', '130px');

        HyperlinkUtil.addEditModeListener(app.getModel(), hyperlinkPopup, $.noop);

        var view = app.getView();

        view.on('change:selection change:layoutdata',  function () {

            var selection = view.getSelection();

            if (selection.drawings && selection.drawings.length) {
                hyperlinkPopup.hide();
                return;
            }

            var address = selection.activeCell;
            var cellData = cellCollection.getCellEntry(address);

            if (!cellData) {
                hyperlinkPopup.hide();
                return;
            }

            var url = cellData.attributes.character.url;
            if (url && url.length > 0 && HyperlinkUtil.isLegalUrl(url)) {
                HyperlinkUtil.updatePopup(url, hyperlinkPopup);

                var node = gridPane.getCellNode(address);

                var height = node.css('height');
                hyperlinkPopup.css('top', height);

                node.append(hyperlinkPopup);

                hyperlinkPopup.show();
            } else {
                hyperlinkPopup.hide();
            }

        });

        view.on('celledit:enter', function () {
            hyperlinkPopup.hide();
        });


        /** this function is called by gridpane before 'celledit:leave'
         * so all hyperlink-features are in the same operation like the other changes
         *
         * @param {String} value is the display-text
         *
         * @param {Object|Undefined} attributes are the old attributes, will be filled with hyperlink info
         *
         * @return {Object} attributes
         */
        this.checkForHyperlink = function (value, attributes) {
            if (!attributes || !attributes.character || !attributes.character.url) {
                var url = HyperlinkUtil.checkForHyperlink(value);
                if (url && url.length) {
                    return fillAttributes(attributes, url);
                }
            }
            return attributes;
        };

    } // class Hyperlink


    function fillAttributes(attributes, url) {
        if (!attributes) {
            attributes = {};
        }
        if (!attributes.character) {
            attributes.character = {};
        }
        attributes.character.url = url;
        attributes.character.underline = true;
        attributes.character.color = {
            type: 'scheme',
            value: 'hyperlink'
        };
        return attributes;
    }

    /**
     * set the Hyperlink direct in the cell
     *
     * @param {Baseapplication} app
     *
     * @param {String} url
     *
     */
    function setHyperlink(app, text, url) {
        var view = app.getView();

        var attributes = {};
        if (url && url.length) {

            var oldURL = null;
            var oldContent = view.getCellContents();
            if (oldContent && oldContent.attributes && oldContent.attributes.character) {
                oldURL = oldContent.attributes.character.url;
            }

            attributes.character = { url: url };

            if (!oldURL) {
                attributes.character.underline = true;
                attributes.character.color = {
                    type: 'scheme',
                    value: 'hyperlink'
                };
            }
        } else {
            attributes.character = { url: null, underline : null, color: null };
        }
        view.fillCellRanges(text, attributes);

    }

    /**
     * Opens a Hyperlink Dialog Frame with the data from current Selection
     * @param {Baseapplication} app
     *
     * @returns {Jquery.Deferred}
     * this object can be used to observe the Dialog Frame, when url is changed
     */
    Hyperlink.editHyperlink = function (app) {
        var view = app.getView();
        var item = view.getCellContents();
        var display = item.display;
        var url = item.attributes.character.url;

        view.grabFocus();

        if (!url || !url.length) {
            url = HyperlinkUtil.checkForHyperlink(display);
        }

        var def = HyperlinkUtil.showDialog(display, url, app);

        return def.done(function (data) {
            var text;
            if (data.text && data.text.length && data.text !== display) {
                text = data.text;
            }
            setHyperlink(app, text, data.url);

        });

    };

    /**
     * removes the url and its style from the selected cell.
     * is called by the hyperlink-info-popup
     * @param {Baseapplication} app
     */
    Hyperlink.removeHyperlink = function (app) {
        setHyperlink(app, undefined, null);
    };

    return Hyperlink;

});
