/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/tooltip',
     'io.ox/office/tk/popup/listmenu',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/drawinglayer/model/drawingutils',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/model/tokenarray',
     'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
     'io.ox/office/spreadsheet/view/cellcollection',
     'io.ox/office/spreadsheet/view/hyperlink',
     'io.ox/office/spreadsheet/utils/clipboard',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, KeyCodes, ToolTip, ListMenu, TriggerObject, BaseView, Color, DrawingUtils, DrawingFrame, SheetUtils, PaneUtils, TokenArray, GridTrackingMixin, CellCollection, Hyperlink, Clipboard, gt) {

    'use strict';

    var // number of colors for range highlighting
        RANGE_HIGHLIGHT_COLORS = 7,

        // maximum number of cells with any contents to be rendered at once in the background loop
        MAX_RENDER_CELL_COUNT = Modernizr.touch ? 75 : _.browser.IE ? 100 : 150;

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The logical address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The logical address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The logical address of the last cell.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a
     *      selection range has been modified by tracking the resizer handles.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Object} range
     *          The new logical address of the specified range.
     *      (4) {Number[]} [activeCell]
     *          The logical address of the active cell. If omitted, the current
     *          active cell will not be changed, unless the new active range
     *          does not contain it anymore.
     * - 'celledit:enter'
     *      After entering cell in-place edit mode.
     * - 'celledit:change'
     *      After the value has changed in cell in-place edit mode.
     * - 'celledit:leave'
     *      After leaving cell in-place edit mode.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     *
     * @param {CellCollection} cellCollection
     *  The cell collection containing layout information about the cells shown
     *  in this grid pane.
     */
    function GridPane(app, panePos, cellCollection) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            model = null,
            view = null,

            // the style sheet container of the document
            documentStyles = null,
            fontCollection = null,
            numberFormatter = null,

            // the model and collections of the active sheet
            sheetModel = null,
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            validationCollection = null,
            drawingCollection = null,

            // the column and row header pane associated to the grid pane
            colHeaderPane = null,
            rowHeaderPane = null,

            // the container node of this grid pane (the 'tabindex' attribute makes the node focusable)
            rootNode = $('<div>').addClass('abs grid-pane f6-target'),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            scrollNode = $('<div>').addClass('grid-scroll').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div>').addClass('grid-scroll-size').appendTo(scrollNode),

            // the root node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div>').addClass('grid-layer-root').appendTo(scrollNode),

            // the drawing layer (container for the drawing frames)
            drawingLayerNode = $('<div>').addClass('abs grid-drawing-layer noI18n').appendTo(layerRootNode),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = $('<div>').addClass('abs grid-selection-layer').appendTo(layerRootNode),

            // the range highlight layer (container for the highlighted ranges of formulas)
            highlightLayerNode = $('<div>').addClass('abs grid-highlight-layer').appendTo(layerRootNode),

            // the active range layer (container for the active range selected in cell edit mode)
            activeRangeLayerNode = $('<div>').addClass('abs grid-active-range-layer').appendTo(layerRootNode),

            // the cell layer (container for the cell nodes), insert above other layers for better DOM debugging
            cellLayerNode = $('<div>').addClass('abs grid-cell-layer noI18n').appendTo(layerRootNode),

            // drop-down menu for auto-completion and list validation of active cell
            cellListMenu = null,

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = null,
            clipboardFocusMethod = null,

            // the column and row indexes of the cells shown in this grid pane
            colIndexes = null,
            rowIndexes = null,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1, scrollTopRatio = 1,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0, hiddenHeight = 0,

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // all cells to be rendered in a background loop
            pendingCells = {},

            // the text area control for in-place cell editing (will be inserted into the root node)
            textArea = null,
            textAreaUnderlay = null,

            // pop-up menu and tooltip attached to the text area
            textAreaToolTip = null,
            textAreaListMenu = null,
            textAreaListItemToolTip = null,

            // all settings for the cell in-place edit mode
            cellEditSettings = {
                // whether the in-place cell edit mode is active
                active: false,
                // the index of the sheet containing the edited cell (active sheet may change during edit mode)
                sheet: null,
                // the cell address of the edited cell
                address: null,
                // initial text at the beginning of the edit mode
                originalText: null,
                // the descriptor of the edited cell (result, attributes, etc.)
                cellData: null,
                // quick edit mode (true), or text cursor mode (false)
                quick: null,
                // whether text area has changed since editing has started
                changed: false,
                // the value at the time the text area has been updated the last time
                lastValue: null,
                // last changed value in text area used to restore (redo)
                restoreValue: null,
                // whether the current value of the text area is a formula
                isFormula: false,
                // the token array representing the current formula in the text area
                tokenArray: null,
                // whether to show auto-completion list for a function or name
                autoComplete: false,
                // token info for function auto-completion
                autoTokenInfo: null,
                // how many closing parenthesis entered manually will be ignored
                autoCloseData: [],
                // the position of the current range address while in range selection mode
                rangeSelection: null,
                // workaround for missing 'input' events for the text area in IE9
                updateTimer: null
            },

            hyperlink = null;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        GridTrackingMixin.call(this, app, panePos, layerRootNode);

        // private methods ----------------------------------------------------

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = sheetModel.getViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = sheetModel.getViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (view.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (PaneUtils.getColPaneSide(panePos) === focusPaneSide) || (PaneUtils.getRowPaneSide(panePos) === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Utils.FOCUSED_CLASS, focused);
        }

        /**
         * Updates the CSS formatting of the text area control used for cell
         * in-place edit mode according to the current formatting of the active
         * cell.
         */
        function updateTextAreaStyle() {

            var // the address of the active cell
                address = cellEditSettings.address,
                // the cell attributes of the cell
                cellAttributes = null,
                // the character attributes of the cell
                charAttributes = null,
                // text area node and the underlay node, as one jQuery collection
                targetNodes = textArea.add(textAreaUnderlay);

            // applies a transformation to the passed color, and returns the CSS color value
            function getModifiedCssColor(color, type, value) {
                if (Color.isAutoColor(color)) { return ''; }
                color = _.copy(color, true);
                (color.transformations || (color.transformations = [])).push({ type: type, value: value });
                return documentStyles.getCssColor(color);
            }

            // Try to receive cell contents from own collection, fall back to cached data
            // of active cell in the spreadsheet view (this data will be updated deferred,
            // thus it is not yet available when double-clicking an unselected cell).
            cellEditSettings.cellData = SheetUtils.rangeContainsCell(cellCollection.getRange(), address) ? cellCollection.getCellEntry(address, { always: true }) : view.getCellContents();
            cellEditSettings.cellData = _.copy(cellEditSettings.cellData, true);
            cellAttributes = cellEditSettings.cellData.attributes.cell;
            charAttributes = cellEditSettings.cellData.attributes.character;
            charAttributes.fontSize = view.getEffectiveFontSize(charAttributes.fontSize);

            // set CSS attributes according to formatting of active cell
            targetNodes.css({
                fontFamily: documentStyles.getCssFontFamily(charAttributes.fontName),
                fontSize: Math.max(6, charAttributes.fontSize) + 'pt',
                fontWeight: charAttributes.bold ? 'bold' : 'normal',
                fontStyle: charAttributes.italic ? 'italic' : 'normal',
                textDecoration: CellCollection.getCssTextDecoration(cellEditSettings.cellData),
                textAlign: CellCollection.getCssTextAlignment(cellEditSettings.cellData, true)
            });

            // set cell text color and fill color to the appropriate nodes
            textArea.css('color', getModifiedCssColor(charAttributes.color, 'shade', 30000));
            textAreaUnderlay.css('background-color', getModifiedCssColor(cellAttributes.fillColor, 'tint', 30000));
        }

        /**
         * Updates the position and size of the text area control for the
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.keepSize=false]
         *      If set to true, the size of the text area will not be changed,
         *      only its position. Used when scrolling the grid pane, when the
         *      contents of the text area do not change.
         */
        function updateTextAreaPosition(options) {

            // do nothing if in-place edit mode is not active
            if (!cellEditSettings.active) { return; }

            var // the position of the edited cell
                position = null,
                // the visible area of this grid pane
                visiblePosition = null,
                // whether the text in the cell wraps automatically
                isWrapped = CellCollection.isWrappedText(cellEditSettings.cellData),
                // the effective horizontal CSS text alignment
                textAlign = textArea.css('text-align'),
                // the effective font height, in pixels
                fontSize = Utils.convertCssLength(textArea.css('font-size'), 'px', 1),
                // text area node and the underlay node, as one jQuery collection
                targetNodes = textArea.add(textAreaUnderlay),
                // the new width and height of the text area
                targetWidth = 0, targetHeight = 0;

            // returns required width needed to show all text lines in a single line
            function getMinimumWidth() {

                var // the current character attributes
                    charAttributes = cellEditSettings.cellData.attributes.character,
                    // required width for all text lines
                    requiredWidth = _(textArea.val().split(/\n/)).reduce(function (memo, textLine) {
                        return Math.max(memo, fontCollection.getTextWidth(textLine, charAttributes));
                    }, 0);

                // round to blocks of 20 pixels, enlarge to cell width
                return Math.max(position.width, Utils.roundUp(requiredWidth + 10, 20));
            }

            // returns maximum width according to horizontal alignment and width of visible area
            function getMaximumWidth() {

                var // the distance from left visible border to right cell border (for right alignment)
                    leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                    // the distance from left cell border to right visible border (for left alignment)
                    rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                    // effective available width
                    availWidth = 0;

                // get available width according to text alignment
                switch (textAlign) {
                case 'right':
                    availWidth = leftWidth;
                    break;
                case 'center':
                    availWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                    break;
                default:
                    availWidth = rightWidth;
                }

                // enlarge to at least half of the width of the visible area
                return Math.max(availWidth, Math.round(visiblePosition.width / 2));
            }

            // get horizontal and vertical cell position and size
            position = sheetModel.getCellRectangle(cellEditSettings.address, { expandMerged: true });
            // use font height as minimum width, enlarge width by one pixel to fit exactly into the selection border
            position.width = Math.max(20, fontSize, position.width + 1);
            // enlarge height by one pixel to fit exactly into the selection border
            position.height += 1;

            // get position and size of the visible area in the grid pane
            visiblePosition = self.getVisibleRectangle();

            // calculate and set the new size of the text area
            if (Utils.getBooleanOption(options, 'keepSize', false)) {
                targetWidth = textArea.outerWidth();
                targetHeight = textArea.outerHeight();
            } else {
                (function () {

                    var // find best size, stick to cell width in auto-wrapping mode
                        minWidth = isWrapped ? position.width : getMinimumWidth(),
                        maxWidth = isWrapped ? visiblePosition.width : getMaximumWidth();

                    // set resulting width to the text area (hide scroll bars temporarily to get the best height)
                    targetWidth = Math.min(minWidth, maxWidth);
                    textArea.css({ overflowY: 'hidden', width: targetWidth, height: 1 });

                    // width is optimal, get resulting height, restricted to visible area
                    targetHeight = Math.min(Math.max(textArea[0].scrollHeight, position.height), visiblePosition.height);
                    targetNodes.css({ width: targetWidth, height: targetHeight, overflowY: '' });
                }());
            }

            // calculate and set the new position of the text area
            (function () {

                var // left and top position of the text area
                    leftOffset = position.left,
                    topOffset = position.top;

                // calculate horizontal position according to alignment
                switch (textAlign) {
                case 'right':
                    leftOffset += position.width - targetWidth;
                    break;
                case 'center':
                    leftOffset += Math.round((position.width - targetWidth) / 2);
                    break;
                }

                // move text area into visible area, set effective position
                leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - targetWidth);
                topOffset = Utils.minMax(topOffset, visiblePosition.top, visiblePosition.top + visiblePosition.height - targetHeight);
                targetNodes.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });
            }());
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionSignatureToolTip() {

            var // the cursor position in the text area (off by one to exclude the equality sign)
                textPos = textArea[0].selectionStart - 1,
                // function data for the cursor position
                functionInfo = cellEditSettings.tokenArray.getFunctionAtPosition(textPos),
                // the help descriptor of the function, containing the description text
                functionHelp = functionInfo ? app.getFunctionHelp(functionInfo.name) : null,
                // current list separator
                SEP = app.getListSeparator();

            // do nothing if cursor is not inside a function, or no help is available for the function
            if (!functionHelp) {
                textAreaToolTip.hide();
                return;
            }

            // build function name link to OX AppSuite static help pages
            // TODO: add URL of real help files when we have it
            var functionHelpLink = $('<a>', { href: '/appsuite/help/en_US/index.html', target: '_blank' })
                .toggleClass('param-active', functionInfo.paramIndex < 0)
                .text(functionInfo.name);

            var varargIndex = functionHelp.varargs,
                optargIndex = functionHelp.optargs,
                isVararg = _.isNumber(varargIndex),
                isOptarg = _.isNumber(optargIndex),
                paramsArray = _.clone(functionHelp.params),
                isPair = isVararg && _.isString(paramsArray[varargIndex + 1]),
                varargArray = [],
                paramsNode = $('<span>'),
                bracketStartIndex = 0;

            // prepare parameters array plus generate custom parameters if we have an infinite parameter
            if (isVararg) {
                // handle special case with infinite parameter pairs like parameters from the function 'SUMIFS'
                var varargName = paramsArray[varargIndex].trim(),
                    secondVarargName = isPair ? paramsArray[varargIndex + 1].trim() : '',
                    varargParamCount = 1,
                    sliceCount = isPair ? 2 : 1;
                // calculate number of variable parameters to generate
                if (functionInfo.paramCount > paramsArray.length) {
                    varargParamCount = isPair ? (functionInfo.paramCount - varargIndex) / 2 : functionInfo.paramCount - varargIndex;
                }
                // generate variable parameters
                var i = 0;
                while (i <= varargParamCount) {
                    varargArray.push(varargName + (i + 1));
                    if (isPair) varargArray.push(secondVarargName + (i + 1));
                    i++;
                }
                // join initial parameter array with generated variable parameters array
                paramsArray = paramsArray.slice(0, paramsArray.length - sliceCount).concat(varargArray);
                bracketStartIndex = isPair ? varargIndex + 1 : varargIndex;
            }

            if (isOptarg) { bracketStartIndex = optargIndex - 1; }

            // build mark-up and highlight active parameter (draw brackets, add highlight class, add delimiters)
            _(paramsArray).each(function (elem, index, list) {
                var elemMarkup = elem,
                    paramNode = $('<span>');
                // add brackets where necessary
                if (index > bracketStartIndex) {
                    if (isPair) {
                        elemMarkup = (index % 2) ? ('[' + elem) : (elem + ']');
                    } else {
                        elemMarkup = (isVararg || isOptarg) ? ('[' + elem + ']') : elem;
                    }
                }
                paramNode.html(elemMarkup);
                if (index === functionInfo.paramIndex) paramNode.addClass('param-active');
                paramsNode.append(paramNode);
                if (index !== (list.length - 1)) paramsNode.append(SEP + ' ');
            });

            // add parentheses around the parameters and ellipsis in the end if we have infinite parameters
            paramsNode.prepend('( ');
            if (isVararg) paramsNode.append(SEP + ' \u2026');
            paramsNode.append(' )');
            textAreaToolTip.clearContents().appendContentNodes(functionHelpLink, paramsNode);

            // check if we have parameter help data, get it and build its mark-up
            if (_.isArray(functionHelp.paramshelp)) {
                // show right help in case of variable parameters
                var paramHelpIndex = 0;
                if (isVararg && (functionInfo.paramIndex > varargIndex)) {
                    paramHelpIndex = varargIndex + (isPair ? ((functionInfo.paramIndex - varargIndex) % 2) : 0);
                } else {
                    paramHelpIndex = functionInfo.paramIndex;
                }
                var activeParamHelp = functionHelp.paramshelp[paramHelpIndex] || '';
                if (activeParamHelp.length > 0) {
                    textAreaToolTip.appendContentNodes($('<div>').addClass('param-help').text(activeParamHelp));
                }
            }

            // and finally, show the tooltip
            textAreaToolTip.show();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            var // the selected button element in the auto-completion menu
                buttonNode = textAreaListMenu.getSelectedItemNodes().first(),
                // the name and type of the selected list item
                itemValue = Utils.getControlValue(buttonNode),
                // the help descriptor for the selected function (nothing for defined names)
                functionHelp = (_.isObject(itemValue) && (itemValue.type === 'func')) ? app.getFunctionHelp(itemValue.name) : null;

            // only show tooltips for supported functions (not for defined names)
            if (functionHelp) {
                textAreaListItemToolTip.setAnchorNode(buttonNode).setText(functionHelp.description).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            textAreaToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         */
        function updateTextAreaPopups() {

            var // the cursor position in the text area (off by one to exclude the equality sign)
                textPos = textArea[0].selectionStart - 1,
                // token info for function auto-completion
                tokenInfo = null,
                // the text to be used for function auto-completion
                autoText = null;

            // show pop-ups only in formula mode, do not show them while dragging
            // highlight ranges, or selecting new ranges for the formula
            if (!cellEditSettings.isFormula ||
                _.isNumber(sheetModel.getViewAttribute('highlightIndex')) ||
                _.isObject(sheetModel.getViewAttribute('activeSelection'))
            ) {
                hideTextAreaPopups();
                return;
            }

            // prefer function auto-completion over function tooltip
            cellEditSettings.autoTokenInfo = null;
            if (cellEditSettings.autoComplete && (tokenInfo = cellEditSettings.tokenArray.getTokenAtPosition(textPos)) && (tokenInfo.end === textPos)) {

                // get the text entered before the text cursor (may be a name token or a function token)
                switch (tokenInfo.token.getType()) {
                case 'name':
                    if (!tokenInfo.token.getSheetRef()) { autoText = tokenInfo.token.getName(); }
                    break;
                case 'func':
                    autoText = tokenInfo.token.getName();
                    break;
                }

                // store token info (needed to replace the entered text with the full function name)
                cellEditSettings.autoTokenInfo = tokenInfo;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // prepare the menu sections
                textAreaListMenu.clearContents();
                textAreaListMenu.createSectionNode('funcs');
                textAreaListMenu.createSectionNode('names');

                // insert all matching function names
                autoText = autoText.toUpperCase();
                _(app.getFunctionNames()).each(function (funcName) {
                    if (funcName.substr(0, autoText.length) === autoText) {
                        textAreaListMenu.createItemNode('funcs', { name: funcName, type: 'func' }, { label: funcName });
                    }
                });

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    textAreaToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show the signature (name and parameters) of the current function
            textAreaListMenu.hide();
            updateFunctionSignatureToolTip();
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            cellEditSettings.autoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            cellEditSettings.autoCloseData = [];

            // defer execution to let the browser process pending key events
            _.defer(function () {
                // fail-safe check that cell edit mode is still active
                if (cellEditSettings.active) { updateTextAreaPopups(); }
            });
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell in-place edit mode is active.
         */
        function updateTextAreaHighlighting() {

            var // info objects for all ranges in the formula
                rangeInfos = cellEditSettings.tokenArray.extractRanges({
                    filterSheet: view.getActiveSheet(),
                    refAddress: cellEditSettings.address,
                    targetAddress: cellEditSettings.address,
                    resolveNames: true
                }),
                // unified token indexes of all ranges
                rangeIndexes = _.chain(rangeInfos).pluck('index').unique().value(),
                // the mark-up of the underlay node (span elements containing token texts)
                markup = '';

            // generate the new mark-up for the underlay node
            if (rangeIndexes.length > 0) {
                markup += '<span>=</span>';
                cellEditSettings.tokenArray.iterateTokens(function (token, index) {
                    var styleIndex = _(rangeIndexes).indexOf(index, true);
                    markup += '<span data-token-type="' + token.getType() + '"';
                    if (styleIndex >= 0) { markup += ' data-style="' + ((styleIndex % RANGE_HIGHLIGHT_COLORS) + 1) + '"'; }
                    markup += '>' + Utils.escapeHTML(token.getText()) + '</span>';
                });
            }
            textAreaUnderlay[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell in-place edit mode. The token
         * array will trigger a 'change:tokens' event that will cause to redraw
         * the highlighted ranges in all grid panes.
         */
        var updateTextAreaTokenArray = app.createDebouncedMethod(function () {

            // update the formula flag immediately
            cellEditSettings.isFormula = /^=/.test(textArea.val());

        }, function () {

            var // the current input string in the text area
                value = textArea.val();

            // the cell edit mode may have been finished already
            if (!cellEditSettings.active) { return; }

            // parse the formula string
            if (cellEditSettings.isFormula) {
                cellEditSettings.tokenArray.parseFormula(value.slice(1));
            } else {
                cellEditSettings.tokenArray.clear();
            }

            // insert highlighting mark-up for the underlay node
            updateTextAreaHighlighting();

            // update pop-up nodes and other settings dependent on the text selection
            updateTextAreaPopups();

        }, { delay: 100, maxDelay: 300 });

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Number} [start]
         *  The new start position of the selection. If omitted, the text
         *  cursor will be moved to the end of the new text.
         *
         * @param {Number} [end=start]
         *  The new end position of the selection. If omitted, a simple text
         *  cursor will be shown at the specified start position.
         */
        function setTextAreaContents(text, start, end) {
            textArea.val(text);
            cellEditSettings.lastValue = text;
            Utils.setTextFieldSelection(textArea, _.isNumber(start) ? start : text.length, end);
            updateTextAreaPosition();
            updateTextAreaTokenArray();
        }

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            var // the auto-fill resizer handle node
                autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]'),
                // the range node containing the auto-fill handle
                autoFillRangeNode = autoFillHandleNode.closest('.range');

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
            case 'r':
                autoFillHandleNode.css({ top: -(layerRootNode.position().top + autoFillRangeNode.position().top) });
                break;
            case 'b':
                autoFillHandleNode.css({ left: -(layerRootNode.position().left + autoFillRangeNode.position().left) });
                break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                case 'l':
                case 'r':
                    handleNode.css({ top: -(layerRootNode.position().top + rangeNode.position().top) + (scrollNode[0].clientHeight / 2) });
                    break;
                case 't':
                case 'b':
                    handleNode.css({ left: -(layerRootNode.position().left + rangeNode.position().left) + (scrollNode[0].clientWidth / 2) });
                    break;
                }
            });
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            var // the available size according to existence of scroll bars
                visibleWidth = scrollNode[0].clientWidth,
                visibleHeight = scrollNode[0].clientHeight,
                // difference of scroll area size between header pane and this grid pane
                widthCorrection = rootNode.width() - visibleWidth,
                heightCorrection = rootNode.height() - visibleHeight,
                // the new absolute width and height of the scrollable area
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection),
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollWidth, scrollHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > visibleWidth) ? ((scrollWidth - visibleWidth) / (effectiveSize.width - visibleWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > visibleHeight) ? ((scrollHeight - visibleHeight) / (effectiveSize.height - visibleHeight)) : 0;
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            var // the new DOM scroll position
                scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0,
                scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0,
                // the effective DOM node offsets according to the scroll position
                offsetLeft = 0,
                offsetTop = 0;

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // calculate the effective offsets according to current scroll position
            offsetLeft = Math.max(-layerRectangle.width * 2, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            offsetTop = Math.max(-layerRectangle.height * 2, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollNode, layerRootNode, offsetLeft, offsetTop);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRectangle.width, height: layerRectangle.height });

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();

            // update position of the text area control for in-place edit mode
            updateTextAreaPosition({ keepSize: true });

            // focus back to text area control (lost e.g. while scrolling with scroll bars)
            if (cellEditSettings.active && (textArea[0] !== window.document.activeElement)) {
                textArea.focus();
            }
        }

        /**
         * Returns whether the passed validation settings as returned by the
         * method ValidationCollection.getValidationSettings() describe a cell
         * that will show a drop-down list menu.
         *
         * @param {Object} [settings]
         *  The validation settings received from the validation collection.
         *
         * @returns {Boolean}
         *  Whether the passed validation settings are valid and describe a
         *  cell with source/list validation with a drop-down list menu.
         */
        function isDropDownValidation(settings) {
            // TODO: filter sends reversed 'showDropDown' attribute value
            return _.isObject(settings) && /^(source|list)$/.test(settings.attributes.type) && !settings.attributes.showDropDown;
        }

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the visible area of this grid pane. The iterator
         * function will receive additional data needed for rendering those
         * ranges in some way.
         *
         * @param {Object|Array}
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Object} range
         *      The logical address of the cell range to be rendered.
         *  (2) {Number} index
         *      The array index of the current cell range.
         *  (3) {Object} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateRangesForRendering(ranges, iterator) {

            var // the bounding rectangle for range nodes (prevent oversized DOM nodes)
                boundRectangle = { width: Utils.MAX_NODE_SIZE, height: Utils.MAX_NODE_SIZE };

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRectangle.left = Math.max(0, layerRectangle.left - (boundRectangle.width - layerRectangle.width) / 2);
            boundRectangle.top = Math.max(0, layerRectangle.top - (boundRectangle.height - layerRectangle.height) / 2);

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            return Utils.iterateArray(_(ranges).getArray(), function (range, index) {

                var // position and size of the selection range, restricted to the bounding rectangle
                    rectangle = DrawingUtils.getIntersectionRectangle(sheetModel.getRangeRectangle(range), boundRectangle);

                // skip ranges completely outside the bounding rectangle
                if (!rectangle) { return; }

                // convert range position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;

                // invoke the iterator
                return iterator.call(self, range, index, rectangle);
            });
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            var // the entire cell selection
                selection = view.getSelection(),
                // the range occupied by the active cell (will not be filled)
                activeRange = null,
                // the position of the active cell, extended to the merged range
                activeRectangle = null,
                // additional data for active auto-fill tracking
                autoFillData = sheetModel.getViewAttribute('autoFillData'),
                // whether the active cell touches the borders of selection ranges
                activeBorders = { left: false, top: false, right: false, bottom: false },
                // hide left/top border of selection ranges, if first column/row is hidden
                firstColHidden = !colCollection.isEntryVisible(0),
                firstRowHidden = !rowCollection.isEntryVisible(0),
                // the HTML mark-up for all selection ranges
                markup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (model.isColRange(range) ? hPos : model.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // add active flag to range object
            selection.ranges[selection.activeRange].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                var // the selected cell range
                    firstRange = selection.ranges[0],
                    // whether to expand/shrink the leading or trailing border
                    leading = /^(left|top)$/.test(autoFillData.border),
                    // whether to expand/shrink columns or rows
                    columns = /^(left|right)$/.test(autoFillData.border),
                    // the array index in the cell address
                    addrIndex = columns ? 0 : 1;

                activeRange = _.copy(firstRange, true);
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start[addrIndex] -= autoFillData.count;
                    } else {
                        firstRange.end[addrIndex] += autoFillData.count;
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < (columns ? SheetUtils.getColCount(firstRange) : SheetUtils.getRowCount(firstRange))) {
                        // range partly covered
                        if (leading) {
                            activeRange.start[addrIndex] -= autoFillData.count;
                        } else {
                            activeRange.end[addrIndex] += autoFillData.count;
                        }
                    } else {
                        // remove active range for deletion of the entire range
                        activeRange = {};
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = _.isObject(activeRange) ? (_.isEmpty(activeRange) ? null : activeRange) :
                mergeCollection.expandRangeToMergedRanges({ start: selection.activeCell, end: selection.activeCell });

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                iterateRangesForRendering(activeRange, function (range, index, rectangle) {
                    activeRectangle = rectangle;
                });
            }

            // render all ranges that are visible in this grid pane
            iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {

                var // position of active range, relative to current range
                    relActiveRect = null,
                    // the style attribute
                    styleAttr = '',
                    // mark-up for the fill elements and other helper nodes
                    fillMarkup = '';

                // add one pixel to the right and bottom border, to get matching
                // border line positions in multi-selection
                rectangle.width += 1;
                rectangle.height += 1;

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    rectangle.left -= 5;
                    rectangle.width += 5;
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    rectangle.top -= 5;
                    rectangle.height += 5;
                }

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && SheetUtils.rangesOverlap(range, activeRange)) {

                    // initialize position of active cell, relative to selection range
                    relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        fillMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        fillMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    fillMarkup += '<div class="abs fill"></div>';
                }

                // build the style attribute for the root node of the selection range
                styleAttr += PaneUtils.getRectangleStyleMarkup(rectangle);

                // generate the HTML mark-up for the selection range
                markup += '<div class="range';
                if (range.active) { markup += ' active'; }
                if (range.tracking) { markup += ' tracking-active'; }
                if (range.collapsed) { markup += ' collapsed'; }
                markup += '" style="' + styleAttr + '" data-index="' + index + '">' + fillMarkup + '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Modernizr.touch && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        markup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        markup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                markup += '</div>';
            });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                activeRectangle.width += 1;
                activeRectangle.height += 1;
                markup += '<div class="active-cell';
                _(activeBorders).each(function (isBorder, borderName) { if (!isBorder) { markup += ' ' + borderName; } });
                markup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '">';

                // add drop-down button for list validation
                if (!_.isObject(autoFillData) && model.getEditMode() && isDropDownValidation(validationCollection.getValidationSettings(selection.activeCell))) {
                    markup += '<div class="cell-dropdown-button skip-tracking"';
                    if (activeBorders.right) { markup += ' style="margin-left:2px;"'; }
                    markup += '>' + Utils.createIconMarkup('docs-caret down') + '</div>';
                }

                // close the active cell node
                markup += '</div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = markup;

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();

            // update the anchor node of the drop-down menu
            cellListMenu.setAnchorNode(selectionLayerNode.find('.active-cell'));
        }

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            var // all ranges currently highlighted
                highlightRanges = sheetModel.getViewAttribute('highlightRanges'),
                // unified range identifiers
                rangeIds = _.chain(highlightRanges).pluck('id').unique().value(),
                // array index of the range currently tracked
                highlightIndex = sheetModel.getViewAttribute('highlightIndex'),
                // the HTML mark-up for all highlight ranges
                markup = '';

            // render all ranges that are visible in this grid pane
            if (_.isArray(highlightRanges)) {
                iterateRangesForRendering(highlightRanges, function (range, index, rectangle) {

                    var // array index of the range identifier (multiple ranges may origin from the same defined name)
                        idIndex = rangeIds.indexOf(range.id),
                        // CSS class name for range color style
                        styleId = (idIndex % RANGE_HIGHLIGHT_COLORS) + 1;

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range ' + ((highlightIndex === index) ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + index + '"><div class="fill"></div>';
                    if (model.getEditMode() && Utils.getBooleanOption(range, 'draggable', false)) {
                        markup += '<div class="borders">';
                        markup += _(['l', 'r', 't', 'b']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _(['tl', 'tr', 'bl', 'br']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        var renderActiveSelection = (function () {

            var // the timer for the wireframe animation
                timer = null;

            return function () {

                var // the active ranges
                    selection = sheetModel.getViewAttribute('activeSelection'),
                    // the HTML mark-up for the range
                    markup = '';

                // generate the HTML mark-up for the ranges
                if (_.isObject(selection)) {
                    iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {
                        markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                        markup += _(['l', 'r', 't', 'b']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div></div>';
                    });
                    if (!timer) {
                        timer = app.repeatDelayed(function (index) {
                            activeRangeLayerNode.attr('data-frame', index % 6);
                        }, { delay: 100 });
                    }
                } else if (timer) {
                    timer.abort();
                    timer = null;
                }

                // insert entire HTML mark-up into the container node
                activeRangeLayerNode[0].innerHTML = markup;
            };
        }());

        /**
         * Updates the position of the passed drawing frame according to the
         * anchor attributes in the passed attribute set.
         */
        function updateDrawingFramePosition(drawingModel, drawingFrame) {

            var // the effective drawing rectangle (null if the drawing is hidden)
                rectangle = drawingModel.getRectangle();

            if (rectangle) {
                // convert position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;
                // show the drawing frame, and set its position and size
                drawingFrame.show().css(rectangle);
            } else {
                // hide the drawing frame if it is located in hidden columns or rows
                drawingFrame.hide();
            }
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         */
        function createDrawingFrame(drawingModel, position) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the logical position
                index = position[0];

            // add the unique identifier of the drawing model for DOM look-up
            drawingFrame.attr('data-model-uid', drawingModel.getUid());

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting
            updateDrawingFramePosition(drawingModel, drawingFrame);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Creates DOM drawing frames for all existing drawing objects in the
         * current active sheet.
         */
        function createDrawingFrames() {
            drawingCollection.iterateModelsByPosition(createDrawingFrame);
            renderDrawingSelection();
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         */
        function updateDrawingFrame(drawingModel) {

            var // the existing DOM drawing frame
                drawingFrame = self.getDrawingFrame(drawingModel.getUid());

            // update position and formatting of frames inside the visible area
            if (SheetUtils.rangesOverlap(layerRange, drawingModel.getRange())) {
                updateDrawingFramePosition(drawingModel, drawingFrame);
                DrawingFrame.updateFormatting(app, drawingFrame, drawingModel.getMergedAttributes());
            } else {
                drawingFrame.hide();
            }
        }

        /**
         * Updates position and formatting of all drawing frames in the current
         * visible area or the passed bounding range.
         */
        function updateDrawingFrames() {
            drawingCollection.iterateModelsByPosition(updateDrawingFrame);
        }

        /**
         * Updates the selection of all drawing frames.
         */
        function renderDrawingSelection() {

            var // all drawing frames currently selected in the DOM (also embedded drawings)
                oldSelectedFrames = drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Utils.SELECTED_SELECTOR),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // disable tracking in read-only mode, or if sheet is protected
                editMode = model.getEditMode() && !sheetModel.isProtected();

            // process all drawings to be selected
            _(view.getSelectedDrawings()).each(function (position) {

                var // the root drawing frame addressed by the current logical position
                    drawingFrame = drawingNodes[position[0]];

                // drawing frame may be missing, e.g. while remotely inserting new drawings
                // TODO: find embedded drawing frames
                if (drawingFrame && (position.length === 1)) {
                    DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                    oldSelectedFrames = oldSelectedFrames.not(drawingFrame);
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.each(function () { DrawingFrame.clearSelection(this); });
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position);
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {
            Utils.warn('delete: remove DOM node');
            self.getDrawingFrame(drawingModel.getUid()).remove();
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object that has been changed in any way.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        /**
         * Returns whether the passed sorted (!) array of index intervals
         * contain a specific index, by performing a fast binary search on the
         * interval array.
         *
         * @param {Array} intervals
         *  The sorted and unified array of (column or row) index intervals.
         *
         * @param {Number} index
         *  A single index to be checked.
         *
         * @returns {Boolean}
         *  Whether one of the intervals contains the passed index.
         */
        function intervalsContainIndex(intervals, index) {
            var arrayIndex = _(intervals).sortedIndex({ last: index }, 'last');
            return (arrayIndex < intervals.length) && (intervals[arrayIndex].first <= index);
        }

        /**
         * Invokes the passed iterator function for all visible adjacent empty
         * cells next to the specified cell. Iteration will be stopped when the
         * first non-empty cell will be found. Cells that are part of a merged
         * range are considered non-empty.
         */
        function iterateEmptyCells(address, direction, iterator) {

            var // iterate horizontally or vertically
                vertical = (direction === 'up') || (direction === 'down'),
                // the row/column intervals of all merged ranges covering the current column/row
                mergedIntervals = vertical ? mergeCollection.getMergedRowIntervals(address[0]) : mergeCollection.getMergedColIntervals(address[1]);

            cellCollection.iterateNextCells(address, direction, function (cellData) {
                var isEmpty = CellCollection.isBlank(cellData) && !intervalsContainIndex(mergedIntervals, cellData.address[vertical ? 1 : 0]);
                return isEmpty ? iterator.call(self, cellData) : Utils.BREAK;
            }, { skipStartCell: true });
        }

        /**
         * Returns the clipping and overflow settings for the specified cell
         * (the additional space at the left and right side of the specified
         * cell that can be used for visible overlapping text contents).
         *
         * @param {Object} cellData
         *  The cell descriptor, as returned e.g. by the cell collection. If
         *  the cell is part of a merged range, this object MUST contain the
         *  additional property 'mergedRange' with the logical address of the
         *  merged range.
         *
         * @returns {Object}
         *  The available space at the left and right side of the cell, in the
         *  following properties:
         *  - {Number} clipLeft
         *      The additional space left of the cell that can be used to show
         *      the text contents.
         *  - {Number} clipRight
         *      The additional space right of the cell that can be used to show
         *      the text contents.
         *  - {Number} hiddenLeft
         *      Additional width left of the visible clip region needed to have
         *      enough space for very long text without reaching the content
         *      node borders.
         *  - {Number} hiddenRight
         *      Additional width right of the visible clip region needed to
         *      have enough space for very long text without reaching the
         *      content node borders.
         */
        function calculateCellClipping(cellData) {

            var // the effective CSS text alignment
                textAlign = CellCollection.getCssTextAlignment(cellData),
                // merged ranges will always be cropped at cell boders
                crop = _.isObject(cellData.mergedRange),
                // the resulting clip settings
                clippingData = { clipLeft: 0, clipRight: 0, hiddenLeft: 0, hiddenRight: 0 };

            // no clipping left of cell needed, if cell is right-aligned or justified
            if ((textAlign === 'left') || (textAlign === 'center')) {
                if (!crop) { iterateEmptyCells(cellData.address, 'right', function (emptyData) { clippingData.clipRight += emptyData.width; }); }
                clippingData.hiddenRight = 500000 - clippingData.clipRight;
            }

            // no clipping right of cell needed, if cell is left-aligned or justified
            if ((textAlign === 'right') || (textAlign === 'center')) {
                if (!crop) { iterateEmptyCells(cellData.address, 'left', function (emptyData) { clippingData.clipLeft += emptyData.width; }); }
                clippingData.hiddenLeft = 500000 - clippingData.clipLeft;
            }

            return clippingData;
        }

        /**
         * Renders the specified number of cells from the 'pendingCells'
         * collection, and removes the information of the rendered cells from
         * that collection.
         *
         * @param {Number} contentCount
         *  The maximum number of cells with any text contents to be rendered
         *  during this invokation. The total number of rendered cells (empty
         *  or with text contents) will be restricted to the fourfold of this
         *  value.
         */
        function renderPendingCells(contentCount) {

            // Bug 31125: to prevent performance problems when rendering cells,
            // modifying the DOM (insert contents, change node size and position),
            // and accessing measument properties of DOM elements (size, position)
            // MUST NOT be performed alernatingly. After any modification of the
            // DOM the browser will update its internal rendering tree before it
            // returns any measurement properties. That update may take a few dozen
            // milliseconds (especially Chrome and Internet Explorer), thus doing
            // that for each cell will result in a massive performance fall-off.

            // This method renders the specified number of cells, and performs each
            // step (create HTML mark-up, insert mark-up, post-process cells) for
            // all rendered cells at once, before continuing with the next step.

            var // all cells to be rendered in this invocation
                allCells = [],
                // all cells with any text contents to be rendered in this invocation
                contentCells = [],
                // maximum number of cells to be rendered (empty or with contents)
                allCount = contentCount * 4,
                // the HTML mark-up of all rendered cells
                markup = '';

            // STEP 1: extact the specified number of cells from the cache of pending cells
            _(pendingCells).any(function (cellData, mapKey) {

                var // the address of the cell
                    address = cellData.address,
                    // the merged range that starts at the current cell
                    mergedRange = cellData.mergedRange,
                    // the absolute position and size of the cell in the sheet, in pixels
                    rectangle = mergedRange ? sheetModel.getRangeRectangle(mergedRange) : sheetModel.getCellRectangle(address);

                // remove the cell form the DOM and the cache of pending cells
                self.getCellNode(address).remove();
                delete pendingCells[mapKey];

                // do not render zero-size cells
                if ((rectangle.width > 0) && (rectangle.height > 0)) {

                    // add the pixel rectangle to the cell data
                    cellData.rectangle = rectangle;

                    // add cell to the array of cells to be rendered in this iteration
                    allCells.push(cellData);
                    allCount -= 1;

                    // additional processing for cells with text contents
                    if (cellData.display.length > 0) {
                        contentCells.push(cellData);
                        contentCount -= 1;
                    }
                }

                // repeat until the specified number of cells has been found
                return (allCount === 0) || (contentCount === 0);
            });

            // STEP 2: create HTML mark-up of the clip node, content node, and text contents (this part
            // uses fontCollection.getTextWidht(), modifications of the DOM are not allowed here!)
            _(contentCells).each(function (cellData) {

                var // style attribute value for the cell clip node
                    clipStyle = '',
                    // style attribute value for the cell content node (parent of the text spans)
                    contentStyle = '',
                    // style attribute value for the text span (for now, only one span for entire text)
                    spanStyle = '',

                    // attribute map for the 'cell' attribute family
                    cellAttributes = cellData.attributes.cell,
                    // attribute map for the 'character' attribute family
                    charAttributes = cellData.attributes.character,

                    // the available width for the cell text (2 pixels padding left/right)
                    availableWidth = Math.max(2, cellData.rectangle.width - 4),
                    // whether to show negative numbers in red text color
                    redText = cellData.format.red && CellCollection.isNumber(cellData) && (cellData.result < 0),
                    // CSS text color, calculated from text color and fill color attributes
                    textColor = documentStyles.getCssTextColor(redText ? Color.RED : charAttributes.color, [cellAttributes.fillColor]),
                    // CSS text decoration, calculated from underline and strike attributes
                    textDecoration = '',
                    // additional width for the left/right border of the clip node
                    clippingData = null,
                    // the mark-up for the final text contents
                    textMarkup = null;

                // cache character attibutes with effective font size for current zoom factor in cell data
                charAttributes = cellData.charAttributes = _.clone(charAttributes);
                charAttributes.fontSize = view.getEffectiveFontSize(charAttributes.fontSize);

                // horizontal alignment will be set later (calculate optimal
                // width of cells with justified alignment before setting it)
                cellData.contentCss = { textAlign: CellCollection.getCssTextAlignment(cellData) };

                // add vertical alignment
                switch (cellAttributes.alignVert) {
                case 'top':
                    contentStyle += 'top:0;';
                    break;
                case 'middle':
                    // alignment will be updated dynamically below (needs content node in DOM)
                    break;
                case 'justify':
                    // alignment will be updated dynamically below (needs content node in DOM)
                    contentStyle += 'top:0;line-height:100%;';
                    break;
                default:
                    // 'bottom' alignment, and fall-back for unknown alignments
                    contentStyle += 'bottom:0;';
                }

                // create CSS formatting for character attributes
                spanStyle += 'font-family:' + documentStyles.getCssFontFamily(charAttributes.fontName).replace(/"/g, '\'') + ';';
                if (charAttributes.fontSize !== 11) { spanStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }
                if (charAttributes.bold) { spanStyle += 'font-weight:bold;'; }
                if (charAttributes.italic) { spanStyle += 'font-style:italic;'; }
                textDecoration = CellCollection.getCssTextDecoration(cellData);

                // post-processing of some CSS attributes:
                if (textColor !== '#000000') { spanStyle += 'color:' + textColor + ';'; }
                if (textDecoration !== 'none') { spanStyle += 'text-decoration:' + textDecoration + ';'; }

                // create mark-up for the text contents
                cellData.wrappedText = CellCollection.isWrappedText(cellData);
                if (cellData.wrappedText) {
                    // automatic text wrapping: CSS's 'white-space:pre-wrap' does not work together with
                    // 'text-align:justify', need to process white-space dynamically and use 'white-space:normal' instead
                    contentStyle += 'white-space:normal;word-wrap:break-word;';
                    // create a span element for each text line
                    textMarkup = _(cellData.display.split(/\n/)).map(function (textLine) {
                        textLine = Utils.cleanString(textLine).replace(/\s/g, ' ').replace(/^ /, '\xa0').replace(/ {2}/g, ' \xa0').replace(/ $/, '\xa0');
                        return '<span>' + Utils.escapeHTML(textLine) + '</span>';
                    }).join('<br>');
                } else if (CellCollection.isText(cellData)) {
                    // text cells without wrapping: calculate available space for the cell text
                    // over preceding/following empty cells (but always crop merged ranges)
                    clippingData = calculateCellClipping(cellData);
                    clipStyle += 'left:' + (-clippingData.clipLeft) + 'px;right:' + (-clippingData.clipRight) + 'px;';
                    contentStyle += 'left:' + (-clippingData.hiddenLeft) + 'px;right:' + (-clippingData.hiddenRight) + 'px;';
                    // remove line breaks from text, get optimal width
                    textMarkup = cellData.display.replace(/\n/g, '');
                    cellData.contentWidth = fontCollection.getTextWidth(textMarkup, charAttributes);
                    // create a single span element for the final text
                    textMarkup = '<span>' + Utils.escapeHTML(textMarkup) + '</span>';
                } else {
                    // number cells with 'General' number format
                    if (CellCollection.isNumber(cellData) && _.isFinite(cellData.result) && (cellData.format.cat === 'standard') && (cellData.display !== '\u2026')) {
                        // get the pixel width of the number without any restriction (used for automatic sizing of columns)
                        var formatResult = numberFormatter.formatAsStandard(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_CELL, charAttributes);
                        textMarkup = formatResult.text;
                        cellData.contentWidth = formatResult.width;
                        // get a text representation that fits into the cell
                        if (availableWidth < cellData.contentWidth) {
                            // may become null, if no valid display string can be found for the available width
                            formatResult = numberFormatter.formatAsStandard(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_CELL, charAttributes, availableWidth);
                            textMarkup = formatResult ? formatResult.text : null;
                        }
                    } else {
                        // other numbers, Boolean values, and error codes: get width of display string
                        cellData.contentWidth = fontCollection.getTextWidth(cellData.display, charAttributes);
                        if (cellData.contentWidth <= availableWidth) {
                            textMarkup = cellData.display;
                        }
                    }
                    // set railroad track error for cells where display strng does not fit
                    if (!_.isString(textMarkup)) {
                        var hashCharWidth = fontCollection.getTextWidth('#', charAttributes);
                        textMarkup = Utils.repeatString('#', Math.max(1, Math.floor(availableWidth / hashCharWidth)));
                    }
                    // create a single span element for the final text
                    textMarkup = '<span>' + Utils.escapeHTML(textMarkup) + '</span>';
                }

                // Create the clip node and the content node. For now, character styles
                // from 'spanStyle' go to the content node too (no support for rich-text cells yet).
                cellData.contentMarkup = '<div class="clip" style="' + clipStyle + '"><div class="content" style="' + contentStyle + spanStyle + '">' + textMarkup + '</div></div>';
            });

            // STEP 3: create HTML mark-up for all cells to be rendered
            _(allCells).each(function (cellData) {

                var // the logical address of the cell
                    address = cellData.address,
                    // the merged range the cell is part of
                    mergedRange = cellData.mergedRange,
                    // the location of the cell in the sheet, in pixels
                    rectangle = cellData.rectangle,

                    // style attribute value for the root node of the cell
                    cellStyle = '',
                    // style attribute value for the embedded border node
                    borderStyle = '',

                    // whether any border line of the cell is visible
                    hasBorder = false,
                    // CSS fill color
                    fillColor = documentStyles.getCssColor(cellData.attributes.cell.fillColor, 'fill'),
                    // the effective horizontal alignment
                    textAlign = CellCollection.getCssTextAlignment(cellData);

                // creates and adds CSS attributes for the specified border
                function processBorder(attrName, position, roundUp) {

                    var // the border attribute value
                        border = cellData.attributes.cell[attrName],
                        // the effective CSS attributes
                        cssAttrs = documentStyles.getCssBorderAttributes(border),
                        // the number of pixels the border will be moved out of the cell
                        offset = roundUp ? Math.ceil(cssAttrs.width / 2) : Math.floor(cssAttrs.width / 2);

                    borderStyle += position + ':-' + offset + 'px;';
                    if (cssAttrs.width > 0) {
                        borderStyle += 'border-' + position + ':' + cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color + ';';
                        hasBorder = true;
                    }
                }

                // convert cell position relative to layer node
                rectangle = _.clone(rectangle);
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;

                // restrict width of oversized cells to browser limit, according to horizontal alignment
                if (rectangle.width > Utils.MAX_NODE_SIZE) {
                    var widthDiff = rectangle.width - Utils.MAX_NODE_SIZE,
                        reducedLeftInside = rectangle.left + widthDiff > 0,
                        reducedRightInside = rectangle.left + Utils.MAX_NODE_SIZE < layerRectangle.width;
                    switch (textAlign) {
                    case 'left':
                        // Reduce node width to keep cell contents left-aligned
                        // correctly. If the new right border will be located inside
                        // the layer rectangle after width reduction, adjust left border
                        // (which is far outside the layer rectangle in that case).
                        rectangle.left += reducedRightInside ? widthDiff : 0;
                        break;
                    case 'right':
                        // Adjust left offset of the node to keep cell contents
                        // right-aligned correctly, unless the new left border of
                        // the cell will be located inside the layer rectangle after
                        // width reduction (right border is far outside the layer
                        // rectangle in that case).
                        rectangle.left += reducedLeftInside ? 0 : widthDiff;
                        break;
                    default:
                        // Adjust left offset by half of the width correction to
                        // keep the centered text at its position, unless the left
                        // or right border of the cell will be located inside the
                        // layer rectangle after adjustment or width reduction.
                        rectangle.left += reducedRightInside ? widthDiff : reducedLeftInside ? 0 : Math.round(widthDiff / 2);
                    }
                    rectangle.width = Utils.MAX_NODE_SIZE;
                }

                // restrict height of oversized cells to browser limit, according to vertical alignment
                if (rectangle.height > Utils.MAX_NODE_SIZE) {
                    var heightDiff = rectangle.height - Utils.MAX_NODE_SIZE,
                        reducedTopInside = rectangle.top + heightDiff > 0,
                        reducedBottomInside = rectangle.top + Utils.MAX_NODE_SIZE < layerRectangle.height;
                    switch (cellData.attributes.cell.alignVert) {
                    case 'top':
                        // Reduce node height to keep cell contents top-aligned
                        // correctly. If the new bottom border will be located inside
                        // the layer rectangle after height reduction, adjust top border
                        // (which is far outside the layer rectangle in that case).
                        rectangle.top += reducedBottomInside ? heightDiff : 0;
                        break;
                    case 'bottom':
                        // Adjust top offset of the node to keep cell contents
                        // bottom-aligned correctly, unless the new top border of
                        // the cell will be located inside the layer rectangle after
                        // height reduction (bottom border is far outside the layer
                        // rectangle in that case).
                        rectangle.top += reducedTopInside ? 0 : heightDiff;
                        break;
                    default:
                        // Adjust top offset by half of the height correction to
                        // keep the centered text at its position, unless the top
                        // or bottom border of the cell will be located inside the
                        // layer rectangle after adjustment or height reduction.
                        rectangle.top += reducedBottomInside ? heightDiff : reducedTopInside ? 0 : Math.round(heightDiff / 2);
                    }
                    rectangle.height = Utils.MAX_NODE_SIZE;
                }

                // add effective position and size of the cell to the style attribute
                cellStyle += PaneUtils.getRectangleStyleMarkup(rectangle);

                // offset to left/top cell border: round down (keep middle line for odd line widths on left/top pixel of the OWN cell)
                processBorder('borderLeft', 'left', false);
                processBorder('borderTop', 'top', false);
                // offset to right/bottom cell border: round up (keep middle line for odd line widths on left/top pixel of the NEXT cell)
                processBorder('borderRight', 'right', true);
                processBorder('borderBottom', 'bottom', true);

                // post-processing of some CSS attributes
                if (fillColor !== 'transparent') { cellStyle += 'background-color:' + fillColor + ';'; }

                // generate the HTML mark-up for the cell node
                markup += '<div class="cell" data-col="' + address[0] + '" data-row="' + address[1] + '" data-address="' + address[0] + ',' + address[1] + '"';
                if (mergedRange) {
                    if (mergedRange.start[0] < mergedRange.end[0]) { markup += ' data-col-span="' + SheetUtils.getColCount(mergedRange) + '"'; }
                    if (mergedRange.start[1] < mergedRange.end[1]) { markup += ' data-row-span="' + SheetUtils.getRowCount(mergedRange) + '"'; }
                    if (_.isNumber(cellData.visCol)) { markup += ' data-vis-col="' + cellData.visCol + '"'; }
                    if (_.isNumber(cellData.visRow)) { markup += ' data-vis-row="' + cellData.visRow + '"'; }
                }
                markup += ' style="' + cellStyle + '">';

                // add border node, if any border is visible
                if (hasBorder) {
                    markup += '<div class="border" style="' + borderStyle + '"></div>';
                }

                // add text contents created in the previous step
                if (_.isString(cellData.contentMarkup)) { markup += cellData.contentMarkup; }

                // close the root cell node
                markup += '</div>';
            });

            // STEP 4: insert the entire HTML mark-up into the cell layer node, extract DOM nodes
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            cellLayerNode.append(markup);
            _(allCells).each(function (cellData) { cellData.cellNode = self.getCellNode(cellData.address); });
            _(contentCells).each(function (cellData) { cellData.contentNode = cellData.cellNode.find('>.clip>.content'); });

            // STEP 5: post-processing: update vertical alignment, calculate optimal width of justified cells
            // (this part reads the size of DOM nodes, modifications of the DOM are not allowed here!)
            _(contentCells).each(function (cellData) {

                // update vertical alignment 'middle'
                function updateMiddleAlignment() {
                    cellData.contentCss.top = (cellData.rectangle.height - cellData.contentNode.height()) / 2;
                }

                // update vertical alignment 'justify'
                function updateJustifyAlignment() {

                    var // the total height of the content node
                        contentHeight = cellData.contentNode.height(),
                        // the font size of the cell, in pixels
                        fontSize = Utils.convertLength(cellData.charAttributes.fontSize, 'pt', 'px'),
                        // the 'normal' line height, in pixels
                        normalLineHeight = fontCollection.getNormalLineHeight(cellData.charAttributes),
                        // the free space between text line border and characters with 'normal' line height, in pixels
                        normalLinePadding = (normalLineHeight - fontSize) / 2,
                        // the number of text lines
                        lineCount = Math.round(contentHeight / fontSize),
                        // the free space between text line border and characters with target line height, in pixels
                        targetLinePadding = 0,
                        // the effective line height, in pixels
                        targetLineHeight = 0;

                    if (lineCount > 1) {
                        targetLinePadding = (cellData.rectangle.height - 2 * normalLinePadding - lineCount * fontSize) / (2 * lineCount - 2);
                        targetLineHeight = 2 * targetLinePadding + fontSize;
                    }
                    targetLineHeight = Math.max(targetLineHeight, normalLineHeight);
                    cellData.contentCss.lineHeight = Math.round(targetLineHeight) + 'px';
                    cellData.contentCss.top = Math.round(normalLinePadding - targetLinePadding);
                }

                // update vertical alignment dynamically
                switch (cellData.attributes.cell.alignVert) {
                case 'middle':
                    updateMiddleAlignment();
                    break;
                case 'justify':
                    if (cellData.wrappedText) { updateJustifyAlignment(); }
                    break;
                }

                // write the optimal width of wrapped text cells as data attribute into the cell
                if (cellData.wrappedText) {
                    cellData.contentWidth = _(cellData.contentNode[0].childNodes).reduce(function (memo, textSpan) {
                        return Math.max(memo, textSpan.offsetWidth);
                    }, 0);
                }
            });

            // STEP 6: insert CSS formatting created in the post-processing step
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            _(contentCells).each(function (cellData) {
                cellData.cellNode.attr('data-optimal-width', cellData.contentWidth);
                cellData.contentNode.css(cellData.contentCss);
            });
        }

        /**
         * Renders all cells in the passed cell ranges according to the current
         * column, row, and cell collections. The cells will be stored in the
         * internal cache 'pendingCells' for being rendered in a background
         * loop.
         *
         * @param {Object|Array} [ranges]
         *  The logical address of a single cell range, or an array with cell
         *  range addresses. If omitted, all existing cells will be rendered.
         */
        function renderCellRanges(ranges) {

            var // all merged ranges covered by the rendered cells
                mergedRanges = null,
                // additional cells whose clip regions need to be updated
                updateClipCells = {},
                // all merged ranges not yet rendered, mapped by their reference address
                pendingMergedRanges = {};

            // registers a cell for background rendering, and starts the background loop
            var registerCellForRendering = (function () {

                var // the timer representing the background loop
                    timer = null;

                // registers a new cell for debounced rendering
                function registerCell(cellData) {
                    // map cells by their address key (in case cells are registered repeatedly)
                    pendingCells[SheetUtils.getCellKey(cellData.address)] = cellData;
                }

                // starts a background loop that renders a few cells at a time
                function renderCells() {
                    // check whether the background loop already runs
                    if (timer || _.isEmpty(pendingCells)) { return; }
                    // create a new background loop
                    timer = app.repeatDelayed(function () {
                        renderPendingCells(MAX_RENDER_CELL_COUNT);
                        return _.isEmpty(pendingCells) ? Utils.BREAK : undefined;
                    });
                    // clean up after all cells have been rendered
                    timer.always(function () { timer = null; });
                }

                // create and return the actual debounced method registerCellForRendering()
                return app.createDebouncedMethod(registerCell, renderCells);

            }()); // local scope of registerCellForRendering()

            // prepares the cell data of a single cell for rendering
            function prepareCellForRendering(cellData, mergedRange) {

                var // the cell address
                    address = cellData.address,
                    // the map key of the cell
                    mapKey = SheetUtils.getCellKey(address),
                    // column/row collection entry, needed for merged ranges
                    entryData = null;

                // remove existing outdated entry from the cache of pending cells
                delete pendingCells[mapKey];

                // additional processing for merged ranges
                if (mergedRange) {
                    if (_.isEqual(mergedRange.start, address)) {
                        // this is the reference cell of a merged range
                        delete pendingMergedRanges[mapKey];
                    } else {
                        // this is a cell hidden by a merged range, do not render it
                        self.getCellNode(address).remove();
                        return;
                    }
                }

                // add the merged range address to the cell data
                cellData.mergedRange = mergedRange;
                // add information for merged ranges whose first column/row is hidden
                if (mergedRange) {
                    if ((entryData = colCollection.getNextVisibleEntry(address[0])) && (address[0] < entryData.index)) { cellData.visCol = entryData.index; }
                    if ((entryData = rowCollection.getNextVisibleEntry(address[1])) && (address[1] < entryData.index)) { cellData.visRow = entryData.index; }
                }
                // register cell for rendering in background task
                registerCellForRendering(cellData);
            }

            // restrict to the bounding range of the cell collection
            ranges = SheetUtils.getIntersectionRanges(ranges, cellCollection.getRange());
            if (ranges.length === 0) { return; }

            // process all cells in all passed ranges once (row by row)
            Utils.takeTime('GridPane.renderCellRanges(): panePos=' + panePos + ', preparing rendering', function () {

                var // the index of the last processed row
                    lastRow = -1,
                    // the column intervals of all merged ranges in the current row
                    mergedColIntervals = null;

                // find all merged ranges covered by the rendered cells
                mergedRanges = mergeCollection.getMergedRanges(ranges);

                // initialize the map of merged ranges not yet rendered
                _(mergedRanges).each(function (mergedRange) {
                    pendingMergedRanges[SheetUtils.getCellKey(mergedRange.start)] = mergedRange;
                });

                // finds an additional cell outside the passed ranges whose clip region needs to be updated
                function findUpdateClipCell(address, forward) {
                    cellCollection.iterateNextCells(address, forward ? 'right' : 'left', function (cellData) {

                        // merged cells do not need to be updated (they always clip at cell borders)
                        if (intervalsContainIndex(mergedColIntervals, cellData.address[0])) { return Utils.BREAK; }

                        // skip cells without value
                        if (CellCollection.isBlank(cellData)) { return; }

                        // only text results without automatic wrapping can overflow and thus need to be updated
                        if (CellCollection.isOverflowText(cellData)) {
                            updateClipCells[SheetUtils.getCellKey(cellData.address)] = cellData;
                        }

                        // exit the loop on any non-empty cell
                        return Utils.BREAK;

                    }, { existing: true, skipStartCell: true });
                }

                // generate the HTML mark-up for all cells in the passed ranges
                cellCollection.iterateCellsInRows(ranges, function (cellData, rowBandRange) {

                    var // the cell address
                        address = cellData.address,
                        // the merged range containing the cell
                        mergedRange = _(mergedRanges).find(function (mergedRange) {
                            return SheetUtils.rangeContainsCell(mergedRange, address);
                        });

                    // if a new row has been started, find all merged ranges in that row, sorted by column for binary search
                    if (lastRow !== address[1]) {
                        lastRow = address[1];
                        mergedColIntervals = mergeCollection.getMergedColIntervals(lastRow);
                    }

                    // If cell is located at the left or right border of a row band range,
                    // find preceding/following non-empty cell which needs to be rendered
                    // again if it is not wrapped (its clip region may change due to the
                    // new contents of the current cell).
                    if (address[0] === rowBandRange.start[0]) { findUpdateClipCell(address, false); }
                    if (address[0] === rowBandRange.end[0]) { findUpdateClipCell(address, true); }

                    // remove the old cell node from the DOM, generate the HTML mark-up of the new cell node
                    prepareCellForRendering(cellData, mergedRange);
                });

                // render remaining merged ranges (e.g. starting outside the changed ranges/visible area)
                _(pendingMergedRanges).each(function (mergedRange) {
                    var cellData = cellCollection.getCellEntry(mergedRange.start, { always: true });
                    prepareCellForRendering(cellData, mergedRange);
                });

                // update the clip region of existing cells found while rendering the ranges
                _(updateClipCells).each(function (cellData) {
                    var cellNode = self.getCellNode(cellData.address), // cell node may be missing (if covered by passed cell ranges)
                        clippingData = (cellNode.length === 0) ? null : calculateCellClipping(cellData);
                    if (clippingData) {
                        cellNode.find('>.clip').css({ left: -clippingData.clipLeft, right: -clippingData.clipRight })
                            .find('>.content').css({ left: -clippingData.hiddenLeft, right: -clippingData.hiddenRight });
                    }
                });
            });

            // render the first 5 cells synchronously (looks better when
            // entering multiple single cells manually one-by-one)
            renderPendingCells(5);
        }

        /**
         * Updates the DOM in either horizontal or vertical direction, after
         * the bounding range of the cell collection has changed. Removes DOM
         * cell nodes of cells not visible anymore, and returns the addresses
         * of the cell nodes to be rendered.
         *
         * @returns {Array}
         *  An array with logical range addresses of all cells to be rendered.
         */
        function updateBoundRangeInterval(newInterval, columns, operation) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the old column/row indexes
                oldIndexes = columns ? colIndexes : rowIndexes,
                // the new column/row indexes
                newIndexes = collection.getVisibleEntryIndexes(newInterval),
                // the old column/row interval
                oldInterval = (oldIndexes && (oldIndexes.length > 0)) ? { first: oldIndexes[0], last: _.last(oldIndexes) } : null,
                // the column/row interval of cells to be moved
                moveInterval = null,
                // all cell ranges to be rendered
                renderRanges = [],

                // the model method to get the column/row range from an interval
                makeRangeFunc = _.bind(columns ? model.makeColRange : model.makeRowRange, model),
                // the name of the column/row data attribute
                indexAttrName = columns ? 'data-col' : 'data-row',
                // the name of the additional column/row data attribute for merged ranges with leading hidden columns/rows
                visibleIndexAttrName = columns ? 'data-vis-col' : 'data-vis-row',
                // helper function to update the cell address data attribute of a cell node
                updateCellAddress = columns ?
                    function (cellNode, col) { cellNode.attr('data-address', col + ',' + cellNode.attr('data-row')); } :
                    function (cellNode, row) { cellNode.attr('data-address', cellNode.attr('data-col') + ',' + row); },
                // the array index for cell addresses
                addrIndex = columns ? 0 : 1,
                // the name of the cell data property containing the visible column/row index in partly hidden merged ranges
                visibleIndexPropName = columns ? 'visCol' : 'visRow',
                // the name of the CSS position attribute for cell nodes
                posAttrName = columns ? 'left' : 'top',
                // the leading position of the layer rectangle
                layerOffset = layerRectangle[posAttrName],

                 // whether a 'refresh' operation has been passed
                refreshAll = _.isObject(operation) && (operation.name === 'refresh'),
                // the name of the operation, if it matches the current direction
                operationName = (!refreshAll && _.isObject(operation) && (operation.direction === (columns ? 'columns' : 'rows'))) ? operation.name : null,
                // the number of columns/rows affected by the operation
                operationSize = _.isString(operationName) ? SheetUtils.getIntervalSize(operation) : null,

                // loop variables
                arrayIndex1 = 0, arrayIndex2 = 0, arraySize = 0;

            // removes all cell nodes in the specified columns/rows from the cell layer
            function removeCellNodes(indexes) {
                _(indexes).each(function (index) {

                    // remove cell nodes from the DOM
                    cellLayerNode.find('>[' + indexAttrName + '="' + index + '"]').remove();

                    // remove entries from the cache of pending cells
                    // (iterate entire cache, assuming that it is quite empty most of the time)
                    _(pendingCells).each(function (cellData, mapKey) {
                        if (cellData.address[addrIndex] === index) {
                            delete pendingCells[mapKey];
                        }
                    });
                });
            }

            // updates the address data attributes of all cells in a column/row
            function updateCellNodeAttributes(oldIndex, diff) {

                var // the target column/row index
                    newIndex = oldIndex + diff;

                // update data attributes of cells existing in the DOM
                cellLayerNode.find('>[' + indexAttrName + '="' + oldIndex + '"]')
                    .attr(indexAttrName, newIndex)
                    .each(function () { updateCellAddress($(this), newIndex); });
                cellLayerNode.find('>[' + visibleIndexAttrName + '="' + oldIndex + '"]')
                    .attr(visibleIndexAttrName, newIndex)
                    .each(function () { $(this).attr(indexAttrName, Utils.getElementAttributeAsInteger(this, indexAttrName) + diff); });

                // re-register pending cells for rendering at new position
                // (iterate entire cache, assuming that it is quite empty most of the time)
                _(pendingCells).each(function (cellData, mapKey) {
                    if ((cellData.address[addrIndex] === oldIndex) || (cellData[visibleIndexPropName] === oldIndex)) {
                        delete pendingCells[mapKey];
                        renderRanges.push({ start: cellData.address, end: cellData.address });
                    }
                });
            }

            // set the new column/row indexes
            if (columns) { colIndexes = newIndexes; } else { rowIndexes = newIndexes; }

            // handle 'insert columns/rows' operation
            if ((operationName === 'insert') && oldInterval && (operation.first <= oldInterval.last)) {

                // insert indexes for new columns/rows
                arrayIndex1 = _(oldIndexes).sortedIndex(operation.first);
                oldIndexes.splice.apply(oldIndexes, [arrayIndex1, 0].concat(collection.getVisibleEntryIndexes(operation)));

                // increase column/row indexes in old index array and cell nodes
                arrayIndex1 += operationSize;
                for (arrayIndex2 = oldIndexes.length - 1; arrayIndex2 >= arrayIndex1; arrayIndex2 -= 1) {
                    updateCellNodeAttributes(oldIndexes[arrayIndex2], operationSize);
                    oldIndexes[arrayIndex2] += operationSize;
                }

                // update the old interval
                oldInterval = { first: oldIndexes[0], last: _.last(oldIndexes) };
            }

            // handle 'delete columns/rows' operation
            if ((operationName === 'delete') && oldInterval && (operation.first <= oldInterval.last)) {

                // remove all old indexes and cell nodes in the operation interval
                arrayIndex1 = _(oldIndexes).sortedIndex(operation.first);
                arraySize = _(oldIndexes).sortedIndex(operation.last + 1) - arrayIndex1;
                removeCellNodes(oldIndexes.splice(arrayIndex1, arraySize));

                // decrease column/row indexes in old index array and cell nodes
                for (; arrayIndex1 < oldIndexes.length; arrayIndex1 += 1) {
                    updateCellNodeAttributes(oldIndexes[arrayIndex1], -operationSize);
                    oldIndexes[arrayIndex1] -= operationSize;
                }

                // update the old interval
                oldInterval = { first: oldIndexes[0], last: _.last(oldIndexes) };
            }

            // remove all old cell nodes not visible anymore
            if (oldIndexes) {
                removeCellNodes(_(oldIndexes).difference(newIndexes));
            }

            // move existing cell nodes in the DOM, if cells at the leading border have
            // been removed, or if a column/row operation has caused the change event
            if (oldInterval && !refreshAll) {
                if (oldInterval.first !== newInterval.first) {
                    moveInterval = oldInterval;
                } else if (_.isString(operationName)) {
                    moveInterval = SheetUtils.getIntersectionInterval({ first: operation.last + 1, last: columns ? model.getMaxCol() : model.getMaxRow() }, oldInterval);
                }
                if (moveInterval && (moveInterval = SheetUtils.getIntersectionInterval(moveInterval, newInterval))) {
                    collection.iterateVisibleEntries(moveInterval, function (entryData) {
                        var selector = '>[' + indexAttrName + '="' + entryData.index + '"],>[' + visibleIndexAttrName + '="' + entryData.index + '"]';
                        // move existing cell nodes in the DOM
                        cellLayerNode.find(selector).css(posAttrName, (entryData.offset - layerOffset) + 'px');
                    });
                }
            }

            // get the range addresses of the cells in all new columns/rows
            if (oldInterval && !refreshAll) {
                // new cells moved into the leading border of the bounding range
                if (newInterval.first < oldInterval.first) {
                    renderRanges.push(makeRangeFunc({ first: newInterval.first, last: Math.min(newInterval.last, oldInterval.first - 1) }));
                }
                // new cells moved into the trailing border of the bounding range
                if (oldInterval.last < newInterval.last) {
                    renderRanges.push(makeRangeFunc({ first: Math.max(oldInterval.last + 1, newInterval.first), last: newInterval.last }));
                }
                // operation applied, render all affected cells
                if (_.isString(operationName)) {
                    renderRanges.push(makeRangeFunc(operation));
                }
            } else {
                // render all cells
                renderRanges.push(makeRangeFunc(newInterval));
                cellLayerNode.empty();
                pendingCells = {};
            }

            return renderRanges;
        }

        /**
         * Updates the DOM after the bounding range of the cell collection has
         * changed. Removes DOM cell nodes of cells not visible anymore, and
         * renders new cell nodes now visible in this grid pane.
         */
        function changeBoundRangeHandler(event, boundRange, operation) {

            var // all cell ranges to be rendered
                renderRanges = [];

            Utils.log('GridPane.changeBoundRangeHandler(): panePos=' + panePos + ', boundRange=' + SheetUtils.getRangeName(boundRange) + ', operation=' + JSON.stringify(operation));

            // calculate the new position of the layer nodes
            layerRange = _.copy(boundRange, true);
            layerRectangle = sheetModel.getRangeRectangle(layerRange);

            // update the scroll node and layer root node
            updateScrollAreaSize();
            updateScrollPosition();

            // update all cell nodes
            renderRanges = renderRanges.concat(updateBoundRangeInterval(SheetUtils.getColInterval(boundRange), true, operation));
            renderRanges = renderRanges.concat(updateBoundRangeInterval(SheetUtils.getRowInterval(boundRange), false, operation));
            renderCellRanges(renderRanges);
            renderCellSelection();
            renderHighlightedRanges();
            renderActiveSelection();

            // repaint all visible drawing frames
            updateDrawingFrames();
        }

        /**
         * Updates the DOM after the bounding range of the cell collection has
         * been cleared (after one of the header panes has been hidden).
         */
        function clearBoundRangeHandler() {

            // safely destroy all image nodes to prevent memory problems on iPad
            app.destroyImageNodes(drawingLayerNode.find('img'));

            // clear all DOM layers
            cellLayerNode.empty();
            selectionLayerNode.empty();
            highlightLayerNode.empty();
            activeRangeLayerNode.empty();
            drawingLayerNode.empty();

            // reset all other variables related to the bounding range
            colIndexes = rowIndexes = layerRange = layerRectangle = null;
            pendingCells = {};
        }

        /**
         * Handles change events from the cell collection. Initiates debounced
         * rendering of all affected cells.
         */
        var changeCellsHandler = (function () {

            var // the cell ranges to be rendered after collection updates
                changedRanges = [];

            // direct callback: stores the changed ranges in an internal array
            function registerRanges(event, ranges) {
                if (layerRange) {
                    Utils.log('GridPane.changeCellsHandler(): panePos=' + panePos + ', cells=' + SheetUtils.getRangesName(ranges));
                    changedRanges = changedRanges.concat(ranges);
                }
            }

            // deferred callback: renders all changed cells, updates other layout settings of the grid pane
            function renderChangedRanges() {
                if (layerRange) {
                    renderCellRanges(changedRanges);
                }
                changedRanges = [];
            }

            // create and return the actual debounced 'registerChangedRanges()' method
            return app.createDebouncedMethod(registerRanges, renderChangedRanges);
        }());

        /**
         * Updates the selection and highlight layers after specific sheet view
         * attributes have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell and drawing selection
            if ('selection' in attributes) {
                cellListMenu.hide();
                renderCellSelection();
                renderDrawingSelection();
            }

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                cellListMenu.hide();
                updateFocusDisplay();
            }

            // changed auto-fill tracking data
            if ('autoFillData' in attributes) {
                cellListMenu.hide();
                renderCellSelection();
            }

            // changed highlight ranges
            if (Utils.hasProperty(attributes, /^highlight/)) {
                renderHighlightedRanges();
            }

            // changed active ranges (used while selecting ranges in cell edit mode)
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }
        }

        /**
         * Initializes this grid pane after the active sheet has been changed.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            validationCollection = sheetModel.getValidationCollection();
            drawingCollection = sheetModel.getDrawingCollection();
        }

        /**
         * Initializes the text area used for in-place cell edit mode.
         */
        function changeLayoutDataHandler() {
            if (cellEditSettings.active) {
                updateTextAreaStyle();
                updateTextAreaPosition();
            }
        }

        /**
         * Handles DOM 'scroll' events of the scrollable node.
         */
        function scrollHandler() {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        }

        /**
         * Shows the pop-up menu for list validation.
         */
        var cellDropDownClickHandler = (function () {

            var // the server request currently running
                runningRequest = null;

            // return the actual 'cellDropDownClickHandler()' method
            return function cellDropDownClickHandler() {

                var // the current cell address (will be checked in server response)
                    address = view.getActiveCell(),
                    // the source ranges for the cell (for validation type 'source')
                    sourceRanges = null,
                    // the server request started by this method invokation
                    localRequest = null;

                // leave cell edit mode, or set focus back into clipboard node of this grid pane
                self.leaveCellEditMode('cell');
                self.grabFocus();

                // hide visible pop-up menu without any action
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    return;
                }

                // cancel server request currently running, to prevent showing the menu
                // after the drop-down button has clicked again before the response arrives
                if (runningRequest) {
                    runningRequest.abort();
                    runningRequest = null;
                    return;
                }

                // resolve formula to source ranges
                sourceRanges = validationCollection.getSourceRanges(address);
                if (!sourceRanges) { return; }

                // receive source data from server
                localRequest = runningRequest = app.queryRangeContents([sourceRanges]);

                // fill list menu with values received from server
                localRequest.done(function (resultContents) {

                    // check that this is still the last request (with slow connections,
                    // the response may arrive after selecting another cell with drop-down
                    // menu, and trying to open the dop-down menu of that cell)
                    if (runningRequest !== localRequest) { return; }

                    // check that the cell address of current cell is still the same
                    // (response may arrive after selecting another cell)
                    if (_.isEqual(address, view.getActiveCell())) {
                        cellListMenu.clearContents();
                        _(resultContents[0]).each(function (cellData) {
                            cellListMenu.createItemNode('', cellData.display, { label: _.noI18n(cellData.display) });
                        });
                        cellListMenu.show();
                    }
                });

                // reset the 'runningRequest' variable, if the last response is available
                localRequest.always(function () {
                    if (runningRequest === localRequest) {
                        runningRequest = null;
                    }
                });
            };
        }());

        /**
         * Inserts the value represented by the passed list item from the cell
         * pop-up menu into the cell.
         */
        function applyCellListValue(buttonNode) {

            var // the value of the selected list item
                itemValue = (buttonNode.length > 0) ? Utils.getControlValue(buttonNode) : null;

            // always hide the pop-up menu
            cellListMenu.hide();

            // insert the value into the cell
            if (_.isString(itemValue)) { view.setCellContents(itemValue); }
        }

        /**
         * Updates the layout of the cell pop-up menu. Hides the menu if the
         * anchor cell is not visible anymore, otherwise adjusts the minimum
         * width of the menu node according to the width of the cell.
         */
        function cellListBeforeLayoutHandler() {

            var cellNode = cellListMenu.getAnchorNode(),
                buttonNode = cellNode.find('>.cell-dropdown-button');

            if ((cellNode.length === 0) || (buttonNode.length === 0)) { return; }

            // check the position of the cell node in the scrollable area
            if (!Utils.isChildNodeVisibleInNode(scrollNode, cellNode)) {
                cellListMenu.hide();
                return;
            }

            // update minimum width of the menu sections according to cell width (plus width of drop-down button)
            cellListMenu.getSectionNodes().css({ minWidth: buttonNode.offset().left + buttonNode.outerWidth() - cellNode.offset().left });
        }

        /**
         * Handles clicks in the cell pop-up menu.
         */
        function cellListClickHandler(event) {
            applyCellListValue($(event.currentTarget));
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            var // the list item currently selected in the cell pop-up menu
                itemNode = null;

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // cancel current tracking, but do nothing else
                if ($.cancelTracking()) {
                    return false;
                }

                // drawing selection: back to cell selection
                if (view.hasDrawingSelection()) {
                    view.removeDrawingSelection();
                    return false;
                }

                // close cell pop-up menu
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    return false;
                }

                // close search bar
                if (app.getController().getItemValue('view/searchpane')) {
                    app.getController().executeItem('view/searchpane', false);
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // show cell pop-up menu
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                if (!cellListMenu.isVisible()) {
                    self.scrollToCell(view.getActiveCell());
                    layerRootNode.find('.cell-dropdown-button').click();
                }
                return false;
            }

            // handle keyboard shortcuts for opened cell pop-up menu
            if (cellListMenu.isVisible()) {
                itemNode = cellListMenu.getSelectedItemNodes().first();

                // aply the selected value
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) || KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                    applyCellListValue(itemNode);
                    return; // let event bubble up for selection change
                }

                // change selection
                itemNode = cellListMenu.getItemNodeForKeyEvent(event, itemNode, { selectEmpty: true });
                if (itemNode.length > 0) {
                    cellListMenu.selectItemNode(itemNode, { scroll: true });
                    return false;
                }
            }
        }

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         */
        function grabClipboardFocus() {

            var // the browser selection
                selection = window.getSelection(),
                // a browser selection range object
                docRange = null;

            // calls the native focus() method at the clipboard node
            function grabNativeFocus() {
                clipboardFocusMethod.call(clipboardNode[0]);
            }

            // bug 29948: prevent grabbing to clipboard node while in cell in-place edit mode
            if (cellEditSettings.active) { return; }

            // on touch devices, selecting text nodes in a content-editable node will
            // always show the virtual keyboard
            // TODO: need to find a way to provide clipboard support
            if (Modernizr.touch) {
                grabNativeFocus();
                return;
            }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            if (_.browser.Firefox) {
                grabNativeFocus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            BaseView.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        }

        /**
         * Changes the quick edit mode while in-place cell edit mode is active,
         * and shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} quickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(quickEdit) {

            // change the quick edit state
            if (cellEditSettings.quick === quickEdit) { return; }
            cellEditSettings.quick = quickEdit;

            // show the status label in the view
            view.showStatusLabel(cellEditSettings.quick ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveSelection() {
            sheetModel.setViewAttribute('activeSelection', null);
        }

        /**
         * Returns the target node for the browser focus, which is either the
         * internal clipboard node (used for browser copy/paste events), or the
         * text area control while in-place edit mode is active.
         *
         * @returns {jQuery}
         *  The focus target node, as jQuery object.
         */
        function getFocusTargetNode() {
            return cellEditSettings.active ? textArea : clipboardNode;
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(buttonNode) {

            var // the name and type of the selected list item
                itemValue = Utils.getControlValue(buttonNode),
                // the text to be inserted into the formula, and the new cursor position
                newText = null,
                newStart = 0;

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            newText = itemValue.name;
            newStart = cellEditSettings.autoTokenInfo.start + newText.length + 1;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (cellEditSettings.autoTokenInfo.token.getType() === 'name') {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    cellEditSettings.autoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string
            setTextAreaContents(Utils.replaceSubString(textArea.val(), cellEditSettings.autoTokenInfo.start + 1, cellEditSettings.autoTokenInfo.end + 1, newText), newStart);
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyDownHandler(event) {

            var // the list item currently selected in the auto-completion pop-up menu
                itemNode = textAreaListMenu.getSelectedItemNodes().first();

            // first, try to cancel active tracking mode on ESCAPE key
            if ((event.keyCode === KeyCodes.ESCAPE) && $.cancelTracking()) {
                return false;
            }

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {
                    applyAutoCompleteText(itemNode);
                    return false;
                }

                // try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.leaveCellEditMode();
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!cellEditSettings.quick);
                // forget last cell range selected in the sheet
                resetActiveSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                self.leaveCellEditMode('cell');
                return; // let key event bubble up to move the cell cursor
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert array formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveCellEditMode(event.shiftKey ? 'array' : 'fill');
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Utils.replaceTextInTextFieldSelection(textArea, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                self.leaveCellEditMode('cell');
                return; // let key event bubble up to move the cell cursor
            }

            // any cursor navigation key
            if (_([KeyCodes.LEFT_ARROW, KeyCodes.RIGHT_ARROW, KeyCodes.UP_ARROW, KeyCodes.DOWN_ARROW, KeyCodes.PAGE_UP, KeyCodes.PAGE_DOWN, , KeyCodes.HOME, KeyCodes.END]).contains(event.keyCode)) {
                if (cellEditSettings.quick) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

            // Register an opening parenthesis entered manually. The related closing parenthesis
            // will be inserted even if it occurs among other closing parentheses inserted via
            // function auto-completion (see below).
            case '(':
                cellEditSettings.autoCloseData.push(false);
                break;

            // When a function name has been inserted from the auto-completion pop-up menu, skip
            // a closing parenthesis because it has been inserted already by the auto-completion).
            // But the text cursor needs to be moved one character forward.
            case ')':
                if (cellEditSettings.autoCloseData.pop() === true) {
                    Utils.setTextFieldSelection(textArea, Utils.getTextFieldSelection(textArea).start + 1);
                    updateTextAreaPopups();
                    return false;
                }
                break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            var // the previous value of the text area before the input event
                lastValue = cellEditSettings.lastValue,
                // the current value of the text area
                currValue = textArea.val(),
                // the text selection in the text area control
                selection = Utils.getTextFieldSelection(textArea),
                // whether to show type-ahead suggestion in the text area
                typeAhead = (lastValue.length < currValue.length) && (selection.start === selection.end) && (selection.end === currValue.length);

            // update text area settings (position, formula range highlighting)
            cellEditSettings.changed = true;
            cellEditSettings.lastValue = currValue;
            cellEditSettings.autoComplete = (lastValue !== currValue) && (selection.start === selection.end);
            updateTextAreaPosition();
            updateTextAreaTokenArray();

            // forget last cell range selected in the sheet
            resetActiveSelection();

            /* Start-of autocompleteText */
            // TODO rework needed
            var columnText = [], suggestions = [];

            if (typeAhead && !cellEditSettings.isFormula) { //if user writes(adds) new letter -> auto suggest new text

                cellCollection.iterateNextCells(view.getActiveCell(), 'up down', function textCollector(cellData) {
                    if (CellCollection.isText(cellData) && (cellData.result.length > 0)) {
                        columnText.push(cellData.result);
                    }
                }, { hidden: true, existing: true, skipStartCell: true });

                //if first letter in currValue is LowerCase
                if (currValue.charAt(0) === currValue.charAt(0).toLowerCase() && currValue.charAt(0) !== currValue.charAt(0).toUpperCase()) {
                   //charAt() - returns the character at the specified index in a string.

                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toLowerCase() + columnText[i].substring(1); //make first letter in columText LowerCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) { //loop through each element in "columnText" array, to see if it begins with the text in textArea. indexOf() returns 0 if this is the case.
                            suggestions.push(columnText[i]); //fill an array "suggestions" with all possible suggestions
                        }
                    }
                } //end-of if LowerCase
                else {
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toUpperCase() + columnText[i].substring(1); //make first letter in columText UpperCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                } //end-of else

                if (suggestions.length === 0) { //for mix of LowerCase/UpperCase suggestion
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].toLowerCase();
                    }
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                }
            }

            //sort suggestions by length-ascending
            suggestions.sort(function (a, b) {
                return a.length - b.length;
            });

            //make sure there is at least one suggestion in array "suggestions"
            if (suggestions.length > 0) {
                // suggest new text to user, first best only
                setTextAreaContents(suggestions[0], currValue.length, suggestions[0].length);
                textArea.focus();
            }
            /** End-of autocompleteText */

            // notify listeners
            self.trigger('celledit:change');
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveSelection();
        }

        /**
         * Updates the scroll position of the underlay node, after the scroll
         * position of the text area node has been changed.
         */
        function textAreaScrollHandler() {
            textAreaUnderlay.scrollTop(textArea.scrollTop());
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            var // whether to ignore focus event in the text area (no update of pop-up nodes)
                ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                // text area focused: rebuild all active pop-up nodes
                case 'focus':
                    if (!ignoreFocusEvent) {
                        updateTextAreaPopups();
                    }
                    break;

                // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                // has been clicked, in this case, move focus back to text area immediately)
                case 'blur':
                    _.defer(function () {
                        if (cellEditSettings.active && textAreaListMenu.hasFocus()) {
                            ignoreFocusEvent = true;
                            textArea.focus();
                            // IE sends focus event deferred
                            _.defer(function () { ignoreFocusEvent = false; });
                        } else {
                            hideTextAreaPopups();
                        }
                    });
                }
            };
        }());

        /**
         * Inserts the name clicked in the auto-completion pop-up menu into the
         * current formula text.
         */
        function textAreaListClickHandler(event) {
            applyAutoCompleteText($(event.currentTarget));
        }

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            var // the cursor position in the token array (without leading equality sign)
                textPos = textArea[0].selectionStart - 1,
                // the information for the token at the cursor position
                tokenInfo = null;

            // invalid if not in cell in-place edit mode with a formula
            if (!cellEditSettings.active || !cellEditSettings.isFormula) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || _.isObject(sheetModel.getViewAttribute('activeSelection'))) { return true; }

            // get data of the token located before the text cursor
            tokenInfo = cellEditSettings.tokenArray.getTokenAtPosition(textPos);

            // cursor must be located between tokens; leading token must be of correct
            // type (or cursor must be located at the beginning of the formula)
            return tokenInfo && (textPos === tokenInfo.end) && (/^(op|sep|open)$/).test(tokenInfo.token.getType());
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            var // the current value of the 'activeSelection' view attribute
                activeSelection = sheetModel.getViewAttribute('activeSelection');

            // always return existing view attribute
            if (_.isObject(activeSelection)) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return { ranges: [], activeRange: 0, activeCell: [0, 0] };
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            var // a helper token array to generate reference addresses for the active ranges
                tokenArray = null,
                // the string representation of the active ranges
                references = null,
                // the new formula text
                formulaText = null;

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!_.isObject(sheetModel.getViewAttribute('activeSelection'))) {
                cellEditSettings.rangeSelection = Utils.getTextFieldSelection(textArea);
            }

            // build the new formula term to be inserted into the text
            tokenArray = new TokenArray(app, sheetModel, { silent: true });
            tokenArray.appendRangeList(selection.ranges);
            references = tokenArray.getFormula();
            tokenArray.destroy();
            formulaText = Utils.replaceSubString(textArea.val(), cellEditSettings.rangeSelection.start, cellEditSettings.rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            cellEditSettings.rangeSelection.end = cellEditSettings.rangeSelection.start + references.length;
            setTextAreaContents(formulaText, cellEditSettings.rangeSelection.end);

            // update the view attribute
            sheetModel.setViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        /**
         * Handles a changed document edit mode. If switched to read-only mode,
         * cancels in-place cell edit mode, or a running tracking action to
         * move or resize drawing objects.
         */
        function editModeHandler(event, editMode) {

            // nothing to do, if this grid pane is currently not visible
            if (!layerRange) { return; }

            // leave cell in-place edit mode, if document changes to read-only mode
            if (!editMode) {
                self.leaveCellEditMode();
                cellListMenu.hide();
            }

            // redraw selection according to edit mode
            renderCellSelection();
            renderDrawingSelection();
            renderHighlightedRanges();
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            var // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

            // check for cell protection
            if (sheetModel.isProtected() && view.isSelectionLocked()) {
                view.yell('info', gt('Cutting protected cells is not allowed.'));
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            copyHandler(event);

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // delete values and clear attributes of the selected cells
                view.fillCellRanges(null, undefined, { clear: true });

            } else {
                app.executeDelayed(function () {
                    // delete values and clear attributes of the selected cells
                    view.fillCellRanges(null, undefined, { clear: true });
                });
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        function copyHandler(event) {
            var // current selection in the active sheet
                selection = view.getSelection(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the client clipboard id to identify the copy data when pasting
                clientClipboardId = view.createClientClipboardId(),
                // the html table string to export
                htmlTable,
                // the plain text to export
                plainText;

            // restrict the number of cells to copy at the same time
            if (SheetUtils.getCellCount(selection.ranges[selection.activeRange]) > SheetUtils.MAX_FILL_CELL_COUNT) {
                view.yell('info', gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)));
                event.preventDefault();
                return false;
            }

            // generate HTML table and plain text representation from the active range of the current selection
            htmlTable = Clipboard.getHTMLStringFromSelection(selection, clientClipboardId, cellCollection, mergeCollection, app);
            plainText = Clipboard.getPlainTextFromSelection(selection, cellCollection);

            // initiate clipboard server copy call
            view.clipboardServerCopy(selection);
            Utils.log('clipboard - copy - client id: ' + clientClipboardId);

            // if browser supports clipboard api add data to the event
            if (clipboardData) {
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', plainText);
                clipboardData.setData('text/html', htmlTable);

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }
            } else {
                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    'xmlns': 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // add content and focus
                clipboardNode.empty().append(htmlTable);
                grabClipboardFocus();

                // remove XML name spaces again
                app.executeDelayed(function () {
                    $('html').removeAttr('xmlns xmlns:o xmlns:x');
                });
            }
        }

        /**
         * Handles 'paste' events of the clipboard node.
         */
        function pasteHandler(event) {
            var // current selection in the active sheet
                selection = view.getSelection(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the html data attached to the event
                htmlData,
                // the plain text data attached to the event
                textData;


            function pasteHtmlClipboard(html) {
                var // the client clipboard id to identify the copy data when pasting
                    clientClipboardId = Clipboard.getClientClipboardId(html),
                    // the server clipboard id to identify the copy data when pasting
                    serverClipboardId = view.getServerClipboardId(clientClipboardId);

                if (serverClipboardId) {
                    // server paste call
                    view.clipboardServerPaste(selection, serverClipboardId);
                    Utils.log('clipboard - server side paste');
                } else {
                    // parse HTML
                    handleParserResult(Clipboard.parseHTMLData(html, documentStyles));
                    Utils.log('clipboard - client side html paste');
                }
            }

            function pasteTextClipboard(text) {
                // parse text
                handleParserResult(Clipboard.parseTextData(text));
                Utils.log('clipboard - client side text paste');
            }

            function handleParserResult(result) {
                var // the target range of the paste
                    pasteRange = Clipboard.createRangeFromParserResult(selection.activeCell, result),
                    // the number of cells to paste
                    pasteCellCount = SheetUtils.getCellCount(pasteRange),
                    // the ranges of the merged cells
                    mergeRanges;

                // check for protected cells
                view.areRangesEditable(pasteRange).done(function () {

                    // restrict the number of cells to paste at the same time
                    if (pasteCellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                        view.yell('info', gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)));
                        return;
                    }

                    // unmerge cells if necessary
                    if ((pasteCellCount > 1) && mergeCollection.rangeOverlapsMergedRange(pasteRange)) {
                        sheetModel.mergeRanges(pasteRange, 'unmerge');
                    }

                    // merge cells
                    if (result.mergeCollection && result.mergeCollection.length > 0) {
                        mergeRanges = _.map(result.mergeCollection, function (range) { return SheetUtils.addOffsetToRange(range, selection.activeCell); });
                        sheetModel.mergeRanges(mergeRanges, 'merge');
                    }

                    // set cell contents
                    sheetModel.setCellContents(view.getActiveCell(), result.contents, { parse: true });
                })
                .fail(function () {
                    view.yell('info', gt('Pasting into protected cells is not allowed.'));
                });
            }

            /**
             * Cleans up script elements and event handlers to prevent Cross Site Scripting.
             * And removes position:fixed CSS style to avoid clipboard content being visible.
             */
            function cleanUpHtml(html) {
                var result;

                if (!html) { return null; }

                // remove event handler to prevent script injection and XSS
                result = html.replace(/<[^>]*\s(onload|onunload|onerror|onerrorupdate|onchange)=.*>/gi, '');

                // remove script elements
                result = $.parseHTML(result, null, false /*keepScripts*/);

                // remove position fixed style to avoid clipboard content being visible
                $(result).find('*').andSelf().css('position', '');

                return result;
            }


            // if browser supports clipboard api get data from the event
            if (clipboardData) {
                htmlData = cleanUpHtml(clipboardData.getData('text/html'));
                textData = clipboardData.getData('text/plain');

                if (htmlData || textData) {
                    // prevent default paste handling for desktop browsers, but not for touch devices
                    if (!Modernizr.touch) {
                        event.preventDefault();
                    }

                    if (htmlData && Clipboard.containsHtmlTable(htmlData)) {
                        // render the html to apply css styles provided by Excel or Calc
                        clipboardNode.empty().append(htmlData);
                        // parse and create operations
                        pasteHtmlClipboard(clipboardNode);
                        // clear clipboard
                        clipboardNode.text('\xa0');

                    } else if (textData) {
                        pasteTextClipboard(textData);
                    }

                    return false;
                }
            }

            // focus and select the clipboard node
            grabClipboardFocus();

            // get the plain text data from the IE text clipboard to be used
            // in case the clipboard div contains no html table to parse
            textData = (_.browser.IE && window.clipboardData) ? window.clipboardData.getData('text') : '';

            // read and parse pasted data
            app.executeDelayed(function () {
                if (Clipboard.containsHtmlTable(clipboardNode)) {
                    pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    pasteTextClipboard(textData);
                }
                // clear clipboard
                clipboardNode.text('\xa0');
            });
        }

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this grid pane is visible.
         */
        function registerEventHandler(source, type, handler) {
            source.on(type, function () {
                if (layerRange) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the cell collection instance used by this grid pane.
         *
         * @returns {CellCollection}
         *  The cell collection used by this grid pane.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return getFocusTargetNode().is(window.document.activeElement);
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            getFocusTargetNode().focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  @param {Boolean} colSettings.visible
         *      Whether the column header pane is visible.
         *  @param {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  @param {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels.
         *  @param {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  @param {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  @param {Boolean} rowSettings.visible
         *      Whether the row header pane is visible.
         *  @param {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  @param {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels.
         *  @param {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  @param {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if (colSettings.visible && rowSettings.visible) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                layerRootNode.enableTracking(Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode
                }));

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    bottom: (!colSettings.frozen && !rowSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (!rowSettings.frozen && !colSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_WIDTH) : 0
                });

                // other layout options
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // initialize the layer nodes
                layerRange = SheetUtils.makeRangeFromIntervals(colHeaderPane.getInterval(), rowHeaderPane.getInterval());
                layerRectangle = sheetModel.getRangeRectangle(layerRange);
                updateScrollAreaSize();
                updateScrollPosition();

                // update CSS classes at root node for focus display
                updateFocusDisplay();

                // create all drawing frames if not done yet (e.g. enabled split in current sheet)
                if (!drawingLayerNode[0].hasChildNodes()) {
                    createDrawingFrames();
                }

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                layerRootNode.disableTracking();
            }

            return this;
        };

        /**
         * Returns whether this grid pane has a valid layer cell range and
         * layer rectangle.
         *
         * @returns {Boolean}
         *  Whether this grid pane has a valid layer cell range and layer
         *  rectangle, and thus is ready to render cells etc.
         */
        this.hasLayerRange = function () {
            return _.isObject(layerRange);
        };

        /**
         * Returns the offset and size of the entire sheet rectangle covered by
         * this grid pane, including the ranges around the visible area.
         *
         * @returns {Object}
         *  The offset and size of the sheet area covered by this grid pane, in
         *  pixels.
         */
        this.getLayerRectangle = function () {
            return _.copy(layerRectangle, true);
        };

        /**
         * Returns the effective offset and size of the visible sheet area in
         * this grid pane.
         *
         * @returns {Object}
         *  The offset and size of the sheet area visible in this grid pane, in
         *  pixels.
         */
        this.getVisibleRectangle = function () {
            return {
                left: colHeaderPane.getVisiblePosition().offset,
                top: rowHeaderPane.getVisiblePosition().offset,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        };

        /**
         * Returns the logical address of the top-left cell visible in this
         * grid pane. The cell is considered visible, it it's column and row
         * are at least half visible.
         *
         * @returns {Number[]}
         *  The logical address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return [colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex()];
        };

        /**
         * Returns the DOM cell node at the specified logical address.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {jQuery}
         *  The DOM node of the cell, as jQuery object (empty, if the cell node
         *  does not exist).
         */
        this.getCellNode = function (address) {
            return cellLayerNode.find('>[data-address="' + address[0] + ',' + address[1] + '"]');
        };

        /**
         * Returns the DOM drawing frame node that represents the passed unique
         * drawing model identifier.
         *
         * @param {String} uid
         *  The unique identifier of the drawing model.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object (empty, if the
         *  drawing frame does not exist).
         */
        this.getDrawingFrame = function (uid) {
            return drawingLayerNode.find('>[data-model-uid="' + uid + '"]');
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The logical cell position to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address) {
            var position = sheetModel.getCellRectangle(address, { expandMerged: true });
            colHeaderPane.scrollToPosition(position.left, position.width);
            rowHeaderPane.scrollToPosition(position.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified DOM node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM node to be made visible. If this object is a jQuery
         *  collection, uses the first node it contains. Must be a descendant
         *  node of the layer root node.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToNode = function (node) {

            var // the position of the node in the layer root node
                position = Utils.getChildNodePositionInNode(layerRootNode, node);

            colHeaderPane.scrollToPosition(position.left + layerRectangle.left, position.width);
            rowHeaderPane.scrollToPosition(position.top + layerRectangle.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Number[]} position
         *  The logical document position of the drawing frame.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {
            // TODO: scroll to embedded drawing frames
            return this.scrollToNode(drawingLayerNode[0].childNodes[position[0]]);
        };

        // in-place cell edit mode --------------------------------------------

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return cellEditSettings.active;
        };

        /**
         * Starts the in-place edit mode for the active cell.
         *
         * @param {Object} [options]
         * A map with options controlling the behavior of the edit mode. The
         * following options are supported:
         *  @param {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  @param {Boolean} [options.quick=false]
         *      If set to tur, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *  @param {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.enterCellEditMode = function (options) {

            var // the contents (display string, result, formula) and formatting of the active cell
                cellData = null,
                // the initial text for the text area
                initialText = Utils.getStringOption(options, 'text'),
                // whether the active sheet is protected
                sheetProtected = sheetModel.isProtected();

            // first, check document edit mode
            if (!view.requireEditMode()) { return this; }

            // cancel current tracking, hide cell pop-up menu
            $.cancelTracking();
            cellListMenu.hide();

            // check whether in-place edit mode is already active
            if (cellEditSettings.active) { return this; }

            // store location of edited cell (sheet may change during range selection mode for formulas)
            cellEditSettings.address = view.getActiveCell();
            cellEditSettings.sheet = view.getActiveSheet();

            // initialize CSS formatting of the text area (this initializes the property cellEditSettings.cellData)
            updateTextAreaStyle();
            cellData = cellEditSettings.cellData;

            // scroll to the active cell (before checking edit mode)
            this.scrollToCell(cellEditSettings.address);

            // give user some info and quit if cell is locked
            if (sheetProtected && !cellData.attributes.cell.unlocked) {
                view.yell('info', gt('Protected cells cannot be modified.'));
                return this;
            }

            // initial state of quick edit mode and other settings
            cellEditSettings.active = true;
            setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));
            cellEditSettings.changed = false;
            cellEditSettings.restoreValue = null;

            // the token array stores the parsed formula for range highlighting
            cellEditSettings.tokenArray = new TokenArray(app, sheetModel);
            view.startRangeHighlighting(cellEditSettings.tokenArray, {
                draggable: true,
                refAddress: cellEditSettings.address,
                targetAddress: cellEditSettings.address,
                resolveNames: true
            });

            // update formula string in text area while dragging the highlighted ranges
            cellEditSettings.tokenArray.on('change:token', function (event, token, index) {
                var tokenPos = cellEditSettings.tokenArray.getTokenPosition(index);
                setTextAreaContents('=' + cellEditSettings.tokenArray.getFormula(), tokenPos.end + 1);
                // immediately update the highlighted cell references
                updateTextAreaHighlighting();
                resetActiveSelection();
                cellEditSettings.changed = true;
            });

            // insert and activate the text area
            rootNode.addClass('cell-edit').append(textAreaUnderlay, textArea);
            textArea.focus();
            clipboardNode.detach();

            // IE9 does not trigger 'input' events when deleting characters or
            // pasting text, use a timer interval as a workaround
            if (Utils.IE9) {
                cellEditSettings.updateTimer = app.repeatDelayed(function () {
                    if (textArea.val() !== cellEditSettings.lastValue) {
                        textAreaInputHandler();
                    }
                }, { delay: 250 });
            }

            // calculate initial value for the text area (formula wins over display string)
            if (_.isString(cellData.formula)) {
                // hide formula if the cell contains the hidden attribute
                cellEditSettings.originalText = (sheetProtected && cellData.attributes.cell.hidden) ? '' : cellData.formula;
            } else if (CellCollection.isText(cellData)) {
                // text cells: use plain text instead of the formatted display string
                cellEditSettings.originalText = cellData.result;
            } else if (CellCollection.isBoolean(cellData)) {
                // use plain Boolean literal instead of formatted display string for editing
                cellEditSettings.originalText = app.getBooleanLiteral(cellData.result);
            } else if (CellCollection.isError(cellData)) {
                // use plain error code literal instead of formatted display string for editing
                cellEditSettings.originalText = app.getErrorCodeLiteral(cellData.result);
            } else if (CellCollection.isNumber(cellData)) {
                // use appropriate number representation according to number format category
                switch (cellData.format.cat) {
                case 'standard':
                case 'number':
                case 'scientific':
                case 'currency':
                case 'fraction':
                case 'custom':
                    // use decimal representation with an increased number of characters before switching to scientific
                    cellEditSettings.originalText = numberFormatter.formatAsStandard(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_EDIT).text;
                    break;
                case 'percent':
                    // multiply result by 100, add percent sign
                    cellEditSettings.originalText = numberFormatter.formatAsStandard(cellData.result * 100, SheetUtils.MAX_LENGTH_STANDARD_EDIT).text + '%';
                    break;
                default:
                    // TODO: use matching simple format (date, etc.) for editing, instead of formatted display string
                    cellEditSettings.originalText = cellData.display;
                }
            } else {
                // empty cells
                cellEditSettings.originalText = '';
            }
            textAreaUnderlay.empty();

            // set initial contents, position, size, and selection of the text area
            setTextAreaContents(_.isString(initialText) ? initialText : cellEditSettings.originalText, Utils.getIntegerOption(options, 'pos'));

            // selection handler that changes cell ranges in formaulas in cell edit mode
            view.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

            // notify listeners
            this.trigger('celledit:enter');

            // set last value and invoke input handler to handle custom initial text
            cellEditSettings.lastValue = cellEditSettings.originalText;
            if (cellEditSettings.originalText !== textArea.val()) {
                textAreaInputHandler();
            }

            return this;
        };

        /**
         * Leaves the in-place edit mode for the active cell.
         *
         * @param {String} [commitMode='discard']
         *  Specifies how to leave the cell edit mode. The following values are
         *  supported:
         *  - 'cell': The current value will be committed to the active cell of
         *      the selection.
         *  - 'fill': The current value will be filled to all cells of the
         *      selection.
         *  - 'array': The current value will be inserted as array formula.
         *  - 'discard': The cell in the document will not be changed.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.leaveCellEditMode = function (commitMode) {

            var // the new cell text
                value = textArea.val(),
                // additional cell attributes
                attributes = (value.indexOf('\n') >= 0) ? { cell: { wrapText: true } } : undefined;

            // nothing to do, if in-place edit mode is not active
            if (!cellEditSettings.active) { return this; }

            // deinitialize range highlighting
            cellEditSettings.active = false;
            view.endRangeHighlighting();
            view.unregisterCellSelectionHandlers();
            resetActiveSelection();
            cellEditSettings.tokenArray.destroy();
            cellEditSettings.tokenArray = null;

            // IE9 does not trigger 'input' events when deleting characters or pasting text,
            // remove the timer interval that has been started as a workaround
            if (Utils.IE9) {
                cellEditSettings.updateTimer.abort();
                cellEditSettings.updateTimer = null;
            }

            //watch for a hyperlink in value
            attributes = hyperlink.checkForHyperlink(value, attributes);

            // send current value to document model
            switch (model.getEditMode() ? commitMode : 'discard') {
            case 'cell':
                // single cell mode: do not send operations if cell text did not change
                // (but always send the 'wrapText' attribute for multi-line cells)
                if ((cellEditSettings.originalText !== value) || _.isObject(attributes)) {
                    view.setCellContents(value, attributes, { parse: true });
                    view.setOptimalRowHeightWrap(false);
                }
                break;
            case 'fill':
                // fill mode: always fill all cells in the selection
                view.fillCellRanges(value, attributes, { parse: true });
                break;
            case 'array':
                // TODO
                break;
            default:
                commitMode = 'discard';
            }

            // back to cell selection mode
            cellEditSettings.quick = null;
            rootNode.removeClass('cell-edit');
            hideTextAreaPopups();
            rootNode.append(clipboardNode);
            grabClipboardFocus();
            textAreaUnderlay.detach();
            textArea.detach();

            // notify listeners
            this.trigger('celledit:leave');
            return this;
        };

        /**
         * Returns whether the cell value during in-place cell edit mode has
         * been changed since the edit mode has been started, or since the
         * original value has been restored the last time.
         */
        this.cellEditValueChanged = function () {
            return cellEditSettings.active && cellEditSettings.changed;
        };

        /**
         * Returns whether the cell value during in-place cell edit mode has
         * been restored and has not been changed since then.
         */
        this.cellEditValueRestored = function () {
            return cellEditSettings.active && !cellEditSettings.changed && _.isString(cellEditSettings.restoreValue);
        };

        /**
         * Restores the original cell value, while in-place cell edit mode is
         * active.
         */
        this.restoreOriginalCellEditValue = function () {
            if (this.cellEditValueChanged()) {
                cellEditSettings.restoreValue = textArea.val();
                setTextAreaContents(cellEditSettings.originalText);
                cellEditSettings.changed = false;
            }
            return this;
        };

        /**
         * Restores the last entered cell value before the original cell value
         * has been restored, while in-place cell edit mode is active.
         */
        this.restoreChangedCellEditValue = function () {
            if (this.cellEditValueRestored()) {
                setTextAreaContents(cellEditSettings.restoreValue);
                cellEditSettings.restoreValue = null;
                cellEditSettings.changed = true;
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // initialize class members
        app.on('docs:init', function () {

            // the spreadsheet model and view
            model = app.getModel();
            view = app.getView();

            // the style sheet container of the document
            documentStyles = model.getDocumentStyles();
            fontCollection = model.getFontCollection();
            numberFormatter = model.getNumberFormatter();

            // references to column and row header pane for this grid pane
            colHeaderPane = view.getColHeaderPane(panePos);
            rowHeaderPane = view.getRowHeaderPane(panePos);

            // drop-down menu for auto-completion and list validation of active cell
            cellListMenu = new ListMenu(rootNode);

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = view.createClipboardNode().attr('tabindex', 0).text('\xa0').appendTo(rootNode);
            clipboardFocusMethod = clipboardNode[0].focus;

            // on touch devices, disable the content-editable mode of the clipboard node,
            // to prevent showing the virtual keyboard all the time
            if (Modernizr.touch) { clipboardNode.removeAttr('contenteditable'); }

            // the text area control for in-place cell editing (will be inserted into the root node)
            textArea = Utils.createTextArea().attr({ tabindex: 0, 'data-focus-role': 'textarea' });
            textAreaUnderlay = $('<div>').addClass('textarea-underlay');

            // tooltips and pop-up menus for the text area
            textAreaToolTip = new ToolTip(textArea, { classes: 'textarea-tooltip' });
            textAreaListMenu = new ListMenu(textArea);
            textAreaListItemToolTip = new ToolTip(textArea, { position: 'right left', align: 'center', borderNode: textAreaListMenu.getNode() });

            // listen to layout update events from the view
            view.on({
                'change:activesheet': changeActiveSheetHandler,
                // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
                'celledit:enter': function (event, editPos) { rootNode.toggleClass('f6-target', editPos === panePos); },
                'celledit:leave': function () { rootNode.addClass('f6-target'); }
            });

            // update cell collection after interval changes of the column header pane
            colHeaderPane.on({
                'change:interval': function (event, interval, operation) { cellCollection.setColInterval(interval, operation); },
                'hide:interval': function () { cellCollection.clearColInterval(); }
            });

            // update cell collection after interval changes of the row header pane
            rowHeaderPane.on({
                'change:interval': function (event, interval, operation) { cellCollection.setRowInterval(interval, operation); },
                'hide:interval': function () { cellCollection.clearRowInterval(); }
            });

            // update cell layer node
            cellCollection.on({
                'change:boundrange': changeBoundRangeHandler,
                'clear:boundrange': clearBoundRangeHandler,
                'change:cells': changeCellsHandler
            });

            // update text area formatting (only, if this grid pane is visible)
            registerEventHandler(view, 'change:layoutdata', changeLayoutDataHandler);

            // update selection after the column, row, or cell geometry has changed
            registerEventHandler(view, 'insert:columns delete:columns change:columns insert:rows delete:rows change:rows insert:merged delete:merged', app.createDebouncedMethod($.noop, function () {
                renderCellSelection();
                renderHighlightedRanges();
            }));

            // update drawing layer node (only, if this grid pane is visible)
            registerEventHandler(view, 'insert:drawing', insertDrawingHandler);
            registerEventHandler(view, 'delete:drawing', deleteDrawingHandler);
            registerEventHandler(view, 'change:drawing', changeDrawingHandler);

            // update layers according to changed sheet view attributes (only, if this grid pane is visible)
            registerEventHandler(view, 'change:sheetviewattributes', changeSheetViewAttributesHandler);

            // listen to DOM scroll events
            scrollNode.on('scroll', scrollHandler);

            // listen to changed scroll position/size from header panes
            registerEventHandler(colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(colHeaderPane, 'change:scrollpos', updateScrollPosition);
            registerEventHandler(rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(rowHeaderPane, 'change:scrollpos', updateScrollPosition);

            // activate this pane on focus change
            rootNode.on('focusin', function () {
                // do not change active pane while in cell edit mode (range selection mode in formulas)
                if (!view.isCellEditMode()) {
                    sheetModel.setViewAttribute('activePane', panePos);
                }
            });

            // handle key events
            rootNode.on('keydown', keyDownHandler);

            // show drop-down menu for active cell
            selectionLayerNode.on('click', '.cell-dropdown-button', cellDropDownClickHandler);
            // suppress double-clicks for cell drop-down button
            selectionLayerNode.on('dblclick', '.cell-dropdown-button', false);

            // bug 31479: suppress double-clicks for drawings
            drawingLayerNode.on('dblclick', false);

            // cut, copy & paste events in clipboard node
            clipboardNode.on({
                cut: cutHandler,
                copy: copyHandler,
                paste: pasteHandler
            });

            // handle DOM focus requests at the clipboard node (select clipboard contents)
            clipboardNode[0].focus = grabClipboardFocus;

            // adjust minimum width of the cell pop-up menu
            cellListMenu.on('popup:beforelayout', cellListBeforeLayoutHandler);

            // handle click events in the cell pop-up menu
            cellListMenu.getNode().on('click', Utils.BUTTON_SELECTOR, cellListClickHandler);

            // hide pop-up menu when starting to select (including click on current cell with open menu)
            self.on('select:start', function () { cellListMenu.hide(); });

            // events for in-place edit mode
            textArea.on({
                keydown: textAreaKeyDownHandler,
                keypress: textAreaKeyPressHandler,
                input: textAreaInputHandler,
                mousedown: textAreaMouseTouchHandler,
                touchstart: textAreaMouseTouchHandler,
                mouseup: textAreaMouseTouchHandler,
                touchend: textAreaMouseTouchHandler,
                scroll: textAreaScrollHandler,
                focus: textAreaFocusHandler,
                blur: textAreaFocusHandler
            });

            // handle click events in the auto-completion pop-up menu
            textAreaListMenu.getNode().on('click', Utils.BUTTON_SELECTOR, textAreaListClickHandler);

            // hide function description tooltip when closing the auto-completion menu
            textAreaListMenu.on('popup:hide', function () { textAreaListItemToolTip.hide(); });

            // leave in-place edit mode or cancel tracking, if document goes into read-only mode
            model.on('change:editmode', editModeHandler);

            // hide pop-up menu when application is not active
            app.getWindow().on('beforehide', function () {
                cellListMenu.hide();
            });

            // plug-in for hyperlink handling
            hyperlink = new Hyperlink(app, self);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            cellListMenu.destroy();
            clipboardNode[0].focus = clipboardFocusMethod;
            textAreaToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();

            app.destroyImageNodes(drawingLayerNode.find('img'));

            model = view = null;
            sheetModel = colCollection = rowCollection = mergeCollection = validationCollection = drawingCollection = null;
            cellListMenu = clipboardNode = clipboardFocusMethod = null;
            textAreaToolTip = textAreaListMenu = textAreaListItemToolTip = null;
            pendingCells = null;
        });

    } // class GridPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: GridPane });

});
