/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/editmodel',
     'io.ox/office/editframework/model/undomanager',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/documentstyles',
     'io.ox/office/spreadsheet/model/numberformatter',
     'io.ox/office/spreadsheet/model/namecollection',
     'io.ox/office/spreadsheet/model/viewsettingsmixin',
     'io.ox/office/spreadsheet/model/sheetmodel'
    ], function (Utils, EditModel, UndoManager, SheetUtils, Operations, SpreadsheetDocumentStyles, NumberFormatter, NameCollection, ViewSettingsMixin, SheetModel) {

    'use strict';

    var // default values for global view settings
        VIEW_PROPERTIES = {};

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'insert:sheet:before': Before a new sheet will be inserted into the
     *      document. The event handler receives the zero-based index of the
     *      new sheet it will have in the collection of all sheets.
     * - 'insert:sheet:after': After a new sheet has been inserted into the
     *      document. The event handler receives the zero-based index of the
     *      new sheet in the collection of all sheets.
     * - 'delete:sheet:before': Before a sheet will be removed from the
     *      document. The event handler receives the zero-based index of the
     *      sheet in the collection of all sheets.
     * - 'delete:sheet:after': After a sheet has been removed from the
     *      document. The event handler receives the zero-based index of the
     *      sheet in the collection of all sheets before it has been removed.
     * - 'move:sheet:before': Before a sheet will be moved to a new position.
     *      The event handler receives the zero-based index of the new position
     *      in the collection of all sheets, and the zero-based index of the
     *      old position.
     * - 'move:sheet:after': After a sheet has been moved to a new position.
     *      The event handler receives the zero-based index of the new position
     *      in the collection of all sheets, and the zero-based index of the
     *      old position.
     * - 'rename:sheet': After a sheet has been renamed. The event handler
     *      receives the zero-based index of the renamed sheet in the
     *      collection of all sheets, and the new name of the sheet.
     * These events will always be triggered, also during the document import
     * process.
     *
     * @constructor
     *
     * @extends EditModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    function SpreadsheetModel(app) {

        var // self reference
            self = this,

            // all existing sheets and their names, by zero-based index
            sheetsCollection = [],

            // the undo manager of this document
            undoManager = new UndoManager(app, { remoteUndo: true }),

            // container for all style sheets of all attribute families
            documentStyles = new SpreadsheetDocumentStyles(app),

            // the number formatter for this document
            numberFormatter = new NumberFormatter(app),

            // the collection of globally defined names
            nameCollection = new NameCollection(app),

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [],

            // the maximum column/row index in a sheet
            maxCol = 0, maxRow = 0;

        // base constructors --------------------------------------------------

        EditModel.call(this, app, Operations, undoManager, documentStyles);
        ViewSettingsMixin.call(this, app, VIEW_PROPERTIES);

        // private methods ----------------------------------------------------

        /**
         * Updates the maximum available column and row index.
         */
        function updateMaxColRow(attributes) {
            maxCol = attributes.cols - 1;
            maxRow = attributes.rows - 1;
        }

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {
            sheet = _.isNumber(sheet) ? sheet : _.isString(sheet) ? self.getSheetIndex(sheet) : -1;
            return ((0 <= sheet) && (sheet < sheetsCollection.length)) ? sheetsCollection[sheet] : null;
        }

        /**
         * Returns an error code if the passed sheet name is not valid (empty,
         * or with invalid characters, or already existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {String}
         *  The empty string, if the sheet name is valid; otherwise one of the
         *  following error codes:
         *  - 'empty': The passed sheet name is empty.
         *  - 'invalid': The passed sheet name contains invalid characters.
         *  - 'used': The passed sheet name is already used in the document.
         */
        function validateSheetName(sheetName, sheet) {

            var // an existing sheet with the passed name
                sheetModelByName = self.getSheetModel(sheetName),
                // the model of the specified sheet
                sheetModelByIndex = _.isNumber(sheet) ? self.getSheetModel(sheet) : null;

            // name must not be empty
            if (sheetName.length === 0) {
                return 'empty';
            }

            // name must not start nor end with an apostrophe (embedded apostrophes
            // are allowed) and must not contain any NPCs or other invalid characters
            if (/^'|[\x00-\x1f\x80-\x9f\[\]*?:\/\\]|'$/.test(sheetName)) {
                return 'invalid';
            }

            // name must not be used already for another sheet
            if (_.isObject(sheetModelByName) && (!_.isObject(sheetModelByIndex) || (sheetModelByName !== sheetModelByIndex))) {
                return 'used';
            }

            return '';
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty, no
         * invalid characters, and not existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet name is valid and not used.
         */
        function isValidSheetName(sheetName, sheet) {
            return validateSheetName(sheetName, sheet).length === 0;
        }

        /**
         * Registers own event handlers after a new sheet has been inserted
         * into this document.
         */
        function insertSheetHandler(event, sheet) {

            var // the new sheet model
                sheetModel = self.getSheetModel(sheet);

            // forwards the specified events from the passed event source to the own listeners
            function forwardSheetEvents(eventSource, eventTypeMap) {
                eventSource.on('triggered', function (event2, type) {
                    var triggerType = Utils.getStringOption(eventTypeMap, type, type);
                    self.trigger.apply(self, [triggerType, sheetModel.getIndex()].concat(_.toArray(arguments).slice(2)));
                });
            }

            // trigger all events of the new sheet at this document model
            forwardSheetEvents(sheetModel, { 'change:attributes': 'change:sheetattributes', 'change:viewattributes': 'change:sheetviewattributes' });
            forwardSheetEvents(sheetModel.getColCollection(), { 'insert:entries': 'insert:columns', 'delete:entries': 'delete:columns', 'change:entries': 'change:columns' });
            forwardSheetEvents(sheetModel.getRowCollection(), { 'insert:entries': 'insert:rows', 'delete:entries': 'delete:rows', 'change:entries': 'change:rows' });
            forwardSheetEvents(sheetModel.getMergeCollection());
            forwardSheetEvents(sheetModel.getDrawingCollection());

            // invoke all range transformation handlers
            sheetModel.registerTransformationHandler(function (interval, insert, columns) {
                var sheetIndex = sheetModel.getIndex();
                _(transformationHandlers).each(function (handler) {
                    handler.call(self, sheetIndex, interval, insert, columns);
                });
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * collection for defined names (either the global collection, or a
         * collection of a specific sheet in this document).
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it does not
         *  contain a 'sheet' property (global name collection), or a valid
         *  'sheet' property addressing an existing sheet. Will be called in
         *  the context of this document model instance. Receives the following
         *  parameters:
         *  (1) {NameCollection} nameCollection
         *      The name collection instance addressed by the operation.
         *  (2) {Object} operation
         *      The complete operation object.
         *  (3) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerNameOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var sheet = Utils.getIntegerOption(operation, 'sheet'),
                    model = _.isNumber(sheet) ? this.getSheetModel(sheet) : this;
                return _.isObject(model) && handler.call(this, model.getNameCollection(), operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet. Will be
         *  called in the context of this document model instance. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} operation
         *      The complete operation object.
         *  (3) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerSheetOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var sheetModel = this.getSheetModel(Utils.getIntegerOption(operation, 'sheet', -1));
                return _.isObject(sheetModel) && handler.call(this, sheetModel, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * column/row interval in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {String} type
         *  Whether the operation addresses 'columns' or 'rows'.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing an existing column or row, and an
         *  optional 'end' property addressing an existing column or row that
         *  follows the start column/row. Will be called in the context of this
         *  document model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} interval
         *      The column/row interval, in the properties 'first' and 'last'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerIntervalOperationHandler = function (name, type, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var start = Utils.getIntegerOption(operation, 'start', -1),
                    end = Utils.getIntegerOption(operation, 'end', start),
                    max = (type === 'columns') ? maxCol : maxRow;
                return (0 <= start) && (start <= end) && (end <= max) &&
                    handler.call(this, sheetModel, { first: start, last: end }, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * single cell in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, and a valid
         *  'start' property addressing a cell. Will be called in the context
         *  of this document model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Number[]} address
         *      The logical cell address, from the property 'start'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerCellOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var address = Utils.getArrayOption(operation, 'start');
                return self.isValidAddress(address) && handler.call(this, sheetModel, address, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a specific
         * cell range in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing a start cell, and an optional 'end'
         *  property addressing an end cell that is not located before the
         *  start cell. Will be called in the context of this document model
         *  instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} range
         *      The logical range address, in the properties 'start' and 'end'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerRangeOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var range = { start: Utils.getArrayOption(operation, 'start') };
                range.end = Utils.getArrayOption(operation, 'end', _.clone(range.start));
                return this.isValidRange(range) && handler.call(this, sheetModel, range, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses an entire
         * list of cell ranges in a specific sheet.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, and an array
         *  property 'range' containing valid range addresses (valid 'start'
         *  property addressing a start cell, and an optional 'end' property
         *  addressing an end cell that is not located before the start cell).
         *  Will be called in the context of this document model instance.
         *  Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Array} ranges
         *      The logical range addresses, as array of objects with the
         *      properties 'start' and 'end'. Missing 'end' properties in the
         *      source operation have been added already in this array.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerRangeListOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var ranges = Utils.getArrayOption(operation, 'ranges');
                if (!_.isArray(ranges) || (ranges.length === 0) || !_(ranges).all(_.isObject)) { return false; }
                _(ranges).each(function (range) { if (!('end' in range)) { range.end = _.clone(range.start); } });
                return _(ranges).all(_.bind(this.isValidRange, this)) && handler.call(this, sheetModel, ranges, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name will be applied that addresses a drawing
         * object.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'start' array property addressing an existing sheet with
         *  its first element. Will be called in the context of this document
         *  model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Number} position
         *      The logical drawing position (the 'start' operation property
         *      without the first array element which was the sheet index).
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerDrawingOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var start = Utils.getArrayOption(operation, 'start', []),
                    sheetModel = ((start.length >= 2) && _.isNumber(start[0])) ? this.getSheetModel(start[0]) : null;
                return _.isObject(sheetModel) && handler.call(this, sheetModel, start.slice(1), operation, external);
            });
            return this;
        };

        /**
         * Registers an operation with the specified name that will invoke a
         * method of a drawing model instance addressed by the operation.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {String} methodName
         *  The name of the method that will be invoked at the drawing model
         *  instance. The method receives the entire operation object as first
         *  parameter, and the external state that has been passed to the
         *  method EditModel.applyOperations(). The invoked method must return
         *  a Boolean value indicating whether applying the operation has
         *  succeeded.
         *
         * @param {String} [type]
         *  The type of the drawing object supporting the operation. If
         *  omitted, the operation is supported for all drawing objects.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerSimpleDrawingOperation = function (name, methodName, type) {
            this.registerDrawingOperationHandler(name, function (sheetModel, position, operation, external) {
                var drawingModel = sheetModel.getDrawingCollection().findModel(position, { deep: true, type: type });
                return _.isObject(drawingModel) && _.isFunction(drawingModel[methodName]) && drawingModel[methodName](operation, external);
            });
            return this;
        };

        /**
         * Called from the application's post-process handler when document
         * import has succeeded. Allows to perform additional actions on the
         * document before the 'docs:import:after' event will be triggered, and
         * the event system of all model and view instances will be activated.
         *
         * @internal
         *  Not intended to be called from any other code than the post-process
         *  handler of the application.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  post-processing actions have been executed.
         */
        this.postProcessDocument = function () {
            // create all built-in cell style sheets not imported from the file
            documentStyles.getStyleSheets('cell').createMissingStyleSheets();
            // no deferred code used here
            return $.when();
        };

        /**
         * Called from the applications import-fail handler when document
         * import has failed. Inserts an empty sheet, if this model does not
         * contain any sheets. This helps preventing to write if-else
         * statements for empty documents at thousand places in the view code
         * (there will always be an active sheet with selection etc.).
         *
         * @internal
         *  Not intended to be called from any other code than the fail handler
         *  of the application.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.prepareInvalidDocument = function () {
            if (sheetsCollection.length === 0) {
                self.trigger('insert:sheet:before', 0);
                sheetsCollection.push({ model: new SheetModel(app, 'worksheet'), name: 'Sheet1' });
                self.trigger('insert:sheet:after', 0);
            }
            return this;
        };

        /**
         * Returns the number formatter of this document.
         *
         * @returns {NumberFormatter}
         *  The number formatter of this document.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the collection of all global defined names contained in this
         * document.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this document.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        // sheet range --------------------------------------------------------

        /**
         * Returns the column index of the rightmost cells in a sheet.
         *
         * @returns {Number}
         *  The column index of the rightmost cells in a sheet.
         */
        this.getMaxCol = function () {
            return maxCol;
        };

        /**
         * Returns the row index of the bottom cells in a sheet.
         *
         * @returns {Number}
         *  The row index of the bottom cells in a sheet.
         */
        this.getMaxRow = function () {
            return maxRow;
        };

        /**
         * Returns the logical range address of the entire area in a sheet,
         * from the top-left cell to the bottom-right cell.
         *
         * @returns {Object}
         *  The logical range address of an entire sheet.
         */
        this.getSheetRange = function () {
            return { start: [0, 0], end: [maxCol, maxRow] };
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns in the active sheet.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns.
         */
        this.isColRange = function (range) {
            return (range.start[1] === 0) && (range.end[1] === maxRow);
        };

        /**
         * Returns the logical address of the cell range covering the specified
         * column interval.
         *
         * @param {Object|Number} interval
         *  The column interval, in the zero-based column index properties
         *  'first' and 'last', or a single zero-based column index.
         *
         * @returns {Object}
         *  The logical address of the column range.
         */
        this.makeColRange = function (interval) {
            return {
                start: [_.isNumber(interval) ? interval : interval.first, 0],
                end: [_.isNumber(interval) ? interval : interval.last, maxRow]
            };
        };

        /**
         * Returns the logical addresses of the cell ranges covering the
         * specified column intervals.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals. Each
         *  interval object contains the zero-based index properties 'first'
         *  and 'last'.
         *
         * @returns {Array}
         *  The logical addresses of the column ranges.
         */
        this.makeColRanges = function (intervals) {
            return _.chain(intervals).getArray().map(_.bind(this.makeColRange, this)).value();
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * rows in the active sheet.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire rows.
         */
        this.isRowRange = function (range) {
            return (range.start[0] === 0) && (range.end[0] === maxCol);
        };

        /**
         * Returns the logical address of the cell range covering the specified
         * row interval.
         *
         * @param {Object|Number} interval
         *  The row interval, in the zero-based row index properties 'first'
         *  and 'last', or a single zero-based row index.
         *
         * @returns {Object}
         *  The logical address of the row range.
         */
        this.makeRowRange = function (interval) {
            return {
                start: [0, _.isNumber(interval) ? interval : interval.first],
                end: [maxCol, _.isNumber(interval) ? interval : interval.last]
            };
        };

        /**
         * Returns the logical addresses of the cell ranges covering the
         * specified row intervals.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals. Each interval
         *  object contains the zero-based index properties 'first' and 'last'.
         *
         * @returns {Array}
         *  The logical addresses of the row ranges.
         */
        this.makeRowRanges = function (intervals) {
            return _.chain(intervals).getArray().map(_.bind(this.makeRowRange, this)).value();
        };

        /**
         * Returns whether the passed value is a valid logical cell address.
         *
         * @param {Any} address
         *  Any value that will be checked whether it is a valid logical cell
         *  address.
         *
         * @returns {Boolean}
         *  Whether the passed value is an array with two non-negative integer
         *  elements. The first element (column index) must be less than or
         *  equal to the maximum column index of this document; the second
         *  element (row index) must be less than or equal to the maximum row
         *  index.
         */
        this.isValidAddress = function (address) {
            return _.isArray(address) && (address.length === 2) &&
                _.isFinite(address[0]) && (0 <= address[0]) && (address[0] <= maxCol) &&
                _.isFinite(address[1]) && (0 <= address[1]) && (address[1] <= maxRow);
        };

        /**
         * Returns whether the passed value is a valid logical range address.
         *
         * @param {Any} range
         *  Any value that will be checked whether it is a valid logical cell
         *  range address.
         *
         * @returns {Boolean}
         *  Whether the passed value is an object with the properties 'start'
         *  and 'end' containing valid cell addresses. The column indexes and
         *  row indexes must be ordered.
         */
        this.isValidRange = function (range) {
            return _.isObject(range) &&
                this.isValidAddress(range.start) && this.isValidAddress(range.end) &&
                (range.start[0] <= range.end[0]) && (range.start[1] <= range.end[1]);
        };

        /**
         * Returns a new range that has been moved by the specified distance.
         * This method ensures automatically that the range will not be moved
         * outside the sheet limits.
         *
         * @param {Object} range
         *  The logical address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Object}
         *  The logical address of the moved range. If the specified move
         *  distances are too large, the range will only be moved to the sheet
         *  borders.
         */
        this.getMovedRange = function (range, cols, rows) {

            // reduce move distance to keep the range address valid
            cols = Utils.minMax(cols, -range.start[0], maxCol - range.end[0]);
            rows = Utils.minMax(rows, -range.start[1], maxRow - range.end[1]);

            // return a modified clone of the passed range
            return {
                start: [range.start[0] + cols, range.start[1] + rows],
                end: [range.end[0] + cols, range.end[1] + rows]
            };
        };

        /**
         * Transforms the passed column/row interval according to the specified
         * operation. Used to get the effective interval after columns or rows
         * have been insert into or deleted from a sheet.
         *
         * @param {Object} sourceInterval
         *  The column/row interval to be transformed, with the index
         *  properties 'first' and 'last'.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false). Influences the maximum index allowed in the
         *  transformed interval.
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the interval will be expanded,
         *  if new columns/rows will be inserted exactly behind the interval.
         *
         * @returns {Object|Null}
         *  The transformed column/row interval. If the interval has been
         *  deleted completely while deleting columns/rows, or has been shifted
         *  outside the sheet completely while inserting columns/rows, null
         *  will be returned instead.
         */
        this.transformInterval = function (sourceInterval, operationInterval, insert, columns, expandEnd) {

            var // the length of the interval
                size = SheetUtils.getIntervalSize(operationInterval),
                // maximum valid column/row index
                max = columns ? this.getMaxCol() : this.getMaxRow(),
                // the resulting transformed interval
                targetInterval = _.clone(sourceInterval);

            if (insert) {
                // insert rows/columns (shift, or enlarge the range)
                if (operationInterval.first <= sourceInterval.first) { targetInterval.first += size; }
                if (operationInterval.first <= sourceInterval.last + (expandEnd ? 1 : 0)) { targetInterval.last = Math.min(sourceInterval.last + size, max); }
            } else {
                // delete rows/columns (shift, shrink, or delete the range)
                if (operationInterval.first < sourceInterval.first) { targetInterval.first = Math.max(operationInterval.first, sourceInterval.first - size); }
                if (operationInterval.first <= sourceInterval.last) { targetInterval.last = Math.max(sourceInterval.last - size, operationInterval.first - 1); }
            }

            // delete interval, if resulting start position exceeds end position (happens when deleting
            // the entire interval, or when shifting it outside the sheet while inserting)
            return (targetInterval.first <= targetInterval.last) ? targetInterval : null;
        };

        /**
         * Transforms the passed range according to the specified column or row
         * operation. Used to get the effective range after columns or rows
         * have been insert into or deleted from a sheet.
         *
         * @param {Object} sourceRange
         *  The logical address of the cell range to be transformed.
         *
         * @param {Object} operationInterval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false).
         *
         * @param {Boolean} [expandEnd=false]
         *  If set to true, the end position of the range will be expanded, if
         *  new columns will be inserted exactly right of the range, or new
         *  rows will be inserted exactly below the range.
         *
         * @returns {Object|Null}
         *  The logical address of the transformed cell range. If the range has
         *  been deleted completely while deleting columns/rows, or has been
         *  shifted outside the sheet completely while inserting columns/rows,
         *  null will be returned instead.
         */
        this.transformRange = function (sourceRange, operationInterval, insert, columns, expandEnd) {

            var // the source column/row intervals from the range
                colInterval = SheetUtils.getColInterval(sourceRange),
                rowInterval = SheetUtils.getRowInterval(sourceRange),
                // the transformed interval
                targetInterval = this.transformInterval(columns ? colInterval : rowInterval, operationInterval, insert, columns, expandEnd);

            // assign result to the correct interval
            if (columns) { colInterval = targetInterval; } else { rowInterval = targetInterval; }

            // build and return the resulting range
            return _.isObject(targetInterval) ? SheetUtils.makeRangeFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from any sheet in this
         * document.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this model instance. Receives the following parameters:
         *  (1) {Number} sheet
         *      The zero-based sheet index.
         *  (2) {Object} interval
         *      The column/row interval of the insert/delete operation, with
         *      the index properties 'first' and 'last'.
         *  (3) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (4) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SpreadsheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _(transformationHandlers).without(handler);
            return this;
        };

        // sheets -------------------------------------------------------------

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function () {
            return sheetsCollection.length;
        };

        /**
         * Returns the names of all sheets in the spreadsheet document.
         *
         * @returns {String[]}
         *  The names of all sheets in the document.
         */
        this.getSheetNames = function () {
            return _(sheetsCollection).pluck('name');
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return _.isObject(getSheetInfo(sheet));
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.name : null;
        };

        /**
         * Returns the current index of the sheet with the specified name.
         *
         * @param {String} sheetName
         *  The case-insensitive sheet name.
         *
         * @returns {Number}
         *  The current zero-based index of the sheet with the specified name,
         *  or -1 if a sheet with the passed name does not exist.
         */
        this.getSheetIndex = function (sheetName) {
            sheetName = sheetName.toLowerCase();
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.name.toLowerCase() === sheetName;
            });
        };

        /**
         * Returns the current sheet index of the passed sheet model instance.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model whose index will be returned.
         *
         * @returns {Number}
         *  The current zero-based sheet index of the passed sheet model, or -1
         *  if the passed object is not a valid sheet model.
         */
        this.getSheetIndexOfModel = function (sheetModel) {
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.model === sheetModel;
            });
        };

        /**
         * Returns the model instance for the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.model : null;
        };

        /**
         * Invokes the passed iterator function for all sheet models contained
         * in this document.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all sheet models. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The current sheet model instance.
         *  (2) {Number} sheet
         *      The zero-based sheet index.
         *  (3) {String} name
         *      The name of the sheet.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateSheetModels = function (iterator, options) {
            return Utils.iterateArray(sheetsCollection, function (sheetInfo, sheet) {
                return iterator.call(this, sheetInfo.model, sheet, sheetInfo.name);
            }, options);
        };

        /**
         * Creates a new sheet in this document.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  Attribute set containing initial formatting attributes for the new
         *  sheet.
         *
         * @returns {String}
         *  The empty string, if the sheet has been inserted successfully;
         *  otherwise one of the following error codes:
         *  - 'empty': The passed sheet name is empty.
         *  - 'invalid': The passed sheet name contains invalid characters.
         *  - 'used': The passed sheet name is already used in the document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertSheet = function (sheet, sheetName, attrs) {

            var // the error code for invalid sheet names
                result = '';

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet name can be used (prevent internal application error on failing operation)
            result = validateSheetName(sheetName);
            if (result.length > 0) { return result; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.INSERT_SHEET, sheet: sheet, sheetName: sheetName, attrs: attrs }) ? '' : 'internal';
        };

        /**
         * Removes a sheet from this document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether deleting the sheet was successful.
         */
        this.deleteSheet = function (sheet) {
            // check that the sheet can be removed (prevent internal application error on failing operation)
            return (this.getSheetCount() > 1) && this.applyOperations({ name: Operations.DELETE_SHEET, sheet: sheet });
        };

        /**
         * Moves a sheet in the spreadsheet document to a new position.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be moved.
         *
         * @param {Number} toSheet
         *  The new zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether moving the sheet was successful.
         */
        this.moveSheet = function (fromSheet, toSheet) {
            // generate and apply the operation
            return this.applyOperations({ name: Operations.MOVE_SHEET, sheet: fromSheet, to: toSheet });
        };

        /**
         * Moves a list of sheet-indices in the spreadsheet document to a positions.
         *
         * @param {Array} list
         *  list contains objects with zero-based 'from' and 'to' index
         */
        this.moveSheets = function (moveList) {
            if (moveList.length) {
                var generator = this.createOperationsGenerator();
                _.each(moveList, function (op) {
                    generator.generateOperation(Operations.MOVE_SHEET, {sheet: op.from, to: op.to});
                });
                this.applyOperations(generator.getOperations());
            }
        };

        /**
         * Creates a copy of an existing sheet in the spreadsheet document.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the existing sheet to be copied.
         *
         * @param {Number} toSheet
         *  The zero-based insertion index of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been copied successfully;
         *  otherwise one of the following error codes:
         *  - 'empty': The passed sheet name is empty.
         *  - 'invalid': The passed sheet name contains invalid characters.
         *  - 'used': The passed sheet name is already used in the document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.copySheet = function (fromSheet, toSheet, sheetName) {

            var // the error code for invalid sheet names
                result = null;

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName);
            if (result.length > 0) { return result; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.COPY_SHEET, sheet: fromSheet, to: toSheet, sheetName: sheetName }) ? '' : 'internal';
        };

        /**
         * Creates a sheet name that is not yet used in this document,
         * translated for the current GUI language.
         *
         * @returns {String}
         *  A sheet name not yet used in this document.
         */
        this.generateUnusedSheetName = function () {

            var // the new sheet name
                sheetName = '',
                // counter for the new sheet name
                nameIndex = this.getSheetCount();

            // generate a valid name
            while ((sheetName.length === 0) || this.hasSheet(sheetName)) {
                sheetName = SheetUtils.generateSheetName(nameIndex);
                nameIndex += 1;
            }

            return sheetName;
        };

        /**
         * Renames a sheet in the spreadsheet document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name of the sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {String}
         *  The empty string, if the sheet has been renamed successfully;
         *  otherwise one of the following error codes:
         *  - 'empty': The passed sheet name is empty.
         *  - 'invalid': The passed sheet name contains invalid characters.
         *  - 'used': The passed sheet name is already used in the document.
         *  - 'internal': Internal error while applying the operation.
         */
        this.setSheetName = function (sheet, sheetName) {

            var // the error code for invalid sheet names
                result = '';

            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);

            // check that the sheet can be renamed (prevent internal application error on failing operation)
            result = validateSheetName(sheetName, sheet);
            if (result.length > 0) { return result; }

            // do not generate an operation, if passed name matches old name
            if (this.getSheetName(sheet) === sheetName) { return ''; }

            // generate and apply the operation
            return this.applyOperations({ name: Operations.SET_SHEET_NAME, sheet: sheet, sheetName: sheetName }) ? '' : 'internal';
        };

        // operations registration --------------------------------------------

        this.registerOperationHandler(Operations.INSERT_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', ''),
                // the sheet type passed in the operation
                sheetType = Utils.getStringOption(operation, 'type', 'worksheet'),
                // the sheet attributes passed in the operation
                attributes = Utils.getObjectOption(operation, 'attrs'),
                // the model of the new sheet
                sheetModel = null;

            // check the sheet index and name (must not exist yet)
            if ((sheet < 0) || (sheet > this.getSheetCount()) || !isValidSheetName(sheetName)) {
                return false;
            }

            // check the sheet type
            if (!/^(worksheet|chartsheet|macrosheet|dialogsheet)$/.test(sheetType)) {
                return false;
            }

            // create and insert the new sheet
            self.trigger('insert:sheet:before', sheet);
            sheetModel = new SheetModel(app, sheetType, attributes);
            sheetsCollection.splice(sheet, 0, { model: sheetModel, name: sheetName });
            self.trigger('insert:sheet:after', sheet);
            return true;
        });

        this.registerSheetOperationHandler(Operations.DELETE_SHEET, function (sheetModel, operation) {

            // there must remain at least one sheet in the document
            if (this.getSheetCount() === 1) {
                return false;
            }

            // finally, remove the sheet from the document model
            this.trigger('delete:sheet:before', operation.sheet);
            sheetModel.destroy();
            sheetsCollection.splice(operation.sheet, 1);
            this.trigger('delete:sheet:after', operation.sheet);
            return true;
        });

        this.registerSheetOperationHandler(Operations.MOVE_SHEET, function (sheetModel, operation) {
            var // the source position of the sheet
                from = Utils.getIntegerOption(operation, 'sheet', -1),
                // the target position of the sheet
                to = Utils.getIntegerOption(operation, 'to', -1),
                sheetCount = this.getSheetCount();

            if ((to < 0) || (to > sheetCount) || (from < 0) || (from > sheetCount)) {
                return false;
            }

            var move = sheetsCollection[from];
            self.trigger('move:sheet:before', from, to);
            sheetsCollection.splice(from, 1);
            sheetsCollection.splice(to, 0, move);
            self.trigger('move:sheet:after', from, to);
            return true;

        });

        this.registerSheetOperationHandler(Operations.COPY_SHEET, function (sheetModel, operation) {

            var // the target position for the new sheet
                to = Utils.getIntegerOption(operation, 'to', -1),
                // the sheet descriptor (model and name)
                sheetInfo = getSheetInfo(operation.sheet),
                // the new sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no existing sheet must contain this name)
            if (!_.isObject(sheetInfo) || (to < 0) || (to > this.getSheetCount()) || !isValidSheetName(sheetName)) {
                return false;
            }

            // clone the sheet model, and insert the new model into the collection
            self.trigger('insert:sheet:before', to);
            // first, create the collection entry (this allows to receive the sheet name from this document while cloning the sheet)
            sheetsCollection.splice(to, 0, { model: null, name: sheetName });
            sheetsCollection[to].model = sheetInfo.model.clone(to);
            self.trigger('insert:sheet:after', to);

            return true;
        });

        this.registerSheetOperationHandler(Operations.SET_SHEET_NAME, function (sheetModel, operation) {

            var // the sheet object
                sheetInfo = getSheetInfo(operation.sheet),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no other sheet must contain this name)
            if (!_.isObject(sheetInfo) || !isValidSheetName(sheetName, operation.sheet)) {
                return false;
            }

            // no no events, if passed name matches old name
            if (sheetInfo.name !== sheetName) {
                sheetInfo.name = sheetName;
                this.trigger('rename:sheet', operation.sheet, sheetName);
            }

            return true;
        });

        this.registerSheetOperationHandler(Operations.SET_SHEET_ATTRIBUTES, function (sheetModel, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.setAttributes(attributes);
            return true;
        });

        this.registerNameOperationHandler(Operations.INSERT_NAME, function (sheetNameCollection, operation) {
            var name = Utils.getStringOption(operation, 'exprName', ''),
                formula = Utils.getStringOption(operation, 'formula', '');
            return sheetNameCollection.insertName(name, formula);
        });

        this.registerNameOperationHandler(Operations.DELETE_NAME, function (sheetNameCollection, operation) {
            var name = Utils.getStringOption(operation, 'exprName', '');
            return sheetNameCollection.deleteName(name);
        });

        this.registerIntervalOperationHandler(Operations.INSERT_COLUMNS, 'columns', function (sheetModel, interval) {
            sheetModel.getColCollection().insertEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.DELETE_COLUMNS, 'columns', function (sheetModel, interval) {
            sheetModel.getColCollection().deleteEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.SET_COLUMN_ATTRIBUTES, 'columns', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.getColCollection().setAttributes(interval, attributes, { rangeBorders: operation.rangeBorders, visibleBorders: operation.visibleBorders });
            return true;
        });

        this.registerIntervalOperationHandler(Operations.INSERT_ROWS, 'rows', function (sheetModel, interval) {
            sheetModel.getRowCollection().insertEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.DELETE_ROWS, 'rows', function (sheetModel, interval) {
            sheetModel.getRowCollection().deleteEntries(interval);
            return true;
        });

        this.registerIntervalOperationHandler(Operations.SET_ROW_ATTRIBUTES, 'rows', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            if (!_.isObject(attributes)) { return false; }
            sheetModel.getRowCollection().setAttributes(interval, attributes, { rangeBorders: operation.rangeBorders, visibleBorders: operation.visibleBorders });
            return true;
        });

        this.registerRangeOperationHandler(Operations.INSERT_CELLS, function (/*sheetModel, range*/) {
            // TODO
            return false;
        });

        this.registerRangeOperationHandler(Operations.DELETE_CELLS, function (/*sheetModel, range*/) {
            // TODO
            return false;
        });

        this.registerRangeOperationHandler(Operations.MERGE_CELLS, function (sheetModel, range, operation) {
            var type = Utils.getStringOption(operation, 'type', 'merge');
            sheetModel.getMergeCollection().mergeRange(range, type);
            return true;
        });

        this.registerRangeListOperationHandler(Operations.INSERT_VALIDATION, function (sheetModel, ranges, operation) {
            sheetModel.getValidationCollection().insertValidationRanges(ranges, operation);
            return true;
        });

        this.registerDrawingOperationHandler(Operations.INSERT_DRAWING, function (sheetModel, position, operation) {
            var type = Utils.getStringOption(operation, 'type', ''),
                attributes = Utils.getObjectOption(operation, 'attrs');
            return sheetModel.getDrawingCollection().insertModel(position, type, attributes);
        });

        this.registerDrawingOperationHandler(Operations.DELETE_DRAWING, function (sheetModel, position) {
            return sheetModel.getDrawingCollection().deleteModel(position);
        });

        this.registerDrawingOperationHandler(Operations.SET_DRAWING_ATTRIBUTES, function (sheetModel, position, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs'),
                drawingModel = sheetModel.getDrawingCollection().findModel(position, { deep: true });
            if (!_.isObject(attributes) || !_.isObject(drawingModel)) { return false; }
            drawingModel.setAttributes(attributes);
            return true;
        });

        this.registerSimpleDrawingOperation(Operations.INSERT_CHART_DATASERIES, 'insertDataSeries', 'chart');
        this.registerSimpleDrawingOperation(Operations.DELETE_CHART_DATASERIES, 'deleteDataSeries', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, 'setChartDataSeriesAttributes', 'chart');
        this.registerSimpleDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, 'setAxisAttributes', 'chart');

        // initialization -----------------------------------------------------

        // set the column/row count, listen to document attribute changes
        updateMaxColRow(documentStyles.getAttributes());
        documentStyles.on('change:attributes', function (event, attributes) {
            updateMaxColRow(attributes);
        });

        // register own event handlers after inserting a new sheet
        this.on('insert:sheet:after', insertSheetHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.chain(sheetsCollection).pluck('model').invoke('destroy');
            numberFormatter.destroy();
            nameCollection.destroy();
            sheetsCollection = undoManager = documentStyles = null;
            numberFormatter = nameCollection = transformationHandlers = null;
        });

    } // class SpreadsheetModel

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: SpreadsheetModel });

});
