/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel',
        ['io.ox/office/tk/utils',
         'io.ox/office/drawinglayer/model/drawingmodel',
         'io.ox/office/spreadsheet/utils/sheetutils',
         'io.ox/office/spreadsheet/model/operations',
         'io.ox/office/spreadsheet/model/tokenarray',
         'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
         'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
         'io.ox/office/spreadsheet/model/drawing/axismodel',
         'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
         'gettext!io.ox/office/spreadsheet'
        ], function (Utils, DrawingModel, SheetUtils, Operations, TokenArray, DrawingModelMixin, ManualColor, AxisModel, DataSeriesModel, gt) {

    'use strict';

    var TYPES = {
        'column standard' : 'column',
        'column clustered' : 'column',
        'column stacked' : 'stackedColumn',
        'column percentStacked' : 'stackedColumn100',
        'bar standard' : 'bar',
        'bar clustered' : 'bar',
        'bar stacked' : 'stackedBar',
        'bar percentStacked' : 'stackedBar100',
        'area standard' : 'area',
        'area stacked' : 'stackedArea',
        'area percentStacked' : 'stackedArea100',
        'line standard' : 'line',
        /*
        'line clustered' : 'line',
        'line stacked' : 'line',
        'line percentStacked' : 'line',
        */
        'scatter standard' : 'scatter',
        'bubble standard' : 'bubble',
        'pie standard' : 'pie',
        'donut standard' : 'doughnut'
    };

    var MARKER_TYPES = {
        'circle': 'circle',
        'dot': 'circle',
        'square': 'square',
        'triangle' : 'triangle',
        'x' : 'cross',
        'none': 'none'
    };
    var MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    function SheetChartModel(app, sheetModel, initAttributes) {

        var self = this;

        var docModel = app.getModel();

        var token = null;

        var data = {
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: {},
            axisY: {},
            creditHref: '',
            creditText: '',
            title: {
                text: ''
            },
            data: []
        };

        var axes = {
            x : new AxisModel(app, {}, data.axisX),
            y : new AxisModel(app, {}, data.axisY),
            z : new AxisModel(app, {}, null)
        };

        var colorHandler = new ManualColor(app, self, data);

        // base constructors --------------------------------------------------

        DrawingModel.call(this, app, 'chart', initAttributes, { additionalFamilies: ['chart', 'fill'] });
        DrawingModelMixin.call(this, app, sheetModel);

        var legendPos = self.getMergedAttributes().chart.legendPos;
        switch (legendPos) {
        case 'bottom':
        case 'top':
            data.legend = {
                verticalAlign: legendPos,
                horizontalAlign: 'center',
            };
            break;
        case 'left':
        case 'right':
            data.legend = {
                verticalAlign: 'center',
                horizontalAlign: legendPos
            };
            break;
        case 'topright':
            data.legend = {
                horizontalAlign: 'right',
                verticalAlign: 'top'
            };
            break;
        }

        // private methods ----------------------------------------------------

        function handleColorHandler() {
            var type = self.getChartType().split(' ')[0];

            switch (type) {
            case 'pie':
            case 'donut':
                colorHandler.setSpecialVaryColor(true);
                break;
            default:
                colorHandler.setSpecialVaryColor(false);
                break;
            }

        }

        function getInnerTokenArray(formula) {
            if (!token) {
                token = new TokenArray(app, sheetModel);
            }
            return token.parseFormula(formula);
        }

        function colRowChangeHandler(event, sheet, interval) {
            var change = null;
            switch (event.type) {
            case 'insert:columns':
                change = checkDataInsideInterval(sheet, interval, true, true);
                break;
            case 'delete:columns':
                change = checkDataInsideInterval(sheet, interval, false, true);
                break;
            case 'insert:rows':
                change = checkDataInsideInterval(sheet, interval, true, false);
                break;
            case 'delete:rows':
                change = checkDataInsideInterval(sheet, interval, false, false);
                break;
            }
            if (change) {
                self.trigger('change:drawing', 'series');
            }
        }

        function checkDataInsideInterval(sheetModelIndex, interval, insert, columns) {
            var res = null;



            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var model = dataSeries.model;

                var change = model.checkRangeInsideInterval(sheetModelIndex, interval, insert, columns);
                if (change) {
                    dataSeries.needUpdate = true;
                    res = true;
                }
            }
            return res;
        }

        function parseFormula(formula, sheetIndex) {
            var rangeInfos = getInnerTokenArray(formula).extractRanges({ filterSheet: sheetIndex, resolveNames: true });
            token.clear();
            return rangeInfos;
        }

        /**
         * hangs on the global 'docs:update' Event,
         * if any source cell is changed, the correct Series will be marked,
         * so the correct request can be send out, when the chart is visible the nex time
         */
        function updateNotificationHandler(evt) {
            var me = false;
            if (evt.changed) {
                if (evt.changed.type === 'all') {

                    for (var i = 0; i < data.data.length; i++) {
                        var dataSeries = data.data[i];
                        dataSeries.needUpdate = true;
                    }

                    me = true;
                } else {


                    var sheets = evt.changed.sheets;

                    if (sheets) {
                        //update info only addresses
                        _.each(sheets, function (sheet) {
                            var cells = sheet.cells;
                            _.each(cells, function (cell) {
                                if (overlap(cell)) {
                                    me = true;
                                }
                            });
                        });
                    }

                }
            }
            if (me === true) {
                self.refresh();
            }
        }

        function overlap(range) {
            function iterator(rangeInfo) {
                if (SheetUtils.rangesOverlap(rangeInfo.range, range)) {
                    dataSeries.needUpdate = true;
                    changed = true;
                    return true;
                }
            }

            var changed = null;
            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var formula = dataSeries.model.getFormula();

                var rangeInfos = parseFormula(formula);
                _.find(rangeInfos, iterator);
            }
            return changed;
        }


        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;

            if (!_.isEqual(mc, oc)) {

                updateChartType();
                handleColorHandler();

                changeDrawing(evt, 'series');
//                self.trigger('change:drawing', 'series');
            }
        }


        function changeDrawing(evt, subtype) {
            if (subtype === 'series') {
                var attributes = self.getMergedAttributes();
                var cjst = getCJSType();
                _(data.data).each(function (dataSeries, index) {
                    colorHandler.handleColor(dataSeries.model.getMergedAttributes(), index);
                    if (cjst === 'pie' || cjst === 'doughnut') {
                        dataSeries.startAngle = attributes.chart.rotation - 90;
                        dataSeries.showInLegend = false;
                    } else {
                        if (attributes.chart.legendPos === 'off') {
                            dataSeries.showInLegend = false;
                        } else {
                            dataSeries.showInLegend = true;
                        }
                    }
                });

            }
            refreshAxis();
        }

        function getCJSType() {
            var type = self.getChartType();
            var res = TYPES[type];

            if (!res) {
                Utils.warn('no correct charttype found', type);
                res = 'column';
            }

            if (res === 'line' && self.getMergedAttributes().chart.curved === true) {
                res = 'spline';
            }
            return res;
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.data.length; i++) {
                var series = data.data[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }

        // public methods -----------------------------------------------------

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var stacking = attributes.chart.stacking;
            if (stacking.length) {
                return type + ' ' + stacking;
            } else {
                return type;
            }
        };

        this.getRealChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');
            return type;
        };

        this.getChartSubType = function () {
            var attributes = self.getMergedAttributes();
            return attributes.chart.stacking;
        };

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        /**
         * @param operation
         */
        this.deleteDataSeries = function (operation) {
            var removed = data.data.splice(operation.series, 1);
            removed[0].model.destroy();
            this.trigger('change:drawing');
            return true;
        };

        /**
         * refresh checks if something is changed in the series,
         * if true it calls the 'updateview'-request to the server
         */
        this.refresh = app.createDebouncedMethod($.noop, function () {

            var // all ranges to be querried from server, as array of arrays of range addresses
                sourceRanges = null,
                // maps series indexes and source link types to array elements in 'sourceRanges'
                collectInfo = null;

            _(data.data).each(function (seriesData, series) {
                if (!seriesData.needUpdate) { return; }

                if (!sourceRanges) {
                    sourceRanges = [];
                    collectInfo = [];
                }

                var info = { series: series, indexes: {} };


                seriesData.model.iterateTokens(function (token, name) {
                    var entryRanges = token.resolveRangeList();
                    if (_.isArray(entryRanges)) {
                        info.indexes[name] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });
                collectInfo.push(info);
            });

            if (!_.isArray(sourceRanges)) { return; }

            if (sourceRanges.length > 0) {
                var request = app.queryRangeContents(sourceRanges);

                request.done(function (allContents) {
                    _(collectInfo).each(function (info) {

                        var seriesIndex = info.series,
                            dataSeries = data.data[seriesIndex];

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        dataSeries.needUpdate = false;

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        if (valuesSource.length === 0) { return; }

                        // all other source data may be missing
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');

                        // build series title
                        if (titleSource.length > 0) {
                            // TODO: join title originating from multiple cells
                            dataSeries.name = titleSource[0].display;
                        } else {
                            dataSeries.name =
                            //#. A data series in a chart object (a sequence of multiple data points with a title)
                            //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                            //#. This label should match the default title for data series in other spreadsheet
                            //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
                            //#, c-format
                            gt('Series %1$d', _.noI18n(seriesIndex + 1));
                        }

                        dataSeries.dataPoints = [];
                        for (var n = 0; n < valuesSource.length; n++) {

                            var dataPoint = {};
                            dataSeries.dataPoints.push(dataPoint);

                            // add value of data point (Y value)
                            var valueCell = valuesSource[n];
                            dataPoint.y = isNaN(valueCell.result) ? 0 : valueCell.result;

                            // add title of data point (category name), add X value of data point (in scatter charts)
                            var nameCell = namesSource[n];
                            dataPoint.label = (nameCell && nameCell.display && (nameCell.display.length > 0)) ? nameCell.display : String(n + 1);
                            dataPoint.x = (nameCell && !isNaN(nameCell.result)) ? nameCell.result : (n + 1);

                            // add bubble sizes (bubble charts only)
                            var sizeCell = bubblesSource[n];
                            if (sizeCell) {
                                dataPoint.z = sizeCell.result;
                            }

                            dataPoint.markerType = dataSeries.marker;
                        }
                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.data).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }, { delay: 50, maxDelay: 250 });

        /**
         * saves the Operation-Series in the model, but fetches the correct Data only if its visible
         * via trigger the Events 'change:drawing' and 'insert:series' and mark the current series that ists changed
         * @param operation
         *  @param {Object} [options.attrs]
         *   @param {Object} [options.attrs.series]
         *    the content of the series could be a formula {String}, or explicit Values {Array}.
         *    the values can be taken directy. the formula calls a request to server, the next time the chart visible
         *    @param {String|Array} [options.attrs.series.title]
         *    @param {String|Array} [options.attrs.series.names]
         *    @param {String|Array} [options.attrs.series.bubbles]
         *    @param {String|Array} [options.attrs.series.values]
         *
         * @returns {Boolean}
         */
        this.insertDataSeries = function (operation) {

            data.title = ''; //TODO: from where comes the Title?!
            var dataSeries = {
                type: getCJSType()
            };
            var removed = data.data.splice(operation.series, 0, dataSeries);
            if (removed.length) {
                removed[0].model.destroy();
            }

            var series = operation.attrs.series;

            if (getCJSType() === 'line' || getCJSType() === 'spline') {
                dataSeries.marker = MARKER_TYPES[series.marker];
                if (!dataSeries.marker) {
                    var index = operation.series % MARKER_LIST.length;
                    dataSeries.marker = MARKER_LIST[index];
                }
            }

            var model = new DataSeriesModel(app, sheetModel, operation.attrs);

            dataSeries.model = model;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';

            this.trigger('change:drawing');

            return true;
        };

        this.setAxisAttributes = function (operation) {
            var axisModel = axes[operation.axis];

            axisModel.setAttributes(operation.attrs);
            axisModel.refreshInfo();
            self.trigger('change:drawing', 'series');
        };

        this.setChartDataSeriesAttributes = function (operation) {
            var index = operation.series;
            var attrs = operation.attrs;

            var dataSeries = data.data[index];
            dataSeries.model.setA(attrs);
            dataSeries.needUpdate = true;
            self.trigger('change:drawing', 'series');
            return true;
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         *
         *  @param {Array} array where tokenarrays are pushed in
         */
        this.getTokenArrays = function (res) {
            _.each(data.data, function (dataSerie) {
                dataSerie.model.iterateTokens(function (token) {
                    res.push(token);
                });
            });
        };

        function makeMin(min, max) {
            if (Math.abs(max - min) < 0.1 || (min / max < 5 / 6)) {
                return 0;
            } else {
                return min - ((max - min) / 2);
            }
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function refreshAxis() {
            if (!data.data.length) {
                return;
            }

            var type = getCJSType();

            var givenMin = null;
            var givenMax = null;

            var axisInfo = axes.y.getMergedAttributes();
            if (axisInfo) {
                if (axisInfo.min != 'auto') {
                    givenMin = axes.y.min;
                }
                if (axisInfo.max != 'auto') {
                    givenMax = axes.y.max;
                }
            }

            axes.x.refreshInfo();
            axes.y.refreshInfo();



            var yMin = 999999;
            var yMax = -999999;

            if (type.indexOf('100') > 0) {
                var first = data.data[0];
                for (var j = 0; j < first.dataPoints.length; j++) {
                    var y = 0;
                    for (var i = 0; i < data.data.length; i++) {
                        y += data.data[i].dataPoints[j].y;
                    }
                    yMin = Math.min(yMin, y);
                    yMax = Math.max(yMax, y);
                }
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || ((1 - yMin / yMax) * -100);
                    yMax = givenMax || 100;
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || ((1 - yMax / yMin) * +100);
                    yMin = givenMin || -100;
                } else {
                    //Szenario 3
                    yMax =  givenMax || 100;
                    yMin =  givenMin || -100;
                }
            } else {
                for (var i = 0; i < data.data.length; i++) {
                    var s = data.data[i];
                    for (var j = 0; j < s.dataPoints.length; j++) {
                        var y = s.dataPoints[j].y;
                        yMin = Math.min(yMin, y);
                        yMax = Math.max(yMax, y);
                    }
                }
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || makeMin(yMin, yMax);
                    yMax = givenMax || makeMax(yMin, yMax);
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || -makeMin(-yMax, -yMin);
                    yMin = givenMin || -makeMax(-yMax, -yMin);
                } else {
                    //Szenario 3
                    var ma = yMax;
                    var mi = yMin;
                    yMax =  givenMax || makeMax(mi, ma);
                    yMin =  givenMin || -makeMax(-ma, -mi);
                }
            }

            if (yMin == yMax) {
                yMax = yMin + 1;
            }

            yMin = Math.round(yMin.toPrecision(1));
            yMax = Math.round(yMax.toPrecision(1));

            if (yMin < yMax) {
                //TODO: the random is only here, because canvasjs ignores the values otherwise
                var interval = Math.abs(yMax - yMin) / 8;
                data.axisY.minimum = yMin + Math.random() * 0.001;
                data.axisY.maximum = yMax + Math.random() * 0.001;
                data.axisY.interval = Math.max(1, Math.round(interval)) - 0.0001;
            }
        }

        function getStyleId() {
            return self.getMergedAttributes().chart.chartStyleId;
        }

        function clearSeriesColor(generator, position) {
            _(data.data).each(function (dataSeries, index) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type != 'auto') {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position[0], position[1], { series: index, attrs: { fill: { type: 'auto' } } });
                }
            });
        }

        /**
         * return true if the chart has only one series or is type of bubble or donut (because there you see only the first series)
         * @returns {Boolean}
         */
        this.isVaryColorEnabled = function () {
            return colorHandler.isVaryColorEnabled();
        };


        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            var indiv = _(data.data).find(function (dataSeries) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type != 'auto') {
                    return true;
                }
            });

            if (indiv) {
                return null;
            }

            var styleId = getStyleId() - 1;
            var colorSet = styleId % 8;
            return 'cs' + colorSet;
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            var styleId = getStyleId() - 1;
            var styleSet = (styleId / 8) | 0;

            if (styleSet < 4) {
                return 'ss0';
            } else {
                return 'ss' + styleSet;
            }
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         * @param colorSet
         */
        this.changeColorSet = function (colorSet) {
            var styleId = getStyleId() - 1;
            var csId = colorSet.replace('cs', '') | 0;
            var styleSet = (styleId / 8) | 0;

            styleId = styleSet * 8 + csId + 1;


            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: { chart: {chartStyleId: styleId } } });

            docModel.applyOperations(generator.getOperations());

        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         * @param colorSet
         */
        this.changeStyleSet = function (styleSet) {
            var styleId = getStyleId() - 1;
            var ssId = styleSet.replace('ss', '') | 0;
            var colorSet = styleId % 8;

            styleId = ssId * 8 + colorSet + 1;


            var position = self.getPosition();

            var generator = docModel.createOperationsGenerator();
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: { chart: {chartStyleId: styleId } } });
            docModel.applyOperations(generator.getOperations());

        };

        /**
         * changes the varycolors flag
         * deletes all series colors if existing, so the colors only orient on the patterns
         * @param colorSet
         */
        this.changeVaryColors = function (state) {
            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: { chart: {varyColors: state} } });

            docModel.applyOperations(generator.getOperations());
        };

        this.getSeriesCount = function () {
            return data.data.length;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         */
        this.firstInit = function () {
            updateChartType();
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributes);
        this.on('change:drawing', changeDrawing);
        this.listenTo(app, 'docs:update', updateNotificationHandler);
        this.listenTo(docModel, 'insert:columns delete:columns insert:rows delete:rows', colRowChangeHandler);

        handleColorHandler();

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            var cloneData = { series: [], axes: axes, oldSheet: sheetModel.getIndex(), newSheet: newSheet };
            _.each(data.data, function (dataSeries) {
                var nsp = {};
//                var dsa = dataSeries.model.getMergedAttributes();

                dataSeries.model.iterateTokens(function (token, name) {
                    var ta = getInnerTokenArray(token.getFormula());
                    ta.relocateSheet(cloneData.oldSheet, newSheet);
                    var nf =  ta.getFormula();
                    nsp[name] = nf;
                });

//                nsp.fill = dsa.fill;

                cloneData.series.push(nsp);
            });
            // pass private data as hidden arguments to the constructor
            return new SheetChartModel(app, newSheetModel, this.getExplicitAttributes(), cloneData);
        });

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                app.executeDelayed(function () {
                    _.each(cloneData.axes, function (axisModel, axisType) {
                        self.setAxisAttributes({ axis: axisType, attrs: axisModel.getExplicitAttributes() });
                    });
                    _.each(cloneData.series, function (op, index) {
                        self.insertDataSeries({ series: index, attrs: { series: op } });
                    }, { delay: 1 });
                });

            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (token) {
                token.destroy();
                token = null;
            }
            docModel = null;
            self = null;
        });

    } // class SheetChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: SheetChartModel });

});
