/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/documentstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/documentstyles',
     'io.ox/office/drawinglayer/model/drawingstyles',
     'io.ox/office/spreadsheet/model/cellstylesheets'
    ], function (Utils, Color, Border, DocumentStyles, DrawingStyles, CellStyleSheets) {

    'use strict';

    var // default values of spreadsheet document attributes
        SPREADSHEET_DOCUMENT_ATTRIBUTES = {

            /**
             * Number of columns in a sheet.
             */
            cols: 1024,

            /**
             * Number of rows in a sheet.
             */
            rows: 1048576
        },

        // definitions for character attributes in cells
        CHARACTER_ATTRIBUTE_DEFINITIONS = {
            fontName: { def: 'Arial' },
            fontSize: { def: 11 },
            bold: { def: false },
            italic: { def: false },
            underline: { def: false },
            strike: { def: 'none' },
            color: { def: Color.AUTO },
            url: { def: '' }
        },

        // definitions for cell attributes
        CELL_ATTRIBUTE_DEFINITIONS = {
            alignHor: { def: 'auto' },
            alignVert: { def: 'bottom' },
            wrapText: { def: false },
            numberFormat: { def: { id: 0, code: 'General' }, merge: mergeNumberFormat },
            fillColor: { def: Color.AUTO },
            borderLeft: { def: Border.NONE },
            borderRight: { def: Border.NONE },
            borderTop: { def: Border.NONE },
            borderBottom: { def: Border.NONE },
            borderInsideHor: { def: Border.NONE, scope: 'element' },
            borderInsideVert: { def: Border.NONE, scope: 'element' },
            unlocked: { def: false },
            hidden: { def: false }
        },

        // definitions for column attributes
        COLUMN_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            width: { def: 2500 },
            autoFit: { def: true },
            customWidth: { def: false }
        },

        // definitions for row attributes
        ROW_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            height: { def: 500 },
            customHeight: { def: false },
            customFormat: { def: false }
        },

        // definitions for sheet attributes
        SHEET_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            locked: { def: false }
        },

        // definitions for data validation attributes
        VALIDATION_ATTRIBUTE_DEFINITIONS = {
            type: { def: 'all' },
            compare: { def: '' },
            value1: { def: '' },
            value2: { def: '' },
            showInfo: { def: true },
            infoTitle: { def: '' },
            infoText: { def: '' },
            showError: { def: true },
            errorTitle: { def: '' },
            errorText: { def: '' },
            errorType: { def: 'error' },
            showDropDown: { def: true },
            ignoreEmpty: { def: true }
        },

        // additional definitions for drawing attributes
        DRAWING_ATTRIBUTE_DEFINITIONS = {

            /**
             * Type of the drawing anchor (specifies which anchor attributes
             * have to be used), and runtime behavior of the drawing while
             * inserting, removing, or changing columns and rows. Supported
             * values are 'twoCell', 'oneCell', 'absolute', 'twoCellAsOneCell',
             * and 'twoCellAsAbsolute'.
             */
            anchorType: { def: 'twoCell', scope: 'element' },

            /**
             * Zero-based index of the column containing the leading border of
             * the drawing object.
             */
            startCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the leading border in the column, as ratio
             * of the column width (in the range [0,1]).
             */
            startColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the top border of the
             * drawing object.
             */
            startRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the top border in the row, as ratio of the
             * row height (in the range [0,1]).
             */
            startRowOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the column containing the trailing border of
             * the drawing object.
             */
            endCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the trailing border in the column, as
             * ratio of the column width (in the range [0,1]).
             */
            endColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the bottom border of the
             * drawing object.
             */
            endRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the bottom border in the row, as ratio of
             * the row height (in the range [0,1]).
             */
            endRowOffset: { def: 0, scope: 'element' }
        };

    // global private functions ===============================================

    /**
     * Returns a number format attribute value that results from updating an
     * existing number format with another number format. Keeps the format
     * identifier of the existing number format, if the format codes of both
     * formats are equal, and the new number format does not contain an own
     * identifier.
     */
    function mergeNumberFormat(oldFormat, newFormat) {
        return (!('id' in newFormat) && (oldFormat.code === newFormat.code)) ? oldFormat : newFormat;
    }

    // class SpreadsheetDocumentStyles ========================================

    /**
     * Provides the style sheet containers for all attribute families used in a
     * spreadsheet document, and other containers collecting special formatting
     * information for the spreadsheet document.
     *
     * @constructor
     *
     * @extends DocumentStyles
     *
     * @param {SpreadsheetApplication} app
     *  The root application instance.
     */
    function SpreadsheetDocumentStyles(app) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        DocumentStyles.call(this, app, SPREADSHEET_DOCUMENT_ATTRIBUTES);

        // private methods ----------------------------------------------------

        function getDefaultCharAttributes() {
            var cellStyles = self.getStyleSheets('cell');
            return cellStyles.getStyleSheetAttributes(cellStyles.getDefaultStyleId()).character;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the default column width for all undefined columns in a
         * sheet, according to the font settings in the current default cell
         * style, and the passed base column width.
         *
         * @param {Number} baseWidth
         *  The base column width, as number of zero digits.
         *
         * @returns {Number}
         *  The default column width for all undefined columns, in 1/100 mm.
         */
        this.getDefaultColWidth = function (baseWidth) {
            var charAttributes = getDefaultCharAttributes(),
                digitWidth = this.getFontCollection().getDigitWidth(charAttributes),
                colWidth = digitWidth * baseWidth + 5;
            return Utils.convertLengthToHmm(colWidth, 'px');
        };

        /**
         * Returns the default row height for all undefined rows in a sheet,
         * according to the font settings in the current default cell style.
         *
         * @returns {Number}
         *  The default row height for all undefined rows, in 1/100 mm.
         */
        this.getDefaultRowHeight = function () {
            var charAttributes = getDefaultCharAttributes(),
                lineHeight = this.getFontCollection().getNormalLineHeight(charAttributes),
                rowHeight = Math.floor(lineHeight * 1.1);
            return Utils.convertLengthToHmm(rowHeight, 'px');
        };

        // initialization -----------------------------------------------------

        this.registerAttributeDefinitions('character', CHARACTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('cell', CELL_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('column', COLUMN_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('row', ROW_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sheet', SHEET_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('validation', VALIDATION_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('drawing', DRAWING_ATTRIBUTE_DEFINITIONS);

        this.addStyleSheetsContainer(new CellStyleSheets(app, this))
            .addStyleSheetsContainer(new DrawingStyles(app, this));

    } // class SpreadsheetDocumentStyles

    // exports ================================================================

    // derive this class from class DocumentStyles
    return DocumentStyles.extend({ constructor: SpreadsheetDocumentStyles });

});
