/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/mixedborder',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border'
    ], function (Utils, Color, Border) {

    'use strict';

    // static class MixedBorder ===============================================

    /**
     * Provides helper function to mix several border attributes, e.g. from a
     * selection containing multiple elements with borders, and to convert
     * between mixed borders and an abstract border mode used in GUI border
     * controls.
     */
    var MixedBorder = {};

    /**
     * The names of all border properties in cells, mapped by border mode
     * properties.
     *
     * @constant
     */
    MixedBorder.CELL_BORDER_ATTRIBUTES = {
        left: 'borderLeft',
        right: 'borderRight',
        top: 'borderTop',
        bottom: 'borderBottom',
        insideh: 'borderInsideHor',
        insidev: 'borderInsideVert'
    };

    /**
     * Returns whether the passed mixed border object is valid and visible
     * (line style ambiguous or different to 'none', and line width ambiguous
     * or greater than 0).
     *
     * @param {Object} mixedBorder
     *  The mixed border object, as returned for example by the method
     *  'MixedBorder.mixBorders()'.
     *
     * @returns {Boolean}
     *  Whether the mixed border object represents a visible border style.
     */
    MixedBorder.isVisibleBorder = function (mixedBorder) {
        return _.isObject(mixedBorder) &&
            // in mixed state, there are always some borders visible
            (mixedBorder.mixed || (mixedBorder.style !== 'none')) &&
            // border width may be ambiguous, but must not be 0)
            (_.isNull(mixedBorder.width) || (_.isNumber(mixedBorder.width) && (mixedBorder.width > 0)));
    };

    /**
     * Mixes the passed border objects, and returns a mixed border object
     * representing the states of the single border properties.
     *
     * @param {Array} borders
     *  An array of mixed border objects and/or a regular border attribute
     *  values.
     *
     * @returns {Object}
     *  A mixed border object representing the states of the properties of all
     *  passed borders. Contains the following properties:
     *  - {Boolean} mixed
     *      Whether the visibility state of the borders is ambiguous. If true,
     *      a few but not all borders are visible, and the other borders are
     *      invisible.
     *  - {String|Null} style
     *      The line style of all visible border lines. Will be null, if the
     *      line style is ambiguous. Will be 'none', if ALL borders are
     *      invisible.
     *  - {Number|Null} width
     *      The line width of all visible border lines, in 1/100 mm. Will be
     *      null, if the line width is ambiguous.
     *  - {Object|Null} color
     *      The line color of all visible border lines. Will be null, if the
     *      line color is ambiguous.
     */
    MixedBorder.mixBorders = function (borders) {

        var // the resulting mixed border object
            mixedBorder = { mixed: false };

        // process the elements of the array
        _(borders).each(function (border) {

            var // whether the passed border is visible
                visible = ('mixed' in border) ? MixedBorder.isVisibleBorder(border) : Border.isVisibleBorder(border);

            // update the mixed visibility state
            if (!mixedBorder.mixed && ('style' in mixedBorder) && _.isObject(border)) {
                mixedBorder.mixed = (border.mixed === true) || (MixedBorder.isVisibleBorder(mixedBorder) !== visible);
            }

            // skip the other border line properties, if the border is invisible
            if (!visible) { return; }

            // update line style (null represents mixed state)
            if (!('style' in mixedBorder)) {
                mixedBorder.style = border.style;
            } else if (mixedBorder.style !== border.style) {
                mixedBorder.style = null;
            }

            // update line width (null represents mixed state)
            if (!('width' in mixedBorder)) {
                mixedBorder.width = border.width;
            } else if (mixedBorder.width !== border.width) {
                mixedBorder.width = null;
            }

            // update line color (null represents mixed state)
            if (!('color' in mixedBorder)) {
                mixedBorder.color = border.color;
            } else if (!Color.isEqual(mixedBorder.color, border.color)) {
                mixedBorder.color = null;
            }
        });

        // add style 'none', if style property is still missing (none of the borders is visible)
        return _({ style: 'none', width: null, color: null }).extend(mixedBorder);
    };

    /**
     * Converts the passed mixed border attributes to an object describing the
     * visibility states of all borders.
     *
     * @param {Object} borderAttributes
     *  The attribute map containing mixed border objects, as returned for
     *  example by the method 'MixedBorder.mixBorders()', mapped by border
     *  attribute names (e.g. 'borderLeft', 'borderTop', etc.).
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If the visibility state of a mixed border object in the passed
     *  attribute map is missing or ambiguous (property 'mixed' is true), the
     *  map value will be null. Otherwise, the Boolean value of the property
     *  states whether the border is visible.
     */
    MixedBorder.getBorderMode = function (borderAttributes) {

        var // result mapping border attribute names to boolean values
            borderMode = {};

        _(MixedBorder.CELL_BORDER_ATTRIBUTES).each(function (attrName, propName) {
            var border = borderAttributes[attrName];
            if (_.isObject(border)) {
                borderMode[propName] = (border.mixed === true) ? null : MixedBorder.isVisibleBorder(border);
            }
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of cell
     * border attributes to a table attribute map containing all border
     * attributes.
     *
     * @param {Object} borderMode
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  specifying whether the respective borders are visible. Properties not
     *  contained in this object will not occur in the returned attribute map.
     *  If the passed object contains a single property set to true, the
     *  current visibility state of the respective border attribute will be
     *  toggled.
     *
     * @param {Object} origAttributes
     *  The original attribute map of the target object (table, cell, ...).
     *
     * @returns {Object}
     *  The table/cell attributes containing all border attributes that are
     *  mentioned in the passed object.
     */
    MixedBorder.getBorderAttributes = function (borderMode, origAttributes) {

        var // the resulting border attributes
            borderAttributes = {},
            // whether to toggle the visibility of a single border
            toggle = _.isObject(origAttributes) && (_.size(borderMode) === 1);

        _(borderMode).each(function (visible, propName) {

            var // the name of the attribute
                borderAttrName = MixedBorder.CELL_BORDER_ATTRIBUTES[propName],
                // the original border attribute value or mixed border object
                origBorder = origAttributes[borderAttrName];

            visible = toggle ? !MixedBorder.isVisibleBorder(origBorder) : visible;
            borderAttributes[borderAttrName] = visible ? _.clone(Border.SINGLE) : _.clone(Border.NONE);

            // copy existing (unambiguous) values for style, width, and color
            if (_.isObject(origBorder)) {
                if (visible && _.isString(origBorder.style) && (origBorder.style !== 'none')) {
                    borderAttributes[borderAttrName].style = origBorder.style;
                }
                if (_.isNumber(origBorder.width)) {
                    borderAttributes[borderAttrName].width = origBorder.width;
                }
                if (_.isObject(origBorder.color)) {
                    borderAttributes[borderAttrName].color = origBorder.color;
                }
            }
        });

        return borderAttributes;
    };

    // exports ================================================================

    return MixedBorder;

});
