/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */




define('io.ox/office/drawinglayer/view/imageutil',
    ['io.ox/core/config',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/tk/dialogs',
     'io.ox/core/tk/attachments',
     'gettext!io.ox/office/main'
    ], function (Config, Utils, IO, Dialogs, Attachments, gt) {

    'use strict';

    var CRC32_TABLE = [
            0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
            0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,
            0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
            0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,
            0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
            0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
            0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
            0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,
            0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
            0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,
            0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
            0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
            0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
            0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,
            0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
            0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,
            0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
            0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
            0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
            0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,
            0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
            0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,
            0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
            0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
            0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
            0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,
            0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
            0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,
            0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
            0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
            0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
            0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D
        ],

        BASE64_CHAR_TABLE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=',

        IMAGE_EXTENSIONS = {
            jpg:  'image/jpeg',
            jpe:  'image/jpeg',
            jpeg: 'image/jpeg',
            png:  'image/png',
            gif:  'image/gif',
            bmp:  'image/bmp'
        },

        dialog = null;

    // private static functions ===============================================

    function toHex32(n) {
        if (n < 0) {
            n = 0xFFFFFFFF + n + 1;
        }
        return ('00000000' + n.toString(16).toUpperCase()).substr(-8);
    }

    function crc32(n, iCRC) {
        return (iCRC >>> 8) ^ CRC32_TABLE[(iCRC & 0xFF) ^ n];
    }

    /**
     * calculates crc32 value of base64 encoded data (crc32 is then calculated from the decoded data)
     *
     * @param {String} data
     *  The String must contain base64 encoded data. The data length is determined by data.length - offset;
     *  It is not necessary to include '=' padding bytes at the end, the String size is sufficient.
     *
     * @param {Number} offset
     *  Specifies offset to the beginning of base64 encoded data.
     *
     * @returns {Number} crc32
     */
    function getCRC32fromBase64(data, offset) {

        var dataLength = data.length - offset;

        // data length must be at least two bytes to encode one character (if no pad bytes are used)
        if (dataLength < 2)
            return;

        // taking care of padbytes
        if (data.charAt(offset + dataLength - 1) === '=')
            dataLength--;
        if (data.charAt(offset + dataLength - 1) === '=')
            dataLength--;

        var groups = dataLength >> 2;
        var hangover = dataLength - (groups << 2);

        var bits, n1, n2, n3, n4, iCRC = 0xffffffff, i = offset;

        while (groups--) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n3 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n4 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = n1 << 18 | n2 << 12 | n3 << 6 | n4;
            iCRC = crc32(bits >> 16 & 0xff, iCRC);
            iCRC = crc32(bits >> 8 & 0xff, iCRC);
            iCRC = crc32(bits & 0xff, iCRC);
        }
        // taking care of pad bytes
        if (hangover === 3) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n3 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = n1 << 12 | n2 << 6 | n3;
            iCRC = crc32(bits >> 10 & 0xff, iCRC);
            iCRC = crc32(bits >> 2 & 0xff, iCRC);
        } else if (hangover === 2) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            n2 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            bits = n1 << 6 | n2;
            iCRC = crc32(bits >> 4 & 0xff, iCRC);

        // shouldn't be 1, we are padding the least significant two bits with zero
        } else if (hangover === 1) {
            n1 = BASE64_CHAR_TABLE.indexOf(data.charAt(i++));
            iCRC = crc32(bits << 2 & 0xff, iCRC);
        }
        return iCRC ^ 0xFFFFFFFF;
    }

    function insertFile(app, file) {

        return IO.readClientFileAsDataUrl(file).then(function (dataUrl) {

            var offset = dataUrl.indexOf('base64,') + 7,
                crc32 = toHex32(getCRC32fromBase64(dataUrl, offset)),
                extension = file.name.substring(file.name.lastIndexOf('.') + 1);

//            console.error('sendFilterRequest', crc32, file.name, extension, dataUrl);


            return app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'addfile',
                    add_crc32: crc32,
                    add_ext: extension,
                    alt_filename: file.name,
                    add_filedata: dataUrl,
                    requestdata: 'hallo'
                }
            });
        });
    }

    /**
     * Special handling for IE 9 for inserting the image from the specified
     * file into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {Object} file
     *  The simulated file object describing the image. It contains only the
     *  name.
     *
     * @param {Object} form
     *  The form used to upload the file.
     */
    function insertFileIE9(app, file, form) {

        var def = $.Deferred(),
            name = 'formpost_' + _.now(),
            description = '</body>',
            extension = _.isString(file.name) ? file.name.substring(file.name.lastIndexOf('.') + 1) : '',
            options = {
                form: form,
                data: { description: description },
                field: 'json'
            },
            data = JSON.stringify(options.data);

        $('#tmp').append(
            $('<iframe>', { name: name, id: name, height: 1, width: 1, src: ox.base + '/blank.html' })
        );

        // callback function, that is executed after the file is completely uploaded.
        // This is a process similar to the defereds, that are not supported by form submit.
        // Instead the server writes some JavaScript code into the <iframe> to start this callback.
        window.callback_addfile = function (response) {
            window.callback_addfile = null;
            $('#' + name).remove();
            if (response && !response.error) {
                def.resolve(response);
            } else {
                def.reject(response);
            }
        };


        var action = app.getFilterModuleUrl({ action: 'addfile', add_ext: extension, alt_filename: file.name });
        console.error('getFilterModuleUrl', extension, file.name, action);

        form.append(
                $('<input type="hidden" name="' + options.field + '">').val(data)
            ).attr({
                method: 'post',
                enctype: 'multipart/form-data',
                action: action,
                target: name
            }).submit();

        // extract the data object from the response
        return def.then(IO.createDeferredFilter(function (response) {
            return Utils.getObjectOption(response, 'data');
        })).promise();
    }

    // static class Image =====================================================

    /**
     * Provides static helper methods for manipulation and calculation
     * of image nodes.
     */
    var Image = {};

    // static methods ---------------------------------------------------------

    /**
     * Checks if a file name has an image type extension.
     *
     * @param {String} file
     *
     * @return {Boolean}
     *  Returns true if there is an extension and if has been detected
     *  as a image extension, otherwise false.
     */
    Image.hasFileImageExtension = function (file) {

        var ext = (_.isString(file)) ? file.split('.').pop().toLowerCase() : '';
        return IMAGE_EXTENSIONS.hasOwnProperty(ext);
    };

    /**
     * Extracts the fileName from a hierarchical URI and checks if the extension
     * defines an supported image format.
     *
     * @param {String} uri
     *  A hierarchical uri
     *
     * @returns {Boolean}
     *  TRUE if the uri references a file which has a supported image extensions,
     *  otherwise FALSE.
     */
    Image.hasUrlImageExtension = function (uri) {
        return Image.hasFileImageExtension(Image.getFileNameFromImageUri(uri));
    };

    /**
     * Extracts the fileName from a hierarchical URI.
     *
     * @param {String} uri
     *  A hierarchical uri
     *
     * @returns {Boolean}
     *  TRUE if the uri references a file which has a supported image extensions,
     *  otherwise FALSE.
     */
    Image.getFileNameFromImageUri = function (uri) {

        // Removes the fragment part (#), the query part (?) and chooses the last
        // segment
        var fileName = uri.substring(0, (uri.indexOf('#') === -1) ? uri.length : uri.indexOf('#'));
        fileName = fileName.substring(0, (fileName.indexOf('?') === -1) ? fileName.length : fileName.indexOf('?'));
        fileName = fileName.substring(fileName.lastIndexOf('/') + 1, fileName.length);
        return fileName;
    };

    Image.getFileNameWithoutExt = function (url) {
        var filename = Image.getFileNameFromImageUri(url);
        var pIndex = filename.lastIndexOf('.');
        return filename.substring(0, pIndex);
    };

    /**
     * checks if url is a 'bas64 data url' or an 'absolute / external url'
     * if not, it returns the documenturl of the cacheserver
     *
     *  @param {BaseApplication} app
     *
     *  @param {String} url
     */
    Image.getFileUrl = function (app, url) {
        if (url.substring(0, 10) === 'data:image' || (/:\/\//.test(url))) {
            // base64 data URL, or absolute / external URL
            return url;
        } else {
            // document URL
            return app.getFilterModuleUrl({ action: 'getfile', get_filename: url });
        }
    };

    /**
     * Returns the MIME type for the given image uri according to the image file name extension.
     *
     * @param {String} uri
     *  A hierarchical uri
     *
     * @returns {String}
     *  The MIME type or an empty String.
     */
    Image.getMimeTypeFromImageUri = function (uri) {
        var fileName = Image.getFileNameFromImageUri(uri),
            ext = fileName ? fileName.split('.').pop().toLowerCase() : '';
        return IMAGE_EXTENSIONS.hasOwnProperty(ext) ? IMAGE_EXTENSIONS[ext] : '';
    };

    /**
     * Post processes the attributes of insert drawing operations.
     * Based on the drawing attributes and the editor instance the operation is applied to:
     *      - an image URL to an external location is always used.
     *      - the image media URL is used if the operation is applied within the same editor instance.
     *      - the base64 image data is used if the operation is applied in another editor instance.
     * Finally the attributes will contain either the imageUrl or the imageData attribute.
     *
     * @param drawingAttrs
     *  The drawing attrs of the operation
     *
     * @param fileId
     *  The id of the file where the operation is applied to
     */
    Image.postProcessOperationAttributes = function (drawingAttrs, fileId) {

        if (drawingAttrs.fileId === fileId &&
            drawingAttrs.sessionId === ox.session &&
            _.isString(drawingAttrs.imageUrl)) {

            // we are targeting the same editor instance
            // so we use the image URL and remove the image base64 data.
            delete drawingAttrs.imageData;

        } else if (_.isString(drawingAttrs.imageUrl) && /:\/\//.test(drawingAttrs.imageUrl)) {

            // we aren't targeting the same editor instance but have an external image URL
            // that we can use and remove the image base64 data.
            delete drawingAttrs.imageData;

        } else if (_.isString(drawingAttrs.imageData) && drawingAttrs.imageData.substring(0, 10) === 'data:image') {

            // we aren't targeting the same editor instance and have base64 data
            // so we remove the image URL.
            delete drawingAttrs.imageUrl;

        } else {

            // remove invalid image data attribute.
            delete drawingAttrs.imageData;
        }

        delete drawingAttrs.fileId;
        delete drawingAttrs.sessionId;
    };

    /**
     * Inserts the image from the specified file into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {File} file
     *  The file object describing the image file to be inserted.
     *
     * @param {jQuery} form
     *  The form node (required for IE 9 file upload).
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the image
     *  has been uploaded to the server; or rejected, in case of an error.
     */
    Image.insertFile = function (app, file, form) {
        // the callback methods for insertFile/insertFileIE9
        return (form ? insertFileIE9(app, file, form) : insertFile(app, file))
        .then(function (data) {
            return {
                url: Utils.getStringOption(data, 'added_filename', ''),
                name : Image.getFileNameWithoutExt(file.name)
            };
        })
        .promise();
    };

    /**
     * Shows an insert image dialog
     *
     * @param {TextSpplication} app
     *  The current application.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the dialog
     *  has been closed with the default action; or rejected, if the dialog has
     *  been canceled.
     */
    Image.insertImageDialog = function (app) {

        var asyncMode = !Utils.IE9;  // do not use asynchronous mode for IE9 file upload

        return Image.showInsertImageDialog(app, {title: gt('Insert Image'), async: asyncMode, enter: Dialogs.defaultKeyEnterHandler})
        .then(function (file, isFileUpload, form) {
            if (isFileUpload) {
                return Image.insertFile(app, file, form);
            } else {
                return Image.insertURL(app, file.trim());
            }
        })
        .done(function () {
            if (dialog) {
                // we need to close the dialog now
                if (asyncMode) {
                    dialog.close();
                }
                dialog = null;
            }
        })
        .fail(function (res) {

            if (res !== 'cancel') {
                // cancel already closed the dialog
                if (asyncMode) {
                    dialog.close();
                }
            }
            dialog = null;
        })
        .promise();
    };

    /**
     * Inserts the image specified by a URL into a document.
     *
     * @param {ox.ui.App} app
     *  The application object representing the edited document.
     *
     * @param {String} url
     *  The full URL of the image to be inserted.
     */
    Image.insertURL = function (app, url) {
        return Image.checkURL(url).then(function () {
            return {
                url: url,
                name : Image.getFileNameWithoutExt(url)
            };
        });
    };

    /**
     * Checks if the URL is valid
     *
     * @param {String} url
     *  The full URL of the image to be checked.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved if the URL is valid
     *  and otherwise will be rejected.
     */
    Image.checkURL = function (url) {

        var def = $.Deferred(),
            urlRegEx = /^(?:([A-Za-z]+):)?(\/{0,3})([0-9.\-A-Za-z]+)(?::(\d+))?(?:\/([^?#]*))?(?:\?([^#]*))?(?:#(.*))?$/;

        if (_.isString(url) && urlRegEx.test(url)) {
            return def.resolve();
        }

        return def.reject();
    };

    /**
     * Shows an insert image dialog where the user can choose to enter a URL or
     * upload a local file.
     *
     * @param options
     *  @param {Boolean} [options.async=false]
     *      If true, the dialog is kept open on ok. And the deferred returned
     *      by the show function will also not be resolved.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button or the remove button have been activated, or rejected if the
     *  dialog has been canceled. The done handlers registered at the promise
     *  object will receive a object containing the text and URL entered by the
     *  user. The object contains null for text and the URL if remove has been
     *  clicked.
     */
    Image.showInsertImageDialog = function (app, options) {

        var inputWrap = Attachments.fileUploadWidget({displayLabel: false, multi: false}),
            ieBugFix = $('<input>', { type: 'text', value: '' }).css('display', 'none'),
            fileInput,

            urlInputId = _.uniqueId('url'),
            urlInput = $('<input>', {
                placeholder: Utils.getStringOption(options, 'placeholder', ''),
                value: Utils.getStringOption(options, 'value', '')
            }).addClass('span12 nice-input').attr({ id: urlInputId, tabindex: 0, 'data-default-enter': true, 'aria-labelledby': 'label-url-04', role: 'textbox' }).css({ width: '95%' }),

            // tabs for the dialog object
            tabs = $('<ul>').attr('role', 'tablist').addClass('nav nav-tabs'),
            tabsContent = $('<div>').addClass('tab-content').css({'margin-left': '5px', 'margin-bottom': '4px'}),

            // the file descriptor of the file currently selected
            fileUpload = null,
            isFileUpload = false,
            uploadFileName = null,
            lastTab = null,
            okButton = null,

            // the result deferred
            def = $.Deferred(),

            checkOkButton = function () {
                // enable / disable OK button
                if (app.getModel().getEditMode() && ((lastTab === '#upload' && (fileUpload || uploadFileName)) || (lastTab === '#url' && urlInput.val().length > 0))) {
                    okButton.removeAttr('disabled').attr('aria-disabled', false);
                } else {
                    okButton.attr({'disabled': 'disabled', 'aria-disabled': true, 'aria-label': gt('Ok')});
                }
            },

            editModeChangeHandler = function () { checkOkButton(); };

        fileInput = inputWrap.find('input[type="file"]').attr({
            name: 'files[]',
            'aria-labelledby': 'label-image-03',
            accept: Utils.getStringOption(options, 'filter', 'image/*')
        });

        // special handling for iPad to make nice-input and nice-input:focus work
        if (_.browser.iOS && _.browser.WebKit) {
            urlInput.css('-webkit-appearance', 'none');
        }

        // create and configure the dialog
        dialog = Dialogs.createDialog(_.extend(options, { width: 430, 'data-default-enter': true }))
        .append(
            tabs.append($('<li>').append($('<a>').attr({ 'href': '#upload', 'data-toggle': 'tab', tabindex: 0, 'id': 'tab-1', 'role': 'tab', 'aria-selected': 'false', 'aria-controls': 'upload' }).text(gt('Upload image'))))

                .append($('<li>').append($('<a>').attr({ 'href': '#url', 'data-toggle': 'tab', tabindex: 0, 'id': 'tab-2', 'role': 'tab', 'aria-selected': 'false', 'aria-controls': 'url' }).text(gt('Enter URL')))))
        .append(
            tabsContent
            .append($('<div>').addClass('tab-pane row-fluid').attr({'id': 'upload', 'aria-labelledby': 'tab-1', 'role': 'tabpanel' }).css({'margin-top': '10px', 'min-height': '80px'})
                    .append($('<form>').css({'margin-left': '5px'})  // required for upload in IE 9
                         .append($('<label>').addClass('control-label').attr('id', 'label-image-03').text(gt('Image:')).css({'margin-bottom': '20px', 'margin-right': '4px'}))
                        .append(inputWrap)))
            .append($('<div>').addClass('tab-pane row-fluid').attr({'id': 'url', 'aria-labelledby': 'tab-2', 'role': 'tabpanel' }).css({'margin-top': '10px', 'min-height': '80px'})
                    .append($('<div>').addClass('control-group').css({'margin-left': '5px', 'margin-bottom': '0px'})
                        .append($('<label>').addClass('control-label').attr({'for': urlInputId, 'id': 'label-url-04'}).text(gt('URL:')).css({'margin-bottom': '20px', 'margin-right': '4px'}))
                        .append(urlInput))));

        // IE has a special handling for single inputs inside a form.
        // Pressing enter directly submits the form. Adding a hidden input solves this.
        if (_.browser.IE) {
            dialog.getBody().find('form').append(ieBugFix);
        }

        // add OK and Cancel buttons
        dialog.addPrimaryButton('ok', Utils.getStringOption(options, 'okLabel', gt('OK')), null, { tabIndex: 0 });
        dialog.addButton('cancel', Utils.getStringOption(options, 'cancelLabel', gt('Cancel')), null, { tabIndex: 0 });

        okButton = dialog.getFooter().children('.btn-primary');
        var cancelButton = dialog.getFooter().children('.btn');
        cancelButton.attr('aria-label', gt('Cancel'));

        // add a progress indicator
        dialog.getHeader().find('h4').first().busy().css('background-position', '-100% -100%');

        // register tab-shown handler to know which data must be processed on 'ok'
        $(tabs).find('a[data-toggle="tab"]').on('shown', function (e) {
            lastTab = $(e.target).attr('href'); // activated tab
            if (lastTab === '#upload') {
                $('#tab-1').attr('aria-selected', 'true');
                $('#tab-2').attr('aria-selected', 'false');
                fileInput.focus();
            }
            else if (lastTab === '#url') {
                $('#tab-2').attr('aria-selected', 'true');
                $('#tab-1').attr('aria-selected', 'false');
                urlInput.focus();
            }
            checkOkButton();
        });

        // register a change handler at the input field that extracts the file descriptor
        fileInput.change(function (event) {
            fileUpload = (event.target && event.target.files && event.target.files[0]) || null;  // requires IE 10+

            // in IE 9 fileUpload will not be defined, because this is not supported
            if ((fileUpload === null) && (fileInput[0].value)) {
                uploadFileName = fileInput[0].value.substring(fileInput[0].value.lastIndexOf('\\') + 1);  // removing 'c:\fakepath\'
            }

            checkOkButton();
        });

        // register a handler at the url input field
        urlInput.on('input', checkOkButton);

        // register listeners for 'ok',
        // the deferred returned on dialog.show() gets not resolved when the dialog is in async mode
        dialog.on('ok', function () {

            var file = null,
                form = null;

            // show progress indicator
            dialog.getHeader().find('h4').first().css('background-position', 'right center');

            if (lastTab === '#upload') {
                if (! fileUpload) {
                    form = dialog.getBody().find('form').first();
                    file = { name: uploadFileName };  // simulating file object
                } else {
                    file = fileUpload;
                }
                isFileUpload = true;
            }
            else if (lastTab === '#url') {
                file = urlInput.val();
                isFileUpload = false;
            }
            if (_.isObject(file) || _.isString(file) || (_.isObject(form))) {
                def.resolve(file, isFileUpload, form);
            } else {
                def.reject();
            }

            app.getModel().off('change:editmode', editModeChangeHandler);
        });

        // register listeners for 'cancel'
        dialog.on('cancel', function () {
            def.reject('cancel');
            app.getModel().off('change:editmode', editModeChangeHandler);
        });

        // show the dialog
        dialog.show(function () {
            $(tabs).find('a').first().tab('show');
            checkOkButton();

            // disable the ok button if the dialog is already opened but we lost the edit rights.
            app.getModel().on('change:editmode', editModeChangeHandler);
        });

        return def.promise();
    };


    // exports ================================================================

    return Image;

});
