/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/attributedmodel'
    ], function (Utils, AttributedModel) {

    'use strict';

    // class DrawingModel =====================================================

    /**
     * The model of a drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {EditApplication} app
     *  The application instance containing this drawing object.
     *
     * @param {String} type
     *  The type of this drawing object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the drawing
     *  object.
     *
     * @param {Object} [initOptions]
     *  A map with additional options for the drawing object. The following
     *  options are supported:
     *  @param {String|Array} [options.additionalFamilies]
     *      Additional explicit attribute families supported by this drawing
     *      model object.
     */
    function DrawingModel(app, type, initAttributes, initOptions) {

        var // the globally unique identifier of this model
            uid = 'dr' + _.uniqueId();

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, initAttributes, {
            styleFamily: 'drawing',
            additionalFamilies: Utils.getOption(initOptions, 'additionalFamilies')
        });

        // methods ------------------------------------------------------------

        /**
         * Returns the generated globally unique identifier of this drawing
         * model object.
         *
         * @returns {String}
         *  The globally unique identifier of this drawing model.
         */
        this.getUid = function () {
            return uid;
        };

        /**
         * Returns the type of this drawing model, as specified by the
         * operation that has created the drawing model.
         *
         * @returns {String}
         *  The type of this drawing model.
         */
        this.getType = function () {
            return type;
        };

        this.getCollection = function () {
            return null;
        };

        // initialization -----------------------------------------------------

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function () {
            return new DrawingModel(app, type, this.getExplicitAttributes(), initOptions);
        });

    } // class DrawingModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: DrawingModel });

});
