/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/tablerowstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/stylesheets'
    ], function (Utils, StyleSheets) {

    'use strict';

    var // definitions for table row attributes
        DEFINITIONS = {

            /**
             * The height of the table row. If 'auto' then the row is auto sized.
             */
            height: { def: 'auto' }

        };

    // private global functions ===============================================

    /**
     * Will be called for every table row element whose attributes have been
     * changed. Repositions and reformats the table row according to the passed
     * attributes.
     *
     * @param {jQuery} row
     *  The <tr> element whose table row attributes have been changed,
     *  as jQuery object.
     *
     * @param {Object} mergedAttributes
     *  A map of attribute maps (name/value pairs), keyed by attribute family,
     *  containing the effective attribute values merged from style sheets and
     *  explicit attributes.
     */
    function updateTableRowFormatting(row, mergedAttributes) {

        var // the row attributes of the passed attribute map
            rowAttributes = mergedAttributes.row,
            rowHeight = null,
            cellHeight = null,
            maxHeight = 0,
            paddingHeight = 0;

        if (rowAttributes.height !== 'auto') {
            if (_.browser.WebKit || _.browser.IE) {

                // IE requires setting of height of div.cell inside the <td>
                if (_.browser.IE) {
                    // min-height of div.cell must not influence the height of the <td> element -> resetting before resizing
                    row.children('td').children('div.cell').css('min-height', '0px');
                }

                // Chrome requires row height at the cells, setting height at <tr> is ignored.
                rowHeight = Utils.convertHmmToLength(rowAttributes.height, 'px', 1);
                row.children('td').each(function () {
                    paddingHeight =  Utils.convertCssLength($(this).css('padding-top'), 'px', 1) +
                                     Utils.convertCssLength($(this).css('padding-bottom'), 'px', 1) +
                                     Utils.convertCssLength($(this).css('border-bottom-width'), 'px', 1);

                    cellHeight = (rowHeight - paddingHeight) + 'px';
                    if (_.browser.IE) {
                        maxHeight = Math.max(rowHeight - paddingHeight, maxHeight);
                        $(this).css('min-height', cellHeight);
                    } else {
                        $(this).css('height', cellHeight);
                    }
                });

                // IE requires setting of height of div.cell inside the <td>
                if (_.browser.IE) {
                    // div.cell needs the height of the maximum height of all cells
                    row.children('td').children('div.cell').css('min-height', maxHeight + 'px');
                }

            } else {
                // FireFox requires row height at the rows. Setting height at cells, makes
                // height of cells unchanged, even if text leaves the cell.
                row.css('height', Utils.convertHmmToCssLength(rowAttributes.height, 'px', 1));
            }
        } else {
            // required for undo, if the table cell was not set before
            if (_.browser.WebKit) {
                // Chrome requires row height at the cells.
                row.children('td').css('height', 0);
            } else if (_.browser.IE) {
                row.children('td').css('min-height', '').children('div.cell').css('min-height', '');
            } else {
                // FireFox requires row height at the rows.
                row.css('height', '');
            }
        }
    }

    // class TableRowStyles ===================================================

    /**
     * Contains the style sheets for table row formatting attributes. The CSS
     * formatting will be read from and written to <tr> elements.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function TableRowStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'row', {
            styleSheetSupport: false,
            formatHandler: updateTableRowFormatting
        });

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('row', DEFINITIONS);

    } // class TableRowStyles

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: TableRowStyles });

});
